/**
  ******************************************************************************
  * @file    py32t031x8.h
  * @brief   CMSIS Cortex-M0+ Device Peripheral Access Layer Header File.
  *          This file contains all the peripheral register's definitions, bits
  *          definitions and memory mapping for PY32T0xx devices.
  * @version v1.0.1
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/** @addtogroup CMSIS_Device
  * @{
  */
/** @addtogroup py32t031x8
  * @{
  */

#ifndef __PY32T031X8_H
#define __PY32T031X8_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** @addtogroup Configuration_section_for_CMSIS
  * @{
  */

/**
  * @brief Configuration of the Cortex-M0+ Processor and Core Peripherals
   */
#define __CM0PLUS_REV             0 /*!< Core Revision r0p0                            */
#define __MPU_PRESENT             0 /*!< PY32T0xx do not provide MPU                  */
#define __VTOR_PRESENT            1 /*!< Vector  Table  Register supported             */
#define __NVIC_PRIO_BITS          2 /*!< PY32T0xx uses 2 Bits for the Priority Levels */
#define __Vendor_SysTickConfig    0 /*!< Set to 1 if different SysTick Config is used  */

/**
  * @}
  */

/** @addtogroup Peripheral_interrupt_number_definition
  * @{
  */

/**
 * @brief PY32T0xx Interrupt Number Definition, according to the selected device
 *        in @ref Library_configuration_section
 */

/*!< Interrupt Number Definition */
typedef enum
{
  /******  Cortex-M0+ Processor Exceptions Numbers ***************************************************************/
  NonMaskableInt_IRQn         = -14,    /*!< 2 Non Maskable Interrupt                                          */
  HardFault_IRQn              = -13,    /*!< 3 Cortex-M Hard Fault Interrupt                                   */
  SVC_IRQn                    = -5,     /*!< 11 Cortex-M SV Call Interrupt                                     */
  PendSV_IRQn                 = -2,     /*!< 14 Cortex-M Pend SV Interrupt                                     */
  SysTick_IRQn                = -1,     /*!< 15 Cortex-M System Tick Interrupt                                 */
  /******  PY32T0 specific Interrupt Numbers *********************************************************************/
  WWDG_IWDG_IRQn              = 0,      /*!< Window & Independent WatchDog Interrupt                           */
  PVD_IRQn                    = 1,      /*!< PVD through EXTI Line detection Interrupt(EXTI line 16)           */
  RTC_IRQn                    = 2,      /*!< RTC interrupt through the EXTI line 19                            */
  FLASH_IRQn                  = 3,      /*!< FLASH global Interrupt                                            */
  RCC_IRQn                    = 4,      /*!< RCC global Interrupt                                              */
  EXTI0_1_IRQn                = 5,      /*!< EXTI 0 and 1 Interrupts                                           */
  EXTI2_3_IRQn                = 6,      /*!< EXTI Line 2 and 3 Interrupts                                      */
  EXTI4_15_IRQn               = 7,      /*!< EXTI Line 4 to 15 Interrupts                                      */
  LCD_CANINT1_IRQn            = 8,      /*!< LCD global Interrupt and CAN global intrrupt1                     */
  DMA_Channel1_IRQn           = 9,      /*!< DMA Channel 1 Interrupt                                           */
  DMA_Channel2_3_IRQn         = 10,     /*!< DMA Channel 2 and Channel 3 Interrupts                            */
  DMA_Channel4_IRQn           = 11,     /*!< DMA Channel 4 Interrupt                                           */
  ADC_COMP_IRQn               = 12,     /*!< ADC&COMP Interrupts                                               */
  TIM1_BRK_UP_TRG_COM_IRQn    = 13,     /*!< TIM1 Break, Update, Trigger and Commutation Interrupts            */
  TIM1_CC_IRQn                = 14,     /*!< TIM1 Capture Compare Interrupt                                    */
  TIM2_IRQn                   = 15,     /*!< TIM2 global Interrupt                                             */
  TIM3_IRQn                   = 16,     /*!< TIM3 global Interrupt                                             */
  TIM6_DAC_LPTIM1_IRQn        = 17,     /*!< TIM6, DAC,and LPTIM1 global Interrupts                            */
  TIM7_LPTIM2_IRQn            = 18,     /*!< TIM7 global Interrupt                                             */
  PWM_CANINT2_IRQn            = 19,     /*!< PWM global Interrupt and CAN global Interrupt2                    */
  TIM15_IRQn                  = 20,     /*!< TIM15 global Interrupt                                            */
  TIM16_IRQn                  = 21,     /*!< TIM16 global Interrupt                                            */
  TIM17_IRQn                  = 22,     /*!< TIM17 global Interrupt                                            */
  I2C1_IRQn                   = 23,     /*!< I2C1 global Interrupt                                             */
  I2C2_IRQn                   = 24,     /*!< I2C2 global Interrupt                                             */
  SPI1_IRQn                   = 25,     /*!< SPI1 Interrupt                                                    */
  SPI2_IRQn                   = 26,     /*!< SPI2 Interrupt                                                    */
  USART1_IRQn                 = 27,     /*!< USART1 Interrupt                                                  */
  USART2_IRQn                 = 28,     /*!< USART2 Interrupt                                                  */
  UART1_LPUART1_IRQn          = 29,     /*!< UART1 and LPUART1 Interrupt                                       */
  UART2_LPUART2_IRQn          = 30,     /*!< UART2 and LPUART2 Interrupt                                       */
  TK_IRQn                     = 31,     /*!< TK global Interrupt                                               */
} IRQn_Type;

/**
  * @}
  */

#include "core_cm0plus.h"               /* Cortex-M0+ processor and core peripherals */
#include "system_py32t0xx.h"            /* PY32T0xx System Header */
#include <stdint.h>

/** @addtogroup Peripheral_registers_structures
  * @{
  */

/**
  * @brief Analog to Digital Converter
  */
typedef struct
{
   __IO uint32_t ISR;              /*!< ADC ISR Register,                   Address offset: 0x00  */
   __IO uint32_t IER;              /*!< ADC IER Register,                   Address offset: 0x04  */
   __IO uint32_t CR;                 /*!<   Register,                         Address offset: 0x08  */
   __IO uint32_t CFGR;             /*!< ADC CFGR Register,                  Address offset: 0x0C  */
   __IO uint32_t CFGR2;            /*!< ADC CFGR2 Register,                 Address offset: 0x10  */
   __IO uint32_t SMPR1;            /*!< ADC SMPR1 Register,                 Address offset: 0x14  */
   __IO uint32_t SMPR2;            /*!< ADC SMPR2 Register,                 Address offset: 0x18  */
   __IO uint32_t SMPR3;            /*!< ADC SMPR3 Register,                 Address offset: 0x1C  */
   __IO uint32_t SMPR4;            /*!< ADC SMPR4 Register,                 Address offset: 0x20  */
   __IO uint32_t TR1;              /*!< ADC TR1 Register,                   Address offset: 0x24  */
   __IO uint32_t RESERVED1[2];      
   __IO uint32_t SQR1;             /*!< ADC SQR1 Register,                  Address offset: 0x30  */
   __IO uint32_t SQR2;             /*!< ADC SQR2 Register,                  Address offset: 0x34  */
   __IO uint32_t SQR3;             /*!< ADC SQR3 Register,                  Address offset: 0x38  */
   __IO uint32_t SQR4;             /*!< ADC SQR4 Register,                  Address offset: 0x3C  */
   __IO uint32_t DR;               /*!< ADC DR Register,                    Address offset: 0x40  */
   __IO uint32_t RESERVED2[2];      
   __IO uint32_t JSQR;             /*!< ADC JSQR Register,                  Address offset: 0x4C  */
   __IO uint32_t RESERVED3[4];      
   __IO uint32_t OFR1;             /*!< ADC OFR1 Register,                  Address offset: 0x60  */
   __IO uint32_t OFR2;             /*!< ADC OFR2 Register,                  Address offset: 0x64  */
   __IO uint32_t OFR3;             /*!< ADC OFR3 Register,                  Address offset: 0x68  */
   __IO uint32_t OFR4;             /*!< ADC OFR4 Register,                  Address offset: 0x6C  */
   __IO uint32_t RESERVED4[4];      
   __IO uint32_t JDR1;             /*!< ADC JDR1 Register,                  Address offset: 0x80  */
   __IO uint32_t JDR2;             /*!< ADC JDR2 Register,                  Address offset: 0x84  */
   __IO uint32_t JDR3;             /*!< ADC JDR3 Register,                  Address offset: 0x88  */
   __IO uint32_t JDR4;             /*!< ADC JDR4 Register,                  Address offset: 0x8C  */
   __IO uint32_t RESERVED5[9];      
   __IO uint32_t CALFACT;          /*!< ADC CALFACT Register,               Address offset: 0xB4  */
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t CCR;              /*!< ADC common configuration register,             Address offset: ADC1 base address + 0x308 */
} ADC_Common_TypeDef;

/**
* @brief TK Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< TK CR1 Register,                    Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< TK CR2 Register,                    Address offset: 0x04  */
   __IO uint32_t IDACR;            /*!< TK IDACR Register,                  Address offset: 0x08  */
   __IO uint32_t KR;               /*!< TK KR Register,                     Address offset: 0x0C  */
   __IO uint32_t CSR;              /*!< TK CSR Register,                    Address offset: 0x10  */
   __IO uint32_t UCR;              /*!< TK UCR Register,                    Address offset: 0x14  */
   __IO uint32_t STR;              /*!< TK STR Register,                    Address offset: 0x18  */
   __IO uint32_t RESERVED;         /*!< TK RESERVED Register,               Address offset: 0x1C  */
   __IO uint32_t WR;               /*!< TK WR Register,                     Address offset: 0x20  */
   __IO uint32_t SDR;              /*!< TK SDR Register,                    Address offset: 0x24  */
   __IO uint32_t PRSR;             /*!< TK PRSR Register,                   Address offset: 0x28  */
   __IO uint32_t DR;               /*!< TK DR Register,                     Address offset: 0x2C  */
   __IO uint32_t SR;               /*!< TK SR Register,                     Address offset: 0x30  */
   __IO uint32_t IER;              /*!< TK IER Register,                    Address offset: 0x34  */
   __IO uint32_t ICR;              /*!< TK ICR Register,                    Address offset: 0x38  */
   __IO uint32_t CDR;              /*!< TK CDR Register,                    Address offset: 0x3C  */
   __IO uint32_t LPDR;             /*!< TK LPDR Register,                   Address offset: 0x40  */
   __IO uint32_t LPTR;             /*!< TK LPTR Register,                   Address offset: 0x44  */
   __IO uint32_t OPR;              /*!< TK OPR Register,                    Address offset: 0x48  */
   __IO uint32_t TR;               /*!< TK TR Register,                     Address offset: 0x4C  */
   __IO uint32_t DDR;              /*!< TK DDR Register,                    Address offset: 0x50  */
} TK_TypeDef;


/**
* @brief CRC Registers
*/
typedef struct
{
   __IO uint32_t DR;               /*!< CRC DR Register,                    Address offset: 0x00  */
   __IO uint32_t IDR;              /*!< CRC IDR Register,                   Address offset: 0x04  */
   __IO uint32_t CR;               /*!< CRC CR Register,                    Address offset: 0x08  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t INIT;             /*!< CRC INIT Register,                  Address offset: 0x10  */
   __IO uint32_t POL;              /*!< CRC POL Register,                   Address offset: 0x14  */
} CRC_TypeDef;

/**
  * @brief Comparator
  */
typedef struct
{
  __IO uint32_t CSR;              /*!< COMP control and status register,           Address offset: 0x00 */
  __IO uint32_t FR;               /*!< COMP filter register,                       Address offset: 0x04 */
} COMP_TypeDef;

typedef struct
{
  __IO uint32_t CSR_ODD;          /*!< COMP control and status register located in register of comparator instance odd, used for bits common to several COMP instances, Address offset: 0x00 */
  __IO uint32_t FR_ODD;
  __IO uint32_t CCSR;
       uint32_t RESERVED1;      /*Reserved*/
  __IO uint32_t CSR_EVEN;         /*!< COMP control and status register located in register of comparator instance even, used for bits common to several COMP instances, Address offset: 0x04 */
  __IO uint32_t FR_EVEN;
} COMP_Common_TypeDef;

/**
* @brief OPA Registers
*/
typedef struct
{
   __IO uint32_t RESERVED1[12];      
   __IO uint32_t OCR;              /*!< OPA OCR Register,                   Address offset: 0x30  */
   __IO uint32_t CR;               /*!< OPA CR Register,                    Address offset: 0x34  */
   __IO uint32_t RESERVED2[2];      
   __IO uint32_t TMR;              /*!< OPA TMR Register,                   Address offset: 0x40  */
} OPA_TypeDef;


/**
  * @brief Debug MCU
  */
typedef struct
{
  __IO uint32_t IDCODE;           /*!< MCU device ID code,              Address offset: 0x00 */
  __IO uint32_t CR;               /*!< Debug configuration register,    Address offset: 0x04 */
  __IO uint32_t APBFZ1;           /*!< Debug APB freeze register 1,     Address offset: 0x08 */
  __IO uint32_t APBFZ2;           /*!< Debug APB freeze register 2,     Address offset: 0x0C */
} DBGMCU_TypeDef;

/**
  * @brief DMA Controller
  */
typedef struct
{
  __IO uint32_t ISR;              /*!< DMA interrupt status register,                 Address offset: 0x00 */
  __IO uint32_t IFCR;             /*!< DMA interrupt flag clear register,             Address offset: 0x04 */
} DMA_TypeDef;

typedef struct
{
  __IO uint32_t CCR;              /*!< DMA channel x configuration register        */
  __IO uint32_t CNDTR;            /*!< DMA channel x number of data register       */
  __IO uint32_t CPAR;             /*!< DMA channel x peripheral address register   */
  __IO uint32_t CMAR;             /*!< DMA channel x memory address register       */
  __IO uint32_t CNBTR;            /*!< DMA channel x  register       */
} DMA_Channel_TypeDef;

/**
* @brief EXTI Registers
*/
typedef struct
{
   __IO uint32_t RTSR;             /*!< EXTI RTSR Register,                 Address offset: 0x00  */
   __IO uint32_t FTSR;             /*!< EXTI FTSR Register,                 Address offset: 0x04  */
   __IO uint32_t SWIER;            /*!< EXTI SWIER Register,                Address offset: 0x08  */
   __IO uint32_t PR;               /*!< EXTI PR Register,                   Address offset: 0x0C  */
   __IO uint32_t IOSEL[2];           /*!< EXTI IOSEL Register,               Address offset: 0x10 ~0x14 */
   __IO uint32_t IMR;              /*!< EXTI IMR Register,                  Address offset: 0x18  */
   __IO uint32_t EMR;              /*!< EXTI EMR Register,                  Address offset: 0x1C  */
} EXTI_TypeDef;

/**
* @brief VREFBUF Registers
*/
typedef struct
{
   __IO uint32_t CR;              /*!< PWR CR Register,                   Address offset: 0x00  */
} VREFBUF_TypeDef;

/**
  * @brief FLASH Registers
  */
typedef struct
{
   __IO uint32_t ACR;              /*!< FLASH ACR Register,                 Address offset: 0x00  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t KEYR;             /*!< FLASH KEYR Register,                Address offset: 0x08  */
   __IO uint32_t OPTKEYR;          /*!< FLASH OPTKEYR Register,             Address offset: 0x0C  */
   __IO uint32_t SR;               /*!< FLASH SR Register,                  Address offset: 0x10  */
   __IO uint32_t CR;               /*!< FLASH CR Register,                  Address offset: 0x14  */
   __IO uint32_t ECCR;             /*!< FLASH ECCR Register,                Address offset: 0x18  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t OPTR;             /*!< FLASH OPTR Register,                Address offset: 0x20  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t WRPR;             /*!< FLASH WRPR Register,                Address offset: 0x28  */
   __IO uint32_t RESERVED4;         
   __IO uint32_t PCROPR0;          /*!< FLASH PCROPR0 Register,             Address offset: 0x30  */
   __IO uint32_t PCROPR1;          /*!< FLASH PCROPR1 Register,             Address offset: 0x34  */
   __IO uint32_t RESERVED5[22];     
   __IO uint32_t LPCR;             /*!< FLASH LPCR Register,                Address offset: 0x90  */
   __IO uint32_t RESERVED6[27];     
   __IO uint32_t TS0;              /*!< FLASH TS0 Register,                 Address offset: 0x100  */
   __IO uint32_t TS1;              /*!< FLASH TS1 Register,                 Address offset: 0x104  */
   __IO uint32_t TS2P;             /*!< FLASH TS2P Register,                Address offset: 0x108  */
   __IO uint32_t TPS3;             /*!< FLASH TPS3 Register,                Address offset: 0x10C  */
   __IO uint32_t TS3;              /*!< FLASH TS3 Register,                 Address offset: 0x110  */
   __IO uint32_t PERTPE;           /*!< FLASH PERTPE Register,              Address offset: 0x114  */
   __IO uint32_t SMERTPE;          /*!< FLASH SMERTPE Register,             Address offset: 0x118  */
   __IO uint32_t PRGTPE;           /*!< FLASH PRGTPE Register,              Address offset: 0x11C  */
   __IO uint32_t PRETPE;           /*!< FLASH PRETPE Register,              Address offset: 0x120  */
   __IO uint32_t TRD2WPL;          /*!< FLASH TRD2WPL Register,             Address offset: 0x124  */
} FLASH_TypeDef;

///**
//  * @brief Option Bytes
//  */
//typedef struct
//{
//  __IO uint8_t RDP;               /*!< FLASH option byte Read protection,                 Address offset: 0x00 */
//  __IO uint8_t USER;              /*!< FLASH option byte user options,                    Address offset: 0x01 */
//  __IO uint8_t nRDP;              /*!< Complemented FLASH option byte Read protection,    Address offset: 0x02 */
//  __IO uint8_t nUSER;             /*!< Complemented FLASH option byte user options,       Address offset: 0x03 */
//  __IO uint8_t BOR;               /*!< FLASH option byte BOR Configuration                Address offset: 0x04 */
//       uint8_t RESERVED1;              /*!< RESERVED1,                                         Address offset: 0x05 */
//  __IO uint8_t nBOR;              /*!< Complemented FLASH option byte BOR Configuration   Address offset: 0x06 */
//       uint8_t RESERVED2;              /*!< RESERVED2                                          Address offset: 0x07 */
//  __IO uint8_t SDK_STRT;          /*!< SDK area start address(stored in SDK[4:0]),        Address offset: 0x08 */
//  __IO uint8_t SDK_END;           /*!< SDK area end address(stored in SDK[12:8]),         Address offset: 0x09 */
//  __IO uint8_t nSDK_STRT;         /*!< Complemented SDK area start address,               Address offset: 0x0A */
//  __IO uint8_t nSDK_END;          /*!< Complemented SDK area end address,                 Address offset: 0x0B */
//       uint32_t RESERVED3;             /*!< RESERVED3,                                                           */
//  __IO uint16_t WRP;              /*!< FLASH option byte write protection,                Address offset: 0x0C */
//  __IO uint16_t nWRP;             /*!< Complemented FLASH option byte write protection    Address offset: 0x0E */
//} OB_TypeDef;

/**
  * @brief General Purpose I/O
  */
typedef struct
{
  __IO uint32_t MODER;            /*!< GPIO port mode register,               Address offset: 0x00      */
  __IO uint32_t OTYPER;           /*!< GPIO port output type register,        Address offset: 0x04      */
  __IO uint32_t OSPEEDR;          /*!< GPIO port output speed register,       Address offset: 0x08      */
  __IO uint32_t PUPDR;            /*!< GPIO port pull-up/pull-down register,  Address offset: 0x0C      */
  __IO uint32_t IDR;              /*!< GPIO port input data register,         Address offset: 0x10      */
  __IO uint32_t ODR;              /*!< GPIO port output data register,        Address offset: 0x14      */
  __IO uint32_t BSRR;             /*!< GPIO port bit set/reset  register,     Address offset: 0x18      */
  __IO uint32_t LCKR;             /*!< GPIO port configuration lock register, Address offset: 0x1C      */
  __IO uint32_t AFR[2];           /*!< GPIO alternate function registers,     Address offset: 0x20-0x24 */
  __IO uint32_t BRR;              /*!< GPIO Bit Reset register,               Address offset: 0x28      */
} GPIO_TypeDef;

/**
  * @brief Inter-integrated Circuit Interface
  */
typedef struct
{
  __IO uint32_t CR1;              /*!< I2C Control register 1,                Address offset: 0x00 */
  __IO uint32_t CR2;              /*!< I2C Control register 2,                Address offset: 0x04 */
  __IO uint32_t OAR1;             /*!< I2C Own address register 1,            Address offset: 0x08 */
  __IO uint32_t OAR2;             /*!< I2C Own address register 1,            Address offset: 0x08 */
  __IO uint32_t DR;               /*!< I2C Data register,                     Address offset: 0x10 */
  __IO uint32_t SR1;              /*!< I2C Status register 1,                 Address offset: 0x14 */
  __IO uint32_t SR2;              /*!< I2C Status register 2,                 Address offset: 0x18 */
  __IO uint32_t CCR;              /*!< I2C Clock control register,            Address offset: 0x1C */
  __IO uint32_t TRISE;            /*!< I2C TRISE register,                    Address offset: 0x20 */
  __IO uint32_t TIMEOUTR;              /*!< I2C  register,            Address offset: 0x1C */
  __IO uint32_t TR;            /*!< I2C  register,                    Address offset: 0x20 */
} I2C_TypeDef;

/**
* @brief FDCAN Registers
*/
typedef struct
{
   __IO uint32_t CREL;             /*!< FDCAN CREL Register,                Address offset: 0x00  */
   __IO uint32_t ENDN;             /*!< FDCAN ENDN Register,                Address offset: 0x04  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t DBTP;             /*!< FDCAN DBTP Register,                Address offset: 0x0C  */
   __IO uint32_t TEST;             /*!< FDCAN TEST Register,                Address offset: 0x010  */
   __IO uint32_t RWD;              /*!< FDCAN RWD Register,                 Address offset: 0x14  */
   __IO uint32_t CCCR;             /*!< FDCAN CCCR Register,                Address offset: 0x18  */
   __IO uint32_t NBTP;             /*!< FDCAN NBTP Register,                Address offset: 0x1C  */
   __IO uint32_t TSCC;             /*!< FDCAN TSCC Register,                Address offset: 0x20  */
   __IO uint32_t TSCV;             /*!< FDCAN TSCV Register,                Address offset: 0x24  */
   __IO uint32_t TOCC;             /*!< FDCAN TOCC Register,                Address offset: 0x28  */
   __IO uint32_t TOCV;             /*!< FDCAN TOCV Register,                Address offset: 0x2C  */
   __IO uint32_t RESERVED2[4];      
   __IO uint32_t ECR;              /*!< FDCAN ECR Register,                 Address offset: 0x40  */
   __IO uint32_t PSR;              /*!< FDCAN PSR Register,                 Address offset: 0x44  */
   __IO uint32_t TDCR;             /*!< FDCAN TDCR Register,                Address offset: 0x48  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t IR;               /*!< FDCAN IR Register,                  Address offset: 0x50  */
   __IO uint32_t IE;               /*!< FDCAN IE Register,                  Address offset: 0x54  */
   __IO uint32_t ILS;              /*!< FDCAN ILS Register,                 Address offset: 0x58  */
   __IO uint32_t ILE;              /*!< FDCAN ILE Register,                 Address offset: 0x5C  */
   __IO uint32_t RESERVED4[8];      
   __IO uint32_t GFC;              /*!< FDCAN GFC Register,                 Address offset: 0x80  */
   __IO uint32_t SIDFC;            /*!< FDCAN SIDFC Register,               Address offset: 0x84  */
   __IO uint32_t XIDFC;            /*!< FDCAN XIDFC Register,               Address offset: 0x88  */
   __IO uint32_t RESERVED5;         
   __IO uint32_t XIDAM;            /*!< FDCAN XIDAM Register,               Address offset: 0x90  */
   __IO uint32_t HPMS;             /*!< FDCAN HPMS Register,                Address offset: 0x94  */
   __IO uint32_t NDAT1;            /*!< FDCAN NDAT1 Register,               Address offset: 0x98  */
   __IO uint32_t NDAT2;            /*!< FDCAN NDAT2 Register,               Address offset: 0x9C  */
   __IO uint32_t RXF0C;            /*!< FDCAN RXF0C Register,               Address offset: 0xA0  */
   __IO uint32_t RXF0S;            /*!< FDCAN RXF0S Register,               Address offset: 0xA4  */
   __IO uint32_t RXF0A;            /*!< FDCAN RXF0A Register,               Address offset: 0xA8  */
   __IO uint32_t RXBC;             /*!< FDCAN RXBC Register,                Address offset: 0xAC  */
   __IO uint32_t RXF1C;            /*!< FDCAN RXF1C Register,               Address offset: 0xB0  */
   __IO uint32_t RXF1S;            /*!< FDCAN RXF1S Register,               Address offset: 0xB4  */
   __IO uint32_t RXF1A;            /*!< FDCAN RXF1A Register,               Address offset: 0xB8  */
   __IO uint32_t RXESC;            /*!< FDCAN RXESC Register,               Address offset: 0xBC  */
   __IO uint32_t TXBC;             /*!< FDCAN TXBC Register,                Address offset: 0xC0  */
   __IO uint32_t TXFQS;            /*!< FDCAN TXFQS Register,               Address offset: 0xC4  */
   __IO uint32_t TXESC;            /*!< FDCAN TXESC Register,               Address offset: 0xC8  */
   __IO uint32_t TXBRP;            /*!< FDCAN TXBRP Register,               Address offset: 0xCC  */
   __IO uint32_t TXBAR;            /*!< FDCAN TXBAR Register,               Address offset: 0xD0  */
   __IO uint32_t TXBCR;            /*!< FDCAN TXBCR Register,               Address offset: 0xD4  */
   __IO uint32_t TXBTO;            /*!< FDCAN TXBTO Register,               Address offset: 0xD8  */
   __IO uint32_t TXBCF;            /*!< FDCAN TXBCF Register,               Address offset: 0xDC  */
   __IO uint32_t TXBTIE;           /*!< FDCAN TXBTIE Register,              Address offset: 0xE0  */
   __IO uint32_t TXBCIE;           /*!< FDCAN TXBCIE Register,              Address offset: 0xE4  */
   __IO uint32_t RESERVED6[2];      
   __IO uint32_t TXEFC;            /*!< FDCAN TXEFC Register,               Address offset: 0xF0  */
   __IO uint32_t TXEFS;            /*!< FDCAN TXEFS Register,               Address offset: 0xF4  */
   __IO uint32_t TXEFA;            /*!< FDCAN TXEFA Register,               Address offset: 0xF8  */
} FDCAN_TypeDef;

/**
* @brief IWDG Registers
*/
typedef struct
{
   __IO uint32_t KR;               /*!< IWDG KR Register,                   Address offset: 0x0  */
   __IO uint32_t PR;               /*!< IWDG PR Register,                   Address offset: 0x4  */
   __IO uint32_t RLR;              /*!< IWDG RLR Register,                  Address offset: 0x8  */
   __IO uint32_t SR;               /*!< IWDG SR Register,                   Address offset: 0xC  */
} IWDG_TypeDef;

/**
  * @brief LPTIMER
  */
typedef struct
{
  __IO uint32_t ISR;              /*!< LPTIM Interrupt and Status register,                Address offset: 0x00 */
  __IO uint32_t ICR;              /*!< LPTIM Interrupt Clear register,                     Address offset: 0x04 */
  __IO uint32_t IER;              /*!< LPTIM Interrupt Enable register,                    Address offset: 0x08 */
  __IO uint32_t CFGR;             /*!< LPTIM Configuration register,                       Address offset: 0x0C */
  __IO uint32_t CR;               /*!< LPTIM Control register,                             Address offset: 0x10 */
  __IO uint32_t CMP;        /*!< RESERVED1,                                          Address offset: 0x14 */
  __IO uint32_t ARR;              /*!< LPTIM Autoreload register,                          Address offset: 0x18 */
  __IO uint32_t CNT;              /*!< LPTIM Counter register,                             Address offset: 0x1C */
  __IO uint32_t OR;              /*!< LPTIM Counter register,                             Address offset: 0x1C */
} LPTIM_TypeDef;

/**
* @brief LPUART Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< LPUART CR1 Register,                Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< LPUART CR2 Register,                Address offset: 0x04  */
   __IO uint32_t CR3;              /*!< LPUART CR3 Register,                Address offset: 0x08  */
   __IO uint32_t BRR;              /*!< LPUART BRR Register,                Address offset: 0x0C  */
   __IO uint32_t LPWKUP_BRR;       /*!< LPUART LPWKUP_BRR Register,         Address offset: 0x10  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t RQR;              /*!< LPUART RQR Register,                Address offset: 0x18  */
   __IO uint32_t ISR;              /*!< LPUART ISR Register,                Address offset: 0x1C  */
   __IO uint32_t ICR;              /*!< LPUART ICR Register,                Address offset: 0x20  */
   __IO uint32_t RDR;              /*!< LPUART RDR Register,                Address offset: 0x24  */
   __IO uint32_t TDR;              /*!< LPUART TDR Register,                Address offset: 0x28  */
   __IO uint32_t PRESC;            /*!< LPUART PRESC Register,              Address offset: 0x2C  */
   __IO uint32_t DBG;              /*!< LPUART DBG Register,                Address offset: 0x30  */
} LPUART_TypeDef;

/**
* @brief BKP Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< TAMP CR1 Register,                  Address offset: 0x0  */
   __IO uint32_t CR2;              /*!< TAMP CR2 Register,                  Address offset: 0x4  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t FLTCR;            /*!< TAMP FLTCR Register,                Address offset: 0xC  */
   __IO uint32_t RESERVED2[7];      
   __IO uint32_t IER;              /*!< TAMP IER Register,                  Address offset: 0x2C  */
   __IO uint32_t SR;               /*!< TAMP SR Register,                   Address offset: 0x30  */
   __IO uint32_t MISR;             /*!< TAMP MISR Register,                 Address offset: 0x34  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t SCR;              /*!< TAMP SCR Register,                  Address offset: 0x3C  */
   __IO uint32_t RESERVED4[48];     
   __IO uint32_t BKP0R;            /*!< TAMP BKP0R Register,                Address offset: 0x100  */
   __IO uint32_t BKP1R;            /*!< TAMP BKP1R Register,                Address offset: 0x104  */
   __IO uint32_t BKP2R;            /*!< TAMP BKP2R Register,                Address offset: 0x108  */
   __IO uint32_t BKP3R;            /*!< TAMP BKP3R Register,                Address offset: 0x10C  */
   __IO uint32_t BKP4R;            /*!< TAMP BKP4R Register,                Address offset: 0x110  */
   __IO uint32_t BKP5R;            /*!< TAMP BKP5R Register,                Address offset: 0x114  */
   __IO uint32_t BKP6R;            /*!< TAMP BKP6R Register,                Address offset: 0x118  */
   __IO uint32_t BKP7R;            /*!< TAMP BKP7R Register,                Address offset: 0x11C  */
   __IO uint32_t BKP8R;            /*!< TAMP BKP8R Register,                Address offset: 0x120  */
   __IO uint32_t BKP9R;            /*!< TAMP BKP9R Register,                Address offset: 0x124  */
   __IO uint32_t BKP10R;           /*!< TAMP BKP10R Register,               Address offset: 0x128  */
   __IO uint32_t BKP11R;           /*!< TAMP BKP11R Register,               Address offset: 0x12C  */
   __IO uint32_t BKP12R;           /*!< TAMP BKP12R Register,               Address offset: 0x130  */
   __IO uint32_t BKP13R;           /*!< TAMP BKP13R Register,               Address offset: 0x134  */
   __IO uint32_t BKP14R;           /*!< TAMP BKP14R Register,               Address offset: 0x138  */
   __IO uint32_t BKP15R;           /*!< TAMP BKP15R Register,               Address offset: 0x13C  */
   __IO uint32_t BKP16R;           /*!< TAMP BKP16R Register,               Address offset: 0x140  */
   __IO uint32_t BKP17R;           /*!< TAMP BKP17R Register,               Address offset: 0x144  */
   __IO uint32_t BKP18R;           /*!< TAMP BKP18R Register,               Address offset: 0x148  */
   __IO uint32_t BKP19R;           /*!< TAMP BKP19R Register,               Address offset: 0x14C  */
   __IO uint32_t BKP20R;           /*!< TAMP BKP20R Register,               Address offset: 0x150  */
   __IO uint32_t BKP21R;           /*!< TAMP BKP21R Register,               Address offset: 0x154  */
   __IO uint32_t BKP22R;           /*!< TAMP BKP22R Register,               Address offset: 0x158  */
   __IO uint32_t BKP23R;           /*!< TAMP BKP23R Register,               Address offset: 0x15C  */
   __IO uint32_t BKP24R;           /*!< TAMP BKP24R Register,               Address offset: 0x160  */
   __IO uint32_t BKP25R;           /*!< TAMP BKP25R Register,               Address offset: 0x164  */
   __IO uint32_t BKP26R;           /*!< TAMP BKP26R Register,               Address offset: 0x168  */
   __IO uint32_t BKP27R;           /*!< TAMP BKP27R Register,               Address offset: 0x16C  */
   __IO uint32_t BKP28R;           /*!< TAMP BKP28R Register,               Address offset: 0x170  */
   __IO uint32_t BKP29R;           /*!< TAMP BKP29R Register,               Address offset: 0x174  */
   __IO uint32_t BKP30R;           /*!< TAMP BKP30R Register,               Address offset: 0x178  */
   __IO uint32_t BKP31R;           /*!< TAMP BKP31R Register,               Address offset: 0x17C  */
} TAMP_TypeDef;

/**
* @brief PWR Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< PWR CR1 Register,                   Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< PWR CR2 Register,                   Address offset: 0x04  */
   __IO uint32_t CR3;              /*!< PWR CR3 Register,                   Address offset: 0x08  */
   __IO uint32_t CR4;              /*!< PWR CR4 Register,                   Address offset: 0x0C  */
   __IO uint32_t SR;               /*!< PWR SR Register,                    Address offset: 0x10  */
   __IO uint32_t CSR;              /*!< PWR CSR Register,                   Address offset: 0x14  */
} PWR_TypeDef;


/**
* @brief DAC Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< DAC CR1 Register,                   Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< DAC CR2 Register,                   Address offset: 0x04  */
   __IO uint32_t DHR12R1;          /*!< DAC DHR12R1 Register,               Address offset: 0x08  */
   __IO uint32_t DHR12L1;          /*!< DAC DHR12L1 Register,               Address offset: 0x0C  */
   __IO uint32_t DHR8R1;           /*!< DAC DHR8R1 Register,                Address offset: 0x10  */
   __IO uint32_t RESERVED1[6];      
   __IO uint32_t DOR1;             /*!< DAC DOR1 Register,                  Address offset: 0x2C  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t SR;               /*!< DAC SR Register,                    Address offset: 0x34  */
   __IO uint32_t CALR;             /*!< DAC CALR Register,                  Address offset: 0x38  */
} DAC_TypeDef;

/**
* @brief RCC Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< RCC CR Register,                    Address offset: 0x00  */
   __IO uint32_t ICSCR;            /*!< RCC ICSCR Register,                 Address offset: 0x04  */
   __IO uint32_t CFGR;             /*!< RCC CFGR Register,                  Address offset: 0x08  */
   __IO uint32_t PLLCFGR;          /*!< RCC PLLCFGR Register,               Address offset: 0x0C  */
   __IO uint32_t ECSCR;            /*!< RCC ECSCR Register,                 Address offset: 0x10  */
   __IO uint32_t RESERVED1;         
   __IO uint32_t CIER;             /*!< RCC CIER Register,                  Address offset: 0x18  */
   __IO uint32_t CIFR;             /*!< RCC CIFR Register,                  Address offset: 0x1C  */
   __IO uint32_t CICR;             /*!< RCC CICR Register,                  Address offset: 0x20  */
   __IO uint32_t IOPRSTR;          /*!< RCC IOPRSTR Register,               Address offset: 0x24  */
   __IO uint32_t AHBRSTR;          /*!< RCC AHBRSTR Register,               Address offset: 0x28  */
   __IO uint32_t APBRSTR1;         /*!< RCC APBRSTR1 Register,              Address offset: 0x2C  */
   __IO uint32_t APBRSTR2;         /*!< RCC APBRSTR2 Register,              Address offset: 0x30  */
   __IO uint32_t IOPENR;           /*!< RCC IOPENR Register,                Address offset: 0x34  */
   __IO uint32_t AHBENR;           /*!< RCC AHBENR Register,                Address offset: 0x38  */
   __IO uint32_t APBENR1;          /*!< RCC APBENR1 Register,               Address offset: 0x3C  */
   __IO uint32_t APBENR2;          /*!< RCC APBENR2 Register,               Address offset: 0x40  */
   __IO uint32_t RESERVED2[4];      
   __IO uint32_t CCIPR;            /*!< RCC CCIPR Register,                 Address offset: 0x54  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t BDCR;             /*!< RCC BDCR Register,                  Address offset: 0x5C  */
   __IO uint32_t CSR;              /*!< RCC CSR Register,                   Address offset: 0x60  */
} RCC_TypeDef;


/**
* @brief RTC Registers
*/
typedef struct
{
   __IO uint32_t TR;               /*!< RTC TR Register,                    Address offset: 0x00  */
   __IO uint32_t DR;               /*!< RTC DR Register,                    Address offset: 0x04  */
   __IO uint32_t SSR;              /*!< RTC SSR Register,                   Address offset: 0x08  */
   __IO uint32_t ICSR;             /*!< RTC ICSR Register,                  Address offset: 0x0C  */
   __IO uint32_t PRER;             /*!< RTC PRER Register,                  Address offset: 0x10  */
   __IO uint32_t WUTR;             /*!< RTC WUTR Register,                  Address offset: 0x14  */
   __IO uint32_t CR;               /*!< RTC CR Register,                    Address offset: 0x18  */
   __IO uint32_t RESERVED1[2];      
   __IO uint32_t WPR;              /*!< RTC WPR Register,                   Address offset: 0x24  */
   __IO uint32_t CALR;             /*!< RTC CALR Register,                  Address offset: 0x28  */
   __IO uint32_t SHIFTR;           /*!< RTC SHIFTR Register,                Address offset: 0x2C  */
   __IO uint32_t TSTR;             /*!< RTC TSTR Register,                  Address offset: 0x30  */
   __IO uint32_t TSDR;             /*!< RTC TSDR Register,                  Address offset: 0x34  */
   __IO uint32_t TSSSR;            /*!< RTC TSSSR Register,                 Address offset: 0x38  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t ALRMAR;           /*!< RTC ALRMAR Register,                Address offset: 0x40  */
   __IO uint32_t ALRMASSR;         /*!< RTC ALRMASSR Register,              Address offset: 0x44  */
   __IO uint32_t ALRMBR;           /*!< RTC ALRMBR Register,                Address offset: 0x48  */
   __IO uint32_t ALRMBSSR;         /*!< RTC ALRMBSSR Register,              Address offset: 0x4C  */
   __IO uint32_t SR;               /*!< RTC SR Register,                    Address offset: 0x50  */
   __IO uint32_t MISR;             /*!< RTC MISR Register,                  Address offset: 0x54  */
   __IO uint32_t SCR;              /*!< RTC SCR Register,                   Address offset: 0x58  */
} RTC_TypeDef;


/**
* @brief SPI Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< SPI CR1 Register,                   Address offset: 0x00  */
   __IO uint32_t CR2;              /*!< SPI CR2 Register,                   Address offset: 0x04  */
   __IO uint32_t SR;               /*!< SPI SR Register,                    Address offset: 0x08  */
   __IO uint32_t DR;               /*!< SPI DR Register,                    Address offset: 0x0C  */
   __IO uint32_t CRCPR;            /*!< SPI CRCPR Register,                 Address offset: 0x10  */
   __IO uint32_t RXCRCR;           /*!< SPI RXCRCR Register,                Address offset: 0x14  */
   __IO uint32_t TXCRCR;           /*!< SPI TXCRCR Register,                Address offset: 0x18  */
   __IO uint32_t I2SCFGR;         /*!< SPI I2S_CFGR Register,              Address offset: 0x1C  */
   __IO uint32_t I2SPR;            /*!< SPI I2SPR Register,                 Address offset: 0x20  */
} SPI_TypeDef;


/**
* @brief SYSCFG Registers
*/
typedef struct
{
   __IO uint32_t CFGR1;            /*!< SYSCFG CFGR1 Register,              Address offset: 0x00  */
   __IO uint32_t CFGR2;            /*!< SYSCFG CFGR2 Register,              Address offset: 0x04  */
   __IO uint32_t CFGR3;            /*!< SYSCFG CFGR3 Register,              Address offset: 0x08  */
   __IO uint32_t RESERVED1[5];      
   __IO uint32_t PA_ENS;          /*!< PA ENS_REG Register,                Address offset: 0x20  */
   __IO uint32_t PB_ENS;          /*!< PB ENS_REG Register,                Address offset: 0x24  */
   __IO uint32_t PC_ENS;          /*!< PC ENS_REG Register,                Address offset: 0x28  */
   __IO uint32_t PD_ENS;          /*!< PD ENS_REG Register,                Address offset: 0x2C  */
   __IO uint32_t RESERVED2[2];      
   __IO uint32_t IOCFG;            /*!< SYSCFG IOCFG Register,              Address offset: 0x38  */
   __IO uint32_t LEDCFG;           /*!< SYSCFG LEDCFG Register,             Address offset: 0x3C  */
   __IO uint32_t PA_ANA2EN;       /*!< PA ANA2EN_REG Register,             Address offset: 0x40  */
   __IO uint32_t PB_ANA2EN;       /*!< PB ANA2EN_REG Register,             Address offset: 0x44  */
   __IO uint32_t PC_ANA2EN;       /*!< PC ANA2EN_REG Register,             Address offset: 0x48  */
   __IO uint32_t PD_ANA2EN;       /*!< PD ANA2EN_REG Register,             Address offset: 0x4C  */
   __IO uint32_t RESERVED3[4];      
   __IO uint32_t PA_IORP;             /*!< PA IORP Register,                   Address offset: 0x60  */
   __IO uint32_t PB_IORP;             /*!< PB IORP Register,                   Address offset: 0x64  */
   __IO uint32_t PC_IORP;             /*!< PC IORP Register,                   Address offset: 0x68  */
   __IO uint32_t PD_IORP;             /*!< PD IORP Register,                   Address offset: 0x6C  */
   __IO uint32_t SCSR;              /*!< SRAM CSR Register,                  Address offset: 0x70  */
   __IO uint32_t SWPR;            /*!< SRAM PXWPR Register,                Address offset: 0x74  */
} SYSCFG_TypeDef;


/**
* @brief TIM Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< TIM_ CR1 Register,                  Address offset: 0x0000  */
   __IO uint32_t CR2;              /*!< TIM_ CR2 Register,                  Address offset: 0x0004  */
   __IO uint32_t SMCR;             /*!< TIM_ SMCR Register,                 Address offset: 0x0008  */
   __IO uint32_t DIER;             /*!< TIM_ DIER Register,                 Address offset: 0x000C  */
   __IO uint32_t SR;               /*!< TIM_ SR Register,                   Address offset: 0x0010  */
   __IO uint32_t EGR;              /*!< TIM_ EGR Register,                  Address offset: 0x0014  */
   __IO uint32_t CCMR1;            /*!< TIM_ CCMR1 Register,                Address offset: 0x0018  */
   __IO uint32_t CCMR2;            /*!< TIM_ CCMR2 Register,                Address offset: 0x001C  */
   __IO uint32_t CCER;             /*!< TIM_ CCER Register,                 Address offset: 0x0020  */
   __IO uint32_t CNT;              /*!< TIM_ CNT Register,                  Address offset: 0x0024  */
   __IO uint32_t PSC;              /*!< TIM_ PSC Register,                  Address offset: 0x0028  */
   __IO uint32_t ARR;              /*!< TIM_ ARR Register,                  Address offset: 0x002C  */
   __IO uint32_t RCR;              /*!< TIM_ RCR Register,                  Address offset: 0x0030  */
   __IO uint32_t CCR1;             /*!< TIM_ CCR1 Register,                 Address offset: 0x0034  */
   __IO uint32_t CCR2;             /*!< TIM_ CCR2 Register,                 Address offset: 0x0038  */
   __IO uint32_t CCR3;             /*!< TIM_ CCR3 Register,                 Address offset: 0x003C  */
   __IO uint32_t CCR4;             /*!< TIM_ CCR4 Register,                 Address offset: 0x0040  */
   __IO uint32_t BDTR;             /*!< TIM_ BDTR Register,                 Address offset: 0x0044  */
   __IO uint32_t RESERVED6[5];      
   __IO uint32_t TISEL;            /*!< TIM_ TISEL Register,                Address offset: 0x005C  */
   __IO uint32_t AF1;              /*!< TIM_ AF1 Register,                  Address offset: 0x0060  */
   __IO uint32_t AF2;              /*!< TIM_ AF2 Register,                  Address offset: 0x0064  */
   __IO uint32_t RESERVED7[221];    
   __IO uint32_t DCR;              /*!< TIM_ DCR Register,                  Address offset: 0x03DC  */
   __IO uint32_t RESERVED8[3];      
   __IO uint32_t DMAR;             /*!< TIM_ DMAR Register,                 Address offset: 0x03E0  */
} TIM_TypeDef;


/**
* @brief PWM Registers
*/
typedef struct
{
   __IO uint32_t CR1;              /*!< PWM CR1 Register,                   Address offset: 0x0000  */
   __IO uint32_t CR2;              /*!< PWM CR2 Register,                   Address offset: 0x0004  */
   __IO uint32_t SMCR;             /*!< PWM SMCR Register,                  Address offset: 0x0008  */
   __IO uint32_t DIER;             /*!< PWM DIER Register,                  Address offset: 0x000C  */
   __IO uint32_t RESERVED1[3];      
   __IO uint32_t SR;               /*!< PWM SR Register,                    Address offset: 0x0010  */
   __IO uint32_t RESERVED2;         
   __IO uint32_t CMR;              /*!< PWM CMR Register,                   Address offset: 0x0018  */
   __IO uint32_t RESERVED3;         
   __IO uint32_t CER;              /*!< PWM CER Register,                   Address offset: 0x0020  */
   __IO uint32_t RESERVED4[3];      
   __IO uint32_t CNT;              /*!< PWM CNT Register,                   Address offset: 0x0024  */
   __IO uint32_t RESERVED5[3];      
   __IO uint32_t PSC;              /*!< PWM PSC Register,                   Address offset: 0x0028  */
   __IO uint32_t RESERVED6[3];      
   __IO uint32_t ARR;              /*!< PWM ARR Register,                   Address offset: 0x002C  */
   __IO uint32_t RESERVED7[7];      
   __IO uint32_t CCR1;             /*!< PWM CCR1 Register,                  Address offset: 0x0034  */
   __IO uint32_t RESERVED8[3];      
   __IO uint32_t CCR2;             /*!< PWM CCR2 Register,                  Address offset: 0x0038  */
   __IO uint32_t RESERVED9[3];      
   __IO uint32_t CCR3;             /*!< PWM CCR3 Register,                  Address offset: 0x003C  */
   __IO uint32_t RESERVED10[3];     
   __IO uint32_t CCR4;             /*!< PWM CCR4 Register,                  Address offset: 0x0040  */
   __IO uint32_t RESERVED11[3];     
   __IO uint32_t BDTR;             /*!< PWM BDTR Register,                  Address offset: 0x0044  */
   __IO uint32_t DCR;              /*!< PWM DCR Register,                   Address offset: 0x48  */
   __IO uint32_t RESERVED12[3];     
   __IO uint32_t DMAR;             /*!< PWM DMAR Register,                  Address offset: 0x4C  */
} PWM_TypeDef;

/**
* @brief USART Registers
*/
typedef struct
{
   __IO uint32_t SR;               /*!< USART SR Register,                  Address offset: 0x00  */
   __IO uint32_t DR;               /*!< USART DR Register,                  Address offset: 0x04  */
   __IO uint32_t BRR;              /*!< USART BRR Register,                 Address offset: 0x08  */
   __IO uint32_t CR1;              /*!< USART CR1 Register,                 Address offset: 0x0C  */
   __IO uint32_t CR2;              /*!< USART CR2 Register,                 Address offset: 0x10  */
   __IO uint32_t CR3;              /*!< USART CR3 Register,                 Address offset: 0x14  */
   __IO uint32_t GTPR;             /*!< USART GTPR Register,                Address offset: 0x18  */
   __IO uint32_t RTOR;             /*!< USART RTOR Register,                Address offset: 0x1C  */
} USART_TypeDef;

/**
* @brief UART Registers
*/
typedef struct
{
   __IO uint32_t DR;               /*!< UART DR Register,                   Address offset: 0x00  */
   __IO uint32_t BRR;              /*!< UART BRR Register,                  Address offset: 0x04  */
   __IO uint32_t RESERVED1[2];      
   __IO uint32_t SR;               /*!< UART SR Register,                   Address offset: 0x10  */
   __IO uint32_t CR1;              /*!< UART CR1 Register,                  Address offset: 0x14  */
   __IO uint32_t CR2;              /*!< UART CR2 Register,                  Address offset: 0x18  */
   __IO uint32_t CR3;              /*!< UART CR3 Register,                  Address offset: 0x1C  */
   __IO uint32_t RAR;              /*!< UART RAR Register,                  Address offset: 0x20  */
   __IO uint32_t TAR;              /*!< UART TAR Register,                  Address offset: 0x24  */
   __IO uint32_t BRRF;             /*!< UART BRRF Register,                 Address offset: 0x28  */
} UART_TypeDef;

/**
* @brief WWDG Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< WWDR CR Register,                   Address offset: 0x0  */
   __IO uint32_t CFR;              /*!< WWDG CFR Register,                  Address offset: 0x4  */
   __IO uint32_t SR;               /*!< WWDG SR Register,                   Address offset: 0x8  */
} WWDG_TypeDef;

/**
* @brief LCD Registers
*/
typedef struct
{
   __IO uint32_t CR;               /*!< LCD CR Register,                    Address offset: 0x00  */
   __IO uint32_t CSR;              /*!< LCD CSR Register,                   Address offset: 0x04  */
   __IO uint32_t INTCLR;           /*!< LCD INTCLR Register,                Address offset: 0x08  */
   __IO uint32_t POEN0;            /*!< LCD POEN0 Register,                 Address offset: 0x0C  */
   __IO uint32_t POEN1;            /*!< LCD POEN1 Register,                 Address offset: 0x10  */
   __IO uint32_t RAM[15];          /*!< LCD RAM0 Register,                  Address offset: 0x14 ~ 0x50 */
} LCD_TypeDef;


/** @addtogroup Peripheral_memory_map
  * @{
  */
#define FLASH_BASE            (0x08000000UL)  /*!< FLASH base address */
#define FLASH_END             (0x0803FFFFUL)  /*!< FLASH end address */
#define FLASH_SIZE            (FLASH_END - FLASH_BASE + 1)
#define FLASH_PAGE_SIZE       0x00000100U     /*!< FLASH Page Size, 256 Bytes */
#define FLASH_PAGE_NB         (FLASH_SIZE / FLASH_PAGE_SIZE)
#define FLASH_SECTOR_SIZE     0x00002000U     /*!< FLASH Sector Size, 8k Bytes */
#define FLASH_SECTOR_NB       (FLASH_SIZE / FLASH_SECTOR_SIZE)
#define SRAM_BASE             (0x20000000UL)  /*!< SRAM base address */
#define SRAM_END              (0x20007FFFUL)  /*!< SRAM end address */
#define PERIPH_BASE           (0x40000000UL)  /*!< Peripheral base address */
#define IOPORT_BASE           (0x50000000UL)  /*!< IOPORT base address */
#define FLASH_USERDATA_BASE   (0x1FFF1A00UL)  /*!< Base address of : (up to 128 Bytes) embedded FLASH Userdata Area  */
#define FLASH_USERDATA_END    (0x1FFF1AFFUL)  /*!< End address of : (up to 128 Bytes) embedded FLASH Userdata Area   */

/*!< Peripheral memory map */
#define APBPERIPH_BASE        (PERIPH_BASE)
#define AHBPERIPH_BASE        (PERIPH_BASE + 0x00020000UL)

/*!< APB peripherals */
#define TIM2_BASE             (APBPERIPH_BASE + 0x00000000UL)
#define TIM3_BASE             (APBPERIPH_BASE + 0x00000400UL)
#define TIM6_BASE             (APBPERIPH_BASE + 0x00001000UL)
#define TIM7_BASE             (APBPERIPH_BASE + 0x00001400UL)
#define LCD_BASE              (APBPERIPH_BASE + 0x00002400UL)
#define RTC_BASE              (APBPERIPH_BASE + 0x00002800UL)
#define WWDG_BASE             (APBPERIPH_BASE + 0x00002C00UL)
#define IWDG_BASE             (APBPERIPH_BASE + 0x00003000UL)
#define SPI2_BASE             (APBPERIPH_BASE + 0x00003800UL)
#define USART2_BASE           (APBPERIPH_BASE + 0x00004400UL)
#define UART1_BASE            (APBPERIPH_BASE + 0x00004800UL)
#define UART2_BASE            (APBPERIPH_BASE + 0x00004C00UL)
#define I2C1_BASE             (APBPERIPH_BASE + 0x00005400UL)
#define I2C2_BASE             (APBPERIPH_BASE + 0x00005800UL)
#define FDCAN_BASE            (APBPERIPH_BASE + 0x00006400UL)
#define PWR_BASE              (APBPERIPH_BASE + 0x00007000UL)
#define DAC_BASE              (APBPERIPH_BASE + 0x00007400UL)
#define LPTIM1_BASE           (APBPERIPH_BASE + 0x00007C00UL)
#define LPUART1_BASE          (APBPERIPH_BASE + 0x00008000UL)
#define LPTIM2_BASE           (APBPERIPH_BASE + 0x00009400UL)
#define LPUART2_BASE          (APBPERIPH_BASE + 0x00009800UL)
#define TAMP_BASE              (APBPERIPH_BASE + 0x0000B000UL)
#define SYSCFG_BASE           (APBPERIPH_BASE + 0x00010000UL)
#define COMP1_BASE            (APBPERIPH_BASE + 0x00010200UL)
#define COMP2_BASE            (APBPERIPH_BASE + 0x00010210UL)
#define OPA_BASE              (APBPERIPH_BASE + 0x00010300UL)
#define EXTI_BASE             (APBPERIPH_BASE + 0x00010400UL)
#define VREFBUF_BASE          (APBPERIPH_BASE + 0x00010800UL)
#define ADC_BASE              (APBPERIPH_BASE + 0x00012400UL)
#define TK_BASE               (APBPERIPH_BASE + 0x00012800UL)
#define TIM1_BASE             (APBPERIPH_BASE + 0x00012C00UL)
#define SPI1_BASE             (APBPERIPH_BASE + 0x00013000UL)
#define USART1_BASE           (APBPERIPH_BASE + 0x00013800UL)
#define TIM15_BASE            (APBPERIPH_BASE + 0x00014000UL)
#define TIM16_BASE            (APBPERIPH_BASE + 0x00014400UL)
#define TIM17_BASE            (APBPERIPH_BASE + 0x00014800UL)
#define PWM_BASE              (APBPERIPH_BASE + 0x00014C00UL)
#define DBGMCU_BASE           (APBPERIPH_BASE + 0x00015800UL)

/*!< AHB peripherals */
#define DMA_BASE              (AHBPERIPH_BASE + 0x00000000UL)
#define DMA_Channel1_BASE     (DMA_BASE + 0x00000008UL)
#define DMA_Channel2_BASE     (DMA_BASE + 0x0000001CUL)
#define DMA_Channel3_BASE     (DMA_BASE + 0x00000030UL)
#define DMA_Channel4_BASE     (DMA_BASE + 0x00000044UL)
#define RCC_BASE              (AHBPERIPH_BASE + 0x00001000UL)
#define FLASH_R_BASE          (AHBPERIPH_BASE + 0x00002000UL) /*!< FLASH registers base address */
#define OB_BASE               0x1FFF1D00UL       /*!< FLASH Option Bytes base address */
// #define FLASHSIZE_BASE        0x1FFF0FFCUL       /*!< FLASH Size register base address */
#define UID_BASE              0x1FFF1C00UL       /*!< Unique device ID register base address */
#define CRC_BASE              (AHBPERIPH_BASE + 0x00003000UL)

/*!< IOPORT */
#define GPIOA_BASE            (IOPORT_BASE + 0x00000000UL)
#define GPIOB_BASE            (IOPORT_BASE + 0x00000400UL)
#define GPIOC_BASE            (IOPORT_BASE + 0x00000800UL)
#define GPIOD_BASE            (IOPORT_BASE + 0x00000C00UL)

/**
  * @}
  */

/** @addtogroup Peripheral_declaration
  * @{
  */
#define TIM2                ((TIM_TypeDef *) TIM2_BASE)
#define TIM3                ((TIM_TypeDef *) TIM3_BASE)
#define TIM6                ((TIM_TypeDef *) TIM6_BASE)
#define TIM7                ((TIM_TypeDef *) TIM7_BASE)
#define LCD                 ((LCD_TypeDef *) LCD_BASE)
#define RTC                 ((RTC_TypeDef *) RTC_BASE)
#define WWDG                ((WWDG_TypeDef *) WWDG_BASE)
#define IWDG                ((IWDG_TypeDef *) IWDG_BASE)
#define SPI2                ((SPI_TypeDef *) SPI2_BASE)
#define USART2              ((USART_TypeDef *) USART2_BASE)
#define UART1               ((UART_TypeDef *) UART1_BASE)
#define UART2               ((UART_TypeDef *) UART2_BASE)
#define I2C1                ((I2C_TypeDef *) I2C1_BASE)
#define I2C                 ((I2C_TypeDef *) I2C1_BASE)        /* Kept for legacy purpose */
#define I2C2                ((I2C_TypeDef *) I2C2_BASE)
#define FDCAN               ((FDCAN_TypeDef *) FDCAN_BASE)
#define PWR                 ((PWR_TypeDef *) PWR_BASE)
#define DAC                 ((DAC_TypeDef *) DAC_BASE)
#define LPTIM1              ((LPTIM_TypeDef *) LPTIM1_BASE)
#define LPTIM               ((LPTIM_TypeDef *) LPTIM1_BASE)    /* Kept for legacy purpose */
#define LPUART1             ((LPUART_TypeDef *) LPUART1_BASE)
#define LPTIM2              ((LPTIM_TypeDef *) LPTIM2_BASE)
#define LPUART2             ((LPUART_TypeDef *) LPUART2_BASE)
#define TAMP                ((TAMP_TypeDef *) TAMP_BASE)
#define SYSCFG              ((SYSCFG_TypeDef *) SYSCFG_BASE)
#define COMP1               ((COMP_TypeDef *) COMP1_BASE)
#define COMP2               ((COMP_TypeDef *) COMP2_BASE)
#define COMP12_COMMON       ((COMP_Common_TypeDef *) COMP1_BASE)
#define OPA                 ((OPA_TypeDef *) OPA_BASE)
#define EXTI                ((EXTI_TypeDef *) EXTI_BASE)
#define VREFBUF             ((VREFBUF_TypeDef *) VREFBUF_BASE)
#define ADC1                ((ADC_TypeDef *) ADC_BASE)
#define ADC1_COMMON         ((ADC_Common_TypeDef *) ADC_BASE)
#define ADC                 ((ADC_Common_TypeDef *) ADC_BASE) /* Kept for legacy purpose */
#define TK                  ((TK_TypeDef *) TK_BASE)
#define TIM1                ((TIM_TypeDef *) TIM1_BASE)
#define SPI1                ((SPI_TypeDef *) SPI1_BASE)
#define USART1              ((USART_TypeDef *) USART1_BASE)
#define TIM15               ((TIM_TypeDef *) TIM15_BASE)
#define TIM16               ((TIM_TypeDef *) TIM16_BASE)
#define TIM17               ((TIM_TypeDef *) TIM17_BASE)
#define PWM                 ((PWM_TypeDef *) PWM_BASE)
#define DBGMCU              ((DBGMCU_TypeDef *) DBGMCU_BASE)
#define DMA                 ((DMA_TypeDef *) DMA_BASE)
#define DMA_Channel1        ((DMA_Channel_TypeDef *) DMA_Channel1_BASE)
#define DMA_Channel2        ((DMA_Channel_TypeDef *) DMA_Channel2_BASE)
#define DMA_Channel3        ((DMA_Channel_TypeDef *) DMA_Channel3_BASE)
#define DMA_Channel4        ((DMA_Channel_TypeDef *) DMA_Channel4_BASE)
#define RCC                 ((RCC_TypeDef *) RCC_BASE)
#define FLASH               ((FLASH_TypeDef *) FLASH_R_BASE)
#define CRC                 ((CRC_TypeDef *) CRC_BASE)
//#define OB                  ((OB_TypeDef *) OB_BASE)
#define CRC                 ((CRC_TypeDef *) CRC_BASE)
#define GPIOA               ((GPIO_TypeDef *) GPIOA_BASE)
#define GPIOB               ((GPIO_TypeDef *) GPIOB_BASE)
#define GPIOC               ((GPIO_TypeDef *) GPIOC_BASE)
#define GPIOD               ((GPIO_TypeDef *) GPIOD_BASE)

/**
  * @}
  */

/** @addtogroup Exported_constants
  * @{
  */

/** @addtogroup Peripheral_Registers_Bits_Definition
* @{
*/

/******************************************************************************/
/*                         Peripheral Registers Bits Definition               */
/******************************************************************************/

/********************************************************************************************************************/
/********************************* ADC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for ADC_ISR register **********************************************/
#define ADC_ISR_ADRDY_Pos                         (0U)
#define ADC_ISR_ADRDY_Msk                         (0x1UL<<ADC_ISR_ADRDY_Pos)                        /*!< 0x00000001 */
#define ADC_ISR_ADRDY                             ADC_ISR_ADRDY_Msk                                 
#define ADC_ISR_EOSMP_Pos                         (1U)
#define ADC_ISR_EOSMP_Msk                         (0x1UL<<ADC_ISR_EOSMP_Pos)                        /*!< 0x00000002 */
#define ADC_ISR_EOSMP                             ADC_ISR_EOSMP_Msk                                 
#define ADC_ISR_EOC_Pos                           (2U)
#define ADC_ISR_EOC_Msk                           (0x1UL<<ADC_ISR_EOC_Pos)                          /*!< 0x00000004 */
#define ADC_ISR_EOC                               ADC_ISR_EOC_Msk                                   
#define ADC_ISR_EOS_Pos                           (3U)
#define ADC_ISR_EOS_Msk                           (0x1UL<<ADC_ISR_EOS_Pos)                          /*!< 0x00000008 */
#define ADC_ISR_EOS                               ADC_ISR_EOS_Msk                                   
#define ADC_ISR_OVR_Pos                           (4U)
#define ADC_ISR_OVR_Msk                           (0x1UL<<ADC_ISR_OVR_Pos)                          /*!< 0x00000010 */
#define ADC_ISR_OVR                               ADC_ISR_OVR_Msk                                   
#define ADC_ISR_JEOC_Pos                          (5U)
#define ADC_ISR_JEOC_Msk                          (0x1UL<<ADC_ISR_JEOC_Pos)                         /*!< 0x00000020 */
#define ADC_ISR_JEOC                              ADC_ISR_JEOC_Msk                                  
#define ADC_ISR_JEOS_Pos                          (6U)
#define ADC_ISR_JEOS_Msk                          (0x1UL<<ADC_ISR_JEOS_Pos)                         /*!< 0x00000040 */
#define ADC_ISR_JEOS                              ADC_ISR_JEOS_Msk                                  
#define ADC_ISR_AWD1_Pos                          (7U)
#define ADC_ISR_AWD1_Msk                          (0x1UL<<ADC_ISR_AWD1_Pos)                         /*!< 0x00000080 */
#define ADC_ISR_AWD1                              ADC_ISR_AWD1_Msk                                  

/********************************* Bit definition for ADC_IER register **********************************************/
#define ADC_IER_ADRDYIE_Pos                       (0U)
#define ADC_IER_ADRDYIE_Msk                       (0x1UL<<ADC_IER_ADRDYIE_Pos)                      /*!< 0x00000001 */
#define ADC_IER_ADRDYIE                           ADC_IER_ADRDYIE_Msk                               
#define ADC_IER_EOSMPIE_Pos                       (1U)
#define ADC_IER_EOSMPIE_Msk                       (0x1UL<<ADC_IER_EOSMPIE_Pos)                      /*!< 0x00000002 */
#define ADC_IER_EOSMPIE                           ADC_IER_EOSMPIE_Msk                               
#define ADC_IER_EOCIE_Pos                         (2U)
#define ADC_IER_EOCIE_Msk                         (0x1UL<<ADC_IER_EOCIE_Pos)                        /*!< 0x00000004 */
#define ADC_IER_EOCIE                             ADC_IER_EOCIE_Msk                                 
#define ADC_IER_EOSIE_Pos                         (3U)
#define ADC_IER_EOSIE_Msk                         (0x1UL<<ADC_IER_EOSIE_Pos)                        /*!< 0x00000008 */
#define ADC_IER_EOSIE                             ADC_IER_EOSIE_Msk                                 
#define ADC_IER_OVRIE_Pos                         (4U)
#define ADC_IER_OVRIE_Msk                         (0x1UL<<ADC_IER_OVRIE_Pos)                        /*!< 0x00000010 */
#define ADC_IER_OVRIE                             ADC_IER_OVRIE_Msk                                 
#define ADC_IER_JEOCIE_Pos                        (5U)
#define ADC_IER_JEOCIE_Msk                        (0x1UL<<ADC_IER_JEOCIE_Pos)                       /*!< 0x00000020 */
#define ADC_IER_JEOCIE                            ADC_IER_JEOCIE_Msk                                
#define ADC_IER_JEOSIE_Pos                        (6U)
#define ADC_IER_JEOSIE_Msk                        (0x1UL<<ADC_IER_JEOSIE_Pos)                       /*!< 0x00000040 */
#define ADC_IER_JEOSIE                            ADC_IER_JEOSIE_Msk                                
#define ADC_IER_AWD1IE_Pos                        (7U)
#define ADC_IER_AWD1IE_Msk                        (0x1UL<<ADC_IER_AWD1IE_Pos)                       /*!< 0x00000080 */
#define ADC_IER_AWD1IE                            ADC_IER_AWD1IE_Msk                                

/********************************* Bit definition for ADC_CR register ***********************************************/
#define ADC_CR_ADEN_Pos                           (0U)
#define ADC_CR_ADEN_Msk                           (0x1UL<<ADC_CR_ADEN_Pos)                          /*!< 0x00000001 */
#define ADC_CR_ADEN                               ADC_CR_ADEN_Msk                                   
#define ADC_CR_ADDIS_Pos                          (1U)
#define ADC_CR_ADDIS_Msk                          (0x1UL<<ADC_CR_ADDIS_Pos)                         /*!< 0x00000002 */
#define ADC_CR_ADDIS                              ADC_CR_ADDIS_Msk                                  
#define ADC_CR_ADSTART_Pos                        (2U)
#define ADC_CR_ADSTART_Msk                        (0x1UL<<ADC_CR_ADSTART_Pos)                       /*!< 0x00000004 */
#define ADC_CR_ADSTART                            ADC_CR_ADSTART_Msk                                
#define ADC_CR_JADSTART_Pos                       (3U)
#define ADC_CR_JADSTART_Msk                       (0x1UL<<ADC_CR_JADSTART_Pos)                      /*!< 0x00000008 */
#define ADC_CR_JADSTART                           ADC_CR_JADSTART_Msk                               
#define ADC_CR_ADSTP_Pos                          (4U)
#define ADC_CR_ADSTP_Msk                          (0x1UL<<ADC_CR_ADSTP_Pos)                         /*!< 0x00000010 */
#define ADC_CR_ADSTP                              ADC_CR_ADSTP_Msk                                  
#define ADC_CR_JADSTP_Pos                         (5U)
#define ADC_CR_JADSTP_Msk                         (0x1UL<<ADC_CR_JADSTP_Pos)                        /*!< 0x00000020 */
#define ADC_CR_JADSTP                             ADC_CR_JADSTP_Msk                                 
#define ADC_CR_ADCAL_Pos                          (31U)
#define ADC_CR_ADCAL_Msk                          (0x1UL<<ADC_CR_ADCAL_Pos)                         /*!< 0x80000000 */
#define ADC_CR_ADCAL                              ADC_CR_ADCAL_Msk                                  

/********************************* Bit definition for ADC_CFGR register *********************************************/
#define ADC_CFGR_DMAEN_Pos                        (0U)
#define ADC_CFGR_DMAEN_Msk                        (0x1UL<<ADC_CFGR_DMAEN_Pos)                       /*!< 0x00000001 */
#define ADC_CFGR_DMAEN                            ADC_CFGR_DMAEN_Msk                                
#define ADC_CFGR_DMACFG_Pos                       (1U)
#define ADC_CFGR_DMACFG_Msk                       (0x1UL<<ADC_CFGR_DMACFG_Pos)                      /*!< 0x00000002 */
#define ADC_CFGR_DMACFG                           ADC_CFGR_DMACFG_Msk                               
#define ADC_CFGR_RES_Pos                          (3U)
#define ADC_CFGR_RES_Msk                          (0x3UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000018 */
#define ADC_CFGR_RES                              ADC_CFGR_RES_Msk
#define ADC_CFGR_RES_0                            (0x1UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000008 */
#define ADC_CFGR_RES_1                            (0x2UL<<ADC_CFGR_RES_Pos)                         /*!< 0x00000010 */
#define ADC_CFGR_EXTSEL_Pos                       (5U)
#define ADC_CFGR_EXTSEL_Msk                       (0xFUL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x000001E0 */
#define ADC_CFGR_EXTSEL                           ADC_CFGR_EXTSEL_Msk
#define ADC_CFGR_EXTSEL_0                         (0x1UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000020 */
#define ADC_CFGR_EXTSEL_1                         (0x2UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000040 */
#define ADC_CFGR_EXTSEL_2                         (0x4UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000080 */
#define ADC_CFGR_EXTSEL_3                         (0x8UL<<ADC_CFGR_EXTSEL_Pos)                      /*!< 0x00000100 */
#define ADC_CFGR_EXTEN_Pos                        (10U)
#define ADC_CFGR_EXTEN_Msk                        (0x3UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000C00 */
#define ADC_CFGR_EXTEN                            ADC_CFGR_EXTEN_Msk
#define ADC_CFGR_EXTEN_0                          (0x1UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000400 */
#define ADC_CFGR_EXTEN_1                          (0x2UL<<ADC_CFGR_EXTEN_Pos)                       /*!< 0x00000800 */
#define ADC_CFGR_OVRMOD_Pos                       (12U)
#define ADC_CFGR_OVRMOD_Msk                       (0x1UL<<ADC_CFGR_OVRMOD_Pos)                      /*!< 0x00001000 */
#define ADC_CFGR_OVRMOD                           ADC_CFGR_OVRMOD_Msk                               
#define ADC_CFGR_CONT_Pos                         (13U)
#define ADC_CFGR_CONT_Msk                         (0x1UL<<ADC_CFGR_CONT_Pos)                        /*!< 0x00002000 */
#define ADC_CFGR_CONT                             ADC_CFGR_CONT_Msk                                 
#define ADC_CFGR_AUTDLY_Pos                       (14U)
#define ADC_CFGR_AUTDLY_Msk                       (0x1UL<<ADC_CFGR_AUTDLY_Pos)                      /*!< 0x00004000 */
#define ADC_CFGR_AUTDLY                           ADC_CFGR_AUTDLY_Msk                               
#define ADC_CFGR_ALIGN_Pos                        (15U)
#define ADC_CFGR_ALIGN_Msk                        (0x1UL<<ADC_CFGR_ALIGN_Pos)                       /*!< 0x00008000 */
#define ADC_CFGR_ALIGN                            ADC_CFGR_ALIGN_Msk                                
#define ADC_CFGR_DISCEN_Pos                       (16U)
#define ADC_CFGR_DISCEN_Msk                       (0x1UL<<ADC_CFGR_DISCEN_Pos)                      /*!< 0x00010000 */
#define ADC_CFGR_DISCEN                           ADC_CFGR_DISCEN_Msk                               
#define ADC_CFGR_DISCNUM_Pos                      (17U)
#define ADC_CFGR_DISCNUM_Msk                      (0x7UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x000E0000 */
#define ADC_CFGR_DISCNUM                          ADC_CFGR_DISCNUM_Msk
#define ADC_CFGR_DISCNUM_0                        (0x1UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00020000 */
#define ADC_CFGR_DISCNUM_1                        (0x2UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00040000 */
#define ADC_CFGR_DISCNUM_2                        (0x4UL<<ADC_CFGR_DISCNUM_Pos)                     /*!< 0x00080000 */
#define ADC_CFGR_JDISCEN_Pos                      (20U)
#define ADC_CFGR_JDISCEN_Msk                      (0x1UL<<ADC_CFGR_JDISCEN_Pos)                     /*!< 0x00100000 */
#define ADC_CFGR_JDISCEN                          ADC_CFGR_JDISCEN_Msk                              
#define ADC_CFGR_AWD1SGL_Pos                      (22U)
#define ADC_CFGR_AWD1SGL_Msk                      (0x1UL<<ADC_CFGR_AWD1SGL_Pos)                     /*!< 0x00400000 */
#define ADC_CFGR_AWD1SGL                          ADC_CFGR_AWD1SGL_Msk                              
#define ADC_CFGR_AWD1EN_Pos                       (23U)
#define ADC_CFGR_AWD1EN_Msk                       (0x1UL<<ADC_CFGR_AWD1EN_Pos)                      /*!< 0x00800000 */
#define ADC_CFGR_AWD1EN                           ADC_CFGR_AWD1EN_Msk                               
#define ADC_CFGR_JAWD1EN_Pos                      (24U)
#define ADC_CFGR_JAWD1EN_Msk                      (0x1UL<<ADC_CFGR_JAWD1EN_Pos)                     /*!< 0x01000000 */
#define ADC_CFGR_JAWD1EN                          ADC_CFGR_JAWD1EN_Msk                              
#define ADC_CFGR_JAUTO_Pos                        (25U)
#define ADC_CFGR_JAUTO_Msk                        (0x1UL<<ADC_CFGR_JAUTO_Pos)                       /*!< 0x02000000 */
#define ADC_CFGR_JAUTO                            ADC_CFGR_JAUTO_Msk                                
#define ADC_CFGR_AWD1CH_Pos                       (26U)
#define ADC_CFGR_AWD1CH_Msk                       (0x1FUL<<ADC_CFGR_AWD1CH_Pos)                     /*!< 0x7C000000 */
#define ADC_CFGR_AWD1CH                           ADC_CFGR_AWD1CH_Msk
#define ADC_CFGR_AWD1CH_0                         (0x1UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x04000000 */
#define ADC_CFGR_AWD1CH_1                         (0x2UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x08000000 */
#define ADC_CFGR_AWD1CH_2                         (0x4UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x10000000 */
#define ADC_CFGR_AWD1CH_3                         (0x8UL<<ADC_CFGR_AWD1CH_Pos)                      /*!< 0x20000000 */
#define ADC_CFGR_AWD1CH_4                         (0x10UL<<ADC_CFGR_AWD1CH_Pos)                     /*!< 0x40000000 */

/********************************* Bit definition for ADC_CFGR2 register ********************************************/
#define ADC_CFGR2_ROVSE_Pos                       (0U)
#define ADC_CFGR2_ROVSE_Msk                       (0x1UL<<ADC_CFGR2_ROVSE_Pos)                      /*!< 0x00000001 */
#define ADC_CFGR2_ROVSE                           ADC_CFGR2_ROVSE_Msk                               
#define ADC_CFGR2_JOVSE_Pos                       (1U)
#define ADC_CFGR2_JOVSE_Msk                       (0x1UL<<ADC_CFGR2_JOVSE_Pos)                      /*!< 0x00000002 */
#define ADC_CFGR2_JOVSE                           ADC_CFGR2_JOVSE_Msk                               
#define ADC_CFGR2_OVSR_Pos                        (2U)
#define ADC_CFGR2_OVSR_Msk                        (0x7UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x0000001C */
#define ADC_CFGR2_OVSR                            ADC_CFGR2_OVSR_Msk
#define ADC_CFGR2_OVSR_0                          (0x1UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000004 */
#define ADC_CFGR2_OVSR_1                          (0x2UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000008 */
#define ADC_CFGR2_OVSR_2                          (0x4UL<<ADC_CFGR2_OVSR_Pos)                       /*!< 0x00000010 */
#define ADC_CFGR2_OVSS_Pos                        (5U)
#define ADC_CFGR2_OVSS_Msk                        (0xFUL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x000001E0 */
#define ADC_CFGR2_OVSS                            ADC_CFGR2_OVSS_Msk
#define ADC_CFGR2_OVSS_0                          (0x1UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000020 */
#define ADC_CFGR2_OVSS_1                          (0x2UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000040 */
#define ADC_CFGR2_OVSS_2                          (0x4UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000080 */
#define ADC_CFGR2_OVSS_3                          (0x8UL<<ADC_CFGR2_OVSS_Pos)                       /*!< 0x00000100 */
#define ADC_CFGR2_TROVS_Pos                       (9U)
#define ADC_CFGR2_TROVS_Msk                       (0x1UL<<ADC_CFGR2_TROVS_Pos)                      /*!< 0x00000200 */
#define ADC_CFGR2_TROVS                           ADC_CFGR2_TROVS_Msk                               
#define ADC_CFGR2_ROVSM_Pos                       (10U)
#define ADC_CFGR2_ROVSM_Msk                       (0x1UL<<ADC_CFGR2_ROVSM_Pos)                      /*!< 0x00000400 */
#define ADC_CFGR2_ROVSM                           ADC_CFGR2_ROVSM_Msk                               
#define ADC_CFGR2_CALNUM_Pos                      (28U)
#define ADC_CFGR2_CALNUM_Msk                      (0x7UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x70000000 */
#define ADC_CFGR2_CALNUM                          ADC_CFGR2_CALNUM_Msk
#define ADC_CFGR2_CALNUM_0                        (0x1UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x10000000 */
#define ADC_CFGR2_CALNUM_1                        (0x2UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x20000000 */
#define ADC_CFGR2_CALNUM_2                        (0x4UL<<ADC_CFGR2_CALNUM_Pos)                     /*!< 0x40000000 */

/********************************* Bit definition for ADC_SMPR1 register ********************************************/
#define ADC_SMPR1_SMP0_Pos                        (0U)
#define ADC_SMPR1_SMP0_Msk                        (0x7UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000007 */
#define ADC_SMPR1_SMP0                            ADC_SMPR1_SMP0_Msk
#define ADC_SMPR1_SMP0_0                          (0x1UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000001 */
#define ADC_SMPR1_SMP0_1                          (0x2UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000002 */
#define ADC_SMPR1_SMP0_2                          (0x4UL<<ADC_SMPR1_SMP0_Pos)                       /*!< 0x00000004 */
#define ADC_SMPR1_SMP1_Pos                        (3U)
#define ADC_SMPR1_SMP1_Msk                        (0x7UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000038 */
#define ADC_SMPR1_SMP1                            ADC_SMPR1_SMP1_Msk
#define ADC_SMPR1_SMP1_0                          (0x1UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000008 */
#define ADC_SMPR1_SMP1_1                          (0x2UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000010 */
#define ADC_SMPR1_SMP1_2                          (0x4UL<<ADC_SMPR1_SMP1_Pos)                       /*!< 0x00000020 */
#define ADC_SMPR1_SMP2_Pos                        (6U)
#define ADC_SMPR1_SMP2_Msk                        (0x7UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x000001C0 */
#define ADC_SMPR1_SMP2                            ADC_SMPR1_SMP2_Msk
#define ADC_SMPR1_SMP2_0                          (0x1UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000040 */
#define ADC_SMPR1_SMP2_1                          (0x2UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000080 */
#define ADC_SMPR1_SMP2_2                          (0x4UL<<ADC_SMPR1_SMP2_Pos)                       /*!< 0x00000100 */
#define ADC_SMPR1_SMP3_Pos                        (9U)
#define ADC_SMPR1_SMP3_Msk                        (0x7UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000E00 */
#define ADC_SMPR1_SMP3                            ADC_SMPR1_SMP3_Msk
#define ADC_SMPR1_SMP3_0                          (0x1UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000200 */
#define ADC_SMPR1_SMP3_1                          (0x2UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000400 */
#define ADC_SMPR1_SMP3_2                          (0x4UL<<ADC_SMPR1_SMP3_Pos)                       /*!< 0x00000800 */
#define ADC_SMPR1_SMP4_Pos                        (12U)
#define ADC_SMPR1_SMP4_Msk                        (0x7UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00007000 */
#define ADC_SMPR1_SMP4                            ADC_SMPR1_SMP4_Msk
#define ADC_SMPR1_SMP4_0                          (0x1UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00001000 */
#define ADC_SMPR1_SMP4_1                          (0x2UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00002000 */
#define ADC_SMPR1_SMP4_2                          (0x4UL<<ADC_SMPR1_SMP4_Pos)                       /*!< 0x00004000 */
#define ADC_SMPR1_SMP5_Pos                        (15U)
#define ADC_SMPR1_SMP5_Msk                        (0x7UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00038000 */
#define ADC_SMPR1_SMP5                            ADC_SMPR1_SMP5_Msk
#define ADC_SMPR1_SMP5_0                          (0x1UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00008000 */
#define ADC_SMPR1_SMP5_1                          (0x2UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00010000 */
#define ADC_SMPR1_SMP5_2                          (0x4UL<<ADC_SMPR1_SMP5_Pos)                       /*!< 0x00020000 */
#define ADC_SMPR1_SMP6_Pos                        (18U)
#define ADC_SMPR1_SMP6_Msk                        (0x7UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x001C0000 */
#define ADC_SMPR1_SMP6                            ADC_SMPR1_SMP6_Msk
#define ADC_SMPR1_SMP6_0                          (0x1UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00040000 */
#define ADC_SMPR1_SMP6_1                          (0x2UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00080000 */
#define ADC_SMPR1_SMP6_2                          (0x4UL<<ADC_SMPR1_SMP6_Pos)                       /*!< 0x00100000 */
#define ADC_SMPR1_SMP7_Pos                        (21U)
#define ADC_SMPR1_SMP7_Msk                        (0x7UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00E00000 */
#define ADC_SMPR1_SMP7                            ADC_SMPR1_SMP7_Msk
#define ADC_SMPR1_SMP7_0                          (0x1UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00200000 */
#define ADC_SMPR1_SMP7_1                          (0x2UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00400000 */
#define ADC_SMPR1_SMP7_2                          (0x4UL<<ADC_SMPR1_SMP7_Pos)                       /*!< 0x00800000 */
#define ADC_SMPR1_SMP8_Pos                        (24U)
#define ADC_SMPR1_SMP8_Msk                        (0x7UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x07000000 */
#define ADC_SMPR1_SMP8                            ADC_SMPR1_SMP8_Msk
#define ADC_SMPR1_SMP8_0                          (0x1UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x01000000 */
#define ADC_SMPR1_SMP8_1                          (0x2UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x02000000 */
#define ADC_SMPR1_SMP8_2                          (0x4UL<<ADC_SMPR1_SMP8_Pos)                       /*!< 0x04000000 */
#define ADC_SMPR1_SMP9_Pos                        (27U)
#define ADC_SMPR1_SMP9_Msk                        (0x7UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x38000000 */
#define ADC_SMPR1_SMP9                            ADC_SMPR1_SMP9_Msk
#define ADC_SMPR1_SMP9_0                          (0x1UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x08000000 */
#define ADC_SMPR1_SMP9_1                          (0x2UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x10000000 */
#define ADC_SMPR1_SMP9_2                          (0x4UL<<ADC_SMPR1_SMP9_Pos)                       /*!< 0x20000000 */

/********************************* Bit definition for ADC_SMPR2 register ********************************************/
#define ADC_SMPR2_SMP10_Pos                       (0U)
#define ADC_SMPR2_SMP10_Msk                       (0x7UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000007 */
#define ADC_SMPR2_SMP10                           ADC_SMPR2_SMP10_Msk
#define ADC_SMPR2_SMP10_0                         (0x1UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000001 */
#define ADC_SMPR2_SMP10_1                         (0x2UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000002 */
#define ADC_SMPR2_SMP10_2                         (0x4UL<<ADC_SMPR2_SMP10_Pos)                      /*!< 0x00000004 */
#define ADC_SMPR2_SMP11_Pos                       (3U)
#define ADC_SMPR2_SMP11_Msk                       (0x7UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000038 */
#define ADC_SMPR2_SMP11                           ADC_SMPR2_SMP11_Msk
#define ADC_SMPR2_SMP11_0                         (0x1UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000008 */
#define ADC_SMPR2_SMP11_1                         (0x2UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000010 */
#define ADC_SMPR2_SMP11_2                         (0x4UL<<ADC_SMPR2_SMP11_Pos)                      /*!< 0x00000020 */
#define ADC_SMPR2_SMP12_Pos                       (6U)
#define ADC_SMPR2_SMP12_Msk                       (0x7UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x000001C0 */
#define ADC_SMPR2_SMP12                           ADC_SMPR2_SMP12_Msk
#define ADC_SMPR2_SMP12_0                         (0x1UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000040 */
#define ADC_SMPR2_SMP12_1                         (0x2UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000080 */
#define ADC_SMPR2_SMP12_2                         (0x4UL<<ADC_SMPR2_SMP12_Pos)                      /*!< 0x00000100 */
#define ADC_SMPR2_SMP13_Pos                       (9U)
#define ADC_SMPR2_SMP13_Msk                       (0x7UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000E00 */
#define ADC_SMPR2_SMP13                           ADC_SMPR2_SMP13_Msk
#define ADC_SMPR2_SMP13_0                         (0x1UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000200 */
#define ADC_SMPR2_SMP13_1                         (0x2UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000400 */
#define ADC_SMPR2_SMP13_2                         (0x4UL<<ADC_SMPR2_SMP13_Pos)                      /*!< 0x00000800 */
#define ADC_SMPR2_SMP14_Pos                       (12U)
#define ADC_SMPR2_SMP14_Msk                       (0x7UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00007000 */
#define ADC_SMPR2_SMP14                           ADC_SMPR2_SMP14_Msk
#define ADC_SMPR2_SMP14_0                         (0x1UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00001000 */
#define ADC_SMPR2_SMP14_1                         (0x2UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00002000 */
#define ADC_SMPR2_SMP14_2                         (0x4UL<<ADC_SMPR2_SMP14_Pos)                      /*!< 0x00004000 */
#define ADC_SMPR2_SMP15_Pos                       (15U)
#define ADC_SMPR2_SMP15_Msk                       (0x7UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00038000 */
#define ADC_SMPR2_SMP15                           ADC_SMPR2_SMP15_Msk
#define ADC_SMPR2_SMP15_0                         (0x1UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00008000 */
#define ADC_SMPR2_SMP15_1                         (0x2UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00010000 */
#define ADC_SMPR2_SMP15_2                         (0x4UL<<ADC_SMPR2_SMP15_Pos)                      /*!< 0x00020000 */
#define ADC_SMPR2_SMP16_Pos                       (18U)
#define ADC_SMPR2_SMP16_Msk                       (0x7UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x001C0000 */
#define ADC_SMPR2_SMP16                           ADC_SMPR2_SMP16_Msk
#define ADC_SMPR2_SMP16_0                         (0x1UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00040000 */
#define ADC_SMPR2_SMP16_1                         (0x2UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00080000 */
#define ADC_SMPR2_SMP16_2                         (0x4UL<<ADC_SMPR2_SMP16_Pos)                      /*!< 0x00100000 */
#define ADC_SMPR2_SMP17_Pos                       (21U)
#define ADC_SMPR2_SMP17_Msk                       (0x7UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00E00000 */
#define ADC_SMPR2_SMP17                           ADC_SMPR2_SMP17_Msk
#define ADC_SMPR2_SMP17_0                         (0x1UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00200000 */
#define ADC_SMPR2_SMP17_1                         (0x2UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00400000 */
#define ADC_SMPR2_SMP17_2                         (0x4UL<<ADC_SMPR2_SMP17_Pos)                      /*!< 0x00800000 */
#define ADC_SMPR2_SMP18_Pos                       (24U)
#define ADC_SMPR2_SMP18_Msk                       (0x7UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x07000000 */
#define ADC_SMPR2_SMP18                           ADC_SMPR2_SMP18_Msk
#define ADC_SMPR2_SMP18_0                         (0x1UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x01000000 */
#define ADC_SMPR2_SMP18_1                         (0x2UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x02000000 */
#define ADC_SMPR2_SMP18_2                         (0x4UL<<ADC_SMPR2_SMP18_Pos)                      /*!< 0x04000000 */
#define ADC_SMPR2_SMP19_Pos                       (27U)
#define ADC_SMPR2_SMP19_Msk                       (0x7UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x38000000 */
#define ADC_SMPR2_SMP19                           ADC_SMPR2_SMP19_Msk
#define ADC_SMPR2_SMP19_0                         (0x1UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x08000000 */
#define ADC_SMPR2_SMP19_1                         (0x2UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x10000000 */
#define ADC_SMPR2_SMP19_2                         (0x4UL<<ADC_SMPR2_SMP19_Pos)                      /*!< 0x20000000 */

/********************************* Bit definition for ADC_SMPR3 register ********************************************/
#define ADC_SMPR3_SMP20_Pos                       (0U)
#define ADC_SMPR3_SMP20_Msk                       (0x7UL<<ADC_SMPR3_SMP20_Pos)                      /*!< 0x00000007 */
#define ADC_SMPR3_SMP20                           ADC_SMPR3_SMP20_Msk
#define ADC_SMPR3_SMP20_0                         (0x1UL<<ADC_SMPR3_SMP20_Pos)                      /*!< 0x00000001 */
#define ADC_SMPR3_SMP20_1                         (0x2UL<<ADC_SMPR3_SMP20_Pos)                      /*!< 0x00000002 */
#define ADC_SMPR3_SMP20_2                         (0x4UL<<ADC_SMPR3_SMP20_Pos)                      /*!< 0x00000004 */
#define ADC_SMPR3_SMP21_Pos                       (3U)
#define ADC_SMPR3_SMP21_Msk                       (0x7UL<<ADC_SMPR3_SMP21_Pos)                      /*!< 0x00000038 */
#define ADC_SMPR3_SMP21                           ADC_SMPR3_SMP21_Msk
#define ADC_SMPR3_SMP21_0                         (0x1UL<<ADC_SMPR3_SMP21_Pos)                      /*!< 0x00000008 */
#define ADC_SMPR3_SMP21_1                         (0x2UL<<ADC_SMPR3_SMP21_Pos)                      /*!< 0x00000010 */
#define ADC_SMPR3_SMP21_2                         (0x4UL<<ADC_SMPR3_SMP21_Pos)                      /*!< 0x00000020 */
#define ADC_SMPR3_SMP22_Pos                       (6U)
#define ADC_SMPR3_SMP22_Msk                       (0x7UL<<ADC_SMPR3_SMP22_Pos)                      /*!< 0x000001C0 */
#define ADC_SMPR3_SMP22                           ADC_SMPR3_SMP22_Msk
#define ADC_SMPR3_SMP22_0                         (0x1UL<<ADC_SMPR3_SMP22_Pos)                      /*!< 0x00000040 */
#define ADC_SMPR3_SMP22_1                         (0x2UL<<ADC_SMPR3_SMP22_Pos)                      /*!< 0x00000080 */
#define ADC_SMPR3_SMP22_2                         (0x4UL<<ADC_SMPR3_SMP22_Pos)                      /*!< 0x00000100 */
#define ADC_SMPR3_SMP23_Pos                       (9U)
#define ADC_SMPR3_SMP23_Msk                       (0x7UL<<ADC_SMPR3_SMP23_Pos)                      /*!< 0x00000E00 */
#define ADC_SMPR3_SMP23                           ADC_SMPR3_SMP23_Msk
#define ADC_SMPR3_SMP23_0                         (0x1UL<<ADC_SMPR3_SMP23_Pos)                      /*!< 0x00000200 */
#define ADC_SMPR3_SMP23_1                         (0x2UL<<ADC_SMPR3_SMP23_Pos)                      /*!< 0x00000400 */
#define ADC_SMPR3_SMP23_2                         (0x4UL<<ADC_SMPR3_SMP23_Pos)                      /*!< 0x00000800 */
#define ADC_SMPR3_SMP24_Pos                       (12U)
#define ADC_SMPR3_SMP24_Msk                       (0x7UL<<ADC_SMPR3_SMP24_Pos)                      /*!< 0x00007000 */
#define ADC_SMPR3_SMP24                           ADC_SMPR3_SMP24_Msk
#define ADC_SMPR3_SMP24_0                         (0x1UL<<ADC_SMPR3_SMP24_Pos)                      /*!< 0x00001000 */
#define ADC_SMPR3_SMP24_1                         (0x2UL<<ADC_SMPR3_SMP24_Pos)                      /*!< 0x00002000 */
#define ADC_SMPR3_SMP24_2                         (0x4UL<<ADC_SMPR3_SMP24_Pos)                      /*!< 0x00004000 */
#define ADC_SMPR3_SMP25_Pos                       (15U)
#define ADC_SMPR3_SMP25_Msk                       (0x7UL<<ADC_SMPR3_SMP25_Pos)                      /*!< 0x00038000 */
#define ADC_SMPR3_SMP25                           ADC_SMPR3_SMP25_Msk
#define ADC_SMPR3_SMP25_0                         (0x1UL<<ADC_SMPR3_SMP25_Pos)                      /*!< 0x00008000 */
#define ADC_SMPR3_SMP25_1                         (0x2UL<<ADC_SMPR3_SMP25_Pos)                      /*!< 0x00010000 */
#define ADC_SMPR3_SMP25_2                         (0x4UL<<ADC_SMPR3_SMP25_Pos)                      /*!< 0x00020000 */
#define ADC_SMPR3_SMP26_Pos                       (18U)
#define ADC_SMPR3_SMP26_Msk                       (0x7UL<<ADC_SMPR3_SMP26_Pos)                      /*!< 0x001C0000 */
#define ADC_SMPR3_SMP26                           ADC_SMPR3_SMP26_Msk
#define ADC_SMPR3_SMP26_0                         (0x1UL<<ADC_SMPR3_SMP26_Pos)                      /*!< 0x00040000 */
#define ADC_SMPR3_SMP26_1                         (0x2UL<<ADC_SMPR3_SMP26_Pos)                      /*!< 0x00080000 */
#define ADC_SMPR3_SMP26_2                         (0x4UL<<ADC_SMPR3_SMP26_Pos)                      /*!< 0x00100000 */
#define ADC_SMPR3_SMP27_Pos                       (21U)
#define ADC_SMPR3_SMP27_Msk                       (0x7UL<<ADC_SMPR3_SMP27_Pos)                      /*!< 0x00E00000 */
#define ADC_SMPR3_SMP27                           ADC_SMPR3_SMP27_Msk
#define ADC_SMPR3_SMP27_0                         (0x1UL<<ADC_SMPR3_SMP27_Pos)                      /*!< 0x00200000 */
#define ADC_SMPR3_SMP27_1                         (0x2UL<<ADC_SMPR3_SMP27_Pos)                      /*!< 0x00400000 */
#define ADC_SMPR3_SMP27_2                         (0x4UL<<ADC_SMPR3_SMP27_Pos)                      /*!< 0x00800000 */
#define ADC_SMPR3_SMP28_Pos                       (24U)
#define ADC_SMPR3_SMP28_Msk                       (0x7UL<<ADC_SMPR3_SMP28_Pos)                      /*!< 0x07000000 */
#define ADC_SMPR3_SMP28                           ADC_SMPR3_SMP28_Msk
#define ADC_SMPR3_SMP28_0                         (0x1UL<<ADC_SMPR3_SMP28_Pos)                      /*!< 0x01000000 */
#define ADC_SMPR3_SMP28_1                         (0x2UL<<ADC_SMPR3_SMP28_Pos)                      /*!< 0x02000000 */
#define ADC_SMPR3_SMP28_2                         (0x4UL<<ADC_SMPR3_SMP28_Pos)                      /*!< 0x04000000 */
#define ADC_SMPR3_SMP29_Pos                       (27U)
#define ADC_SMPR3_SMP29_Msk                       (0x7UL<<ADC_SMPR3_SMP29_Pos)                      /*!< 0x38000000 */
#define ADC_SMPR3_SMP29                           ADC_SMPR3_SMP29_Msk
#define ADC_SMPR3_SMP29_0                         (0x1UL<<ADC_SMPR3_SMP29_Pos)                      /*!< 0x08000000 */
#define ADC_SMPR3_SMP29_1                         (0x2UL<<ADC_SMPR3_SMP29_Pos)                      /*!< 0x10000000 */
#define ADC_SMPR3_SMP29_2                         (0x4UL<<ADC_SMPR3_SMP29_Pos)                      /*!< 0x20000000 */

/********************************* Bit definition for ADC_SMPR4 register ********************************************/
#define ADC_SMPR4_SMP30_Pos                       (0U)
#define ADC_SMPR4_SMP30_Msk                       (0x7UL<<ADC_SMPR4_SMP30_Pos)                      /*!< 0x00000007 */
#define ADC_SMPR4_SMP30                           ADC_SMPR4_SMP30_Msk
#define ADC_SMPR4_SMP30_0                         (0x1UL<<ADC_SMPR4_SMP30_Pos)                      /*!< 0x00000001 */
#define ADC_SMPR4_SMP30_1                         (0x2UL<<ADC_SMPR4_SMP30_Pos)                      /*!< 0x00000002 */
#define ADC_SMPR4_SMP30_2                         (0x4UL<<ADC_SMPR4_SMP30_Pos)                      /*!< 0x00000004 */
#define ADC_SMPR4_SMP31_Pos                       (3U)
#define ADC_SMPR4_SMP31_Msk                       (0x7UL<<ADC_SMPR4_SMP31_Pos)                      /*!< 0x00000038 */
#define ADC_SMPR4_SMP31                           ADC_SMPR4_SMP31_Msk
#define ADC_SMPR4_SMP31_0                         (0x1UL<<ADC_SMPR4_SMP31_Pos)                      /*!< 0x00000008 */
#define ADC_SMPR4_SMP31_1                         (0x2UL<<ADC_SMPR4_SMP31_Pos)                      /*!< 0x00000010 */
#define ADC_SMPR4_SMP31_2                         (0x4UL<<ADC_SMPR4_SMP31_Pos)                      /*!< 0x00000020 */
#define ADC_SMPR4_SMP32_Pos                       (6U)
#define ADC_SMPR4_SMP32_Msk                       (0x7UL<<ADC_SMPR4_SMP32_Pos)                      /*!< 0x000001C0 */
#define ADC_SMPR4_SMP32                           ADC_SMPR4_SMP32_Msk
#define ADC_SMPR4_SMP32_0                         (0x1UL<<ADC_SMPR4_SMP32_Pos)                      /*!< 0x00000040 */
#define ADC_SMPR4_SMP32_1                         (0x2UL<<ADC_SMPR4_SMP32_Pos)                      /*!< 0x00000080 */
#define ADC_SMPR4_SMP32_2                         (0x4UL<<ADC_SMPR4_SMP32_Pos)                      /*!< 0x00000100 */

/********************************* Bit definition for ADC_TR1 register **********************************************/
#define ADC_TR1_LT1_Pos                           (0U)
#define ADC_TR1_LT1_Msk                           (0xFFFUL<<ADC_TR1_LT1_Pos)                        /*!< 0x00000FFF */
#define ADC_TR1_LT1                               ADC_TR1_LT1_Msk
#define ADC_TR1_AWDFILT_Pos                       (12U)
#define ADC_TR1_AWDFILT_Msk                       (0x7UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00007000 */
#define ADC_TR1_AWDFILT                           ADC_TR1_AWDFILT_Msk
#define ADC_TR1_AWDFILT_0                         (0x1UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00001000 */
#define ADC_TR1_AWDFILT_1                         (0x2UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00002000 */
#define ADC_TR1_AWDFILT_2                         (0x4UL<<ADC_TR1_AWDFILT_Pos)                      /*!< 0x00004000 */
#define ADC_TR1_HT1_Pos                           (16U)
#define ADC_TR1_HT1_Msk                           (0xFFFUL<<ADC_TR1_HT1_Pos)                        /*!< 0x0FFF0000 */
#define ADC_TR1_HT1                               ADC_TR1_HT1_Msk

/********************************* Bit definition for ADC_SQR1 register *********************************************/
#define ADC_SQR1_L_Pos                            (0U)
#define ADC_SQR1_L_Msk                            (0xFUL<<ADC_SQR1_L_Pos)                           /*!< 0x0000000F */
#define ADC_SQR1_L                                ADC_SQR1_L_Msk
#define ADC_SQR1_L_0                              (0x1UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000001 */
#define ADC_SQR1_L_1                              (0x2UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000002 */
#define ADC_SQR1_L_2                              (0x4UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000004 */
#define ADC_SQR1_L_3                              (0x8UL<<ADC_SQR1_L_Pos)                           /*!< 0x00000008 */
#define ADC_SQR1_SQ1_Pos                          (6U)
#define ADC_SQR1_SQ1_Msk                          (0x3FUL<<ADC_SQR1_SQ1_Pos)                        /*!< 0x00000FC0 */
#define ADC_SQR1_SQ1                              ADC_SQR1_SQ1_Msk
#define ADC_SQR1_SQ1_0                            (0x1UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000040 */
#define ADC_SQR1_SQ1_1                            (0x2UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000080 */
#define ADC_SQR1_SQ1_2                            (0x4UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000100 */
#define ADC_SQR1_SQ1_3                            (0x8UL<<ADC_SQR1_SQ1_Pos)                         /*!< 0x00000200 */
#define ADC_SQR1_SQ1_4                            (0x10UL<<ADC_SQR1_SQ1_Pos)                        /*!< 0x00000400 */
#define ADC_SQR1_SQ1_5                            (0x20UL<<ADC_SQR1_SQ1_Pos)                        /*!< 0x00000800 */
#define ADC_SQR1_SQ2_Pos                          (12U)
#define ADC_SQR1_SQ2_Msk                          (0x3FUL<<ADC_SQR1_SQ2_Pos)                        /*!< 0x0003F000 */
#define ADC_SQR1_SQ2                              ADC_SQR1_SQ2_Msk
#define ADC_SQR1_SQ2_0                            (0x1UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00001000 */
#define ADC_SQR1_SQ2_1                            (0x2UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00002000 */
#define ADC_SQR1_SQ2_2                            (0x4UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00004000 */
#define ADC_SQR1_SQ2_3                            (0x8UL<<ADC_SQR1_SQ2_Pos)                         /*!< 0x00008000 */
#define ADC_SQR1_SQ2_4                            (0x10UL<<ADC_SQR1_SQ2_Pos)                        /*!< 0x00010000 */
#define ADC_SQR1_SQ2_5                            (0x20UL<<ADC_SQR1_SQ2_Pos)                        /*!< 0x00020000 */
#define ADC_SQR1_SQ3_Pos                          (18U)
#define ADC_SQR1_SQ3_Msk                          (0x3FUL<<ADC_SQR1_SQ3_Pos)                        /*!< 0x00FC0000 */
#define ADC_SQR1_SQ3                              ADC_SQR1_SQ3_Msk
#define ADC_SQR1_SQ3_0                            (0x1UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00040000 */
#define ADC_SQR1_SQ3_1                            (0x2UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00080000 */
#define ADC_SQR1_SQ3_2                            (0x4UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00100000 */
#define ADC_SQR1_SQ3_3                            (0x8UL<<ADC_SQR1_SQ3_Pos)                         /*!< 0x00200000 */
#define ADC_SQR1_SQ3_4                            (0x10UL<<ADC_SQR1_SQ3_Pos)                        /*!< 0x00400000 */
#define ADC_SQR1_SQ3_5                            (0x20UL<<ADC_SQR1_SQ3_Pos)                        /*!< 0x00800000 */
#define ADC_SQR1_SQ4_Pos                          (24U)
#define ADC_SQR1_SQ4_Msk                          (0x3FUL<<ADC_SQR1_SQ4_Pos)                        /*!< 0x3F000000 */
#define ADC_SQR1_SQ4                              ADC_SQR1_SQ4_Msk
#define ADC_SQR1_SQ4_0                            (0x1UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x01000000 */
#define ADC_SQR1_SQ4_1                            (0x2UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x02000000 */
#define ADC_SQR1_SQ4_2                            (0x4UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x04000000 */
#define ADC_SQR1_SQ4_3                            (0x8UL<<ADC_SQR1_SQ4_Pos)                         /*!< 0x08000000 */
#define ADC_SQR1_SQ4_4                            (0x10UL<<ADC_SQR1_SQ4_Pos)                        /*!< 0x10000000 */
#define ADC_SQR1_SQ4_5                            (0x20UL<<ADC_SQR1_SQ4_Pos)                        /*!< 0x20000000 */

/********************************* Bit definition for ADC_SQR2 register *********************************************/
#define ADC_SQR2_SQ5_Pos                          (0U)
#define ADC_SQR2_SQ5_Msk                          (0x3FUL<<ADC_SQR2_SQ5_Pos)                        /*!< 0x0000003F */
#define ADC_SQR2_SQ5                              ADC_SQR2_SQ5_Msk
#define ADC_SQR2_SQ5_0                            (0x1UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000001 */
#define ADC_SQR2_SQ5_1                            (0x2UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000002 */
#define ADC_SQR2_SQ5_2                            (0x4UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000004 */
#define ADC_SQR2_SQ5_3                            (0x8UL<<ADC_SQR2_SQ5_Pos)                         /*!< 0x00000008 */
#define ADC_SQR2_SQ5_4                            (0x10UL<<ADC_SQR2_SQ5_Pos)                        /*!< 0x00000010 */
#define ADC_SQR2_SQ5_5                            (0x20UL<<ADC_SQR2_SQ5_Pos)                        /*!< 0x00000020 */
#define ADC_SQR2_SQ6_Pos                          (6U)
#define ADC_SQR2_SQ6_Msk                          (0x3FUL<<ADC_SQR2_SQ6_Pos)                        /*!< 0x00000FC0 */
#define ADC_SQR2_SQ6                              ADC_SQR2_SQ6_Msk
#define ADC_SQR2_SQ6_0                            (0x1UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000040 */
#define ADC_SQR2_SQ6_1                            (0x2UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000080 */
#define ADC_SQR2_SQ6_2                            (0x4UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000100 */
#define ADC_SQR2_SQ6_3                            (0x8UL<<ADC_SQR2_SQ6_Pos)                         /*!< 0x00000200 */
#define ADC_SQR2_SQ6_4                            (0x10UL<<ADC_SQR2_SQ6_Pos)                        /*!< 0x00000400 */
#define ADC_SQR2_SQ6_5                            (0x20UL<<ADC_SQR2_SQ6_Pos)                        /*!< 0x00000800 */
#define ADC_SQR2_SQ7_Pos                          (12U)
#define ADC_SQR2_SQ7_Msk                          (0x3FUL<<ADC_SQR2_SQ7_Pos)                        /*!< 0x0003F000 */
#define ADC_SQR2_SQ7                              ADC_SQR2_SQ7_Msk
#define ADC_SQR2_SQ7_0                            (0x1UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00001000 */
#define ADC_SQR2_SQ7_1                            (0x2UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00002000 */
#define ADC_SQR2_SQ7_2                            (0x4UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00004000 */
#define ADC_SQR2_SQ7_3                            (0x8UL<<ADC_SQR2_SQ7_Pos)                         /*!< 0x00008000 */
#define ADC_SQR2_SQ7_4                            (0x10UL<<ADC_SQR2_SQ7_Pos)                        /*!< 0x00010000 */
#define ADC_SQR2_SQ7_5                            (0x20UL<<ADC_SQR2_SQ7_Pos)                        /*!< 0x00020000 */
#define ADC_SQR2_SQ8_Pos                          (18U)
#define ADC_SQR2_SQ8_Msk                          (0x3FUL<<ADC_SQR2_SQ8_Pos)                        /*!< 0x00FC0000 */
#define ADC_SQR2_SQ8                              ADC_SQR2_SQ8_Msk
#define ADC_SQR2_SQ8_0                            (0x1UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00040000 */
#define ADC_SQR2_SQ8_1                            (0x2UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00080000 */
#define ADC_SQR2_SQ8_2                            (0x4UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00100000 */
#define ADC_SQR2_SQ8_3                            (0x8UL<<ADC_SQR2_SQ8_Pos)                         /*!< 0x00200000 */
#define ADC_SQR2_SQ8_4                            (0x10UL<<ADC_SQR2_SQ8_Pos)                        /*!< 0x00400000 */
#define ADC_SQR2_SQ8_5                            (0x20UL<<ADC_SQR2_SQ8_Pos)                        /*!< 0x00800000 */
#define ADC_SQR2_SQ9_Pos                          (24U)
#define ADC_SQR2_SQ9_Msk                          (0x3FUL<<ADC_SQR2_SQ9_Pos)                        /*!< 0x3F000000 */
#define ADC_SQR2_SQ9                              ADC_SQR2_SQ9_Msk
#define ADC_SQR2_SQ9_0                            (0x1UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x01000000 */
#define ADC_SQR2_SQ9_1                            (0x2UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x02000000 */
#define ADC_SQR2_SQ9_2                            (0x4UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x04000000 */
#define ADC_SQR2_SQ9_3                            (0x8UL<<ADC_SQR2_SQ9_Pos)                         /*!< 0x08000000 */
#define ADC_SQR2_SQ9_4                            (0x10UL<<ADC_SQR2_SQ9_Pos)                        /*!< 0x10000000 */
#define ADC_SQR2_SQ9_5                            (0x20UL<<ADC_SQR2_SQ9_Pos)                        /*!< 0x20000000 */

/********************************* Bit definition for ADC_SQR3 register *********************************************/
#define ADC_SQR3_SQ10_Pos                         (0U)
#define ADC_SQR3_SQ10_Msk                         (0x3FUL<<ADC_SQR3_SQ10_Pos)                       /*!< 0x0000003F */
#define ADC_SQR3_SQ10                             ADC_SQR3_SQ10_Msk
#define ADC_SQR3_SQ10_0                           (0x1UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000001 */
#define ADC_SQR3_SQ10_1                           (0x2UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000002 */
#define ADC_SQR3_SQ10_2                           (0x4UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000004 */
#define ADC_SQR3_SQ10_3                           (0x8UL<<ADC_SQR3_SQ10_Pos)                        /*!< 0x00000008 */
#define ADC_SQR3_SQ10_4                           (0x10UL<<ADC_SQR3_SQ10_Pos)                       /*!< 0x00000010 */
#define ADC_SQR3_SQ10_5                           (0x20UL<<ADC_SQR3_SQ10_Pos)                       /*!< 0x00000020 */
#define ADC_SQR3_SQ11_Pos                         (6U)
#define ADC_SQR3_SQ11_Msk                         (0x3FUL<<ADC_SQR3_SQ11_Pos)                       /*!< 0x00000FC0 */
#define ADC_SQR3_SQ11                             ADC_SQR3_SQ11_Msk
#define ADC_SQR3_SQ11_0                           (0x1UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000040 */
#define ADC_SQR3_SQ11_1                           (0x2UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000080 */
#define ADC_SQR3_SQ11_2                           (0x4UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000100 */
#define ADC_SQR3_SQ11_3                           (0x8UL<<ADC_SQR3_SQ11_Pos)                        /*!< 0x00000200 */
#define ADC_SQR3_SQ11_4                           (0x10UL<<ADC_SQR3_SQ11_Pos)                       /*!< 0x00000400 */
#define ADC_SQR3_SQ11_5                           (0x20UL<<ADC_SQR3_SQ11_Pos)                       /*!< 0x00000800 */
#define ADC_SQR3_SQ12_Pos                         (12U)
#define ADC_SQR3_SQ12_Msk                         (0x3FUL<<ADC_SQR3_SQ12_Pos)                       /*!< 0x0003F000 */
#define ADC_SQR3_SQ12                             ADC_SQR3_SQ12_Msk
#define ADC_SQR3_SQ12_0                           (0x1UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00001000 */
#define ADC_SQR3_SQ12_1                           (0x2UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00002000 */
#define ADC_SQR3_SQ12_2                           (0x4UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00004000 */
#define ADC_SQR3_SQ12_3                           (0x8UL<<ADC_SQR3_SQ12_Pos)                        /*!< 0x00008000 */
#define ADC_SQR3_SQ12_4                           (0x10UL<<ADC_SQR3_SQ12_Pos)                       /*!< 0x00010000 */
#define ADC_SQR3_SQ12_5                           (0x20UL<<ADC_SQR3_SQ12_Pos)                       /*!< 0x00020000 */
#define ADC_SQR3_SQ13_Pos                         (18U)
#define ADC_SQR3_SQ13_Msk                         (0x3FUL<<ADC_SQR3_SQ13_Pos)                       /*!< 0x00FC0000 */
#define ADC_SQR3_SQ13                             ADC_SQR3_SQ13_Msk
#define ADC_SQR3_SQ13_0                           (0x1UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00040000 */
#define ADC_SQR3_SQ13_1                           (0x2UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00080000 */
#define ADC_SQR3_SQ13_2                           (0x4UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00100000 */
#define ADC_SQR3_SQ13_3                           (0x8UL<<ADC_SQR3_SQ13_Pos)                        /*!< 0x00200000 */
#define ADC_SQR3_SQ13_4                           (0x10UL<<ADC_SQR3_SQ13_Pos)                       /*!< 0x00400000 */
#define ADC_SQR3_SQ13_5                           (0x20UL<<ADC_SQR3_SQ13_Pos)                       /*!< 0x00800000 */
#define ADC_SQR3_SQ14_Pos                         (24U)
#define ADC_SQR3_SQ14_Msk                         (0x3FUL<<ADC_SQR3_SQ14_Pos)                       /*!< 0x3F000000 */
#define ADC_SQR3_SQ14                             ADC_SQR3_SQ14_Msk
#define ADC_SQR3_SQ14_0                           (0x1UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x01000000 */
#define ADC_SQR3_SQ14_1                           (0x2UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x02000000 */
#define ADC_SQR3_SQ14_2                           (0x4UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x04000000 */
#define ADC_SQR3_SQ14_3                           (0x8UL<<ADC_SQR3_SQ14_Pos)                        /*!< 0x08000000 */
#define ADC_SQR3_SQ14_4                           (0x10UL<<ADC_SQR3_SQ14_Pos)                       /*!< 0x10000000 */
#define ADC_SQR3_SQ14_5                           (0x20UL<<ADC_SQR3_SQ14_Pos)                       /*!< 0x20000000 */

/********************************* Bit definition for ADC_SQR4 register *********************************************/
#define ADC_SQR4_SQ15_Pos                         (0U)
#define ADC_SQR4_SQ15_Msk                         (0x3FUL<<ADC_SQR4_SQ15_Pos)                       /*!< 0x0000003F */
#define ADC_SQR4_SQ15                             ADC_SQR4_SQ15_Msk
#define ADC_SQR4_SQ15_0                           (0x1UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000001 */
#define ADC_SQR4_SQ15_1                           (0x2UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000002 */
#define ADC_SQR4_SQ15_2                           (0x4UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000004 */
#define ADC_SQR4_SQ15_3                           (0x8UL<<ADC_SQR4_SQ15_Pos)                        /*!< 0x00000008 */
#define ADC_SQR4_SQ15_4                           (0x10UL<<ADC_SQR4_SQ15_Pos)                       /*!< 0x00000010 */
#define ADC_SQR4_SQ15_5                           (0x20UL<<ADC_SQR4_SQ15_Pos)                       /*!< 0x00000020 */
#define ADC_SQR4_SQ16_Pos                         (6U)
#define ADC_SQR4_SQ16_Msk                         (0x3FUL<<ADC_SQR4_SQ16_Pos)                       /*!< 0x00000FC0 */
#define ADC_SQR4_SQ16                             ADC_SQR4_SQ16_Msk
#define ADC_SQR4_SQ16_0                           (0x1UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000040 */
#define ADC_SQR4_SQ16_1                           (0x2UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000080 */
#define ADC_SQR4_SQ16_2                           (0x4UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000100 */
#define ADC_SQR4_SQ16_3                           (0x8UL<<ADC_SQR4_SQ16_Pos)                        /*!< 0x00000200 */
#define ADC_SQR4_SQ16_4                           (0x10UL<<ADC_SQR4_SQ16_Pos)                       /*!< 0x00000400 */
#define ADC_SQR4_SQ16_5                           (0x20UL<<ADC_SQR4_SQ16_Pos)                       /*!< 0x00000800 */

/********************************* Bit definition for ADC_DR register ***********************************************/
#define ADC_DR_RDATA_Pos                          (0U)
#define ADC_DR_RDATA_Msk                          (0xFFFFUL<<ADC_DR_RDATA_Pos)                      /*!< 0x0000FFFF */
#define ADC_DR_RDATA                              ADC_DR_RDATA_Msk

/********************************* Bit definition for ADC_JSQR register *********************************************/
#define ADC_JSQR_JL_Pos                           (0U)
#define ADC_JSQR_JL_Msk                           (0x3UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000003 */
#define ADC_JSQR_JL                               ADC_JSQR_JL_Msk
#define ADC_JSQR_JL_0                             (0x1UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000001 */
#define ADC_JSQR_JL_1                             (0x2UL<<ADC_JSQR_JL_Pos)                          /*!< 0x00000002 */
#define ADC_JSQR_JEXTSEL_Pos                      (2U)
#define ADC_JSQR_JEXTSEL_Msk                      (0xFUL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x0000003C */
#define ADC_JSQR_JEXTSEL                          ADC_JSQR_JEXTSEL_Msk
#define ADC_JSQR_JEXTSEL_0                        (0x1UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000004 */
#define ADC_JSQR_JEXTSEL_1                        (0x2UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000008 */
#define ADC_JSQR_JEXTSEL_2                        (0x4UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000010 */
#define ADC_JSQR_JEXTSEL_3                        (0x8UL<<ADC_JSQR_JEXTSEL_Pos)                     /*!< 0x00000020 */
#define ADC_JSQR_JEXTEN_Pos                       (6U)
#define ADC_JSQR_JEXTEN_Msk                       (0x3UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x000000C0 */
#define ADC_JSQR_JEXTEN                           ADC_JSQR_JEXTEN_Msk
#define ADC_JSQR_JEXTEN_0                         (0x1UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x00000040 */
#define ADC_JSQR_JEXTEN_1                         (0x2UL<<ADC_JSQR_JEXTEN_Pos)                      /*!< 0x00000080 */
#define ADC_JSQR_JSQ1_Pos                         (8U)
#define ADC_JSQR_JSQ1_Msk                         (0x3FUL<<ADC_JSQR_JSQ1_Pos)                       /*!< 0x00003F00 */
#define ADC_JSQR_JSQ1                             ADC_JSQR_JSQ1_Msk
#define ADC_JSQR_JSQ1_0                           (0x1UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000100 */
#define ADC_JSQR_JSQ1_1                           (0x2UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000200 */
#define ADC_JSQR_JSQ1_2                           (0x4UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000400 */
#define ADC_JSQR_JSQ1_3                           (0x8UL<<ADC_JSQR_JSQ1_Pos)                        /*!< 0x00000800 */
#define ADC_JSQR_JSQ1_4                           (0x10UL<<ADC_JSQR_JSQ1_Pos)                       /*!< 0x00001000 */
#define ADC_JSQR_JSQ1_5                           (0x20UL<<ADC_JSQR_JSQ1_Pos)                       /*!< 0x00002000 */
#define ADC_JSQR_JSQ2_Pos                         (14U)
#define ADC_JSQR_JSQ2_Msk                         (0x3FUL<<ADC_JSQR_JSQ2_Pos)                       /*!< 0x000FC000 */
#define ADC_JSQR_JSQ2                             ADC_JSQR_JSQ2_Msk
#define ADC_JSQR_JSQ2_0                           (0x1UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00004000 */
#define ADC_JSQR_JSQ2_1                           (0x2UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00008000 */
#define ADC_JSQR_JSQ2_2                           (0x4UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00010000 */
#define ADC_JSQR_JSQ2_3                           (0x8UL<<ADC_JSQR_JSQ2_Pos)                        /*!< 0x00020000 */
#define ADC_JSQR_JSQ2_4                           (0x10UL<<ADC_JSQR_JSQ2_Pos)                       /*!< 0x00040000 */
#define ADC_JSQR_JSQ2_5                           (0x20UL<<ADC_JSQR_JSQ2_Pos)                       /*!< 0x00080000 */
#define ADC_JSQR_JSQ3_Pos                         (20U)
#define ADC_JSQR_JSQ3_Msk                         (0x3FUL<<ADC_JSQR_JSQ3_Pos)                       /*!< 0x03F00000 */
#define ADC_JSQR_JSQ3                             ADC_JSQR_JSQ3_Msk
#define ADC_JSQR_JSQ3_0                           (0x1UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00100000 */
#define ADC_JSQR_JSQ3_1                           (0x2UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00200000 */
#define ADC_JSQR_JSQ3_2                           (0x4UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00400000 */
#define ADC_JSQR_JSQ3_3                           (0x8UL<<ADC_JSQR_JSQ3_Pos)                        /*!< 0x00800000 */
#define ADC_JSQR_JSQ3_4                           (0x10UL<<ADC_JSQR_JSQ3_Pos)                       /*!< 0x01000000 */
#define ADC_JSQR_JSQ3_5                           (0x20UL<<ADC_JSQR_JSQ3_Pos)                       /*!< 0x02000000 */
#define ADC_JSQR_JSQ4_Pos                         (26U)
#define ADC_JSQR_JSQ4_Msk                         (0x3FUL<<ADC_JSQR_JSQ4_Pos)                       /*!< 0xFC000000 */
#define ADC_JSQR_JSQ4                             ADC_JSQR_JSQ4_Msk
#define ADC_JSQR_JSQ4_0                           (0x1UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x04000000 */
#define ADC_JSQR_JSQ4_1                           (0x2UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x08000000 */
#define ADC_JSQR_JSQ4_2                           (0x4UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x10000000 */
#define ADC_JSQR_JSQ4_3                           (0x8UL<<ADC_JSQR_JSQ4_Pos)                        /*!< 0x20000000 */
#define ADC_JSQR_JSQ4_4                           (0x10UL<<ADC_JSQR_JSQ4_Pos)                       /*!< 0x40000000 */
#define ADC_JSQR_JSQ4_5                           (0x20UL<<ADC_JSQR_JSQ4_Pos)                       /*!< 0x80000000 */

/********************************* Bit definition for ADC_OFR1 register *********************************************/
#define ADC_OFR1_OFFSET1_Pos                      (0U)
#define ADC_OFR1_OFFSET1_Msk                      (0xFFFUL<<ADC_OFR1_OFFSET1_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR1_OFFSET1                          ADC_OFR1_OFFSET1_Msk
#define ADC_OFR1_OFFSETPOS_Pos                    (23U)
#define ADC_OFR1_OFFSETPOS_Msk                    (0x1UL<<ADC_OFR1_OFFSETPOS_Pos)                   /*!< 0x00800000 */
#define ADC_OFR1_OFFSETPOS                        ADC_OFR1_OFFSETPOS_Msk                            
#define ADC_OFR1_SATEN_Pos                        (24U)
#define ADC_OFR1_SATEN_Msk                        (0x1UL<<ADC_OFR1_SATEN_Pos)                       /*!< 0x01000000 */
#define ADC_OFR1_SATEN                            ADC_OFR1_SATEN_Msk                                
#define ADC_OFR1_OFFSET1_CH_Pos                   (25U)
#define ADC_OFR1_OFFSET1_CH_Msk                   (0x3FUL<<ADC_OFR1_OFFSET1_CH_Pos)                 /*!< 0x7E000000 */
#define ADC_OFR1_OFFSET1_CH                       ADC_OFR1_OFFSET1_CH_Msk
#define ADC_OFR1_OFFSET1_CH_0                     (0x1UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x02000000 */
#define ADC_OFR1_OFFSET1_CH_1                     (0x2UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR1_OFFSET1_CH_2                     (0x4UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR1_OFFSET1_CH_3                     (0x8UL<<ADC_OFR1_OFFSET1_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR1_OFFSET1_CH_4                     (0x10UL<<ADC_OFR1_OFFSET1_CH_Pos)                 /*!< 0x20000000 */
#define ADC_OFR1_OFFSET1_CH_5                     (0x20UL<<ADC_OFR1_OFFSET1_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR1_OFFSET1_EN_Pos                   (31U)
#define ADC_OFR1_OFFSET1_EN_Msk                   (0x1UL<<ADC_OFR1_OFFSET1_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR1_OFFSET1_EN                       ADC_OFR1_OFFSET1_EN_Msk                           

/********************************* Bit definition for ADC_OFR2 register *********************************************/
#define ADC_OFR2_OFFSET2_Pos                      (0U)
#define ADC_OFR2_OFFSET2_Msk                      (0xFFFUL<<ADC_OFR2_OFFSET2_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR2_OFFSET2                          ADC_OFR2_OFFSET2_Msk
#define ADC_OFR2_OFFSETPOS_Pos                    (23U)
#define ADC_OFR2_OFFSETPOS_Msk                    (0x1UL<<ADC_OFR2_OFFSETPOS_Pos)                   /*!< 0x00800000 */
#define ADC_OFR2_OFFSETPOS                        ADC_OFR2_OFFSETPOS_Msk                            
#define ADC_OFR2_SATEN_Pos                        (24U)
#define ADC_OFR2_SATEN_Msk                        (0x1UL<<ADC_OFR2_SATEN_Pos)                       /*!< 0x01000000 */
#define ADC_OFR2_SATEN                            ADC_OFR2_SATEN_Msk                                
#define ADC_OFR2_OFFSET2_CH_Pos                   (25U)
#define ADC_OFR2_OFFSET2_CH_Msk                   (0x3FUL<<ADC_OFR2_OFFSET2_CH_Pos)                 /*!< 0x7E000000 */
#define ADC_OFR2_OFFSET2_CH                       ADC_OFR2_OFFSET2_CH_Msk
#define ADC_OFR2_OFFSET2_CH_0                     (0x1UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x02000000 */
#define ADC_OFR2_OFFSET2_CH_1                     (0x2UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR2_OFFSET2_CH_2                     (0x4UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR2_OFFSET2_CH_3                     (0x8UL<<ADC_OFR2_OFFSET2_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR2_OFFSET2_CH_4                     (0x10UL<<ADC_OFR2_OFFSET2_CH_Pos)                 /*!< 0x20000000 */
#define ADC_OFR2_OFFSET2_CH_5                     (0x20UL<<ADC_OFR2_OFFSET2_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR2_OFFSET2_EN_Pos                   (31U)
#define ADC_OFR2_OFFSET2_EN_Msk                   (0x1UL<<ADC_OFR2_OFFSET2_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR2_OFFSET2_EN                       ADC_OFR2_OFFSET2_EN_Msk                           

/********************************* Bit definition for ADC_OFR3 register *********************************************/
#define ADC_OFR3_OFFSET3_Pos                      (0U)
#define ADC_OFR3_OFFSET3_Msk                      (0xFFFUL<<ADC_OFR3_OFFSET3_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR3_OFFSET3                          ADC_OFR3_OFFSET3_Msk
#define ADC_OFR3_OFFSETPOS_Pos                    (23U)
#define ADC_OFR3_OFFSETPOS_Msk                    (0x1UL<<ADC_OFR3_OFFSETPOS_Pos)                   /*!< 0x00800000 */
#define ADC_OFR3_OFFSETPOS                        ADC_OFR3_OFFSETPOS_Msk                            
#define ADC_OFR3_SATEN_Pos                        (24U)
#define ADC_OFR3_SATEN_Msk                        (0x1UL<<ADC_OFR3_SATEN_Pos)                       /*!< 0x01000000 */
#define ADC_OFR3_SATEN                            ADC_OFR3_SATEN_Msk                                
#define ADC_OFR3_OFFSET3_CH_Pos                   (25U)
#define ADC_OFR3_OFFSET3_CH_Msk                   (0x3FUL<<ADC_OFR3_OFFSET3_CH_Pos)                 /*!< 0x7E000000 */
#define ADC_OFR3_OFFSET3_CH                       ADC_OFR3_OFFSET3_CH_Msk
#define ADC_OFR3_OFFSET3_CH_0                     (0x1UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x02000000 */
#define ADC_OFR3_OFFSET3_CH_1                     (0x2UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR3_OFFSET3_CH_2                     (0x4UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR3_OFFSET3_CH_3                     (0x8UL<<ADC_OFR3_OFFSET3_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR3_OFFSET3_CH_4                     (0x10UL<<ADC_OFR3_OFFSET3_CH_Pos)                 /*!< 0x20000000 */
#define ADC_OFR3_OFFSET3_CH_5                     (0x20UL<<ADC_OFR3_OFFSET3_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR3_OFFSET3_EN_Pos                   (31U)
#define ADC_OFR3_OFFSET3_EN_Msk                   (0x1UL<<ADC_OFR3_OFFSET3_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR3_OFFSET3_EN                       ADC_OFR3_OFFSET3_EN_Msk                           

/********************************* Bit definition for ADC_OFR4 register *********************************************/
#define ADC_OFR4_OFFSET4_Pos                      (0U)
#define ADC_OFR4_OFFSET4_Msk                      (0xFFFUL<<ADC_OFR4_OFFSET4_Pos)                   /*!< 0x00000FFF */
#define ADC_OFR4_OFFSET4                          ADC_OFR4_OFFSET4_Msk
#define ADC_OFR4_OFFSETPOS_Pos                    (23U)
#define ADC_OFR4_OFFSETPOS_Msk                    (0x1UL<<ADC_OFR4_OFFSETPOS_Pos)                   /*!< 0x00800000 */
#define ADC_OFR4_OFFSETPOS                        ADC_OFR4_OFFSETPOS_Msk                            
#define ADC_OFR4_SATEN_Pos                        (24U)
#define ADC_OFR4_SATEN_Msk                        (0x1UL<<ADC_OFR4_SATEN_Pos)                       /*!< 0x01000000 */
#define ADC_OFR4_SATEN                            ADC_OFR4_SATEN_Msk                                
#define ADC_OFR4_OFFSET4_CH_Pos                   (25U)
#define ADC_OFR4_OFFSET4_CH_Msk                   (0x3FUL<<ADC_OFR4_OFFSET4_CH_Pos)                 /*!< 0x7E000000 */
#define ADC_OFR4_OFFSET4_CH                       ADC_OFR4_OFFSET4_CH_Msk
#define ADC_OFR4_OFFSET4_CH_0                     (0x1UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x02000000 */
#define ADC_OFR4_OFFSET4_CH_1                     (0x2UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x04000000 */
#define ADC_OFR4_OFFSET4_CH_2                     (0x4UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x08000000 */
#define ADC_OFR4_OFFSET4_CH_3                     (0x8UL<<ADC_OFR4_OFFSET4_CH_Pos)                  /*!< 0x10000000 */
#define ADC_OFR4_OFFSET4_CH_4                     (0x10UL<<ADC_OFR4_OFFSET4_CH_Pos)                 /*!< 0x20000000 */
#define ADC_OFR4_OFFSET4_CH_5                     (0x20UL<<ADC_OFR4_OFFSET4_CH_Pos)                 /*!< 0x40000000 */
#define ADC_OFR4_OFFSET4_EN_Pos                   (31U)
#define ADC_OFR4_OFFSET4_EN_Msk                   (0x1UL<<ADC_OFR4_OFFSET4_EN_Pos)                  /*!< 0x80000000 */
#define ADC_OFR4_OFFSET4_EN                       ADC_OFR4_OFFSET4_EN_Msk                           

/********************************* Bit definition for ADC_JDR1 register *********************************************/
#define ADC_JDR1_JDATA_Pos                        (0U)
#define ADC_JDR1_JDATA_Msk                        (0xFFFFUL<<ADC_JDR1_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR1_JDATA                            ADC_JDR1_JDATA_Msk

/********************************* Bit definition for ADC_JDR2 register *********************************************/
#define ADC_JDR2_JDATA_Pos                        (0U)
#define ADC_JDR2_JDATA_Msk                        (0xFFFFUL<<ADC_JDR2_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR2_JDATA                            ADC_JDR2_JDATA_Msk

/********************************* Bit definition for ADC_JDR3 register *********************************************/
#define ADC_JDR3_JDATA_Pos                        (0U)
#define ADC_JDR3_JDATA_Msk                        (0xFFFFUL<<ADC_JDR3_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR3_JDATA                            ADC_JDR3_JDATA_Msk

/********************************* Bit definition for ADC_JDR4 register *********************************************/
#define ADC_JDR4_JDATA_Pos                        (0U)
#define ADC_JDR4_JDATA_Msk                        (0xFFFFUL<<ADC_JDR4_JDATA_Pos)                    /*!< 0x0000FFFF */
#define ADC_JDR4_JDATA                            ADC_JDR4_JDATA_Msk

/********************************* Bit definition for ADC_CALFACT register ******************************************/
#define ADC_CALFACT_OFFSUC_Pos                    (20U)
#define ADC_CALFACT_OFFSUC_Msk                    (0x1UL<<ADC_CALFACT_OFFSUC_Pos)                   /*!< 0x00100000 */
#define ADC_CALFACT_OFFSUC                        ADC_CALFACT_OFFSUC_Msk                            
#define ADC_CALFACT_CAPSUC_Pos                    (21U)
#define ADC_CALFACT_CAPSUC_Msk                    (0x1UL<<ADC_CALFACT_CAPSUC_Pos)                   /*!< 0x00200000 */
#define ADC_CALFACT_CAPSUC                        ADC_CALFACT_CAPSUC_Msk                            

/********************************* Bit definition for ADC_CCR register *********************************************/
#define ADC_CCR_VREFBUFEN_Pos                    (5U)
#define ADC_CCR_VREFBUFEN_Msk                    (0x1UL<<ADC_CCR_VREFBUFEN_Pos)                   /*!< 0x00000020 */
#define ADC_CCR_VREFBUFEN                        ADC_CCR_VREFBUFEN_Msk                            
#define ADC_CCR_VREFBUFSEL_Pos                   (6U)
#define ADC_CCR_VREFBUFSEL_Msk                   (0x7UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x000001C0 */
#define ADC_CCR_VREFBUFSEL                       ADC_CCR_VREFBUFSEL_Msk
#define ADC_CCR_VREFBUFSEL_0                     (0x1UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x00000040 */
#define ADC_CCR_VREFBUFSEL_1                     (0x2UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x00000080 */
#define ADC_CCR_VREFBUFSEL_2                     (0x4UL<<ADC_CCR_VREFBUFSEL_Pos)                  /*!< 0x00000100 */
#define ADC_CCR_CKMODE_Pos                       (16U)
#define ADC_CCR_CKMODE_Msk                       (0x3UL<<ADC_CCR_CKMODE_Pos)                      /*!< 0x00030000 */
#define ADC_CCR_CKMODE                           ADC_CCR_CKMODE_Msk
#define ADC_CCR_CKMODE_0                         (0x1UL<<ADC_CCR_CKMODE_Pos)                      /*!< 0x00010000 */
#define ADC_CCR_CKMODE_1                         (0x2UL<<ADC_CCR_CKMODE_Pos)                      /*!< 0x00020000 */
#define ADC_CCR_PRESC_Pos                        (18U)
#define ADC_CCR_PRESC_Msk                        (0xFUL<<ADC_CCR_PRESC_Pos)                       /*!< 0x003C0000 */
#define ADC_CCR_PRESC                            ADC_CCR_PRESC_Msk
#define ADC_CCR_PRESC_0                          (0x1UL<<ADC_CCR_PRESC_Pos)                       /*!< 0x00040000 */
#define ADC_CCR_PRESC_1                          (0x2UL<<ADC_CCR_PRESC_Pos)                       /*!< 0x00080000 */
#define ADC_CCR_PRESC_2                          (0x4UL<<ADC_CCR_PRESC_Pos)                       /*!< 0x00100000 */
#define ADC_CCR_PRESC_3                          (0x8UL<<ADC_CCR_PRESC_Pos)                       /*!< 0x00200000 */
#define ADC_CCR_VREFEN_Pos                       (22U)
#define ADC_CCR_VREFEN_Msk                       (0x1UL<<ADC_CCR_VREFEN_Pos)                      /*!< 0x00400000 */
#define ADC_CCR_VREFEN                           ADC_CCR_VREFEN_Msk                               
#define ADC_CCR_VSENSESEL_Pos                    (23U)
#define ADC_CCR_VSENSESEL_Msk                    (0x1UL<<ADC_CCR_VSENSESEL_Pos)                   /*!< 0x00800000 */
#define ADC_CCR_VSENSESEL                        ADC_CCR_VSENSESEL_Msk                            

/********************************************************************************************************************/
/********************************* BKP ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for TAMP_CR1 register *********************************************/
#define TAMP_CR1_TAMP1EN_Pos                      (0U)
#define TAMP_CR1_TAMP1EN_Msk                      (0x1UL<<TAMP_CR1_TAMP1EN_Pos)                     /*!< 0x00000001 */
#define TAMP_CR1_TAMP1EN                          TAMP_CR1_TAMP1EN_Msk                              /*!< TAMP enable */
#define TAMP_CR1_ITAMP1EN_Pos                     (16U)
#define TAMP_CR1_ITAMP1EN_Msk                     (0x1UL<<TAMP_CR1_ITAMP1EN_Pos)                    /*!< 0x00010000 */
#define TAMP_CR1_ITAMP1EN                         TAMP_CR1_ITAMP1EN_Msk                             /*!< TAMPER pin interrupt enable */

/********************************* Bit definition for TAMP_CR2 register *********************************************/
#define TAMP_CR2_TAMP1NOER_Pos                    (0U)
#define TAMP_CR2_TAMP1NOER_Msk                    (0x1UL<<TAMP_CR2_TAMP1NOER_Pos)                   /*!< 0x00000001 */
#define TAMP_CR2_TAMP1NOER                        TAMP_CR2_TAMP1NOER_Msk                            /*!< TAMP enable */
#define TAMP_CR2_TAMP1MSK_Pos                     (16U)
#define TAMP_CR2_TAMP1MSK_Msk                     (0x1UL<<TAMP_CR2_TAMP1MSK_Pos)                    /*!< 0x00010000 */
#define TAMP_CR2_TAMP1MSK                         TAMP_CR2_TAMP1MSK_Msk                             /*!< TAMP enable */
#define TAMP_CR2_TAMP1TRG_Pos                     (24U)
#define TAMP_CR2_TAMP1TRG_Msk                     (0x1UL<<TAMP_CR2_TAMP1TRG_Pos)                    /*!< 0x01000000 */
#define TAMP_CR2_TAMP1TRG                         TAMP_CR2_TAMP1TRG_Msk                             /*!< TAMPER pin interrupt enable */

/********************************* Bit definition for TAMP_FLTCR register *******************************************/
#define TAMP_FLTCR_TAMPFREQ_Pos                   (0U)
#define TAMP_FLTCR_TAMPFREQ_Msk                   (0x7UL<<TAMP_FLTCR_TAMPFREQ_Pos)                  /*!< 0x00000007 */
#define TAMP_FLTCR_TAMPFREQ                       TAMP_FLTCR_TAMPFREQ_Msk
#define TAMP_FLTCR_TAMPFREQ_0                     (0x1UL<<TAMP_FLTCR_TAMPFREQ_Pos)                  /*!< 0x00000001 */
#define TAMP_FLTCR_TAMPFREQ_1                     (0x2UL<<TAMP_FLTCR_TAMPFREQ_Pos)                  /*!< 0x00000002 */
#define TAMP_FLTCR_TAMPFREQ_2                     (0x4UL<<TAMP_FLTCR_TAMPFREQ_Pos)                  /*!< 0x00000004 */
#define TAMP_FLTCR_RAMPFLT_Pos                    (3U)
#define TAMP_FLTCR_RAMPFLT_Msk                    (0x3UL<<TAMP_FLTCR_RAMPFLT_Pos)                   /*!< 0x00000018 */
#define TAMP_FLTCR_RAMPFLT                        TAMP_FLTCR_RAMPFLT_Msk
#define TAMP_FLTCR_RAMPFLT_0                      (0x1UL<<TAMP_FLTCR_RAMPFLT_Pos)                   /*!< 0x00000008 */
#define TAMP_FLTCR_RAMPFLT_1                      (0x2UL<<TAMP_FLTCR_RAMPFLT_Pos)                   /*!< 0x00000010 */
#define TAMP_FLTCR_RAMPPRCH_Pos                   (5U)
#define TAMP_FLTCR_RAMPPRCH_Msk                   (0x3UL<<TAMP_FLTCR_RAMPPRCH_Pos)                  /*!< 0x00000060 */
#define TAMP_FLTCR_RAMPPRCH                       TAMP_FLTCR_RAMPPRCH_Msk                           /*!< TAMP event flag */
#define TAMP_FLTCR_RAMPPRCH_0                     (0x1UL<<TAMP_FLTCR_RAMPPRCH_Pos)                  /*!< 0x00000020 */
#define TAMP_FLTCR_RAMPPRCH_1                     (0x2UL<<TAMP_FLTCR_RAMPPRCH_Pos)                  /*!< 0x00000040 */
#define TAMP_FLTCR_RAMPPUDIS_Pos                  (7U)
#define TAMP_FLTCR_RAMPPUDIS_Msk                  (0x1UL<<TAMP_FLTCR_RAMPPUDIS_Pos)                 /*!< 0x00000080 */
#define TAMP_FLTCR_RAMPPUDIS                      TAMP_FLTCR_RAMPPUDIS_Msk                          /*!< �� */

/********************************* Bit definition for TAMP_IER register *********************************************/
#define TAMP_IER_TAMP1IE_Pos                      (0U)
#define TAMP_IER_TAMP1IE_Msk                      (0x1UL<<TAMP_IER_TAMP1IE_Pos)                     /*!< 0x00000001 */
#define TAMP_IER_TAMP1IE                          TAMP_IER_TAMP1IE_Msk                              
#define TAMP_IER_ITAMP1IE_Pos                     (16U)
#define TAMP_IER_ITAMP1IE_Msk                     (0x1UL<<TAMP_IER_ITAMP1IE_Pos)                    /*!< 0x00010000 */
#define TAMP_IER_ITAMP1IE                         TAMP_IER_ITAMP1IE_Msk                             

/********************************* Bit definition for TAMP_SR register **********************************************/
#define TAMP_SR_TAMP1IF_Pos                       (0U)
#define TAMP_SR_TAMP1IF_Msk                       (0x1UL<<TAMP_SR_TAMP1IF_Pos)                      /*!< 0x00000001 */
#define TAMP_SR_TAMP1IF                           TAMP_SR_TAMP1IF_Msk                               
#define TAMP_SR_ITAMP1F_Pos                       (16U)
#define TAMP_SR_ITAMP1F_Msk                       (0x1UL<<TAMP_SR_ITAMP1F_Pos)                      /*!< 0x00010000 */
#define TAMP_SR_ITAMP1F                           TAMP_SR_ITAMP1F_Msk                               

/********************************* Bit definition for TAMP_MISR register ********************************************/
#define TAMP_MISR_TAMP1MF_Pos                     (0U)
#define TAMP_MISR_TAMP1MF_Msk                     (0x1UL<<TAMP_MISR_TAMP1MF_Pos)                    /*!< 0x00000001 */
#define TAMP_MISR_TAMP1MF                         TAMP_MISR_TAMP1MF_Msk                             
#define TAMP_MISR_ITAMP1MF_Pos                    (16U)
#define TAMP_MISR_ITAMP1MF_Msk                    (0x1UL<<TAMP_MISR_ITAMP1MF_Pos)                   /*!< 0x00010000 */
#define TAMP_MISR_ITAMP1MF                        TAMP_MISR_ITAMP1MF_Msk                            

/********************************* Bit definition for TAMP_SCR register *********************************************/
#define TAMP_SCR_CTAMP1F_Pos                      (0U)
#define TAMP_SCR_CTAMP1F_Msk                      (0x1UL<<TAMP_SCR_CTAMP1F_Pos)                     /*!< 0x00000001 */
#define TAMP_SCR_CTAMP1F                          TAMP_SCR_CTAMP1F_Msk                              
#define TAMP_SCR_CITAMP1F_Pos                     (16U)
#define TAMP_SCR_CITAMP1F_Msk                     (0x1UL<<TAMP_SCR_CITAMP1F_Pos)                    /*!< 0x00010000 */
#define TAMP_SCR_CITAMP1F                         TAMP_SCR_CITAMP1F_Msk                             

/********************************* Bit definition for TAMP_BKP0R register *******************************************/
#define TAMP_BKP0R_D_Pos                          (0U)
#define TAMP_BKP0R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP0R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP0R_D                              TAMP_BKP0R_D_Msk                                  /*!< bakeup data word0 */

/********************************* Bit definition for TAMP_BKP1R register *******************************************/
#define TAMP_BKP1R_D_Pos                          (0U)
#define TAMP_BKP1R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP1R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP1R_D                              TAMP_BKP1R_D_Msk                                  /*!< bakeup data word1 */

/********************************* Bit definition for TAMP_BKP2R register *******************************************/
#define TAMP_BKP2R_D_Pos                          (0U)
#define TAMP_BKP2R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP2R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP2R_D                              TAMP_BKP2R_D_Msk                                  /*!< bakeup data word2 */

/********************************* Bit definition for TAMP_BKP3R register *******************************************/
#define TAMP_BKP3R_D_Pos                          (0U)
#define TAMP_BKP3R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP3R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP3R_D                              TAMP_BKP3R_D_Msk                                  /*!< bakeup data word3 */

/********************************* Bit definition for TAMP_BKP4R register *******************************************/
#define TAMP_BKP4R_D_Pos                          (0U)
#define TAMP_BKP4R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP4R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP4R_D                              TAMP_BKP4R_D_Msk                                  /*!< bakeup data word4 */

/********************************* Bit definition for TAMP_BKP5R register *******************************************/
#define TAMP_BKP5R_D_Pos                          (0U)
#define TAMP_BKP5R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP5R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP5R_D                              TAMP_BKP5R_D_Msk                                  /*!< bakeup data word5 */

/********************************* Bit definition for TAMP_BKP6R register *******************************************/
#define TAMP_BKP6R_D_Pos                          (0U)
#define TAMP_BKP6R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP6R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP6R_D                              TAMP_BKP6R_D_Msk                                  /*!< bakeup data word6 */

/********************************* Bit definition for TAMP_BKP7R register *******************************************/
#define TAMP_BKP7R_D_Pos                          (0U)
#define TAMP_BKP7R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP7R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP7R_D                              TAMP_BKP7R_D_Msk                                  /*!< bakeup data word7 */

/********************************* Bit definition for TAMP_BKP8R register *******************************************/
#define TAMP_BKP8R_D_Pos                          (0U)
#define TAMP_BKP8R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP8R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP8R_D                              TAMP_BKP8R_D_Msk                                  /*!< bakeup data word8 */

/********************************* Bit definition for TAMP_BKP9R register *******************************************/
#define TAMP_BKP9R_D_Pos                          (0U)
#define TAMP_BKP9R_D_Msk                          (0xFFFFFFFFUL<<TAMP_BKP9R_D_Pos)                  /*!< 0xFFFFFFFF */
#define TAMP_BKP9R_D                              TAMP_BKP9R_D_Msk                                  /*!< bakeup data word9 */

/********************************* Bit definition for TAMP_BKP10R register ******************************************/
#define TAMP_BKP10R_D_Pos                         (0U)
#define TAMP_BKP10R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP10R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP10R_D                             TAMP_BKP10R_D_Msk                                 /*!< bakeup data word10 */

/********************************* Bit definition for TAMP_BKP11R register ******************************************/
#define TAMP_BKP11R_D_Pos                         (0U)
#define TAMP_BKP11R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP11R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP11R_D                             TAMP_BKP11R_D_Msk                                 /*!< bakeup data word11 */

/********************************* Bit definition for TAMP_BKP12R register ******************************************/
#define TAMP_BKP12R_D_Pos                         (0U)
#define TAMP_BKP12R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP12R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP12R_D                             TAMP_BKP12R_D_Msk                                 /*!< bakeup data word12 */

/********************************* Bit definition for TAMP_BKP13R register ******************************************/
#define TAMP_BKP13R_D_Pos                         (0U)
#define TAMP_BKP13R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP13R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP13R_D                             TAMP_BKP13R_D_Msk                                 /*!< bakeup data word13 */

/********************************* Bit definition for TAMP_BKP14R register ******************************************/
#define TAMP_BKP14R_D_Pos                         (0U)
#define TAMP_BKP14R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP14R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP14R_D                             TAMP_BKP14R_D_Msk                                 /*!< bakeup data word14 */

/********************************* Bit definition for TAMP_BKP15R register ******************************************/
#define TAMP_BKP15R_D_Pos                         (0U)
#define TAMP_BKP15R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP15R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP15R_D                             TAMP_BKP15R_D_Msk                                 /*!< bakeup data word15 */

/********************************* Bit definition for TAMP_BKP16R register ******************************************/
#define TAMP_BKP16R_D_Pos                         (0U)
#define TAMP_BKP16R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP16R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP16R_D                             TAMP_BKP16R_D_Msk                                 /*!< bakeup data word16 */

/********************************* Bit definition for TAMP_BKP17R register ******************************************/
#define TAMP_BKP17R_D_Pos                         (0U)
#define TAMP_BKP17R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP17R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP17R_D                             TAMP_BKP17R_D_Msk                                 /*!< bakeup data word17 */

/********************************* Bit definition for TAMP_BKP18R register ******************************************/
#define TAMP_BKP18R_D_Pos                         (0U)
#define TAMP_BKP18R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP18R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP18R_D                             TAMP_BKP18R_D_Msk                                 /*!< bakeup data word18 */

/********************************* Bit definition for TAMP_BKP19R register ******************************************/
#define TAMP_BKP19R_D_Pos                         (0U)
#define TAMP_BKP19R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP19R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP19R_D                             TAMP_BKP19R_D_Msk                                 /*!< bakeup data word19 */

/********************************* Bit definition for TAMP_BKP20R register ******************************************/
#define TAMP_BKP20R_D_Pos                         (0U)
#define TAMP_BKP20R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP20R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP20R_D                             TAMP_BKP20R_D_Msk                                 /*!< bakeup data word20 */

/********************************* Bit definition for TAMP_BKP21R register ******************************************/
#define TAMP_BKP21R_D_Pos                         (0U)
#define TAMP_BKP21R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP21R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP21R_D                             TAMP_BKP21R_D_Msk                                 /*!< bakeup data word21 */

/********************************* Bit definition for TAMP_BKP22R register ******************************************/
#define TAMP_BKP22R_D_Pos                         (0U)
#define TAMP_BKP22R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP22R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP22R_D                             TAMP_BKP22R_D_Msk                                 /*!< bakeup data word22 */

/********************************* Bit definition for TAMP_BKP23R register ******************************************/
#define TAMP_BKP23R_D_Pos                         (0U)
#define TAMP_BKP23R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP23R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP23R_D                             TAMP_BKP23R_D_Msk                                 /*!< bakeup data word23 */

/********************************* Bit definition for TAMP_BKP24R register ******************************************/
#define TAMP_BKP24R_D_Pos                         (0U)
#define TAMP_BKP24R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP24R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP24R_D                             TAMP_BKP24R_D_Msk                                 /*!< bakeup data word24 */

/********************************* Bit definition for TAMP_BKP25R register ******************************************/
#define TAMP_BKP25R_D_Pos                         (0U)
#define TAMP_BKP25R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP25R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP25R_D                             TAMP_BKP25R_D_Msk                                 /*!< bakeup data word25 */

/********************************* Bit definition for TAMP_BKP26R register ******************************************/
#define TAMP_BKP26R_D_Pos                         (0U)
#define TAMP_BKP26R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP26R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP26R_D                             TAMP_BKP26R_D_Msk                                 /*!< bakeup data word26 */

/********************************* Bit definition for TAMP_BKP27R register ******************************************/
#define TAMP_BKP27R_D_Pos                         (0U)
#define TAMP_BKP27R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP27R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP27R_D                             TAMP_BKP27R_D_Msk                                 /*!< bakeup data word27 */

/********************************* Bit definition for TAMP_BKP28R register ******************************************/
#define TAMP_BKP28R_D_Pos                         (0U)
#define TAMP_BKP28R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP28R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP28R_D                             TAMP_BKP28R_D_Msk                                 /*!< bakeup data word28 */

/********************************* Bit definition for TAMP_BKP29R register ******************************************/
#define TAMP_BKP29R_D_Pos                         (0U)
#define TAMP_BKP29R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP29R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP29R_D                             TAMP_BKP29R_D_Msk                                 /*!< bakeup data word29 */

/********************************* Bit definition for TAMP_BKP30R register ******************************************/
#define TAMP_BKP30R_D_Pos                         (0U)
#define TAMP_BKP30R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP30R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP30R_D                             TAMP_BKP30R_D_Msk                                 /*!< bakeup data word30 */

/********************************* Bit definition for TAMP_BKP31R register ******************************************/
#define TAMP_BKP31R_D_Pos                         (0U)
#define TAMP_BKP31R_D_Msk                         (0xFFFFFFFFUL<<TAMP_BKP31R_D_Pos)                 /*!< 0xFFFFFFFF */
#define TAMP_BKP31R_D                             TAMP_BKP31R_D_Msk                                 /*!< bakeup data word31 */

/********************************************************************************************************************/
/********************************* COMP *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for COMP_CSR register ********************************************/
#define COMP_CSR_EN_Pos                          (0U)
#define COMP_CSR_EN_Msk                          (0x1UL<<COMP_CSR_EN_Pos)                         /*!< 0x00000001 */
#define COMP_CSR_EN                              COMP_CSR_EN_Msk                                  /*!< COMP1 enable bit */
#define COMP_CSR_COMP1_EN                        COMP_CSR_EN
#define COMP_CSR_COMP2_EN                        COMP_CSR_EN
#define COMP_CSR_INMSEL_Pos                      (2U)
#define COMP_CSR_INMSEL_Msk                      (0x7UL<<COMP_CSR_INMSEL_Pos)                     /*!< 0x0000001C */
#define COMP_CSR_INMSEL                          COMP_CSR_INMSEL_Msk                              /*!< COMP1 negative input select */
#define COMP_CSR_INMSEL_0                        (0x1UL<<COMP_CSR_INMSEL_Pos)                     /*!< 0x00000004 */
#define COMP_CSR_INMSEL_1                        (0x2UL<<COMP_CSR_INMSEL_Pos)                     /*!< 0x00000008 */
#define COMP_CSR_INMSEL_2                        (0x4UL<<COMP_CSR_INMSEL_Pos)                     /*!< 0x00000010 */
#define COMP_CSR_INPSEL_Pos                      (5U)
#define COMP_CSR_INPSEL_Msk                      (0x7UL<<COMP_CSR_INPSEL_Pos)                     /*!< 0x000000E0 */
#define COMP_CSR_INPSEL                          COMP_CSR_INPSEL_Msk                              /*!< COMP1 positive input select */
#define COMP_CSR_INPSEL_0                        (0x1UL<<COMP_CSR_INPSEL_Pos)                     /*!< 0x00000020 */
#define COMP_CSR_INPSEL_1                        (0x2UL<<COMP_CSR_INPSEL_Pos)                     /*!< 0x00000040 */
#define COMP_CSR_INPSEL_2                        (0x4UL<<COMP_CSR_INPSEL_Pos)                     /*!< 0x00000080 */
#define COMP_CSR_WINMODE_Pos                     (11U)
#define COMP_CSR_WINMODE_Msk                     (0x1UL<<COMP_CSR_WINMODE_Pos)                    /*!< 0x00000800 */
#define COMP_CSR_WINMODE                         COMP_CSR_WINMODE_Msk                             /*!< COMP1 window mode enable */
#define COMP_CSR_BLANKSEL_Pos                    (12U)
#define COMP_CSR_BLANKSEL_Msk                    (0x7UL<<COMP_CSR_BLANKSEL_Pos)                   /*!< 0x00007000 */
#define COMP_CSR_BLANKSEL                        COMP_CSR_BLANKSEL_Msk                            /*!< Comparator 1 blanking signal selection */
#define COMP_CSR_BLANKSEL_0                      (0x1UL<<COMP_CSR_BLANKSEL_Pos)                   /*!< 0x00001000 */
#define COMP_CSR_BLANKSEL_1                      (0x2UL<<COMP_CSR_BLANKSEL_Pos)                   /*!< 0x00002000 */
#define COMP_CSR_BLANKSEL_2                      (0x4UL<<COMP_CSR_BLANKSEL_Pos)                   /*!< 0x00004000 */
#define COMP_CSR_POLARITY_Pos                    (15U)
#define COMP_CSR_POLARITY_Msk                    (0x1UL<<COMP_CSR_POLARITY_Pos)                   /*!< 0x00008000 */
#define COMP_CSR_POLARITY                        COMP_CSR_POLARITY_Msk                            /*!< COMP1 output polarity selection */
#define COMP_CSR_HYST_Pos                        (16U)
#define COMP_CSR_HYST_Msk                        (0x1UL<<COMP_CSR_HYST_Pos)                       /*!< 0x00010000 */
#define COMP_CSR_HYST                            COMP_CSR_HYST_Msk                                /*!< COMP1 hysteresis function enable control */
#define COMP_CSR_PWRMODE_Pos                     (18U)
#define COMP_CSR_PWRMODE_Msk                     (0x1UL<<COMP_CSR_PWRMODE_Pos)                    /*!< 0x00040000 */
#define COMP_CSR_PWRMODE                         COMP_CSR_PWRMODE_Msk                             /*!< COMP1 power mode selection */
#define COMP_CSR_INT_OUT_SEL_Pos                 (29U)
#define COMP_CSR_INT_OUT_SEL_Msk                 (0x1UL<<COMP_CSR_INT_OUT_SEL_Pos)                /*!< 0x20000000 */
#define COMP_CSR_INT_OUT_SEL                     COMP_CSR_INT_OUT_SEL_Msk                         /*!< COMP1 output to TIM/EXTI selection */
#define COMP_CSR_COMP_OUT_Pos                    (30U)
#define COMP_CSR_COMP_OUT_Msk                    (0x1UL<<COMP_CSR_COMP_OUT_Pos)                   /*!< 0x40000000 */
#define COMP_CSR_COMP_OUT                        COMP_CSR_COMP_OUT_Msk                            /*!< COMP1 output status */
#define COMP_CSR_LOCK_Pos                        (31U)
#define COMP_CSR_LOCK_Msk                        (0x1UL<<COMP_CSR_LOCK_Pos)                       /*!< 0x80000000 */
#define COMP_CSR_LOCK                            COMP_CSR_LOCK_Msk                                /*!< COMP_CSR registerlock */

/********************************* Bit definition for COMP_FR register *********************************************/
#define COMP_FR_FLTEN1_Pos                       (0U)
#define COMP_FR_FLTEN1_Msk                       (0x1UL<<COMP_FR_FLTEN1_Pos)                      /*!< 0x00000001 */
#define COMP_FR_FLTEN1                           COMP_FR_FLTEN1_Msk                               /*!< Comparator 1 digital filter configuration */
#define COMP_FR_FLTCNT1_Pos                      (16U)
#define COMP_FR_FLTCNT1_Msk                      (0xFFFFUL<<COMP_FR_FLTCNT1_Pos)                  /*!< 0xFFFF0000 */
#define COMP_FR_FLTCNT1                          COMP_FR_FLTCNT1_Msk                              /*!< Comparator 1 sampling filter counter */

/********************************* Bit definition for COMP_CCSR register *********************************************/
#define COMP_CCSR_VREFCMP_EN_Pos                   (0U)
#define COMP_CCSR_VREFCMP_EN_Msk                   (0x1UL<<COMP_CCSR_VREFCMP_EN_Pos)                  /*!< 0x00000001 */
#define COMP_CCSR_VREFCMP_EN                       COMP_CCSR_VREFCMP_EN_Msk                           /*!< VREFCMP voltage divider enable signal */
#define COMP_CCSR_VCDIV_Pos                        (1U)
#define COMP_CCSR_VCDIV_Msk                        (0x3FUL<<COMP_CCSR_VCDIV_Pos)                      /*!< 0x0000007E */
#define COMP_CCSR_VCDIV                            COMP_CCSR_VCDIV_Msk
#define COMP_CCSR_VCDIV_0                          (0x1UL<<COMP_CCSR_VCDIV_Pos)                       /*!< 0x00000002 */
#define COMP_CCSR_VCDIV_1                          (0x2UL<<COMP_CCSR_VCDIV_Pos)                       /*!< 0x00000004 */
#define COMP_CCSR_VCDIV_2                          (0x4UL<<COMP_CCSR_VCDIV_Pos)                       /*!< 0x00000008 */
#define COMP_CCSR_VCDIV_3                          (0x8UL<<COMP_CCSR_VCDIV_Pos)                       /*!< 0x00000010 */
#define COMP_CCSR_VCDIV_4                          (0x10UL<<COMP_CCSR_VCDIV_Pos)                      /*!< 0x00000020 */
#define COMP_CCSR_VCDIV_5                          (0x20UL<<COMP_CCSR_VCDIV_Pos)                      /*!< 0x00000040 */
#define COMP_CCSR_SSEL_Pos                         (7U)
#define COMP_CCSR_SSEL_Msk                         (0x1UL<<COMP_CCSR_SSEL_Pos)                        /*!< 0x00000080 */
#define COMP_CCSR_SSEL                             COMP_CCSR_SSEL_Msk                                 
#define COMP_CCSR_VSEL_Pos                         (8U)
#define COMP_CCSR_VSEL_Msk                         (0x7UL<<COMP_CCSR_VSEL_Pos)                        /*!< 0x00000700 */
#define COMP_CCSR_VSEL                             COMP_CCSR_VSEL_Msk                                 /*!< VREFCMP reference source selection */
#define COMP_CCSR_VSEL_0                           (0x1UL<<COMP_CCSR_VSEL_Pos)                        /*!< 0x00000100 */
#define COMP_CCSR_VSEL_1                           (0x2UL<<COMP_CCSR_VSEL_Pos)                        /*!< 0x00000200 */
#define COMP_CCSR_VSEL_2                           (0x4UL<<COMP_CCSR_VSEL_Pos)                        /*!< 0x00000400 */

/********************************************************************************************************************/
/********************************* CRC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for CRC_DR register ***********************************************/
#define CRC_DR_DR_Pos                             (0U)
#define CRC_DR_DR_Msk                             (0xFFFFFFFFUL<<CRC_DR_DR_Pos)                     /*!< 0xFFFFFFFF */
#define CRC_DR_DR                                 CRC_DR_DR_Msk                                     /*!< CRC data */

/********************************* Bit definition for CRC_IDR register **********************************************/
#define CRC_IDR_IDR_Pos                           (0U)
#define CRC_IDR_IDR_Msk                           (0xFFFFFFFFUL<<CRC_IDR_IDR_Pos)                   /*!< 0xFFFFFFFF */
#define CRC_IDR_IDR                               CRC_IDR_IDR_Msk                                   /*!< CRC independent data */

/********************************* Bit definition for CRC_CR register ***********************************************/
#define CRC_CR_RESET_Pos                          (0U)
#define CRC_CR_RESET_Msk                          (0x1UL<<CRC_CR_RESET_Pos)                         /*!< 0x00000001 */
#define CRC_CR_RESET                              CRC_CR_RESET_Msk                                  /*!< CRC software reset */
#define CRC_CR_POLYSIZE_Pos                       (3U)
#define CRC_CR_POLYSIZE_Msk                       (0x3UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000018 */
#define CRC_CR_POLYSIZE                           CRC_CR_POLYSIZE_Msk                               /*!< Polynomial size */
#define CRC_CR_POLYSIZE_0                         (0x1UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000008 */
#define CRC_CR_POLYSIZE_1                         (0x2UL<<CRC_CR_POLYSIZE_Pos)                      /*!< 0x00000010 */
#define CRC_CR_REV_IN_Pos                         (5U)
#define CRC_CR_REV_IN_Msk                         (0x3UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000060 */
#define CRC_CR_REV_IN                             CRC_CR_REV_IN_Msk                                 /*!< Reverse input data */
#define CRC_CR_REV_IN_0                           (0x1UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000020 */
#define CRC_CR_REV_IN_1                           (0x2UL<<CRC_CR_REV_IN_Pos)                        /*!< 0x00000040 */
#define CRC_CR_REV_OUT_Pos                        (7U)
#define CRC_CR_REV_OUT_Msk                        (0x1UL<<CRC_CR_REV_OUT_Pos)                       /*!< 0x00000080 */
#define CRC_CR_REV_OUT                            CRC_CR_REV_OUT_Msk                                /*!< Reverse output data */

/********************************* Bit definition for CRC_INIT register *********************************************/
#define CRC_INIT_INIT_Pos                         (0U)
#define CRC_INIT_INIT_Msk                         (0xFFFFFFFFUL<<CRC_INIT_INIT_Pos)                 /*!< 0xFFFFFFFF */
#define CRC_INIT_INIT                             CRC_INIT_INIT_Msk                                 /*!< initial value */

/********************************* Bit definition for CRC_POL register **********************************************/
#define CRC_POL_POL_Pos                           (0U)
#define CRC_POL_POL_Msk                           (0xFFFFFFFFUL<<CRC_POL_POL_Pos)                   /*!< 0xFFFFFFFF */
#define CRC_POL_POL                               CRC_POL_POL_Msk                                   /*!< polynomial  */

/********************************************************************************************************************/
/********************************* DAC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for DAC_CR1 register **********************************************/
#define DAC_CR1_EN1_Pos                           (0U)
#define DAC_CR1_EN1_Msk                           (0x1UL<<DAC_CR1_EN1_Pos)                          /*!< 0x00000001 */
#define DAC_CR1_EN1                               DAC_CR1_EN1_Msk                                   
#define DAC_CR1_BOFF1_Pos                         (1U)
#define DAC_CR1_BOFF1_Msk                         (0x1UL<<DAC_CR1_BOFF1_Pos)                        /*!< 0x00000002 */
#define DAC_CR1_BOFF1                             DAC_CR1_BOFF1_Msk                                 
#define DAC_CR1_TEN1_Pos                          (2U)
#define DAC_CR1_TEN1_Msk                          (0x1UL<<DAC_CR1_TEN1_Pos)                         /*!< 0x00000004 */
#define DAC_CR1_TEN1                              DAC_CR1_TEN1_Msk                                  
#define DAC_CR1_TSEL1_Pos                         (3U)
#define DAC_CR1_TSEL1_Msk                         (0x7UL<<DAC_CR1_TSEL1_Pos)                        /*!< 0x00000038 */
#define DAC_CR1_TSEL1                             DAC_CR1_TSEL1_Msk
#define DAC_CR1_TSEL1_0                           (0x1UL<<DAC_CR1_TSEL1_Pos)                        /*!< 0x00000008 */
#define DAC_CR1_TSEL1_1                           (0x2UL<<DAC_CR1_TSEL1_Pos)                        /*!< 0x00000010 */
#define DAC_CR1_TSEL1_2                           (0x4UL<<DAC_CR1_TSEL1_Pos)                        /*!< 0x00000020 */
#define DAC_CR1_WAVE1_Pos                         (6U)
#define DAC_CR1_WAVE1_Msk                         (0x3UL<<DAC_CR1_WAVE1_Pos)                        /*!< 0x000000C0 */
#define DAC_CR1_WAVE1                             DAC_CR1_WAVE1_Msk
#define DAC_CR1_WAVE1_0                           (0x1UL<<DAC_CR1_WAVE1_Pos)                        /*!< 0x00000040 */
#define DAC_CR1_WAVE1_1                           (0x2UL<<DAC_CR1_WAVE1_Pos)                        /*!< 0x00000080 */
#define DAC_CR1_MAMP1_Pos                         (8U)
#define DAC_CR1_MAMP1_Msk                         (0xFUL<<DAC_CR1_MAMP1_Pos)                        /*!< 0x00000F00 */
#define DAC_CR1_MAMP1                             DAC_CR1_MAMP1_Msk
#define DAC_CR1_MAMP1_0                           (0x1UL<<DAC_CR1_MAMP1_Pos)                        /*!< 0x00000100 */
#define DAC_CR1_MAMP1_1                           (0x2UL<<DAC_CR1_MAMP1_Pos)                        /*!< 0x00000200 */
#define DAC_CR1_MAMP1_2                           (0x4UL<<DAC_CR1_MAMP1_Pos)                        /*!< 0x00000400 */
#define DAC_CR1_MAMP1_3                           (0x8UL<<DAC_CR1_MAMP1_Pos)                        /*!< 0x00000800 */
#define DAC_CR1_DMAEN1_Pos                        (12U)
#define DAC_CR1_DMAEN1_Msk                        (0x1UL<<DAC_CR1_DMAEN1_Pos)                       /*!< 0x00001000 */
#define DAC_CR1_DMAEN1                            DAC_CR1_DMAEN1_Msk                                
#define DAC_CR1_DMAUDRIE1_Pos                     (13U)
#define DAC_CR1_DMAUDRIE1_Msk                     (0x1UL<<DAC_CR1_DMAUDRIE1_Pos)                    /*!< 0x00002000 */
#define DAC_CR1_DMAUDRIE1                         DAC_CR1_DMAUDRIE1_Msk                             
#define DAC_CR1_DAC1CEN_Pos                       (14U)
#define DAC_CR1_DAC1CEN_Msk                       (0x1UL<<DAC_CR1_DAC1CEN_Pos)                      /*!< 0x00004000 */
#define DAC_CR1_DAC1CEN                           DAC_CR1_DAC1CEN_Msk                               

/********************************* Bit definition for DAC_CR2 register **********************************************/
#define DAC_CR2_SWTRIG1_Pos                       (0U)
#define DAC_CR2_SWTRIG1_Msk                       (0x1UL<<DAC_CR2_SWTRIG1_Pos)                      /*!< 0x00000001 */
#define DAC_CR2_SWTRIG1                           DAC_CR2_SWTRIG1_Msk                               
#define DAC_CR2_VREFSEL1_Pos                      (1U)
#define DAC_CR2_VREFSEL1_Msk                      (0x1UL<<DAC_CR2_VREFSEL1_Pos)                     /*!< 0x00000002 */
#define DAC_CR2_VREFSEL1                          DAC_CR2_VREFSEL1_Msk                              
#define DAC_CR2_INTEREN1_Pos                      (2U)
#define DAC_CR2_INTEREN1_Msk                      (0x1UL<<DAC_CR2_INTEREN1_Pos)                     /*!< 0x00000004 */
#define DAC_CR2_INTEREN1                          DAC_CR2_INTEREN1_Msk                              
#define DAC_CR2_OUTSEL_Pos                        (3U)
#define DAC_CR2_OUTSEL_Msk                        (0x1UL<<DAC_CR2_OUTSEL_Pos)                       /*!< 0x00000008 */
#define DAC_CR2_OUTSEL                            DAC_CR2_OUTSEL_Msk                                

/********************************* Bit definition for DAC_DHR12R1 register ******************************************/
#define DAC_DHR12R1_DACC1DHR_Pos                  (0U)
#define DAC_DHR12R1_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12R1_DACC1DHR_Pos)               /*!< 0x00000FFF */
#define DAC_DHR12R1_DACC1DHR                      DAC_DHR12R1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DHR12L1 register ******************************************/
#define DAC_DHR12L1_DACC1DHR_Pos                  (4U)
#define DAC_DHR12L1_DACC1DHR_Msk                  (0xFFFUL<<DAC_DHR12L1_DACC1DHR_Pos)               /*!< 0x0000FFF0 */
#define DAC_DHR12L1_DACC1DHR                      DAC_DHR12L1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DHR8R1 register *******************************************/
#define DAC_DHR8R1_DACC1DHR_Pos                   (0U)
#define DAC_DHR8R1_DACC1DHR_Msk                   (0xFFUL<<DAC_DHR8R1_DACC1DHR_Pos)                 /*!< 0x000000FF */
#define DAC_DHR8R1_DACC1DHR                       DAC_DHR8R1_DACC1DHR_Msk

/********************************* Bit definition for DAC_DOR1 register *********************************************/
#define DAC_DOR1_DACC1DOR_Pos                     (0U)
#define DAC_DOR1_DACC1DOR_Msk                     (0xFFFUL<<DAC_DOR1_DACC1DOR_Pos)                  /*!< 0x00000FFF */
#define DAC_DOR1_DACC1DOR                         DAC_DOR1_DACC1DOR_Msk

/********************************* Bit definition for DAC_SR register ***********************************************/
#define DAC_SR_DMAUDR1_Pos                        (13U)
#define DAC_SR_DMAUDR1_Msk                        (0x1UL<<DAC_SR_DMAUDR1_Pos)                       /*!< 0x00002000 */
#define DAC_SR_DMAUDR1                            DAC_SR_DMAUDR1_Msk                                

/********************************* Bit definition for DAC_CALR register *********************************************/
#define DAC_CALR_DAC_OS_TRIMCR_VCC_Pos            (0U)
#define DAC_CALR_DAC_OS_TRIMCR_VCC_Msk            (0x3FUL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)          /*!< 0x0000003F */
#define DAC_CALR_DAC_OS_TRIMCR_VCC                DAC_CALR_DAC_OS_TRIMCR_VCC_Msk
#define DAC_CALR_DAC_OS_TRIMCR_VCC_0              (0x1UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)           /*!< 0x00000001 */
#define DAC_CALR_DAC_OS_TRIMCR_VCC_1              (0x2UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)           /*!< 0x00000002 */
#define DAC_CALR_DAC_OS_TRIMCR_VCC_2              (0x4UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)           /*!< 0x00000004 */
#define DAC_CALR_DAC_OS_TRIMCR_VCC_3              (0x8UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)           /*!< 0x00000008 */
#define DAC_CALR_DAC_OS_TRIMCR_VCC_4              (0x10UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)          /*!< 0x00000010 */
#define DAC_CALR_DAC_OS_TRIMCR_VCC_5              (0x20UL<<DAC_CALR_DAC_OS_TRIMCR_VCC_Pos)          /*!< 0x00000020 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos        (6U)
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Msk        (0x3FUL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)      /*!< 0x00000FC0 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF            DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Msk
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_0          (0x1UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)       /*!< 0x00000040 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_1          (0x2UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)       /*!< 0x00000080 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_2          (0x4UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)       /*!< 0x00000100 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_3          (0x8UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)       /*!< 0x00000200 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_4          (0x10UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)      /*!< 0x00000400 */
#define DAC_CALR_DAC_OS_TRIMCR_VREFBUF_5          (0x20UL<<DAC_CALR_DAC_OS_TRIMCR_VREFBUF_Pos)      /*!< 0x00000800 */
#define DAC_CALR_DAC_CR_Pos                       (16U)
#define DAC_CALR_DAC_CR_Msk                       (0x3FUL<<DAC_CALR_DAC_CR_Pos)                     /*!< 0x003F0000 */
#define DAC_CALR_DAC_CR                           DAC_CALR_DAC_CR_Msk
#define DAC_CALR_DAC_CR_0                         (0x1UL<<DAC_CALR_DAC_CR_Pos)                      /*!< 0x00010000 */
#define DAC_CALR_DAC_CR_1                         (0x2UL<<DAC_CALR_DAC_CR_Pos)                      /*!< 0x00020000 */
#define DAC_CALR_DAC_CR_2                         (0x4UL<<DAC_CALR_DAC_CR_Pos)                      /*!< 0x00040000 */
#define DAC_CALR_DAC_CR_3                         (0x8UL<<DAC_CALR_DAC_CR_Pos)                      /*!< 0x00080000 */
#define DAC_CALR_DAC_CR_4                         (0x10UL<<DAC_CALR_DAC_CR_Pos)                     /*!< 0x00100000 */
#define DAC_CALR_DAC_CR_5                         (0x20UL<<DAC_CALR_DAC_CR_Pos)                     /*!< 0x00200000 */


/******************************************************************************/
/*                                                                            */
/*                                Debug MCU (DBGMCU)                          */
/*                                                                            */
/******************************************************************************/
/********************  Bit definition for DBG_IDCODE register  *************/
#define DBGMCU_IDCODE_DEV_ID_Pos                          (0U)
#define DBGMCU_IDCODE_DEV_ID_Msk                          (0xFFFUL << DBGMCU_IDCODE_DEV_ID_Pos)  /*!< 0x00000FFF */
#define DBGMCU_IDCODE_DEV_ID                              DBGMCU_IDCODE_DEV_ID_Msk
#define DBGMCU_IDCODE_REV_ID_Pos                          (16U)
#define DBGMCU_IDCODE_REV_ID_Msk                          (0xFFFFUL << DBGMCU_IDCODE_REV_ID_Pos) /*!< 0xFFFF0000 */
#define DBGMCU_IDCODE_REV_ID                              DBGMCU_IDCODE_REV_ID_Msk

/********************  Bit definition for DBGMCU_CR register  *****************/
#define DBGMCU_CR_DBG_SLEEP_Pos                           (0U)
#define DBGMCU_CR_DBG_SLEEP_Msk                           (0x1UL << DBGMCU_CR_DBG_SLEEP_Pos)     /*!< 0x00000001 */
#define DBGMCU_CR_DBG_SLEEP                               DBGMCU_CR_DBG_SLEEP_Msk

#define DBGMCU_CR_DBG_STOP_Pos                            (1U)
#define DBGMCU_CR_DBG_STOP_Msk                            (0x1UL << DBGMCU_CR_DBG_STOP_Pos)      /*!< 0x00000002 */
#define DBGMCU_CR_DBG_STOP                                DBGMCU_CR_DBG_STOP_Msk

#define DBGMCU_CR_DBG_STDBY_Pos                           (2U)
#define DBGMCU_CR_DBG_STDBY_Msk                           (0x1UL << DBGMCU_CR_DBG_STDBY_Pos)     /*!< 0x00000004 */
#define DBGMCU_CR_DBG_STDBY                               DBGMCU_CR_DBG_STDBY_Msk

/********************  Bit definition for DBGMCU_APB_FZ1 register  ***********/
#define DBGMCU_APB_FZ1_DBG_TIM2_STOP_Pos                  (0U)
#define DBGMCU_APB_FZ1_DBG_TIM2_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_TIM2_STOP_Pos) /*!< 0x00000001 */
#define DBGMCU_APB_FZ1_DBG_TIM2_STOP                      DBGMCU_APB_FZ1_DBG_TIM2_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_TIM3_STOP_Pos                  (1U)
#define DBGMCU_APB_FZ1_DBG_TIM3_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_TIM3_STOP_Pos) /*!< 0x00000002 */
#define DBGMCU_APB_FZ1_DBG_TIM3_STOP                      DBGMCU_APB_FZ1_DBG_TIM3_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_TIM6_STOP_Pos                  (4U)
#define DBGMCU_APB_FZ1_DBG_TIM6_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_TIM6_STOP_Pos) /*!< 0x00000010 */
#define DBGMCU_APB_FZ1_DBG_TIM6_STOP                      DBGMCU_APB_FZ1_DBG_TIM6_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_TIM7_STOP_Pos                  (5U)
#define DBGMCU_APB_FZ1_DBG_TIM7_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_TIM7_STOP_Pos) /*!< 0x00000020 */
#define DBGMCU_APB_FZ1_DBG_TIM7_STOP                      DBGMCU_APB_FZ1_DBG_TIM7_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_RTC_STOP_Pos                   (10U)
#define DBGMCU_APB_FZ1_DBG_RTC_STOP_Msk                   (0x1UL << DBGMCU_APB_FZ1_DBG_RTC_STOP_Pos)  /*!< 0x00000400 */
#define DBGMCU_APB_FZ1_DBG_RTC_STOP                       DBGMCU_APB_FZ1_DBG_RTC_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_WWDG_STOP_Pos                  (11U)
#define DBGMCU_APB_FZ1_DBG_WWDG_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_WWDG_STOP_Pos) /*!< 0x00000800 */
#define DBGMCU_APB_FZ1_DBG_WWDG_STOP                      DBGMCU_APB_FZ1_DBG_WWDG_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_IWDG_STOP_Pos                  (12U)
#define DBGMCU_APB_FZ1_DBG_IWDG_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_IWDG_STOP_Pos) /*!< 0x00001000 */
#define DBGMCU_APB_FZ1_DBG_IWDG_STOP                      DBGMCU_APB_FZ1_DBG_IWDG_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_I2C1_STOP_Pos                  (21U)
#define DBGMCU_APB_FZ1_DBG_I2C1_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_I2C1_STOP_Pos) /*!< 0x00200000 */
#define DBGMCU_APB_FZ1_DBG_I2C1_STOP                      DBGMCU_APB_FZ1_DBG_I2C1_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_I2C2_STOP_Pos                  (22U)
#define DBGMCU_APB_FZ1_DBG_I2C2_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ1_DBG_I2C2_STOP_Pos) /*!< 0x00400000 */
#define DBGMCU_APB_FZ1_DBG_I2C2_STOP                      DBGMCU_APB_FZ1_DBG_I2C2_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_LPTIM2_STOP_Pos                (30U)
#define DBGMCU_APB_FZ1_DBG_LPTIM2_STOP_Msk                (0x1UL << DBGMCU_APB_FZ1_DBG_LPTIM2_STOP_Pos) /*!< 0x40000000 */
#define DBGMCU_APB_FZ1_DBG_LPTIM2_STOP                    DBGMCU_APB_FZ1_DBG_LPTIM2_STOP_Msk
#define DBGMCU_APB_FZ1_DBG_LPTIM1_STOP_Pos                (31U)
#define DBGMCU_APB_FZ1_DBG_LPTIM1_STOP_Msk                (0x1UL << DBGMCU_APB_FZ1_DBG_LPTIM1_STOP_Pos) /*!< 0x80000000 */
#define DBGMCU_APB_FZ1_DBG_LPTIM1_STOP                    DBGMCU_APB_FZ1_DBG_LPTIM1_STOP_Msk

/********************  Bit definition for DBGMCU_APB_FZ2 register  ************/
#define DBGMCU_APB_FZ2_DBG_TIM1_STOP_Pos                  (11U)
#define DBGMCU_APB_FZ2_DBG_TIM1_STOP_Msk                  (0x1UL << DBGMCU_APB_FZ2_DBG_TIM1_STOP_Pos)  /*!< 0x00000800 */
#define DBGMCU_APB_FZ2_DBG_TIM1_STOP                      DBGMCU_APB_FZ2_DBG_TIM1_STOP_Msk
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP_Pos                 (17U)
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP_Msk                 (0x1UL << DBGMCU_APB_FZ2_DBG_TIM16_STOP_Pos) /*!< 0x00020000 */
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP                     DBGMCU_APB_FZ2_DBG_TIM16_STOP_Msk
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP_Pos                 (18U)
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP_Msk                 (0x1UL << DBGMCU_APB_FZ2_DBG_TIM17_STOP_Pos) /*!< 0x00040000 */
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP                     DBGMCU_APB_FZ2_DBG_TIM17_STOP_Msk
#define DBGMCU_APB_FZ2_DBG_PWM_STOP_Pos                   (19U)
#define DBGMCU_APB_FZ2_DBG_PWM_STOP_Msk                   (0x1UL << DBGMCU_APB_FZ2_DBG_PWM_STOP_Pos)   /*!< 0x00080000 */
#define DBGMCU_APB_FZ2_DBG_PWM_STOP                       DBGMCU_APB_FZ2_DBG_PWM_STOP_Msk
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP_Pos                 (17U)
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP_Msk                 (0x1UL << DBGMCU_APB_FZ2_DBG_TIM16_STOP_Pos) /*!< 0x00020000 */
#define DBGMCU_APB_FZ2_DBG_TIM16_STOP                     DBGMCU_APB_FZ2_DBG_TIM16_STOP_Msk
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP_Pos                 (18U)
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP_Msk                 (0x1UL << DBGMCU_APB_FZ2_DBG_TIM17_STOP_Pos) /*!< 0x00040000 */
#define DBGMCU_APB_FZ2_DBG_TIM17_STOP                     DBGMCU_APB_FZ2_DBG_TIM17_STOP_Msk


/********************************************************************************************************************/
/********************************* DMA ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for DMA_ISR register **********************************************/
#define DMA_ISR_TCIF0_Pos                         (0U)
#define DMA_ISR_TCIF0_Msk                         (0x1UL<<DMA_ISR_TCIF0_Pos)                        /*!< 0x00000001 */
#define DMA_ISR_TCIF0                             DMA_ISR_TCIF0_Msk                                 /*!< Transfer Complete Flag for Channel 0 */
#define DMA_ISR_BTCIF0_Pos                        (1U)
#define DMA_ISR_BTCIF0_Msk                        (0x1UL<<DMA_ISR_BTCIF0_Pos)                       /*!< 0x00000002 */
#define DMA_ISR_BTCIF0                            DMA_ISR_BTCIF0_Msk                                /*!< Block Transfer Complete Flag for Channel 0 */
#define DMA_ISR_HBTIF0_Pos                        (2U)
#define DMA_ISR_HBTIF0_Msk                        (0x1UL<<DMA_ISR_HBTIF0_Pos)                       /*!< 0x00000004 */
#define DMA_ISR_HBTIF0                            DMA_ISR_HBTIF0_Msk                                /*!< Half Block Transfer Complete Flag for Channel 0 */
#define DMA_ISR_TEIF0_Pos                         (3U)
#define DMA_ISR_TEIF0_Msk                         (0x1UL<<DMA_ISR_TEIF0_Pos)                        /*!< 0x00000008 */
#define DMA_ISR_TEIF0                             DMA_ISR_TEIF0_Msk                                 /*!< Transfer Error Flag for Channel 0 */
#define DMA_ISR_TCIF1_Pos                         (4U)
#define DMA_ISR_TCIF1_Msk                         (0x1UL<<DMA_ISR_TCIF1_Pos)                        /*!< 0x00000010 */
#define DMA_ISR_TCIF1                             DMA_ISR_TCIF1_Msk                                 /*!< Transfer Complete Flag for Channel 1 */
#define DMA_ISR_BTCIF1_Pos                        (5U)
#define DMA_ISR_BTCIF1_Msk                        (0x1UL<<DMA_ISR_BTCIF1_Pos)                       /*!< 0x00000020 */
#define DMA_ISR_BTCIF1                            DMA_ISR_BTCIF1_Msk                                /*!< Block Transfer Complete Flag for Channel 1 */
#define DMA_ISR_HBTIF1_Pos                        (6U)
#define DMA_ISR_HBTIF1_Msk                        (0x1UL<<DMA_ISR_HBTIF1_Pos)                       /*!< 0x00000040 */
#define DMA_ISR_HBTIF1                            DMA_ISR_HBTIF1_Msk                                /*!< Half Block Transfer Complete Flag for Channel 1 */
#define DMA_ISR_TEIF1_Pos                         (7U)
#define DMA_ISR_TEIF1_Msk                         (0x1UL<<DMA_ISR_TEIF1_Pos)                        /*!< 0x00000080 */
#define DMA_ISR_TEIF1                             DMA_ISR_TEIF1_Msk                                 /*!< Transfer Error Flag for Channel 1 */
#define DMA_ISR_TCIF2_Pos                         (8U)
#define DMA_ISR_TCIF2_Msk                         (0x1UL<<DMA_ISR_TCIF2_Pos)                        /*!< 0x00000100 */
#define DMA_ISR_TCIF2                             DMA_ISR_TCIF2_Msk                                 /*!< Transfer Complete Flag for Channel 2 */
#define DMA_ISR_BTCIF2_Pos                        (9U)
#define DMA_ISR_BTCIF2_Msk                        (0x1UL<<DMA_ISR_BTCIF2_Pos)                       /*!< 0x00000200 */
#define DMA_ISR_BTCIF2                            DMA_ISR_BTCIF2_Msk                                /*!< Block Transfer Complete Flag for Channel 2 */
#define DMA_ISR_HBTIF2_Pos                        (10U)
#define DMA_ISR_HBTIF2_Msk                        (0x1UL<<DMA_ISR_HBTIF2_Pos)                       /*!< 0x00000400 */
#define DMA_ISR_HBTIF2                            DMA_ISR_HBTIF2_Msk                                /*!< Half Block Transfer Complete Flag for Channel 2 */
#define DMA_ISR_TEIF2_Pos                         (11U)
#define DMA_ISR_TEIF2_Msk                         (0x1UL<<DMA_ISR_TEIF2_Pos)                        /*!< 0x00000800 */
#define DMA_ISR_TEIF2                             DMA_ISR_TEIF2_Msk                                 /*!< Transfer Error Flag for Channel 2 */
#define DMA_ISR_TCIF3_Pos                         (12U)
#define DMA_ISR_TCIF3_Msk                         (0x1UL<<DMA_ISR_TCIF3_Pos)                        /*!< 0x00001000 */
#define DMA_ISR_TCIF3                             DMA_ISR_TCIF3_Msk                                 /*!< Transfer Complete Flag for Channel 3 */
#define DMA_ISR_BTCIF3_Pos                        (13U)
#define DMA_ISR_BTCIF3_Msk                        (0x1UL<<DMA_ISR_BTCIF3_Pos)                       /*!< 0x00002000 */
#define DMA_ISR_BTCIF3                            DMA_ISR_BTCIF3_Msk                                /*!< Block Transfer Complete Flag for Channel 3 */
#define DMA_ISR_HBTIF3_Pos                        (14U)
#define DMA_ISR_HBTIF3_Msk                        (0x1UL<<DMA_ISR_HBTIF3_Pos)                       /*!< 0x00004000 */
#define DMA_ISR_HBTIF3                            DMA_ISR_HBTIF3_Msk                                /*!< Half Block Transfer Complete Flag for Channel 3 */
#define DMA_ISR_TEIF3_Pos                         (15U)
#define DMA_ISR_TEIF3_Msk                         (0x1UL<<DMA_ISR_TEIF3_Pos)                        /*!< 0x00008000 */
#define DMA_ISR_TEIF3                             DMA_ISR_TEIF3_Msk                                 /*!< Transfer Error Flag for Channel 3 */

/********************************* Bit definition for DMA_IFCR register *********************************************/
#define DMA_IFCR_CTCIF0_Pos                       (0U)
#define DMA_IFCR_CTCIF0_Msk                       (0x1UL<<DMA_IFCR_CTCIF0_Pos)                      /*!< 0x00000001 */
#define DMA_IFCR_CTCIF0                           DMA_IFCR_CTCIF0_Msk                               /*!< Transfer Complete Flag Clear for Channel 0 */
#define DMA_IFCR_CBTCIF0_Pos                      (1U)
#define DMA_IFCR_CBTCIF0_Msk                      (0x1UL<<DMA_IFCR_CBTCIF0_Pos)                     /*!< 0x00000002 */
#define DMA_IFCR_CBTCIF0                          DMA_IFCR_CBTCIF0_Msk                              /*!< Block Transfer Complete Flag Clear for Channel 0 */
#define DMA_IFCR_CHBTIF0_Pos                      (2U)
#define DMA_IFCR_CHBTIF0_Msk                      (0x1UL<<DMA_IFCR_CHBTIF0_Pos)                     /*!< 0x00000004 */
#define DMA_IFCR_CHBTIF0                          DMA_IFCR_CHBTIF0_Msk                              /*!< Half Block Transfer Complete Flag Clear for Channel 0 */
#define DMA_IFCR_CTEIF0_Pos                       (3U)
#define DMA_IFCR_CTEIF0_Msk                       (0x1UL<<DMA_IFCR_CTEIF0_Pos)                      /*!< 0x00000008 */
#define DMA_IFCR_CTEIF0                           DMA_IFCR_CTEIF0_Msk                               /*!< Transfer Error Flag Clear for Channel 0 */
#define DMA_IFCR_CTCIF1_Pos                       (4U)
#define DMA_IFCR_CTCIF1_Msk                       (0x1UL<<DMA_IFCR_CTCIF1_Pos)                      /*!< 0x00000010 */
#define DMA_IFCR_CTCIF1                           DMA_IFCR_CTCIF1_Msk                               /*!< Transfer Complete Flag Clear for Channel 1 */
#define DMA_IFCR_CBTCIF1_Pos                      (5U)
#define DMA_IFCR_CBTCIF1_Msk                      (0x1UL<<DMA_IFCR_CBTCIF1_Pos)                     /*!< 0x00000020 */
#define DMA_IFCR_CBTCIF1                          DMA_IFCR_CBTCIF1_Msk                              /*!< Block Transfer Complete Flag Clear for Channel 1 */
#define DMA_IFCR_CHBTIF1_Pos                      (6U)
#define DMA_IFCR_CHBTIF1_Msk                      (0x1UL<<DMA_IFCR_CHBTIF1_Pos)                     /*!< 0x00000040 */
#define DMA_IFCR_CHBTIF1                          DMA_IFCR_CHBTIF1_Msk                              /*!< Half Block Transfer Complete Flag Clear for Channel 1 */
#define DMA_IFCR_CTEIF1_Pos                       (7U)
#define DMA_IFCR_CTEIF1_Msk                       (0x1UL<<DMA_IFCR_CTEIF1_Pos)                      /*!< 0x00000080 */
#define DMA_IFCR_CTEIF1                           DMA_IFCR_CTEIF1_Msk                               /*!< Transfer Error Flag Clear for Channel 1 */
#define DMA_IFCR_CTCIF2_Pos                       (8U)
#define DMA_IFCR_CTCIF2_Msk                       (0x1UL<<DMA_IFCR_CTCIF2_Pos)                      /*!< 0x00000100 */
#define DMA_IFCR_CTCIF2                           DMA_IFCR_CTCIF2_Msk                               /*!< Transfer Complete Flag Clear for Channel 2 */
#define DMA_IFCR_CBTCIF2_Pos                      (9U)
#define DMA_IFCR_CBTCIF2_Msk                      (0x1UL<<DMA_IFCR_CBTCIF2_Pos)                     /*!< 0x00000200 */
#define DMA_IFCR_CBTCIF2                          DMA_IFCR_CBTCIF2_Msk                              /*!< Block Transfer Complete Flag Clear for Channel 2 */
#define DMA_IFCR_CHBTIF2_Pos                      (10U)
#define DMA_IFCR_CHBTIF2_Msk                      (0x1UL<<DMA_IFCR_CHBTIF2_Pos)                     /*!< 0x00000400 */
#define DMA_IFCR_CHBTIF2                          DMA_IFCR_CHBTIF2_Msk                              /*!< Half Block Transfer Complete Flag Clear for Channel 2 */
#define DMA_IFCR_CTEIF2_Pos                       (11U)
#define DMA_IFCR_CTEIF2_Msk                       (0x1UL<<DMA_IFCR_CTEIF2_Pos)                      /*!< 0x00000800 */
#define DMA_IFCR_CTEIF2                           DMA_IFCR_CTEIF2_Msk                               /*!< Transfer Error Flag Clear for Channel 2 */
#define DMA_IFCR_CTCIF3_Pos                       (12U)
#define DMA_IFCR_CTCIF3_Msk                       (0x1UL<<DMA_IFCR_CTCIF3_Pos)                      /*!< 0x00001000 */
#define DMA_IFCR_CTCIF3                           DMA_IFCR_CTCIF3_Msk                               /*!< Transfer Complete Flag Clear for Channel 3 */
#define DMA_IFCR_CBTCIF3_Pos                      (13U)
#define DMA_IFCR_CBTCIF3_Msk                      (0x1UL<<DMA_IFCR_CBTCIF3_Pos)                     /*!< 0x00002000 */
#define DMA_IFCR_CBTCIF3                          DMA_IFCR_CBTCIF3_Msk                              /*!< Block Transfer Complete Flag Clear for Channel 3 */
#define DMA_IFCR_CHBTIF3_Pos                      (14U)
#define DMA_IFCR_CHBTIF3_Msk                      (0x1UL<<DMA_IFCR_CHBTIF3_Pos)                     /*!< 0x00004000 */
#define DMA_IFCR_CHBTIF3                          DMA_IFCR_CHBTIF3_Msk                              /*!< Half Block Transfer Complete Flag Clear for Channel 3 */
#define DMA_IFCR_CTEIF3_Pos                       (15U)
#define DMA_IFCR_CTEIF3_Msk                       (0x1UL<<DMA_IFCR_CTEIF3_Pos)                      /*!< 0x00008000 */
#define DMA_IFCR_CTEIF3                           DMA_IFCR_CTEIF3_Msk                               /*!< Transfer Error Flag Clear for Channel 3 */


/********************************* Bit definition for DMA_CCR register *********************************************/
#define DMA_CCR_EN_Pos                           (0U)
#define DMA_CCR_EN_Msk                           (0x1UL<<DMA_CCR_EN_Pos)                          /*!< 0x00000001 */
#define DMA_CCR_EN                               DMA_CCR_EN_Msk                                   /*!< Channel Enable */
#define DMA_CCR_TCIE_Pos                         (1U)
#define DMA_CCR_TCIE_Msk                         (0x1UL<<DMA_CCR_TCIE_Pos)                        /*!< 0x00000002 */
#define DMA_CCR_TCIE                             DMA_CCR_TCIE_Msk                                 /*!< Transfer Complete Interrupt Enable */
#define DMA_CCR_BTCIE_Pos                        (2U)
#define DMA_CCR_BTCIE_Msk                        (0x1UL<<DMA_CCR_BTCIE_Pos)                       /*!< 0x00000004 */
#define DMA_CCR_BTCIE                            DMA_CCR_BTCIE_Msk                                /*!< Block Transfer Complete Interrupt Enable */
#define DMA_CCR_HBTIE_Pos                        (3U)
#define DMA_CCR_HBTIE_Msk                        (0x1UL<<DMA_CCR_HBTIE_Pos)                       /*!< 0x00000008 */
#define DMA_CCR_HBTIE                            DMA_CCR_HBTIE_Msk                                /*!< Half Block Transfer Complete Interrupt Enable */
#define DMA_CCR_TEIE_Pos                         (4U)
#define DMA_CCR_TEIE_Msk                         (0x1UL<<DMA_CCR_TEIE_Pos)                        /*!< 0x00000010 */
#define DMA_CCR_TEIE                             DMA_CCR_TEIE_Msk                                 /*!< Transfer Error Interrupt Enable */
#define DMA_CCR_DIR_Pos                          (5U)
#define DMA_CCR_DIR_Msk                          (0x1UL<<DMA_CCR_DIR_Pos)                         /*!< 0x00000020 */
#define DMA_CCR_DIR                              DMA_CCR_DIR_Msk                                  /*!< Data Transfer Direction */
#define DMA_CCR_CIRC_Pos                         (6U)
#define DMA_CCR_CIRC_Msk                         (0x1UL<<DMA_CCR_CIRC_Pos)                        /*!< 0x00000040 */
#define DMA_CCR_CIRC                             DMA_CCR_CIRC_Msk                                 /*!< Circular Mode */
#define DMA_CCR_TRANST_Pos                       (7U)
#define DMA_CCR_TRANST_Msk                       (0x1UL<<DMA_CCR_TRANST_Pos)                      /*!< 0x00000080 */
#define DMA_CCR_TRANST                           DMA_CCR_TRANST_Msk                               /*!< Transfer Type */
#define DMA_CCR_PINC_Pos                         (8U)
#define DMA_CCR_PINC_Msk                         (0x3UL<<DMA_CCR_PINC_Pos)                        /*!< 0x00000300 */
#define DMA_CCR_PINC                             DMA_CCR_PINC_Msk                                 /*!< Peripheral Increment Mode */
#define DMA_CCR_PINC_0                           (0x1UL<<DMA_CCR_PINC_Pos)                        /*!< 0x00000100 */
#define DMA_CCR_PINC_1                           (0x2UL<<DMA_CCR_PINC_Pos)                        /*!< 0x00000200 */
#define DMA_CCR_MINC_Pos                         (10U)
#define DMA_CCR_MINC_Msk                         (0x3UL<<DMA_CCR_MINC_Pos)                        /*!< 0x00000C00 */
#define DMA_CCR_MINC                             DMA_CCR_MINC_Msk                                 /*!< Memory Increment Mode */
#define DMA_CCR_MINC_0                           (0x1UL<<DMA_CCR_MINC_Pos)                        /*!< 0x00000400 */
#define DMA_CCR_MINC_1                           (0x2UL<<DMA_CCR_MINC_Pos)                        /*!< 0x00000800 */
#define DMA_CCR_PSIZE_Pos                        (12U)
#define DMA_CCR_PSIZE_Msk                        (0x3UL<<DMA_CCR_PSIZE_Pos)                       /*!< 0x00003000 */
#define DMA_CCR_PSIZE                            DMA_CCR_PSIZE_Msk                                /*!< Peripheral Size */
#define DMA_CCR_PSIZE_0                          (0x1UL<<DMA_CCR_PSIZE_Pos)                       /*!< 0x00001000 */
#define DMA_CCR_PSIZE_1                          (0x2UL<<DMA_CCR_PSIZE_Pos)                       /*!< 0x00002000 */
#define DMA_CCR_MSIZE_Pos                        (14U)
#define DMA_CCR_MSIZE_Msk                        (0x3UL<<DMA_CCR_MSIZE_Pos)                       /*!< 0x0000C000 */
#define DMA_CCR_MSIZE                            DMA_CCR_MSIZE_Msk                                /*!< Memory Size */
#define DMA_CCR_MSIZE_0                          (0x1UL<<DMA_CCR_MSIZE_Pos)                       /*!< 0x00004000 */
#define DMA_CCR_MSIZE_1                          (0x2UL<<DMA_CCR_MSIZE_Pos)                       /*!< 0x00008000 */
#define DMA_CCR_PL_Pos                           (16U)
#define DMA_CCR_PL_Msk                           (0x7UL<<DMA_CCR_PL_Pos)                          /*!< 0x00070000 */
#define DMA_CCR_PL                               DMA_CCR_PL_Msk                                   /*!< Priority Level */
#define DMA_CCR_PL_0                             (0x1UL<<DMA_CCR_PL_Pos)                          /*!< 0x00010000 */
#define DMA_CCR_PL_1                             (0x2UL<<DMA_CCR_PL_Pos)                          /*!< 0x00020000 */
#define DMA_CCR_PL_2                             (0x4UL<<DMA_CCR_PL_Pos)                          /*!< 0x00040000 */
#define DMA_CCR_MEM2MEM_Pos                      (19U)
#define DMA_CCR_MEM2MEM_Msk                      (0x1UL<<DMA_CCR_MEM2MEM_Pos)                     /*!< 0x00080000 */
#define DMA_CCR_MEM2MEM                          DMA_CCR_MEM2MEM_Msk                              /*!< Memory-to-memory Mode */
#define DMA_CCR_PNORLD_Pos                       (20U)
#define DMA_CCR_PNORLD_Msk                       (0x1UL<<DMA_CCR_PNORLD_Pos)                      /*!< 0x00100000 */
#define DMA_CCR_PNORLD                           DMA_CCR_PNORLD_Msk                               /*!< Peripheral No Reload */
#define DMA_CCR_MNORLD_Pos                       (21U)
#define DMA_CCR_MNORLD_Msk                       (0x1UL<<DMA_CCR_MNORLD_Pos)                      /*!< 0x00200000 */
#define DMA_CCR_MNORLD                           DMA_CCR_MNORLD_Msk                               /*!< Memory No Reload */
#define DMA_CCR_BTCSUSP_Pos                      (22U)
#define DMA_CCR_BTCSUSP_Msk                      (0x1UL<<DMA_CCR_BTCSUSP_Pos)                     /*!< 0x00400000 */
#define DMA_CCR_BTCSUSP                          DMA_CCR_BTCSUSP_Msk                              /*!< Suspend in Block Transfer Complete Interrupt */
#define DMA_CCR_M2MARB_Pos                       (23U)
#define DMA_CCR_M2MARB_Msk                       (0x1UL<<DMA_CCR_M2MARB_Pos)                      /*!< 0x00800000 */
#define DMA_CCR_M2MARB                           DMA_CCR_M2MARB_Msk                               /*!< Memory-to-memory Mode Arbitration */

/********************************* Bit definition for DMA_CNDTR register *******************************************/
#define DMA_CNDTR_NDT_Pos                        (0U)
#define DMA_CNDTR_NDT_Msk                        (0xFFFFUL<<DMA_CNDTR_NDT_Pos)                    /*!< 0x0000FFFF */
#define DMA_CNDTR_NDT                            DMA_CNDTR_NDT_Msk                                /*!< Number of Data to Transfer */

/********************************* Bit definition for DMA_CPAR register ********************************************/
#define DMA_CPAR_PA_Pos                          (0U)
#define DMA_CPAR_PA_Msk                          (0xFFFFFFFFUL<<DMA_CPAR_PA_Pos)                  /*!< 0xFFFFFFFF */
#define DMA_CPAR_PA                              DMA_CPAR_PA_Msk                                  /*!< Peripheral Address */

/********************************* Bit definition for DMA_CMAR register ********************************************/
#define DMA_CMAR_MA_Pos                          (0U)
#define DMA_CMAR_MA_Msk                          (0xFFFFFFFFUL<<DMA_CMAR_MA_Pos)                  /*!< 0xFFFFFFFF */
#define DMA_CMAR_MA                              DMA_CMAR_MA_Msk                                  /*!< Memory Address */

/********************************* Bit definition for DMA_CNBTR register *******************************************/
#define DMA_CNBTR_NBT_Pos                        (0U)
#define DMA_CNBTR_NBT_Msk                        (0xFFUL<<DMA_CNBTR_NBT_Pos)                      /*!< 0x000000FF */
#define DMA_CNBTR_NBT                            DMA_CNBTR_NBT_Msk                                /*!< Number of Block to Loop Transfer */

/********************************************************************************************************************/
/********************************* EXTI *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for EXTI_RTSR register ********************************************/
#define EXTI_RTSR_RT0_Pos                         (0U)
#define EXTI_RTSR_RT0_Msk                         (0x1UL<<EXTI_RTSR_RT0_Pos)                        /*!< 0x00000001 */
#define EXTI_RTSR_RT0                             EXTI_RTSR_RT0_Msk                                 
#define EXTI_RTSR_RT1_Pos                         (1U)
#define EXTI_RTSR_RT1_Msk                         (0x1UL<<EXTI_RTSR_RT1_Pos)                        /*!< 0x00000002 */
#define EXTI_RTSR_RT1                             EXTI_RTSR_RT1_Msk                                 
#define EXTI_RTSR_RT2_Pos                         (2U)
#define EXTI_RTSR_RT2_Msk                         (0x1UL<<EXTI_RTSR_RT2_Pos)                        /*!< 0x00000004 */
#define EXTI_RTSR_RT2                             EXTI_RTSR_RT2_Msk                                 
#define EXTI_RTSR_RT3_Pos                         (3U)
#define EXTI_RTSR_RT3_Msk                         (0x1UL<<EXTI_RTSR_RT3_Pos)                        /*!< 0x00000008 */
#define EXTI_RTSR_RT3                             EXTI_RTSR_RT3_Msk                                 
#define EXTI_RTSR_RT4_Pos                         (4U)
#define EXTI_RTSR_RT4_Msk                         (0x1UL<<EXTI_RTSR_RT4_Pos)                        /*!< 0x00000010 */
#define EXTI_RTSR_RT4                             EXTI_RTSR_RT4_Msk                                 
#define EXTI_RTSR_RT5_Pos                         (5U)
#define EXTI_RTSR_RT5_Msk                         (0x1UL<<EXTI_RTSR_RT5_Pos)                        /*!< 0x00000020 */
#define EXTI_RTSR_RT5                             EXTI_RTSR_RT5_Msk                                 
#define EXTI_RTSR_RT6_Pos                         (6U)
#define EXTI_RTSR_RT6_Msk                         (0x1UL<<EXTI_RTSR_RT6_Pos)                        /*!< 0x00000040 */
#define EXTI_RTSR_RT6                             EXTI_RTSR_RT6_Msk                                 
#define EXTI_RTSR_RT7_Pos                         (7U)
#define EXTI_RTSR_RT7_Msk                         (0x1UL<<EXTI_RTSR_RT7_Pos)                        /*!< 0x00000080 */
#define EXTI_RTSR_RT7                             EXTI_RTSR_RT7_Msk                                 
#define EXTI_RTSR_RT8_Pos                         (8U)
#define EXTI_RTSR_RT8_Msk                         (0x1UL<<EXTI_RTSR_RT8_Pos)                        /*!< 0x00000100 */
#define EXTI_RTSR_RT8                             EXTI_RTSR_RT8_Msk                                 
#define EXTI_RTSR_RT9_Pos                         (9U)
#define EXTI_RTSR_RT9_Msk                         (0x1UL<<EXTI_RTSR_RT9_Pos)                        /*!< 0x00000200 */
#define EXTI_RTSR_RT9                             EXTI_RTSR_RT9_Msk                                 
#define EXTI_RTSR_RT10_Pos                        (10U)
#define EXTI_RTSR_RT10_Msk                        (0x1UL<<EXTI_RTSR_RT10_Pos)                       /*!< 0x00000400 */
#define EXTI_RTSR_RT10                            EXTI_RTSR_RT10_Msk                                
#define EXTI_RTSR_RT11_Pos                        (11U)
#define EXTI_RTSR_RT11_Msk                        (0x1UL<<EXTI_RTSR_RT11_Pos)                       /*!< 0x00000800 */
#define EXTI_RTSR_RT11                            EXTI_RTSR_RT11_Msk                                
#define EXTI_RTSR_RT12_Pos                        (12U)
#define EXTI_RTSR_RT12_Msk                        (0x1UL<<EXTI_RTSR_RT12_Pos)                       /*!< 0x00001000 */
#define EXTI_RTSR_RT12                            EXTI_RTSR_RT12_Msk                                
#define EXTI_RTSR_RT13_Pos                        (13U)
#define EXTI_RTSR_RT13_Msk                        (0x1UL<<EXTI_RTSR_RT13_Pos)                       /*!< 0x00002000 */
#define EXTI_RTSR_RT13                            EXTI_RTSR_RT13_Msk                                
#define EXTI_RTSR_RT14_Pos                        (14U)
#define EXTI_RTSR_RT14_Msk                        (0x1UL<<EXTI_RTSR_RT14_Pos)                       /*!< 0x00004000 */
#define EXTI_RTSR_RT14                            EXTI_RTSR_RT14_Msk                                
#define EXTI_RTSR_RT15_Pos                        (15U)
#define EXTI_RTSR_RT15_Msk                        (0x1UL<<EXTI_RTSR_RT15_Pos)                       /*!< 0x00008000 */
#define EXTI_RTSR_RT15                            EXTI_RTSR_RT15_Msk                                
#define EXTI_RTSR_RT16_Pos                        (16U)
#define EXTI_RTSR_RT16_Msk                        (0x1UL<<EXTI_RTSR_RT16_Pos)                       /*!< 0x00010000 */
#define EXTI_RTSR_RT16                            EXTI_RTSR_RT16_Msk                                
#define EXTI_RTSR_RT17_Pos                        (17U)
#define EXTI_RTSR_RT17_Msk                        (0x1UL<<EXTI_RTSR_RT17_Pos)                       /*!< 0x00020000 */
#define EXTI_RTSR_RT17                            EXTI_RTSR_RT17_Msk                                
#define EXTI_RTSR_RT18_Pos                        (18U)
#define EXTI_RTSR_RT18_Msk                        (0x1UL<<EXTI_RTSR_RT18_Pos)                       /*!< 0x00040000 */
#define EXTI_RTSR_RT18                            EXTI_RTSR_RT18_Msk                                
#define EXTI_RTSR_RT19_Pos                        (19U)
#define EXTI_RTSR_RT19_Msk                        (0x1UL<<EXTI_RTSR_RT19_Pos)                       /*!< 0x00080000 */
#define EXTI_RTSR_RT19                            EXTI_RTSR_RT19_Msk                                

/********************************* Bit definition for EXTI_FTSR register ********************************************/
#define EXTI_FTSR_FT0_Pos                         (0U)
#define EXTI_FTSR_FT0_Msk                         (0x1UL<<EXTI_FTSR_FT0_Pos)                        /*!< 0x00000001 */
#define EXTI_FTSR_FT0                             EXTI_FTSR_FT0_Msk                                 
#define EXTI_FTSR_FT1_Pos                         (1U)
#define EXTI_FTSR_FT1_Msk                         (0x1UL<<EXTI_FTSR_FT1_Pos)                        /*!< 0x00000002 */
#define EXTI_FTSR_FT1                             EXTI_FTSR_FT1_Msk                                 
#define EXTI_FTSR_FT2_Pos                         (2U)
#define EXTI_FTSR_FT2_Msk                         (0x1UL<<EXTI_FTSR_FT2_Pos)                        /*!< 0x00000004 */
#define EXTI_FTSR_FT2                             EXTI_FTSR_FT2_Msk                                 
#define EXTI_FTSR_FT3_Pos                         (3U)
#define EXTI_FTSR_FT3_Msk                         (0x1UL<<EXTI_FTSR_FT3_Pos)                        /*!< 0x00000008 */
#define EXTI_FTSR_FT3                             EXTI_FTSR_FT3_Msk                                 
#define EXTI_FTSR_FT4_Pos                         (4U)
#define EXTI_FTSR_FT4_Msk                         (0x1UL<<EXTI_FTSR_FT4_Pos)                        /*!< 0x00000010 */
#define EXTI_FTSR_FT4                             EXTI_FTSR_FT4_Msk                                 
#define EXTI_FTSR_FT5_Pos                         (5U)
#define EXTI_FTSR_FT5_Msk                         (0x1UL<<EXTI_FTSR_FT5_Pos)                        /*!< 0x00000020 */
#define EXTI_FTSR_FT5                             EXTI_FTSR_FT5_Msk                                 
#define EXTI_FTSR_FT6_Pos                         (6U)
#define EXTI_FTSR_FT6_Msk                         (0x1UL<<EXTI_FTSR_FT6_Pos)                        /*!< 0x00000040 */
#define EXTI_FTSR_FT6                             EXTI_FTSR_FT6_Msk                                 
#define EXTI_FTSR_FT7_Pos                         (7U)
#define EXTI_FTSR_FT7_Msk                         (0x1UL<<EXTI_FTSR_FT7_Pos)                        /*!< 0x00000080 */
#define EXTI_FTSR_FT7                             EXTI_FTSR_FT7_Msk                                 
#define EXTI_FTSR_FT8_Pos                         (8U)
#define EXTI_FTSR_FT8_Msk                         (0x1UL<<EXTI_FTSR_FT8_Pos)                        /*!< 0x00000100 */
#define EXTI_FTSR_FT8                             EXTI_FTSR_FT8_Msk                                 
#define EXTI_FTSR_FT9_Pos                         (9U)
#define EXTI_FTSR_FT9_Msk                         (0x1UL<<EXTI_FTSR_FT9_Pos)                        /*!< 0x00000200 */
#define EXTI_FTSR_FT9                             EXTI_FTSR_FT9_Msk                                 
#define EXTI_FTSR_FT10_Pos                        (10U)
#define EXTI_FTSR_FT10_Msk                        (0x1UL<<EXTI_FTSR_FT10_Pos)                       /*!< 0x00000400 */
#define EXTI_FTSR_FT10                            EXTI_FTSR_FT10_Msk                                
#define EXTI_FTSR_FT11_Pos                        (11U)
#define EXTI_FTSR_FT11_Msk                        (0x1UL<<EXTI_FTSR_FT11_Pos)                       /*!< 0x00000800 */
#define EXTI_FTSR_FT11                            EXTI_FTSR_FT11_Msk                                
#define EXTI_FTSR_FT12_Pos                        (12U)
#define EXTI_FTSR_FT12_Msk                        (0x1UL<<EXTI_FTSR_FT12_Pos)                       /*!< 0x00001000 */
#define EXTI_FTSR_FT12                            EXTI_FTSR_FT12_Msk                                
#define EXTI_FTSR_FT13_Pos                        (13U)
#define EXTI_FTSR_FT13_Msk                        (0x1UL<<EXTI_FTSR_FT13_Pos)                       /*!< 0x00002000 */
#define EXTI_FTSR_FT13                            EXTI_FTSR_FT13_Msk                                
#define EXTI_FTSR_FT14_Pos                        (14U)
#define EXTI_FTSR_FT14_Msk                        (0x1UL<<EXTI_FTSR_FT14_Pos)                       /*!< 0x00004000 */
#define EXTI_FTSR_FT14                            EXTI_FTSR_FT14_Msk                                
#define EXTI_FTSR_FT15_Pos                        (15U)
#define EXTI_FTSR_FT15_Msk                        (0x1UL<<EXTI_FTSR_FT15_Pos)                       /*!< 0x00008000 */
#define EXTI_FTSR_FT15                            EXTI_FTSR_FT15_Msk                                
#define EXTI_FTSR_FT16_Pos                        (16U)
#define EXTI_FTSR_FT16_Msk                        (0x1UL<<EXTI_FTSR_FT16_Pos)                       /*!< 0x00010000 */
#define EXTI_FTSR_FT16                            EXTI_FTSR_FT16_Msk                                
#define EXTI_FTSR_FT17_Pos                        (17U)
#define EXTI_FTSR_FT17_Msk                        (0x1UL<<EXTI_FTSR_FT17_Pos)                       /*!< 0x00020000 */
#define EXTI_FTSR_FT17                            EXTI_FTSR_FT17_Msk                                
#define EXTI_FTSR_FT18_Pos                        (18U)
#define EXTI_FTSR_FT18_Msk                        (0x1UL<<EXTI_FTSR_FT18_Pos)                       /*!< 0x00040000 */
#define EXTI_FTSR_FT18                            EXTI_FTSR_FT18_Msk                                
#define EXTI_FTSR_FT19_Pos                        (19U)
#define EXTI_FTSR_FT19_Msk                        (0x1UL<<EXTI_FTSR_FT19_Pos)                       /*!< 0x00080000 */
#define EXTI_FTSR_FT19                            EXTI_FTSR_FT19_Msk                                

/********************************* Bit definition for EXTI_SWIER register *******************************************/
#define EXTI_SWIER_SWIE0_Pos                      (0U)
#define EXTI_SWIER_SWIE0_Msk                      (0x1UL<<EXTI_SWIER_SWIE0_Pos)                     /*!< 0x00000001 */
#define EXTI_SWIER_SWIE0                          EXTI_SWIER_SWIE0_Msk                              
#define EXTI_SWIER_SWIE1_Pos                      (1U)
#define EXTI_SWIER_SWIE1_Msk                      (0x1UL<<EXTI_SWIER_SWIE1_Pos)                     /*!< 0x00000002 */
#define EXTI_SWIER_SWIE1                          EXTI_SWIER_SWIE1_Msk                              
#define EXTI_SWIER_SWIE2_Pos                      (2U)
#define EXTI_SWIER_SWIE2_Msk                      (0x1UL<<EXTI_SWIER_SWIE2_Pos)                     /*!< 0x00000004 */
#define EXTI_SWIER_SWIE2                          EXTI_SWIER_SWIE2_Msk                              
#define EXTI_SWIER_SWIE3_Pos                      (3U)
#define EXTI_SWIER_SWIE3_Msk                      (0x1UL<<EXTI_SWIER_SWIE3_Pos)                     /*!< 0x00000008 */
#define EXTI_SWIER_SWIE3                          EXTI_SWIER_SWIE3_Msk                              
#define EXTI_SWIER_SWIE4_Pos                      (4U)
#define EXTI_SWIER_SWIE4_Msk                      (0x1UL<<EXTI_SWIER_SWIE4_Pos)                     /*!< 0x00000010 */
#define EXTI_SWIER_SWIE4                          EXTI_SWIER_SWIE4_Msk                              
#define EXTI_SWIER_SWIE5_Pos                      (5U)
#define EXTI_SWIER_SWIE5_Msk                      (0x1UL<<EXTI_SWIER_SWIE5_Pos)                     /*!< 0x00000020 */
#define EXTI_SWIER_SWIE5                          EXTI_SWIER_SWIE5_Msk                              
#define EXTI_SWIER_SWIE6_Pos                      (6U)
#define EXTI_SWIER_SWIE6_Msk                      (0x1UL<<EXTI_SWIER_SWIE6_Pos)                     /*!< 0x00000040 */
#define EXTI_SWIER_SWIE6                          EXTI_SWIER_SWIE6_Msk                              
#define EXTI_SWIER_SWIE7_Pos                      (7U)
#define EXTI_SWIER_SWIE7_Msk                      (0x1UL<<EXTI_SWIER_SWIE7_Pos)                     /*!< 0x00000080 */
#define EXTI_SWIER_SWIE7                          EXTI_SWIER_SWIE7_Msk                              
#define EXTI_SWIER_SWIE8_Pos                      (8U)
#define EXTI_SWIER_SWIE8_Msk                      (0x1UL<<EXTI_SWIER_SWIE8_Pos)                     /*!< 0x00000100 */
#define EXTI_SWIER_SWIE8                          EXTI_SWIER_SWIE8_Msk                              
#define EXTI_SWIER_SWIE9_Pos                      (9U)
#define EXTI_SWIER_SWIE9_Msk                      (0x1UL<<EXTI_SWIER_SWIE9_Pos)                     /*!< 0x00000200 */
#define EXTI_SWIER_SWIE9                          EXTI_SWIER_SWIE9_Msk                              
#define EXTI_SWIER_SWIE10_Pos                     (10U)
#define EXTI_SWIER_SWIE10_Msk                     (0x1UL<<EXTI_SWIER_SWIE10_Pos)                    /*!< 0x00000400 */
#define EXTI_SWIER_SWIE10                         EXTI_SWIER_SWIE10_Msk                             
#define EXTI_SWIER_SWIE11_Pos                     (11U)
#define EXTI_SWIER_SWIE11_Msk                     (0x1UL<<EXTI_SWIER_SWIE11_Pos)                    /*!< 0x00000800 */
#define EXTI_SWIER_SWIE11                         EXTI_SWIER_SWIE11_Msk                             
#define EXTI_SWIER_SWIE12_Pos                     (12U)
#define EXTI_SWIER_SWIE12_Msk                     (0x1UL<<EXTI_SWIER_SWIE12_Pos)                    /*!< 0x00001000 */
#define EXTI_SWIER_SWIE12                         EXTI_SWIER_SWIE12_Msk                             
#define EXTI_SWIER_SWIE13_Pos                     (13U)
#define EXTI_SWIER_SWIE13_Msk                     (0x1UL<<EXTI_SWIER_SWIE13_Pos)                    /*!< 0x00002000 */
#define EXTI_SWIER_SWIE13                         EXTI_SWIER_SWIE13_Msk                             
#define EXTI_SWIER_SWIE14_Pos                     (14U)
#define EXTI_SWIER_SWIE14_Msk                     (0x1UL<<EXTI_SWIER_SWIE14_Pos)                    /*!< 0x00004000 */
#define EXTI_SWIER_SWIE14                         EXTI_SWIER_SWIE14_Msk                             
#define EXTI_SWIER_SWIE15_Pos                     (15U)
#define EXTI_SWIER_SWIE15_Msk                     (0x1UL<<EXTI_SWIER_SWIE15_Pos)                    /*!< 0x00008000 */
#define EXTI_SWIER_SWIE15                         EXTI_SWIER_SWIE15_Msk                             
#define EXTI_SWIER_SWIE16_Pos                     (16U)
#define EXTI_SWIER_SWIE16_Msk                     (0x1UL<<EXTI_SWIER_SWIE16_Pos)                    /*!< 0x00010000 */
#define EXTI_SWIER_SWIE16                         EXTI_SWIER_SWIE16_Msk                             
#define EXTI_SWIER_SWIE17_Pos                     (17U)
#define EXTI_SWIER_SWIE17_Msk                     (0x1UL<<EXTI_SWIER_SWIE17_Pos)                    /*!< 0x00020000 */
#define EXTI_SWIER_SWIE17                         EXTI_SWIER_SWIE17_Msk                             
#define EXTI_SWIER_SWIE18_Pos                     (18U)
#define EXTI_SWIER_SWIE18_Msk                     (0x1UL<<EXTI_SWIER_SWIE18_Pos)                    /*!< 0x00040000 */
#define EXTI_SWIER_SWIE18                         EXTI_SWIER_SWIE18_Msk                             
#define EXTI_SWIER_SWIE19_Pos                     (19U)
#define EXTI_SWIER_SWIE19_Msk                     (0x1UL<<EXTI_SWIER_SWIE19_Pos)                    /*!< 0x00080000 */
#define EXTI_SWIER_SWIE19                         EXTI_SWIER_SWIE19_Msk                             

/********************************* Bit definition for EXTI_PR register **********************************************/
#define EXTI_PR_PR0_Pos                           (0U)
#define EXTI_PR_PR0_Msk                           (0x1UL<<EXTI_PR_PR0_Pos)                          /*!< 0x00000001 */
#define EXTI_PR_PR0                               EXTI_PR_PR0_Msk                                   
#define EXTI_PR_PR1_Pos                           (1U)
#define EXTI_PR_PR1_Msk                           (0x1UL<<EXTI_PR_PR1_Pos)                          /*!< 0x00000002 */
#define EXTI_PR_PR1                               EXTI_PR_PR1_Msk                                   
#define EXTI_PR_PR2_Pos                           (2U)
#define EXTI_PR_PR2_Msk                           (0x1UL<<EXTI_PR_PR2_Pos)                          /*!< 0x00000004 */
#define EXTI_PR_PR2                               EXTI_PR_PR2_Msk                                   
#define EXTI_PR_PR3_Pos                           (3U)
#define EXTI_PR_PR3_Msk                           (0x1UL<<EXTI_PR_PR3_Pos)                          /*!< 0x00000008 */
#define EXTI_PR_PR3                               EXTI_PR_PR3_Msk                                   
#define EXTI_PR_PR4_Pos                           (4U)
#define EXTI_PR_PR4_Msk                           (0x1UL<<EXTI_PR_PR4_Pos)                          /*!< 0x00000010 */
#define EXTI_PR_PR4                               EXTI_PR_PR4_Msk                                   
#define EXTI_PR_PR5_Pos                           (5U)
#define EXTI_PR_PR5_Msk                           (0x1UL<<EXTI_PR_PR5_Pos)                          /*!< 0x00000020 */
#define EXTI_PR_PR5                               EXTI_PR_PR5_Msk                                   
#define EXTI_PR_PR6_Pos                           (6U)
#define EXTI_PR_PR6_Msk                           (0x1UL<<EXTI_PR_PR6_Pos)                          /*!< 0x00000040 */
#define EXTI_PR_PR6                               EXTI_PR_PR6_Msk                                   
#define EXTI_PR_PR7_Pos                           (7U)
#define EXTI_PR_PR7_Msk                           (0x1UL<<EXTI_PR_PR7_Pos)                          /*!< 0x00000080 */
#define EXTI_PR_PR7                               EXTI_PR_PR7_Msk                                   
#define EXTI_PR_PR8_Pos                           (8U)
#define EXTI_PR_PR8_Msk                           (0x1UL<<EXTI_PR_PR8_Pos)                          /*!< 0x00000100 */
#define EXTI_PR_PR8                               EXTI_PR_PR8_Msk                                   
#define EXTI_PR_PR9_Pos                           (9U)
#define EXTI_PR_PR9_Msk                           (0x1UL<<EXTI_PR_PR9_Pos)                          /*!< 0x00000200 */
#define EXTI_PR_PR9                               EXTI_PR_PR9_Msk                                   
#define EXTI_PR_PR10_Pos                          (10U)
#define EXTI_PR_PR10_Msk                          (0x1UL<<EXTI_PR_PR10_Pos)                         /*!< 0x00000400 */
#define EXTI_PR_PR10                              EXTI_PR_PR10_Msk                                  
#define EXTI_PR_PR11_Pos                          (11U)
#define EXTI_PR_PR11_Msk                          (0x1UL<<EXTI_PR_PR11_Pos)                         /*!< 0x00000800 */
#define EXTI_PR_PR11                              EXTI_PR_PR11_Msk                                  
#define EXTI_PR_PR12_Pos                          (12U)
#define EXTI_PR_PR12_Msk                          (0x1UL<<EXTI_PR_PR12_Pos)                         /*!< 0x00001000 */
#define EXTI_PR_PR12                              EXTI_PR_PR12_Msk                                  
#define EXTI_PR_PR13_Pos                          (13U)
#define EXTI_PR_PR13_Msk                          (0x1UL<<EXTI_PR_PR13_Pos)                         /*!< 0x00002000 */
#define EXTI_PR_PR13                              EXTI_PR_PR13_Msk                                  
#define EXTI_PR_PR14_Pos                          (14U)
#define EXTI_PR_PR14_Msk                          (0x1UL<<EXTI_PR_PR14_Pos)                         /*!< 0x00004000 */
#define EXTI_PR_PR14                              EXTI_PR_PR14_Msk                                  
#define EXTI_PR_PR15_Pos                          (15U)
#define EXTI_PR_PR15_Msk                          (0x1UL<<EXTI_PR_PR15_Pos)                         /*!< 0x00008000 */
#define EXTI_PR_PR15                              EXTI_PR_PR15_Msk                                  
#define EXTI_PR_PR16_Pos                          (16U)
#define EXTI_PR_PR16_Msk                          (0x1UL<<EXTI_PR_PR16_Pos)                         /*!< 0x00010000 */
#define EXTI_PR_PR16                              EXTI_PR_PR16_Msk                                  
#define EXTI_PR_PR17_Pos                          (17U)
#define EXTI_PR_PR17_Msk                          (0x1UL<<EXTI_PR_PR17_Pos)                         /*!< 0x00020000 */
#define EXTI_PR_PR17                              EXTI_PR_PR17_Msk                                  
#define EXTI_PR_PR18_Pos                          (18U)
#define EXTI_PR_PR18_Msk                          (0x1UL<<EXTI_PR_PR18_Pos)                         /*!< 0x00040000 */
#define EXTI_PR_PR18                              EXTI_PR_PR18_Msk                                  
#define EXTI_PR_PR19_Pos                          (19U)
#define EXTI_PR_PR19_Msk                          (0x1UL<<EXTI_PR_PR19_Pos)                         /*!< 0x00080000 */
#define EXTI_PR_PR19                              EXTI_PR_PR19_Msk                                  

/********************************* Bit definition for EXTI_IOSEL1 register ******************************************/
#define EXTI_IOSEL1_EXTI0SEL_Pos                  (0U)
#define EXTI_IOSEL1_EXTI0SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI0SEL_Pos)                 /*!< 0x00000003 */
#define EXTI_IOSEL1_EXTI0SEL                      EXTI_IOSEL1_EXTI0SEL_Msk
#define EXTI_IOSEL1_EXTI0SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI0SEL_Pos)                 /*!< 0x00000001 */
#define EXTI_IOSEL1_EXTI0SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI0SEL_Pos)                 /*!< 0x00000002 */
#define EXTI_IOSEL1_EXTI1SEL_Pos                  (4U)
#define EXTI_IOSEL1_EXTI1SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI1SEL_Pos)                 /*!< 0x00000030 */
#define EXTI_IOSEL1_EXTI1SEL                      EXTI_IOSEL1_EXTI1SEL_Msk
#define EXTI_IOSEL1_EXTI1SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI1SEL_Pos)                 /*!< 0x00000010 */
#define EXTI_IOSEL1_EXTI1SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI1SEL_Pos)                 /*!< 0x00000020 */
#define EXTI_IOSEL1_EXTI2SEL_Pos                  (8U)
#define EXTI_IOSEL1_EXTI2SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI2SEL_Pos)                 /*!< 0x00000300 */
#define EXTI_IOSEL1_EXTI2SEL                      EXTI_IOSEL1_EXTI2SEL_Msk
#define EXTI_IOSEL1_EXTI2SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI2SEL_Pos)                 /*!< 0x00000100 */
#define EXTI_IOSEL1_EXTI2SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI2SEL_Pos)                 /*!< 0x00000200 */
#define EXTI_IOSEL1_EXTI3SEL_Pos                  (12U)
#define EXTI_IOSEL1_EXTI3SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI3SEL_Pos)                 /*!< 0x00003000 */
#define EXTI_IOSEL1_EXTI3SEL                      EXTI_IOSEL1_EXTI3SEL_Msk
#define EXTI_IOSEL1_EXTI3SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI3SEL_Pos)                 /*!< 0x00001000 */
#define EXTI_IOSEL1_EXTI3SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI3SEL_Pos)                 /*!< 0x00002000 */
#define EXTI_IOSEL1_EXTI4SEL_Pos                  (16U)
#define EXTI_IOSEL1_EXTI4SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI4SEL_Pos)                 /*!< 0x00030000 */
#define EXTI_IOSEL1_EXTI4SEL                      EXTI_IOSEL1_EXTI4SEL_Msk
#define EXTI_IOSEL1_EXTI4SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI4SEL_Pos)                 /*!< 0x00010000 */
#define EXTI_IOSEL1_EXTI4SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI4SEL_Pos)                 /*!< 0x00020000 */
#define EXTI_IOSEL1_EXTI5SEL_Pos                  (20U)
#define EXTI_IOSEL1_EXTI5SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI5SEL_Pos)                 /*!< 0x00300000 */
#define EXTI_IOSEL1_EXTI5SEL                      EXTI_IOSEL1_EXTI5SEL_Msk
#define EXTI_IOSEL1_EXTI5SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI5SEL_Pos)                 /*!< 0x00100000 */
#define EXTI_IOSEL1_EXTI5SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI5SEL_Pos)                 /*!< 0x00200000 */
#define EXTI_IOSEL1_EXTI6SEL_Pos                  (24U)
#define EXTI_IOSEL1_EXTI6SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI6SEL_Pos)                 /*!< 0x03000000 */
#define EXTI_IOSEL1_EXTI6SEL                      EXTI_IOSEL1_EXTI6SEL_Msk
#define EXTI_IOSEL1_EXTI6SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI6SEL_Pos)                 /*!< 0x01000000 */
#define EXTI_IOSEL1_EXTI6SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI6SEL_Pos)                 /*!< 0x02000000 */
#define EXTI_IOSEL1_EXTI7SEL_Pos                  (28U)
#define EXTI_IOSEL1_EXTI7SEL_Msk                  (0x3UL<<EXTI_IOSEL1_EXTI7SEL_Pos)                 /*!< 0x30000000 */
#define EXTI_IOSEL1_EXTI7SEL                      EXTI_IOSEL1_EXTI7SEL_Msk
#define EXTI_IOSEL1_EXTI7SEL_0                    (0x1UL<<EXTI_IOSEL1_EXTI7SEL_Pos)                 /*!< 0x10000000 */
#define EXTI_IOSEL1_EXTI7SEL_1                    (0x2UL<<EXTI_IOSEL1_EXTI7SEL_Pos)                 /*!< 0x20000000 */

/********************************* Bit definition for EXTI_IOSEL2 register ******************************************/
#define EXTI_IOSEL2_EXTI8SEL_Pos                  (0U)
#define EXTI_IOSEL2_EXTI8SEL_Msk                  (0x3UL<<EXTI_IOSEL2_EXTI8SEL_Pos)                 /*!< 0x00000003 */
#define EXTI_IOSEL2_EXTI8SEL                      EXTI_IOSEL2_EXTI8SEL_Msk
#define EXTI_IOSEL2_EXTI8SEL_0                    (0x1UL<<EXTI_IOSEL2_EXTI8SEL_Pos)                 /*!< 0x00000001 */
#define EXTI_IOSEL2_EXTI8SEL_1                    (0x2UL<<EXTI_IOSEL2_EXTI8SEL_Pos)                 /*!< 0x00000002 */
#define EXTI_IOSEL2_EXTI9SEL_Pos                  (4U)
#define EXTI_IOSEL2_EXTI9SEL_Msk                  (0x3UL<<EXTI_IOSEL2_EXTI9SEL_Pos)                 /*!< 0x00000030 */
#define EXTI_IOSEL2_EXTI9SEL                      EXTI_IOSEL2_EXTI9SEL_Msk
#define EXTI_IOSEL2_EXTI9SEL_0                    (0x1UL<<EXTI_IOSEL2_EXTI9SEL_Pos)                 /*!< 0x00000010 */
#define EXTI_IOSEL2_EXTI9SEL_1                    (0x2UL<<EXTI_IOSEL2_EXTI9SEL_Pos)                 /*!< 0x00000020 */
#define EXTI_IOSEL2_EXTI10SEL_Pos                 (8U)
#define EXTI_IOSEL2_EXTI10SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI10SEL_Pos)                /*!< 0x00000300 */
#define EXTI_IOSEL2_EXTI10SEL                     EXTI_IOSEL2_EXTI10SEL_Msk
#define EXTI_IOSEL2_EXTI10SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI10SEL_Pos)                /*!< 0x00000100 */
#define EXTI_IOSEL2_EXTI10SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI10SEL_Pos)                /*!< 0x00000200 */
#define EXTI_IOSEL2_EXTI11SEL_Pos                 (12U)
#define EXTI_IOSEL2_EXTI11SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI11SEL_Pos)                /*!< 0x00003000 */
#define EXTI_IOSEL2_EXTI11SEL                     EXTI_IOSEL2_EXTI11SEL_Msk
#define EXTI_IOSEL2_EXTI11SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI11SEL_Pos)                /*!< 0x00001000 */
#define EXTI_IOSEL2_EXTI11SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI11SEL_Pos)                /*!< 0x00002000 */
#define EXTI_IOSEL2_EXTI12SEL_Pos                 (16U)
#define EXTI_IOSEL2_EXTI12SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI12SEL_Pos)                /*!< 0x00030000 */
#define EXTI_IOSEL2_EXTI12SEL                     EXTI_IOSEL2_EXTI12SEL_Msk
#define EXTI_IOSEL2_EXTI12SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI12SEL_Pos)                /*!< 0x00010000 */
#define EXTI_IOSEL2_EXTI12SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI12SEL_Pos)                /*!< 0x00020000 */
#define EXTI_IOSEL2_EXTI13SEL_Pos                 (20U)
#define EXTI_IOSEL2_EXTI13SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI13SEL_Pos)                /*!< 0x00300000 */
#define EXTI_IOSEL2_EXTI13SEL                     EXTI_IOSEL2_EXTI13SEL_Msk
#define EXTI_IOSEL2_EXTI13SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI13SEL_Pos)                /*!< 0x00100000 */
#define EXTI_IOSEL2_EXTI13SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI13SEL_Pos)                /*!< 0x00200000 */
#define EXTI_IOSEL2_EXTI14SEL_Pos                 (24U)
#define EXTI_IOSEL2_EXTI14SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI14SEL_Pos)                /*!< 0x03000000 */
#define EXTI_IOSEL2_EXTI14SEL                     EXTI_IOSEL2_EXTI14SEL_Msk
#define EXTI_IOSEL2_EXTI14SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI14SEL_Pos)                /*!< 0x01000000 */
#define EXTI_IOSEL2_EXTI14SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI14SEL_Pos)                /*!< 0x02000000 */
#define EXTI_IOSEL2_EXTI15SEL_Pos                 (28U)
#define EXTI_IOSEL2_EXTI15SEL_Msk                 (0x3UL<<EXTI_IOSEL2_EXTI15SEL_Pos)                /*!< 0x30000000 */
#define EXTI_IOSEL2_EXTI15SEL                     EXTI_IOSEL2_EXTI15SEL_Msk
#define EXTI_IOSEL2_EXTI15SEL_0                   (0x1UL<<EXTI_IOSEL2_EXTI15SEL_Pos)                /*!< 0x10000000 */
#define EXTI_IOSEL2_EXTI15SEL_1                   (0x2UL<<EXTI_IOSEL2_EXTI15SEL_Pos)                /*!< 0x20000000 */

/********************************* Bit definition for EXTI_IMR register *********************************************/
#define EXTI_IMR_IM0_Pos                          (0U)
#define EXTI_IMR_IM0_Msk                          (0x1UL<<EXTI_IMR_IM0_Pos)                         /*!< 0x00000001 */
#define EXTI_IMR_IM0                              EXTI_IMR_IM0_Msk                                  
#define EXTI_IMR_IM1_Pos                          (1U)
#define EXTI_IMR_IM1_Msk                          (0x1UL<<EXTI_IMR_IM1_Pos)                         /*!< 0x00000002 */
#define EXTI_IMR_IM1                              EXTI_IMR_IM1_Msk                                  
#define EXTI_IMR_IM2_Pos                          (2U)
#define EXTI_IMR_IM2_Msk                          (0x1UL<<EXTI_IMR_IM2_Pos)                         /*!< 0x00000004 */
#define EXTI_IMR_IM2                              EXTI_IMR_IM2_Msk                                  
#define EXTI_IMR_IM3_Pos                          (3U)
#define EXTI_IMR_IM3_Msk                          (0x1UL<<EXTI_IMR_IM3_Pos)                         /*!< 0x00000008 */
#define EXTI_IMR_IM3                              EXTI_IMR_IM3_Msk                                  
#define EXTI_IMR_IM4_Pos                          (4U)
#define EXTI_IMR_IM4_Msk                          (0x1UL<<EXTI_IMR_IM4_Pos)                         /*!< 0x00000010 */
#define EXTI_IMR_IM4                              EXTI_IMR_IM4_Msk                                  
#define EXTI_IMR_IM5_Pos                          (5U)
#define EXTI_IMR_IM5_Msk                          (0x1UL<<EXTI_IMR_IM5_Pos)                         /*!< 0x00000020 */
#define EXTI_IMR_IM5                              EXTI_IMR_IM5_Msk                                  
#define EXTI_IMR_IM6_Pos                          (6U)
#define EXTI_IMR_IM6_Msk                          (0x1UL<<EXTI_IMR_IM6_Pos)                         /*!< 0x00000040 */
#define EXTI_IMR_IM6                              EXTI_IMR_IM6_Msk                                  
#define EXTI_IMR_IM7_Pos                          (7U)
#define EXTI_IMR_IM7_Msk                          (0x1UL<<EXTI_IMR_IM7_Pos)                         /*!< 0x00000080 */
#define EXTI_IMR_IM7                              EXTI_IMR_IM7_Msk                                  
#define EXTI_IMR_IM8_Pos                          (8U)
#define EXTI_IMR_IM8_Msk                          (0x1UL<<EXTI_IMR_IM8_Pos)                         /*!< 0x00000100 */
#define EXTI_IMR_IM8                              EXTI_IMR_IM8_Msk                                  
#define EXTI_IMR_IM9_Pos                          (9U)
#define EXTI_IMR_IM9_Msk                          (0x1UL<<EXTI_IMR_IM9_Pos)                         /*!< 0x00000200 */
#define EXTI_IMR_IM9                              EXTI_IMR_IM9_Msk                                  
#define EXTI_IMR_IM10_Pos                         (10U)
#define EXTI_IMR_IM10_Msk                         (0x1UL<<EXTI_IMR_IM10_Pos)                        /*!< 0x00000400 */
#define EXTI_IMR_IM10                             EXTI_IMR_IM10_Msk                                 
#define EXTI_IMR_IM11_Pos                         (11U)
#define EXTI_IMR_IM11_Msk                         (0x1UL<<EXTI_IMR_IM11_Pos)                        /*!< 0x00000800 */
#define EXTI_IMR_IM11                             EXTI_IMR_IM11_Msk                                 
#define EXTI_IMR_IM12_Pos                         (12U)
#define EXTI_IMR_IM12_Msk                         (0x1UL<<EXTI_IMR_IM12_Pos)                        /*!< 0x00001000 */
#define EXTI_IMR_IM12                             EXTI_IMR_IM12_Msk                                 
#define EXTI_IMR_IM13_Pos                         (13U)
#define EXTI_IMR_IM13_Msk                         (0x1UL<<EXTI_IMR_IM13_Pos)                        /*!< 0x00002000 */
#define EXTI_IMR_IM13                             EXTI_IMR_IM13_Msk                                 
#define EXTI_IMR_IM14_Pos                         (14U)
#define EXTI_IMR_IM14_Msk                         (0x1UL<<EXTI_IMR_IM14_Pos)                        /*!< 0x00004000 */
#define EXTI_IMR_IM14                             EXTI_IMR_IM14_Msk                                 
#define EXTI_IMR_IM15_Pos                         (15U)
#define EXTI_IMR_IM15_Msk                         (0x1UL<<EXTI_IMR_IM15_Pos)                        /*!< 0x00008000 */
#define EXTI_IMR_IM15                             EXTI_IMR_IM15_Msk                                 
#define EXTI_IMR_IM16_Pos                         (16U)
#define EXTI_IMR_IM16_Msk                         (0x1UL<<EXTI_IMR_IM16_Pos)                        /*!< 0x00010000 */
#define EXTI_IMR_IM16                             EXTI_IMR_IM16_Msk                                 
#define EXTI_IMR_IM17_Pos                         (17U)
#define EXTI_IMR_IM17_Msk                         (0x1UL<<EXTI_IMR_IM17_Pos)                        /*!< 0x00020000 */
#define EXTI_IMR_IM17                             EXTI_IMR_IM17_Msk                                 
#define EXTI_IMR_IM18_Pos                         (18U)
#define EXTI_IMR_IM18_Msk                         (0x1UL<<EXTI_IMR_IM18_Pos)                        /*!< 0x00040000 */
#define EXTI_IMR_IM18                             EXTI_IMR_IM18_Msk                                 
#define EXTI_IMR_IM19_Pos                         (19U)
#define EXTI_IMR_IM19_Msk                         (0x1UL<<EXTI_IMR_IM19_Pos)                        /*!< 0x00080000 */
#define EXTI_IMR_IM19                             EXTI_IMR_IM19_Msk                                 
#define EXTI_IMR_IM20_Pos                         (20U)
#define EXTI_IMR_IM20_Msk                         (0x1UL<<EXTI_IMR_IM20_Pos)                        /*!< 0x00100000 */
#define EXTI_IMR_IM20                             EXTI_IMR_IM20_Msk                                 
#define EXTI_IMR_IM21_Pos                         (21U)
#define EXTI_IMR_IM21_Msk                         (0x1UL<<EXTI_IMR_IM21_Pos)                        /*!< 0x00200000 */
#define EXTI_IMR_IM21                             EXTI_IMR_IM21_Msk                                 
#define EXTI_IMR_IM22_Pos                         (22U)
#define EXTI_IMR_IM22_Msk                         (0x1UL<<EXTI_IMR_IM22_Pos)                        /*!< 0x00400000 */
#define EXTI_IMR_IM22                             EXTI_IMR_IM22_Msk                                 
#define EXTI_IMR_IM23_Pos                         (23U)
#define EXTI_IMR_IM23_Msk                         (0x1UL<<EXTI_IMR_IM23_Pos)                        /*!< 0x00800000 */
#define EXTI_IMR_IM23                             EXTI_IMR_IM23_Msk                                 
#define EXTI_IMR_IM24_Pos                         (24U)
#define EXTI_IMR_IM24_Msk                         (0x1UL<<EXTI_IMR_IM24_Pos)                        /*!< 0x01000000 */
#define EXTI_IMR_IM24                             EXTI_IMR_IM24_Msk                                 
#define EXTI_IMR_IM25_Pos                         (25U)
#define EXTI_IMR_IM25_Msk                         (0x1UL<<EXTI_IMR_IM25_Pos)                        /*!< 0x02000000 */
#define EXTI_IMR_IM25                             EXTI_IMR_IM25_Msk                                 
#define EXTI_IMR_IM26_Pos                         (26U)
#define EXTI_IMR_IM26_Msk                         (0x1UL<<EXTI_IMR_IM26_Pos)                        /*!< 0x04000000 */
#define EXTI_IMR_IM26                             EXTI_IMR_IM26_Msk                                 
#define EXTI_IMR_IM27_Pos                         (27U)
#define EXTI_IMR_IM27_Msk                         (0x1UL<<EXTI_IMR_IM27_Pos)                        /*!< 0x08000000 */
#define EXTI_IMR_IM27                             EXTI_IMR_IM27_Msk                                 
#define EXTI_IMR_IM28_Pos                         (28U)
#define EXTI_IMR_IM28_Msk                         (0x1UL<<EXTI_IMR_IM28_Pos)                        /*!< 0x10000000 */
#define EXTI_IMR_IM28                             EXTI_IMR_IM28_Msk                                 

/********************************* Bit definition for EXTI_EMR register *********************************************/
#define EXTI_EMR_EM0_Pos                          (0U)
#define EXTI_EMR_EM0_Msk                          (0x1UL<<EXTI_EMR_EM0_Pos)                         /*!< 0x00000001 */
#define EXTI_EMR_EM0                              EXTI_EMR_EM0_Msk                                  
#define EXTI_EMR_EM1_Pos                          (1U)
#define EXTI_EMR_EM1_Msk                          (0x1UL<<EXTI_EMR_EM1_Pos)                         /*!< 0x00000002 */
#define EXTI_EMR_EM1                              EXTI_EMR_EM1_Msk                                  
#define EXTI_EMR_EM2_Pos                          (2U)
#define EXTI_EMR_EM2_Msk                          (0x1UL<<EXTI_EMR_EM2_Pos)                         /*!< 0x00000004 */
#define EXTI_EMR_EM2                              EXTI_EMR_EM2_Msk                                  
#define EXTI_EMR_EM3_Pos                          (3U)
#define EXTI_EMR_EM3_Msk                          (0x1UL<<EXTI_EMR_EM3_Pos)                         /*!< 0x00000008 */
#define EXTI_EMR_EM3                              EXTI_EMR_EM3_Msk                                  
#define EXTI_EMR_EM4_Pos                          (4U)
#define EXTI_EMR_EM4_Msk                          (0x1UL<<EXTI_EMR_EM4_Pos)                         /*!< 0x00000010 */
#define EXTI_EMR_EM4                              EXTI_EMR_EM4_Msk                                  
#define EXTI_EMR_EM5_Pos                          (5U)
#define EXTI_EMR_EM5_Msk                          (0x1UL<<EXTI_EMR_EM5_Pos)                         /*!< 0x00000020 */
#define EXTI_EMR_EM5                              EXTI_EMR_EM5_Msk                                  
#define EXTI_EMR_EM6_Pos                          (6U)
#define EXTI_EMR_EM6_Msk                          (0x1UL<<EXTI_EMR_EM6_Pos)                         /*!< 0x00000040 */
#define EXTI_EMR_EM6                              EXTI_EMR_EM6_Msk                                  
#define EXTI_EMR_EM7_Pos                          (7U)
#define EXTI_EMR_EM7_Msk                          (0x1UL<<EXTI_EMR_EM7_Pos)                         /*!< 0x00000080 */
#define EXTI_EMR_EM7                              EXTI_EMR_EM7_Msk                                  
#define EXTI_EMR_EM8_Pos                          (8U)
#define EXTI_EMR_EM8_Msk                          (0x1UL<<EXTI_EMR_EM8_Pos)                         /*!< 0x00000100 */
#define EXTI_EMR_EM8                              EXTI_EMR_EM8_Msk                                  
#define EXTI_EMR_EM9_Pos                          (9U)
#define EXTI_EMR_EM9_Msk                          (0x1UL<<EXTI_EMR_EM9_Pos)                         /*!< 0x00000200 */
#define EXTI_EMR_EM9                              EXTI_EMR_EM9_Msk                                  
#define EXTI_EMR_EM10_Pos                         (10U)
#define EXTI_EMR_EM10_Msk                         (0x1UL<<EXTI_EMR_EM10_Pos)                        /*!< 0x00000400 */
#define EXTI_EMR_EM10                             EXTI_EMR_EM10_Msk                                 
#define EXTI_EMR_EM11_Pos                         (11U)
#define EXTI_EMR_EM11_Msk                         (0x1UL<<EXTI_EMR_EM11_Pos)                        /*!< 0x00000800 */
#define EXTI_EMR_EM11                             EXTI_EMR_EM11_Msk                                 
#define EXTI_EMR_EM12_Pos                         (12U)
#define EXTI_EMR_EM12_Msk                         (0x1UL<<EXTI_EMR_EM12_Pos)                        /*!< 0x00001000 */
#define EXTI_EMR_EM12                             EXTI_EMR_EM12_Msk                                 
#define EXTI_EMR_EM13_Pos                         (13U)
#define EXTI_EMR_EM13_Msk                         (0x1UL<<EXTI_EMR_EM13_Pos)                        /*!< 0x00002000 */
#define EXTI_EMR_EM13                             EXTI_EMR_EM13_Msk                                 
#define EXTI_EMR_EM14_Pos                         (14U)
#define EXTI_EMR_EM14_Msk                         (0x1UL<<EXTI_EMR_EM14_Pos)                        /*!< 0x00004000 */
#define EXTI_EMR_EM14                             EXTI_EMR_EM14_Msk                                 
#define EXTI_EMR_EM15_Pos                         (15U)
#define EXTI_EMR_EM15_Msk                         (0x1UL<<EXTI_EMR_EM15_Pos)                        /*!< 0x00008000 */
#define EXTI_EMR_EM15                             EXTI_EMR_EM15_Msk                                 
#define EXTI_EMR_EM16_Pos                         (16U)
#define EXTI_EMR_EM16_Msk                         (0x1UL<<EXTI_EMR_EM16_Pos)                        /*!< 0x00010000 */
#define EXTI_EMR_EM16                             EXTI_EMR_EM16_Msk                                 
#define EXTI_EMR_EM17_Pos                         (17U)
#define EXTI_EMR_EM17_Msk                         (0x1UL<<EXTI_EMR_EM17_Pos)                        /*!< 0x00020000 */
#define EXTI_EMR_EM17                             EXTI_EMR_EM17_Msk                                 
#define EXTI_EMR_EM18_Pos                         (18U)
#define EXTI_EMR_EM18_Msk                         (0x1UL<<EXTI_EMR_EM18_Pos)                        /*!< 0x00040000 */
#define EXTI_EMR_EM18                             EXTI_EMR_EM18_Msk                                 
#define EXTI_EMR_EM19_Pos                         (19U)
#define EXTI_EMR_EM19_Msk                         (0x1UL<<EXTI_EMR_EM19_Pos)                        /*!< 0x00080000 */
#define EXTI_EMR_EM19                             EXTI_EMR_EM19_Msk                                 
#define EXTI_EMR_EM20_Pos                         (20U)
#define EXTI_EMR_EM20_Msk                         (0x1UL<<EXTI_EMR_EM20_Pos)                        /*!< 0x00100000 */
#define EXTI_EMR_EM20                             EXTI_EMR_EM20_Msk                                 
#define EXTI_EMR_EM21_Pos                         (21U)
#define EXTI_EMR_EM21_Msk                         (0x1UL<<EXTI_EMR_EM21_Pos)                        /*!< 0x00200000 */
#define EXTI_EMR_EM21                             EXTI_EMR_EM21_Msk                                 
#define EXTI_EMR_EM22_Pos                         (22U)
#define EXTI_EMR_EM22_Msk                         (0x1UL<<EXTI_EMR_EM22_Pos)                        /*!< 0x00400000 */
#define EXTI_EMR_EM22                             EXTI_EMR_EM22_Msk                                 
#define EXTI_EMR_EM23_Pos                         (23U)
#define EXTI_EMR_EM23_Msk                         (0x1UL<<EXTI_EMR_EM23_Pos)                        /*!< 0x00800000 */
#define EXTI_EMR_EM23                             EXTI_EMR_EM23_Msk                                 
#define EXTI_EMR_EM24_Pos                         (24U)
#define EXTI_EMR_EM24_Msk                         (0x1UL<<EXTI_EMR_EM24_Pos)                        /*!< 0x01000000 */
#define EXTI_EMR_EM24                             EXTI_EMR_EM24_Msk                                 
#define EXTI_EMR_EM25_Pos                         (25U)
#define EXTI_EMR_EM25_Msk                         (0x1UL<<EXTI_EMR_EM25_Pos)                        /*!< 0x02000000 */
#define EXTI_EMR_EM25                             EXTI_EMR_EM25_Msk                                 
#define EXTI_EMR_EM26_Pos                         (26U)
#define EXTI_EMR_EM26_Msk                         (0x1UL<<EXTI_EMR_EM26_Pos)                        /*!< 0x04000000 */
#define EXTI_EMR_EM26                             EXTI_EMR_EM26_Msk                                 
#define EXTI_EMR_EM27_Pos                         (27U)
#define EXTI_EMR_EM27_Msk                         (0x1UL<<EXTI_EMR_EM27_Pos)                        /*!< 0x08000000 */
#define EXTI_EMR_EM27                             EXTI_EMR_EM27_Msk                                 
#define EXTI_EMR_EM28_Pos                         (28U)
#define EXTI_EMR_EM28_Msk                         (0x1UL<<EXTI_EMR_EM28_Pos)                        /*!< 0x10000000 */
#define EXTI_EMR_EM28                             EXTI_EMR_EM28_Msk                                 

/********************************************************************************************************************/
/********************************* FDCAN ****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for FDCAN_CREL register *******************************************/
#define FDCAN_CREL_DAY_Pos                        (0U)
#define FDCAN_CREL_DAY_Msk                        (0xFFUL<<FDCAN_CREL_DAY_Pos)                      /*!< 0x000000FF */
#define FDCAN_CREL_DAY                            FDCAN_CREL_DAY_Msk                                /*!< - */
#define FDCAN_CREL_MON_Pos                        (8U)
#define FDCAN_CREL_MON_Msk                        (0xFFUL<<FDCAN_CREL_MON_Pos)                      /*!< 0x0000FF00 */
#define FDCAN_CREL_MON                            FDCAN_CREL_MON_Msk                                /*!< - */
#define FDCAN_CREL_YEAR_Pos                       (16U)
#define FDCAN_CREL_YEAR_Msk                       (0xFUL<<FDCAN_CREL_YEAR_Pos)                      /*!< 0x000F0000 */
#define FDCAN_CREL_YEAR                           FDCAN_CREL_YEAR_Msk                               /*!< - */
#define FDCAN_CREL_YEAR_0                         (0x1UL<<FDCAN_CREL_YEAR_Pos)                      /*!< 0x00010000 */
#define FDCAN_CREL_YEAR_1                         (0x2UL<<FDCAN_CREL_YEAR_Pos)                      /*!< 0x00020000 */
#define FDCAN_CREL_YEAR_2                         (0x4UL<<FDCAN_CREL_YEAR_Pos)                      /*!< 0x00040000 */
#define FDCAN_CREL_YEAR_3                         (0x8UL<<FDCAN_CREL_YEAR_Pos)                      /*!< 0x00080000 */
#define FDCAN_CREL_SUBSTEP_Pos                    (20U)
#define FDCAN_CREL_SUBSTEP_Msk                    (0xFUL<<FDCAN_CREL_SUBSTEP_Pos)                   /*!< 0x00F00000 */
#define FDCAN_CREL_SUBSTEP                        FDCAN_CREL_SUBSTEP_Msk                            /*!< - */
#define FDCAN_CREL_SUBSTEP_0                      (0x1UL<<FDCAN_CREL_SUBSTEP_Pos)                   /*!< 0x00100000 */
#define FDCAN_CREL_SUBSTEP_1                      (0x2UL<<FDCAN_CREL_SUBSTEP_Pos)                   /*!< 0x00200000 */
#define FDCAN_CREL_SUBSTEP_2                      (0x4UL<<FDCAN_CREL_SUBSTEP_Pos)                   /*!< 0x00400000 */
#define FDCAN_CREL_SUBSTEP_3                      (0x8UL<<FDCAN_CREL_SUBSTEP_Pos)                   /*!< 0x00800000 */
#define FDCAN_CREL_STEP_Pos                       (24U)
#define FDCAN_CREL_STEP_Msk                       (0xFUL<<FDCAN_CREL_STEP_Pos)                      /*!< 0x0F000000 */
#define FDCAN_CREL_STEP                           FDCAN_CREL_STEP_Msk                               /*!< - */
#define FDCAN_CREL_STEP_0                         (0x1UL<<FDCAN_CREL_STEP_Pos)                      /*!< 0x01000000 */
#define FDCAN_CREL_STEP_1                         (0x2UL<<FDCAN_CREL_STEP_Pos)                      /*!< 0x02000000 */
#define FDCAN_CREL_STEP_2                         (0x4UL<<FDCAN_CREL_STEP_Pos)                      /*!< 0x04000000 */
#define FDCAN_CREL_STEP_3                         (0x8UL<<FDCAN_CREL_STEP_Pos)                      /*!< 0x08000000 */
#define FDCAN_CREL_REL_Pos                        (28U)
#define FDCAN_CREL_REL_Msk                        (0xFUL<<FDCAN_CREL_REL_Pos)                       /*!< 0xF0000000 */
#define FDCAN_CREL_REL                            FDCAN_CREL_REL_Msk                                /*!< - */
#define FDCAN_CREL_REL_0                          (0x1UL<<FDCAN_CREL_REL_Pos)                       /*!< 0x10000000 */
#define FDCAN_CREL_REL_1                          (0x2UL<<FDCAN_CREL_REL_Pos)                       /*!< 0x20000000 */
#define FDCAN_CREL_REL_2                          (0x4UL<<FDCAN_CREL_REL_Pos)                       /*!< 0x40000000 */
#define FDCAN_CREL_REL_3                          (0x8UL<<FDCAN_CREL_REL_Pos)                       /*!< 0x80000000 */

/********************************* Bit definition for FDCAN_ENDN register *******************************************/
#define FDCAN_ENDN_ETV_Pos                        (0U)
#define FDCAN_ENDN_ETV_Msk                        (0xFFFFFFFFUL<<FDCAN_ENDN_ETV_Pos)                /*!< 0xFFFFFFFF */
#define FDCAN_ENDN_ETV                            FDCAN_ENDN_ETV_Msk                                /*!< Endian Register */

/********************************* Bit definition for FDCAN_DBTP register *******************************************/
#define FDCAN_DBTP_DSJW_Pos                       (0U)
#define FDCAN_DBTP_DSJW_Msk                       (0xFUL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x0000000F */
#define FDCAN_DBTP_DSJW                           FDCAN_DBTP_DSJW_Msk                               /*!< Data (Re)Synchronization Jump Width */
#define FDCAN_DBTP_DSJW_0                         (0x1UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000001 */
#define FDCAN_DBTP_DSJW_1                         (0x2UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000002 */
#define FDCAN_DBTP_DSJW_2                         (0x4UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000004 */
#define FDCAN_DBTP_DSJW_3                         (0x8UL<<FDCAN_DBTP_DSJW_Pos)                      /*!< 0x00000008 */
#define FDCAN_DBTP_DTSEG2_Pos                     (4U)
#define FDCAN_DBTP_DTSEG2_Msk                     (0xFUL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x000000F0 */
#define FDCAN_DBTP_DTSEG2                         FDCAN_DBTP_DTSEG2_Msk                             /*!< Data time segment after sample point */
#define FDCAN_DBTP_DTSEG2_0                       (0x1UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000010 */
#define FDCAN_DBTP_DTSEG2_1                       (0x2UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000020 */
#define FDCAN_DBTP_DTSEG2_2                       (0x4UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000040 */
#define FDCAN_DBTP_DTSEG2_3                       (0x8UL<<FDCAN_DBTP_DTSEG2_Pos)                    /*!< 0x00000080 */
#define FDCAN_DBTP_DTSEG1_Pos                     (8U)
#define FDCAN_DBTP_DTSEG1_Msk                     (0x1FUL<<FDCAN_DBTP_DTSEG1_Pos)                   /*!< 0x00001F00 */
#define FDCAN_DBTP_DTSEG1                         FDCAN_DBTP_DTSEG1_Msk                             /*!< Data time segment before sample point */
#define FDCAN_DBTP_DTSEG1_0                       (0x1UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000100 */
#define FDCAN_DBTP_DTSEG1_1                       (0x2UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000200 */
#define FDCAN_DBTP_DTSEG1_2                       (0x4UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000400 */
#define FDCAN_DBTP_DTSEG1_3                       (0x8UL<<FDCAN_DBTP_DTSEG1_Pos)                    /*!< 0x00000800 */
#define FDCAN_DBTP_DTSEG1_4                       (0x10UL<<FDCAN_DBTP_DTSEG1_Pos)                   /*!< 0x00001000 */
#define FDCAN_DBTP_DBRP_Pos                       (16U)
#define FDCAN_DBTP_DBRP_Msk                       (0x1FUL<<FDCAN_DBTP_DBRP_Pos)                     /*!< 0x001F0000 */
#define FDCAN_DBTP_DBRP                           FDCAN_DBTP_DBRP_Msk                               /*!< Data Bit Rate Prescaler */
#define FDCAN_DBTP_DBRP_0                         (0x1UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00010000 */
#define FDCAN_DBTP_DBRP_1                         (0x2UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00020000 */
#define FDCAN_DBTP_DBRP_2                         (0x4UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00040000 */
#define FDCAN_DBTP_DBRP_3                         (0x8UL<<FDCAN_DBTP_DBRP_Pos)                      /*!< 0x00080000 */
#define FDCAN_DBTP_DBRP_4                         (0x10UL<<FDCAN_DBTP_DBRP_Pos)                     /*!< 0x00100000 */
#define FDCAN_DBTP_TDC_Pos                        (23U)
#define FDCAN_DBTP_TDC_Msk                        (0x1UL<<FDCAN_DBTP_TDC_Pos)                       /*!< 0x00800000 */
#define FDCAN_DBTP_TDC                            FDCAN_DBTP_TDC_Msk                                /*!< Transmitter Delay Compensation */

/********************************* Bit definition for FDCAN_TEST register *******************************************/
#define FDCAN_TEST_LBCK_Pos                       (4U)
#define FDCAN_TEST_LBCK_Msk                       (0x1UL<<FDCAN_TEST_LBCK_Pos)                      /*!< 0x00000010 */
#define FDCAN_TEST_LBCK                           FDCAN_TEST_LBCK_Msk                               /*!< Loop Back Mode */
#define FDCAN_TEST_TX_Pos                         (5U)
#define FDCAN_TEST_TX_Msk                         (0x3UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000060 */
#define FDCAN_TEST_TX                             FDCAN_TEST_TX_Msk                                 /*!< Control of Transmit Pin */
#define FDCAN_TEST_TX_0                           (0x1UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000020 */
#define FDCAN_TEST_TX_1                           (0x2UL<<FDCAN_TEST_TX_Pos)                        /*!< 0x00000040 */
#define FDCAN_TEST_RX_Pos                         (7U)
#define FDCAN_TEST_RX_Msk                         (0x1UL<<FDCAN_TEST_RX_Pos)                        /*!< 0x00000080 */
#define FDCAN_TEST_RX                             FDCAN_TEST_RX_Msk                                 /*!< Receive Pin */
#define FDCAN_TEST_TXBNP_Pos                      (8U)
#define FDCAN_TEST_TXBNP_Msk                      (0x1FUL<<FDCAN_TEST_TXBNP_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TEST_TXBNP                          FDCAN_TEST_TXBNP_Msk                              /*!< Tx Buffer Number Prepared */
#define FDCAN_TEST_TXBNP_0                        (0x1UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000100 */
#define FDCAN_TEST_TXBNP_1                        (0x2UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000200 */
#define FDCAN_TEST_TXBNP_2                        (0x4UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000400 */
#define FDCAN_TEST_TXBNP_3                        (0x8UL<<FDCAN_TEST_TXBNP_Pos)                     /*!< 0x00000800 */
#define FDCAN_TEST_TXBNP_4                        (0x10UL<<FDCAN_TEST_TXBNP_Pos)                    /*!< 0x00001000 */
#define FDCAN_TEST_PVAL_Pos                       (13U)
#define FDCAN_TEST_PVAL_Msk                       (0x1UL<<FDCAN_TEST_PVAL_Pos)                      /*!< 0x00002000 */
#define FDCAN_TEST_PVAL                           FDCAN_TEST_PVAL_Msk                               /*!< Prepared Valid */
#define FDCAN_TEST_TXBNS_Pos                      (16U)
#define FDCAN_TEST_TXBNS_Msk                      (0x1FUL<<FDCAN_TEST_TXBNS_Pos)                    /*!< 0x001F0000 */
#define FDCAN_TEST_TXBNS                          FDCAN_TEST_TXBNS_Msk                              /*!< Tx Buffer Number Started */
#define FDCAN_TEST_TXBNS_0                        (0x1UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00010000 */
#define FDCAN_TEST_TXBNS_1                        (0x2UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00020000 */
#define FDCAN_TEST_TXBNS_2                        (0x4UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00040000 */
#define FDCAN_TEST_TXBNS_3                        (0x8UL<<FDCAN_TEST_TXBNS_Pos)                     /*!< 0x00080000 */
#define FDCAN_TEST_TXBNS_4                        (0x10UL<<FDCAN_TEST_TXBNS_Pos)                    /*!< 0x00100000 */
#define FDCAN_TEST_SVAL_Pos                       (21U)
#define FDCAN_TEST_SVAL_Msk                       (0x1UL<<FDCAN_TEST_SVAL_Pos)                      /*!< 0x00200000 */
#define FDCAN_TEST_SVAL                           FDCAN_TEST_SVAL_Msk                               /*!< Started Valid */

/********************************* Bit definition for FDCAN_RWD register ********************************************/
#define FDCAN_RWD_WDC_Pos                         (0U)
#define FDCAN_RWD_WDC_Msk                         (0xFFUL<<FDCAN_RWD_WDC_Pos)                       /*!< 0x000000FF */
#define FDCAN_RWD_WDC                             FDCAN_RWD_WDC_Msk                                 /*!< Watchdog Configuration */
#define FDCAN_RWD_WDV_Pos                         (8U)
#define FDCAN_RWD_WDV_Msk                         (0xFFUL<<FDCAN_RWD_WDV_Pos)                       /*!< 0x0000FF00 */
#define FDCAN_RWD_WDV                             FDCAN_RWD_WDV_Msk                                 /*!< Watchdog Value */

/********************************* Bit definition for FDCAN_CCCR register *******************************************/
#define FDCAN_CCCR_INIT_Pos                       (0U)
#define FDCAN_CCCR_INIT_Msk                       (0x1UL<<FDCAN_CCCR_INIT_Pos)                      /*!< 0x00000001 */
#define FDCAN_CCCR_INIT                           FDCAN_CCCR_INIT_Msk                               /*!< Initialization */
#define FDCAN_CCCR_CCE_Pos                        (1U)
#define FDCAN_CCCR_CCE_Msk                        (0x1UL<<FDCAN_CCCR_CCE_Pos)                       /*!< 0x00000002 */
#define FDCAN_CCCR_CCE                            FDCAN_CCCR_CCE_Msk                                /*!< Configuration Change Enable */
#define FDCAN_CCCR_ASM_Pos                        (2U)
#define FDCAN_CCCR_ASM_Msk                        (0x1UL<<FDCAN_CCCR_ASM_Pos)                       /*!< 0x00000004 */
#define FDCAN_CCCR_ASM                            FDCAN_CCCR_ASM_Msk                                /*!< Restricted Operation Mode */
#define FDCAN_CCCR_CSA_Pos                        (3U)
#define FDCAN_CCCR_CSA_Msk                        (0x1UL<<FDCAN_CCCR_CSA_Pos)                       /*!< 0x00000008 */
#define FDCAN_CCCR_CSA                            FDCAN_CCCR_CSA_Msk                                /*!< Clock Stop Acknowledge */
#define FDCAN_CCCR_CSR_Pos                        (4U)
#define FDCAN_CCCR_CSR_Msk                        (0x1UL<<FDCAN_CCCR_CSR_Pos)                       /*!< 0x00000010 */
#define FDCAN_CCCR_CSR                            FDCAN_CCCR_CSR_Msk                                /*!< Clock Stop Request */
#define FDCAN_CCCR_MON_Pos                        (5U)
#define FDCAN_CCCR_MON_Msk                        (0x1UL<<FDCAN_CCCR_MON_Pos)                       /*!< 0x00000020 */
#define FDCAN_CCCR_MON                            FDCAN_CCCR_MON_Msk                                /*!< Bus Monitoring Mode */
#define FDCAN_CCCR_DAR_Pos                        (6U)
#define FDCAN_CCCR_DAR_Msk                        (0x1UL<<FDCAN_CCCR_DAR_Pos)                       /*!< 0x00000040 */
#define FDCAN_CCCR_DAR                            FDCAN_CCCR_DAR_Msk                                /*!< Disable Automatic Retransmission */
#define FDCAN_CCCR_TEST_Pos                       (7U)
#define FDCAN_CCCR_TEST_Msk                       (0x1UL<<FDCAN_CCCR_TEST_Pos)                      /*!< 0x00000080 */
#define FDCAN_CCCR_TEST                           FDCAN_CCCR_TEST_Msk                               /*!< Test Mode Enable */
#define FDCAN_CCCR_FDOE_Pos                       (8U)
#define FDCAN_CCCR_FDOE_Msk                       (0x1UL<<FDCAN_CCCR_FDOE_Pos)                      /*!< 0x00000100 */
#define FDCAN_CCCR_FDOE                           FDCAN_CCCR_FDOE_Msk                               /*!< FD Operation Enable */
#define FDCAN_CCCR_BRSE_Pos                       (9U)
#define FDCAN_CCCR_BRSE_Msk                       (0x1UL<<FDCAN_CCCR_BRSE_Pos)                      /*!< 0x00000200 */
#define FDCAN_CCCR_BRSE                           FDCAN_CCCR_BRSE_Msk                               /*!< Bit Rate Switch Enable */
#define FDCAN_CCCR_WMM_Pos                        (11U)
#define FDCAN_CCCR_WMM_Msk                        (0x1UL<<FDCAN_CCCR_WMM_Pos)                       /*!< 0x00000800 */
#define FDCAN_CCCR_WMM                            FDCAN_CCCR_WMM_Msk                                /*!< Wide Message Marker */
#define FDCAN_CCCR_PXHD_Pos                       (12U)
#define FDCAN_CCCR_PXHD_Msk                       (0x1UL<<FDCAN_CCCR_PXHD_Pos)                      /*!< 0x00001000 */
#define FDCAN_CCCR_PXHD                           FDCAN_CCCR_PXHD_Msk                               /*!< Protocol Exception Handling Disable */
#define FDCAN_CCCR_EFBI_Pos                       (13U)
#define FDCAN_CCCR_EFBI_Msk                       (0x1UL<<FDCAN_CCCR_EFBI_Pos)                      /*!< 0x00002000 */
#define FDCAN_CCCR_EFBI                           FDCAN_CCCR_EFBI_Msk                               /*!< Edge Filtering during Bus Integration */
#define FDCAN_CCCR_TXP_Pos                        (14U)
#define FDCAN_CCCR_TXP_Msk                        (0x1UL<<FDCAN_CCCR_TXP_Pos)                       /*!< 0x00004000 */
#define FDCAN_CCCR_TXP                            FDCAN_CCCR_TXP_Msk                                /*!< Transmit Pause */
#define FDCAN_CCCR_NISO_Pos                       (15U)
#define FDCAN_CCCR_NISO_Msk                       (0x1UL<<FDCAN_CCCR_NISO_Pos)                      /*!< 0x00008000 */
#define FDCAN_CCCR_NISO                           FDCAN_CCCR_NISO_Msk                               /*!< Non ISO Operation */

/********************************* Bit definition for FDCAN_NBTP register *******************************************/
#define FDCAN_NBTP_NTSEG2_Pos                     (0U)
#define FDCAN_NBTP_NTSEG2_Msk                     (0x7FUL<<FDCAN_NBTP_NTSEG2_Pos)                   /*!< 0x0000007F */
#define FDCAN_NBTP_NTSEG2                         FDCAN_NBTP_NTSEG2_Msk                             /*!< Nominal Time segment after sample point */
#define FDCAN_NBTP_NTSEG1_Pos                     (8U)
#define FDCAN_NBTP_NTSEG1_Msk                     (0xFFUL<<FDCAN_NBTP_NTSEG1_Pos)                   /*!< 0x0000FF00 */
#define FDCAN_NBTP_NTSEG1                         FDCAN_NBTP_NTSEG1_Msk                             /*!< Nominal Time segment before sample point */
#define FDCAN_NBTP_NBRP_Pos                       (16U)
#define FDCAN_NBTP_NBRP_Msk                       (0x1FFUL<<FDCAN_NBTP_NBRP_Pos)                    /*!< 0x01FF0000 */
#define FDCAN_NBTP_NBRP                           FDCAN_NBTP_NBRP_Msk                               /*!< Nominal Bit Rate Prescaler */
#define FDCAN_NBTP_NSJW_Pos                       (25U)
#define FDCAN_NBTP_NSJW_Msk                       (0x7FUL<<FDCAN_NBTP_NSJW_Pos)                     /*!< 0xFE000000 */
#define FDCAN_NBTP_NSJW                           FDCAN_NBTP_NSJW_Msk                               /*!< Nominal (Re)Synchronization Jump Width */

/********************************* Bit definition for FDCAN_TSCC register *******************************************/
#define FDCAN_TSCC_TSS_Pos                        (0U)
#define FDCAN_TSCC_TSS_Msk                        (0x3UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000003 */
#define FDCAN_TSCC_TSS                            FDCAN_TSCC_TSS_Msk                                /*!< Timestamp Select */
#define FDCAN_TSCC_TSS_0                          (0x1UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000001 */
#define FDCAN_TSCC_TSS_1                          (0x2UL<<FDCAN_TSCC_TSS_Pos)                       /*!< 0x00000002 */
#define FDCAN_TSCC_TCP_Pos                        (16U)
#define FDCAN_TSCC_TCP_Msk                        (0xFUL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x000F0000 */
#define FDCAN_TSCC_TCP                            FDCAN_TSCC_TCP_Msk                                /*!< Timestamp Counter Prescaler */
#define FDCAN_TSCC_TCP_0                          (0x1UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00010000 */
#define FDCAN_TSCC_TCP_1                          (0x2UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00020000 */
#define FDCAN_TSCC_TCP_2                          (0x4UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00040000 */
#define FDCAN_TSCC_TCP_3                          (0x8UL<<FDCAN_TSCC_TCP_Pos)                       /*!< 0x00080000 */

/********************************* Bit definition for FDCAN_TSCV register *******************************************/
#define FDCAN_TSCV_TSC_Pos                        (0U)
#define FDCAN_TSCV_TSC_Msk                        (0xFFFFUL<<FDCAN_TSCV_TSC_Pos)                    /*!< 0x0000FFFF */
#define FDCAN_TSCV_TSC                            FDCAN_TSCV_TSC_Msk                                /*!< Timestamp Counter */

/********************************* Bit definition for FDCAN_TOCC register *******************************************/
#define FDCAN_TOCC_ETOC_Pos                       (0U)
#define FDCAN_TOCC_ETOC_Msk                       (0x1UL<<FDCAN_TOCC_ETOC_Pos)                      /*!< 0x00000001 */
#define FDCAN_TOCC_ETOC                           FDCAN_TOCC_ETOC_Msk                               /*!< Enable Timeout Counter */
#define FDCAN_TOCC_TOS_Pos                        (1U)
#define FDCAN_TOCC_TOS_Msk                        (0x3UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000006 */
#define FDCAN_TOCC_TOS                            FDCAN_TOCC_TOS_Msk                                /*!< Timeout Select */
#define FDCAN_TOCC_TOS_0                          (0x1UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000002 */
#define FDCAN_TOCC_TOS_1                          (0x2UL<<FDCAN_TOCC_TOS_Pos)                       /*!< 0x00000004 */
#define FDCAN_TOCC_TOP_Pos                        (16U)
#define FDCAN_TOCC_TOP_Msk                        (0xFFFFUL<<FDCAN_TOCC_TOP_Pos)                    /*!< 0xFFFF0000 */
#define FDCAN_TOCC_TOP                            FDCAN_TOCC_TOP_Msk                                /*!< Timeout Period */

/********************************* Bit definition for FDCAN_TOCV register *******************************************/
#define FDCAN_TOCV_TOC_Pos                        (0U)
#define FDCAN_TOCV_TOC_Msk                        (0xFFFFUL<<FDCAN_TOCV_TOC_Pos)                    /*!< 0x0000FFFF */
#define FDCAN_TOCV_TOC                            FDCAN_TOCV_TOC_Msk                                /*!< Timeout Counter */

/********************************* Bit definition for FDCAN_ECR register ********************************************/
#define FDCAN_ECR_TEC_Pos                         (0U)
#define FDCAN_ECR_TEC_Msk                         (0xFFUL<<FDCAN_ECR_TEC_Pos)                       /*!< 0x000000FF */
#define FDCAN_ECR_TEC                             FDCAN_ECR_TEC_Msk                                 /*!< Transmit Error Counter */
#define FDCAN_ECR_REC_Pos                         (8U)
#define FDCAN_ECR_REC_Msk                         (0x7FUL<<FDCAN_ECR_REC_Pos)                       /*!< 0x00007F00 */
#define FDCAN_ECR_REC                             FDCAN_ECR_REC_Msk                                 /*!< Receive Error Counter */
#define FDCAN_ECR_RP_Pos                          (15U)
#define FDCAN_ECR_RP_Msk                          (0x1UL<<FDCAN_ECR_RP_Pos)                         /*!< 0x00008000 */
#define FDCAN_ECR_RP                              FDCAN_ECR_RP_Msk                                  /*!< Receive Error Passive */
#define FDCAN_ECR_CEL_Pos                         (16U)
#define FDCAN_ECR_CEL_Msk                         (0xFFUL<<FDCAN_ECR_CEL_Pos)                       /*!< 0x00FF0000 */
#define FDCAN_ECR_CEL                             FDCAN_ECR_CEL_Msk                                 /*!< CAN Error Logging */

/********************************* Bit definition for FDCAN_PSR register ********************************************/
#define FDCAN_PSR_LEC_Pos                         (0U)
#define FDCAN_PSR_LEC_Msk                         (0x7UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000007 */
#define FDCAN_PSR_LEC                             FDCAN_PSR_LEC_Msk                                 /*!< Last Error Code */
#define FDCAN_PSR_LEC_0                           (0x1UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000001 */
#define FDCAN_PSR_LEC_1                           (0x2UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000002 */
#define FDCAN_PSR_LEC_2                           (0x4UL<<FDCAN_PSR_LEC_Pos)                        /*!< 0x00000004 */
#define FDCAN_PSR_ACT_Pos                         (3U)
#define FDCAN_PSR_ACT_Msk                         (0x3UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000018 */
#define FDCAN_PSR_ACT                             FDCAN_PSR_ACT_Msk                                 /*!< Activity */
#define FDCAN_PSR_ACT_0                           (0x1UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000008 */
#define FDCAN_PSR_ACT_1                           (0x2UL<<FDCAN_PSR_ACT_Pos)                        /*!< 0x00000010 */
#define FDCAN_PSR_EP_Pos                          (5U)
#define FDCAN_PSR_EP_Msk                          (0x1UL<<FDCAN_PSR_EP_Pos)                         /*!< 0x00000020 */
#define FDCAN_PSR_EP                              FDCAN_PSR_EP_Msk                                  /*!< Error Passive */
#define FDCAN_PSR_EW_Pos                          (6U)
#define FDCAN_PSR_EW_Msk                          (0x1UL<<FDCAN_PSR_EW_Pos)                         /*!< 0x00000040 */
#define FDCAN_PSR_EW                              FDCAN_PSR_EW_Msk                                  /*!< Warning Status */
#define FDCAN_PSR_BO_Pos                          (7U)
#define FDCAN_PSR_BO_Msk                          (0x1UL<<FDCAN_PSR_BO_Pos)                         /*!< 0x00000080 */
#define FDCAN_PSR_BO                              FDCAN_PSR_BO_Msk                                  /*!< Bus_Off Status */
#define FDCAN_PSR_DLEC_Pos                        (8U)
#define FDCAN_PSR_DLEC_Msk                        (0x7UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000700 */
#define FDCAN_PSR_DLEC                            FDCAN_PSR_DLEC_Msk                                /*!< Data Phase Last Error Code */
#define FDCAN_PSR_DLEC_0                          (0x1UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000100 */
#define FDCAN_PSR_DLEC_1                          (0x2UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000200 */
#define FDCAN_PSR_DLEC_2                          (0x4UL<<FDCAN_PSR_DLEC_Pos)                       /*!< 0x00000400 */
#define FDCAN_PSR_RESI_Pos                        (11U)
#define FDCAN_PSR_RESI_Msk                        (0x1UL<<FDCAN_PSR_RESI_Pos)                       /*!< 0x00000800 */
#define FDCAN_PSR_RESI                            FDCAN_PSR_RESI_Msk                                /*!< ESI flag of last received CAN FD Message */
#define FDCAN_PSR_RBRS_Pos                        (12U)
#define FDCAN_PSR_RBRS_Msk                        (0x1UL<<FDCAN_PSR_RBRS_Pos)                       /*!< 0x00001000 */
#define FDCAN_PSR_RBRS                            FDCAN_PSR_RBRS_Msk                                /*!< BRS flag of last received CAN FD Message */
#define FDCAN_PSR_RFDF_Pos                        (13U)
#define FDCAN_PSR_RFDF_Msk                        (0x1UL<<FDCAN_PSR_RFDF_Pos)                       /*!< 0x00002000 */
#define FDCAN_PSR_RFDF                            FDCAN_PSR_RFDF_Msk                                /*!< Received a CAN FD Message */
#define FDCAN_PSR_PXE_Pos                         (14U)
#define FDCAN_PSR_PXE_Msk                         (0x1UL<<FDCAN_PSR_PXE_Pos)                        /*!< 0x00004000 */
#define FDCAN_PSR_PXE                             FDCAN_PSR_PXE_Msk                                 /*!< Protocol Exception Event */
#define FDCAN_PSR_TDCV_Pos                        (16U)
#define FDCAN_PSR_TDCV_Msk                        (0x7FUL<<FDCAN_PSR_TDCV_Pos)                      /*!< 0x007F0000 */
#define FDCAN_PSR_TDCV                            FDCAN_PSR_TDCV_Msk                                /*!< Transmitter Delay Compensation Value */

/********************************* Bit definition for FDCAN_TDCR register *******************************************/
#define FDCAN_TDCR_TDCF_Pos                       (0U)
#define FDCAN_TDCR_TDCF_Msk                       (0x7FUL<<FDCAN_TDCR_TDCF_Pos)                     /*!< 0x0000007F */
#define FDCAN_TDCR_TDCF                           FDCAN_TDCR_TDCF_Msk                               /*!< Transmitter Delay Compensation Filter Window Length */
#define FDCAN_TDCR_TDCO_Pos                       (8U)
#define FDCAN_TDCR_TDCO_Msk                       (0x7FUL<<FDCAN_TDCR_TDCO_Pos)                     /*!< 0x00007F00 */
#define FDCAN_TDCR_TDCO                           FDCAN_TDCR_TDCO_Msk                               /*!< Transmitter Delay Compensation SSP Offset */

/********************************* Bit definition for FDCAN_IR register *********************************************/
#define FDCAN_IR_RF0N_Pos                         (0U)
#define FDCAN_IR_RF0N_Msk                         (0x1UL<<FDCAN_IR_RF0N_Pos)                        /*!< 0x00000001 */
#define FDCAN_IR_RF0N                             FDCAN_IR_RF0N_Msk                                 /*!< Rx FIFO 0 New Message */
#define FDCAN_IR_RF0W_Pos                         (1U)
#define FDCAN_IR_RF0W_Msk                         (0x1UL<<FDCAN_IR_RF0W_Pos)                        /*!< 0x00000002 */
#define FDCAN_IR_RF0W                             FDCAN_IR_RF0W_Msk                                 /*!< Rx FIFO 0 Watermark Reached */
#define FDCAN_IR_RF0F_Pos                         (2U)
#define FDCAN_IR_RF0F_Msk                         (0x1UL<<FDCAN_IR_RF0F_Pos)                        /*!< 0x00000004 */
#define FDCAN_IR_RF0F                             FDCAN_IR_RF0F_Msk                                 /*!< Rx FIFO 0 Full */
#define FDCAN_IR_RF0L_Pos                         (3U)
#define FDCAN_IR_RF0L_Msk                         (0x1UL<<FDCAN_IR_RF0L_Pos)                        /*!< 0x00000008 */
#define FDCAN_IR_RF0L                             FDCAN_IR_RF0L_Msk                                 /*!< Rx FIFO 0 Message Lost */
#define FDCAN_IR_RF1N_Pos                         (4U)
#define FDCAN_IR_RF1N_Msk                         (0x1UL<<FDCAN_IR_RF1N_Pos)                        /*!< 0x00000010 */
#define FDCAN_IR_RF1N                             FDCAN_IR_RF1N_Msk                                 /*!< Rx FIFO 1 New Message */
#define FDCAN_IR_RF1W_Pos                         (5U)
#define FDCAN_IR_RF1W_Msk                         (0x1UL<<FDCAN_IR_RF1W_Pos)                        /*!< 0x00000020 */
#define FDCAN_IR_RF1W                             FDCAN_IR_RF1W_Msk                                 /*!< Rx FIFO 1 Watermark Reached */
#define FDCAN_IR_RF1F_Pos                         (6U)
#define FDCAN_IR_RF1F_Msk                         (0x1UL<<FDCAN_IR_RF1F_Pos)                        /*!< 0x00000040 */
#define FDCAN_IR_RF1F                             FDCAN_IR_RF1F_Msk                                 /*!< Rx FIFO 1 Full */
#define FDCAN_IR_RF1L_Pos                         (7U)
#define FDCAN_IR_RF1L_Msk                         (0x1UL<<FDCAN_IR_RF1L_Pos)                        /*!< 0x00000080 */
#define FDCAN_IR_RF1L                             FDCAN_IR_RF1L_Msk                                 /*!< Rx FIFO 1 Message Lost */
#define FDCAN_IR_HPM_Pos                          (8U)
#define FDCAN_IR_HPM_Msk                          (0x1UL<<FDCAN_IR_HPM_Pos)                         /*!< 0x00000100 */
#define FDCAN_IR_HPM                              FDCAN_IR_HPM_Msk                                  /*!< High Priority Message */
#define FDCAN_IR_TC_Pos                           (9U)
#define FDCAN_IR_TC_Msk                           (0x1UL<<FDCAN_IR_TC_Pos)                          /*!< 0x00000200 */
#define FDCAN_IR_TC                               FDCAN_IR_TC_Msk                                   /*!< Transmission Completed */
#define FDCAN_IR_TCF_Pos                          (10U)
#define FDCAN_IR_TCF_Msk                          (0x1UL<<FDCAN_IR_TCF_Pos)                         /*!< 0x00000400 */
#define FDCAN_IR_TCF                              FDCAN_IR_TCF_Msk                                  /*!< Transmission Cancellation Finished */
#define FDCAN_IR_TFE_Pos                          (11U)
#define FDCAN_IR_TFE_Msk                          (0x1UL<<FDCAN_IR_TFE_Pos)                         /*!< 0x00000800 */
#define FDCAN_IR_TFE                              FDCAN_IR_TFE_Msk                                  /*!< Tx FIFO Empty */
#define FDCAN_IR_TEFN_Pos                         (12U)
#define FDCAN_IR_TEFN_Msk                         (0x1UL<<FDCAN_IR_TEFN_Pos)                        /*!< 0x00001000 */
#define FDCAN_IR_TEFN                             FDCAN_IR_TEFN_Msk                                 /*!< Tx Event FIFO New Entry */
#define FDCAN_IR_TEFW_Pos                         (13U)
#define FDCAN_IR_TEFW_Msk                         (0x1UL<<FDCAN_IR_TEFW_Pos)                        /*!< 0x00002000 */
#define FDCAN_IR_TEFW                             FDCAN_IR_TEFW_Msk                                 /*!< Tx Event FIFO Watermark Reached */
#define FDCAN_IR_TEFF_Pos                         (14U)
#define FDCAN_IR_TEFF_Msk                         (0x1UL<<FDCAN_IR_TEFF_Pos)                        /*!< 0x00004000 */
#define FDCAN_IR_TEFF                             FDCAN_IR_TEFF_Msk                                 /*!< Tx Event FIFO Full */
#define FDCAN_IR_TEFL_Pos                         (15U)
#define FDCAN_IR_TEFL_Msk                         (0x1UL<<FDCAN_IR_TEFL_Pos)                        /*!< 0x00008000 */
#define FDCAN_IR_TEFL                             FDCAN_IR_TEFL_Msk                                 /*!< Tx Event FIFO Element Lost */
#define FDCAN_IR_TSW_Pos                          (16U)
#define FDCAN_IR_TSW_Msk                          (0x1UL<<FDCAN_IR_TSW_Pos)                         /*!< 0x00010000 */
#define FDCAN_IR_TSW                              FDCAN_IR_TSW_Msk                                  /*!< Timestamp Wraparound */
#define FDCAN_IR_MRAF_Pos                         (17U)
#define FDCAN_IR_MRAF_Msk                         (0x1UL<<FDCAN_IR_MRAF_Pos)                        /*!< 0x00020000 */
#define FDCAN_IR_MRAF                             FDCAN_IR_MRAF_Msk                                 /*!< Message RAM Access Failure */
#define FDCAN_IR_TOO_Pos                          (18U)
#define FDCAN_IR_TOO_Msk                          (0x1UL<<FDCAN_IR_TOO_Pos)                         /*!< 0x00040000 */
#define FDCAN_IR_TOO                              FDCAN_IR_TOO_Msk                                  /*!< Timeout Occurred */
#define FDCAN_IR_DRX_Pos                          (19U)
#define FDCAN_IR_DRX_Msk                          (0x1UL<<FDCAN_IR_DRX_Pos)                         /*!< 0x00080000 */
#define FDCAN_IR_DRX                              FDCAN_IR_DRX_Msk                                  /*!< Message stored to Dedicated Rx Buffer */
#define FDCAN_IR_BEC_Pos                          (20U)
#define FDCAN_IR_BEC_Msk                          (0x1UL<<FDCAN_IR_BEC_Pos)                         /*!< 0x00100000 */
#define FDCAN_IR_BEC                              FDCAN_IR_BEC_Msk                                  /*!< Bit Error Corrected */
#define FDCAN_IR_BEU_Pos                          (21U)
#define FDCAN_IR_BEU_Msk                          (0x1UL<<FDCAN_IR_BEU_Pos)                         /*!< 0x00200000 */
#define FDCAN_IR_BEU                              FDCAN_IR_BEU_Msk                                  /*!< Bit Error Uncorrected */
#define FDCAN_IR_ELO_Pos                          (22U)
#define FDCAN_IR_ELO_Msk                          (0x1UL<<FDCAN_IR_ELO_Pos)                         /*!< 0x00400000 */
#define FDCAN_IR_ELO                              FDCAN_IR_ELO_Msk                                  /*!< Error Logging Overflow */
#define FDCAN_IR_EP_Pos                           (23U)
#define FDCAN_IR_EP_Msk                           (0x1UL<<FDCAN_IR_EP_Pos)                          /*!< 0x00800000 */
#define FDCAN_IR_EP                               FDCAN_IR_EP_Msk                                   /*!< Error Passive */
#define FDCAN_IR_EW_Pos                           (24U)
#define FDCAN_IR_EW_Msk                           (0x1UL<<FDCAN_IR_EW_Pos)                          /*!< 0x01000000 */
#define FDCAN_IR_EW                               FDCAN_IR_EW_Msk                                   /*!< Warning Status */
#define FDCAN_IR_BO_Pos                           (25U)
#define FDCAN_IR_BO_Msk                           (0x1UL<<FDCAN_IR_BO_Pos)                          /*!< 0x02000000 */
#define FDCAN_IR_BO                               FDCAN_IR_BO_Msk                                   /*!< Bus_Off Status */
#define FDCAN_IR_WDI_Pos                          (26U)
#define FDCAN_IR_WDI_Msk                          (0x1UL<<FDCAN_IR_WDI_Pos)                         /*!< 0x04000000 */
#define FDCAN_IR_WDI                              FDCAN_IR_WDI_Msk                                  /*!< Watchdog Interrupt */
#define FDCAN_IR_PEA_Pos                          (27U)
#define FDCAN_IR_PEA_Msk                          (0x1UL<<FDCAN_IR_PEA_Pos)                         /*!< 0x08000000 */
#define FDCAN_IR_PEA                              FDCAN_IR_PEA_Msk                                  /*!< Protocol Error in Arbitration Phase  */
#define FDCAN_IR_PED_Pos                          (28U)
#define FDCAN_IR_PED_Msk                          (0x1UL<<FDCAN_IR_PED_Pos)                         /*!< 0x10000000 */
#define FDCAN_IR_PED                              FDCAN_IR_PED_Msk                                  /*!< Protocol Error in Data Phase (Data Bit Time is used) */
#define FDCAN_IR_ARA_Pos                          (29U)
#define FDCAN_IR_ARA_Msk                          (0x1UL<<FDCAN_IR_ARA_Pos)                         /*!< 0x20000000 */
#define FDCAN_IR_ARA                              FDCAN_IR_ARA_Msk                                  /*!< Access to Reserved Address */

/********************************* Bit definition for FDCAN_IE register *********************************************/
#define FDCAN_IE_RF0NE_Pos                        (0U)
#define FDCAN_IE_RF0NE_Msk                        (0x1UL<<FDCAN_IE_RF0NE_Pos)                       /*!< 0x00000001 */
#define FDCAN_IE_RF0NE                            FDCAN_IE_RF0NE_Msk                                /*!< Rx FIFO 1 New Message Interrupt Enable */
#define FDCAN_IE_RF0WE_Pos                        (1U)
#define FDCAN_IE_RF0WE_Msk                        (0x1UL<<FDCAN_IE_RF0WE_Pos)                       /*!< 0x00000002 */
#define FDCAN_IE_RF0WE                            FDCAN_IE_RF0WE_Msk                                /*!< Rx FIFO 1 Watermark Reached Interrupt Enable */
#define FDCAN_IE_RF0FE_Pos                        (2U)
#define FDCAN_IE_RF0FE_Msk                        (0x1UL<<FDCAN_IE_RF0FE_Pos)                       /*!< 0x00000004 */
#define FDCAN_IE_RF0FE                            FDCAN_IE_RF0FE_Msk                                /*!< Rx FIFO 1 Full Interrupt Enable */
#define FDCAN_IE_RF0LE_Pos                        (3U)
#define FDCAN_IE_RF0LE_Msk                        (0x1UL<<FDCAN_IE_RF0LE_Pos)                       /*!< 0x00000008 */
#define FDCAN_IE_RF0LE                            FDCAN_IE_RF0LE_Msk                                /*!< Rx FIFO 1 Message Lost Interrupt Enable */
#define FDCAN_IE_RF1NE_Pos                        (4U)
#define FDCAN_IE_RF1NE_Msk                        (0x1UL<<FDCAN_IE_RF1NE_Pos)                       /*!< 0x00000010 */
#define FDCAN_IE_RF1NE                            FDCAN_IE_RF1NE_Msk                                /*!< Rx FIFO 2 New Message Interrupt Enable */
#define FDCAN_IE_RF1WE_Pos                        (5U)
#define FDCAN_IE_RF1WE_Msk                        (0x1UL<<FDCAN_IE_RF1WE_Pos)                       /*!< 0x00000020 */
#define FDCAN_IE_RF1WE                            FDCAN_IE_RF1WE_Msk                                /*!< Rx FIFO 2 Watermark Reached Interrupt Enable */
#define FDCAN_IE_RF1FE_Pos                        (6U)
#define FDCAN_IE_RF1FE_Msk                        (0x1UL<<FDCAN_IE_RF1FE_Pos)                       /*!< 0x00000040 */
#define FDCAN_IE_RF1FE                            FDCAN_IE_RF1FE_Msk                                /*!< Rx FIFO 2 Full Interrupt Enable */
#define FDCAN_IE_RF1LE_Pos                        (7U)
#define FDCAN_IE_RF1LE_Msk                        (0x1UL<<FDCAN_IE_RF1LE_Pos)                       /*!< 0x00000080 */
#define FDCAN_IE_RF1LE                            FDCAN_IE_RF1LE_Msk                                /*!< Rx FIFO 2 Message Lost Interrupt Enable */
#define FDCAN_IE_HPME_Pos                         (8U)
#define FDCAN_IE_HPME_Msk                         (0x1UL<<FDCAN_IE_HPME_Pos)                        /*!< 0x00000100 */
#define FDCAN_IE_HPME                             FDCAN_IE_HPME_Msk                                 /*!< High Priority Message Interrupt Enable */
#define FDCAN_IE_TCE_Pos                          (9U)
#define FDCAN_IE_TCE_Msk                          (0x1UL<<FDCAN_IE_TCE_Pos)                         /*!< 0x00000200 */
#define FDCAN_IE_TCE                              FDCAN_IE_TCE_Msk                                  /*!< Transmission Completed Interrupt Enable */
#define FDCAN_IE_TCFE_Pos                         (10U)
#define FDCAN_IE_TCFE_Msk                         (0x1UL<<FDCAN_IE_TCFE_Pos)                        /*!< 0x00000400 */
#define FDCAN_IE_TCFE                             FDCAN_IE_TCFE_Msk                                 /*!< Transmission Cancellation Finished Interrupt Enable */
#define FDCAN_IE_TFEE_Pos                         (11U)
#define FDCAN_IE_TFEE_Msk                         (0x1UL<<FDCAN_IE_TFEE_Pos)                        /*!< 0x00000800 */
#define FDCAN_IE_TFEE                             FDCAN_IE_TFEE_Msk                                 /*!< Tx FIFO Empty Interrupt Enable */
#define FDCAN_IE_TEFNE_Pos                        (12U)
#define FDCAN_IE_TEFNE_Msk                        (0x1UL<<FDCAN_IE_TEFNE_Pos)                       /*!< 0x00001000 */
#define FDCAN_IE_TEFNE                            FDCAN_IE_TEFNE_Msk                                /*!< Tx Event FIFO New Entry Interrupt Enable */
#define FDCAN_IE_TEFWE_Pos                        (13U)
#define FDCAN_IE_TEFWE_Msk                        (0x1UL<<FDCAN_IE_TEFWE_Pos)                       /*!< 0x00002000 */
#define FDCAN_IE_TEFWE                            FDCAN_IE_TEFWE_Msk                                /*!< Tx Event FIFO Watermark Reached Interrupt Enable */
#define FDCAN_IE_TEFFE_Pos                        (14U)
#define FDCAN_IE_TEFFE_Msk                        (0x1UL<<FDCAN_IE_TEFFE_Pos)                       /*!< 0x00004000 */
#define FDCAN_IE_TEFFE                            FDCAN_IE_TEFFE_Msk                                /*!< Tx Event FIFO Full Interrupt Enable */
#define FDCAN_IE_TEFLE_Pos                        (15U)
#define FDCAN_IE_TEFLE_Msk                        (0x1UL<<FDCAN_IE_TEFLE_Pos)                       /*!< 0x00008000 */
#define FDCAN_IE_TEFLE                            FDCAN_IE_TEFLE_Msk                                /*!< Tx Event FIFO Event Lost Interrupt Enable */
#define FDCAN_IE_TSWE_Pos                         (16U)
#define FDCAN_IE_TSWE_Msk                         (0x1UL<<FDCAN_IE_TSWE_Pos)                        /*!< 0x00010000 */
#define FDCAN_IE_TSWE                             FDCAN_IE_TSWE_Msk                                 /*!< Timestamp Wraparound Interrupt Enable */
#define FDCAN_IE_MRAFE_Pos                        (17U)
#define FDCAN_IE_MRAFE_Msk                        (0x1UL<<FDCAN_IE_MRAFE_Pos)                       /*!< 0x00020000 */
#define FDCAN_IE_MRAFE                            FDCAN_IE_MRAFE_Msk                                /*!< Message RAM Access Failure Interrupt Enable */
#define FDCAN_IE_TOOE_Pos                         (18U)
#define FDCAN_IE_TOOE_Msk                         (0x1UL<<FDCAN_IE_TOOE_Pos)                        /*!< 0x00040000 */
#define FDCAN_IE_TOOE                             FDCAN_IE_TOOE_Msk                                 /*!< Timeout Occurred Interrupt Enable */
#define FDCAN_IE_DRXE_Pos                         (19U)
#define FDCAN_IE_DRXE_Msk                         (0x1UL<<FDCAN_IE_DRXE_Pos)                        /*!< 0x00080000 */
#define FDCAN_IE_DRXE                             FDCAN_IE_DRXE_Msk                                 /*!< Message stored to Dedicated Rx Buffer Interrupt Enable */
#define FDCAN_IE_BECE_Pos                         (20U)
#define FDCAN_IE_BECE_Msk                         (0x1UL<<FDCAN_IE_BECE_Pos)                        /*!< 0x00100000 */
#define FDCAN_IE_BECE                             FDCAN_IE_BECE_Msk                                 /*!< Bit Error Corrected Interrupt Enable */
#define FDCAN_IE_BEUE_Pos                         (21U)
#define FDCAN_IE_BEUE_Msk                         (0x1UL<<FDCAN_IE_BEUE_Pos)                        /*!< 0x00200000 */
#define FDCAN_IE_BEUE                             FDCAN_IE_BEUE_Msk                                 /*!< Bit Error Uncorrected Interrupt Enable */
#define FDCAN_IE_ELOE_Pos                         (22U)
#define FDCAN_IE_ELOE_Msk                         (0x1UL<<FDCAN_IE_ELOE_Pos)                        /*!< 0x00400000 */
#define FDCAN_IE_ELOE                             FDCAN_IE_ELOE_Msk                                 /*!< Error Logging Overflow Interrupt Enable */
#define FDCAN_IE_EPE_Pos                          (23U)
#define FDCAN_IE_EPE_Msk                          (0x1UL<<FDCAN_IE_EPE_Pos)                         /*!< 0x00800000 */
#define FDCAN_IE_EPE                              FDCAN_IE_EPE_Msk                                  /*!< Error Passive Interrupt Enable */
#define FDCAN_IE_EWE_Pos                          (24U)
#define FDCAN_IE_EWE_Msk                          (0x1UL<<FDCAN_IE_EWE_Pos)                         /*!< 0x01000000 */
#define FDCAN_IE_EWE                              FDCAN_IE_EWE_Msk                                  /*!< Warning Status Interrupt Enable */
#define FDCAN_IE_BOE_Pos                          (25U)
#define FDCAN_IE_BOE_Msk                          (0x1UL<<FDCAN_IE_BOE_Pos)                         /*!< 0x02000000 */
#define FDCAN_IE_BOE                              FDCAN_IE_BOE_Msk                                  /*!< Bus_Off Status Interrupt Enable */
#define FDCAN_IE_WDIE_Pos                         (26U)
#define FDCAN_IE_WDIE_Msk                         (0x1UL<<FDCAN_IE_WDIE_Pos)                        /*!< 0x04000000 */
#define FDCAN_IE_WDIE                             FDCAN_IE_WDIE_Msk                                 /*!< Watchdog Interrupt Enable */
#define FDCAN_IE_PEAE_Pos                         (27U)
#define FDCAN_IE_PEAE_Msk                         (0x1UL<<FDCAN_IE_PEAE_Pos)                        /*!< 0x08000000 */
#define FDCAN_IE_PEAE                             FDCAN_IE_PEAE_Msk                                 /*!< Protocol Error in Arbitration Phase Enable */
#define FDCAN_IE_PEDE_Pos                         (28U)
#define FDCAN_IE_PEDE_Msk                         (0x1UL<<FDCAN_IE_PEDE_Pos)                        /*!< 0x10000000 */
#define FDCAN_IE_PEDE                             FDCAN_IE_PEDE_Msk                                 /*!< Protocol Error in Data Phase Enable */
#define FDCAN_IE_ARAE_Pos                         (29U)
#define FDCAN_IE_ARAE_Msk                         (0x1UL<<FDCAN_IE_ARAE_Pos)                        /*!< 0x20000000 */
#define FDCAN_IE_ARAE                             FDCAN_IE_ARAE_Msk                                 /*!< Access to Reserved Address Enable */

/********************************* Bit definition for FDCAN_ILS register ********************************************/
#define FDCAN_ILS_RF0NL_Pos                       (0U)
#define FDCAN_ILS_RF0NL_Msk                       (0x1UL<<FDCAN_ILS_RF0NL_Pos)                      /*!< 0x00000001 */
#define FDCAN_ILS_RF0NL                           FDCAN_ILS_RF0NL_Msk                               /*!< Rx FIFO 1 New Message Interrupt Line */
#define FDCAN_ILS_RF0WL_Pos                       (1U)
#define FDCAN_ILS_RF0WL_Msk                       (0x1UL<<FDCAN_ILS_RF0WL_Pos)                      /*!< 0x00000002 */
#define FDCAN_ILS_RF0WL                           FDCAN_ILS_RF0WL_Msk                               /*!< Rx FIFO 1 Watermark Reached Interrupt Line */
#define FDCAN_ILS_RF0FL_Pos                       (2U)
#define FDCAN_ILS_RF0FL_Msk                       (0x1UL<<FDCAN_ILS_RF0FL_Pos)                      /*!< 0x00000004 */
#define FDCAN_ILS_RF0FL                           FDCAN_ILS_RF0FL_Msk                               /*!< Rx FIFO 1 Full Interrupt Line */
#define FDCAN_ILS_RF0LL_Pos                       (3U)
#define FDCAN_ILS_RF0LL_Msk                       (0x1UL<<FDCAN_ILS_RF0LL_Pos)                      /*!< 0x00000008 */
#define FDCAN_ILS_RF0LL                           FDCAN_ILS_RF0LL_Msk                               /*!< Rx FIFO 1 Message Lost Interrupt Line */
#define FDCAN_ILS_RF1NL_Pos                       (4U)
#define FDCAN_ILS_RF1NL_Msk                       (0x1UL<<FDCAN_ILS_RF1NL_Pos)                      /*!< 0x00000010 */
#define FDCAN_ILS_RF1NL                           FDCAN_ILS_RF1NL_Msk                               /*!< Rx FIFO 2 New Message Interrupt Line */
#define FDCAN_ILS_RF1WL_Pos                       (5U)
#define FDCAN_ILS_RF1WL_Msk                       (0x1UL<<FDCAN_ILS_RF1WL_Pos)                      /*!< 0x00000020 */
#define FDCAN_ILS_RF1WL                           FDCAN_ILS_RF1WL_Msk                               /*!< Rx FIFO 2 Watermark Reached Interrupt Line */
#define FDCAN_ILS_RF1FL_Pos                       (6U)
#define FDCAN_ILS_RF1FL_Msk                       (0x1UL<<FDCAN_ILS_RF1FL_Pos)                      /*!< 0x00000040 */
#define FDCAN_ILS_RF1FL                           FDCAN_ILS_RF1FL_Msk                               /*!< Rx FIFO 2 Full Interrupt Line */
#define FDCAN_ILS_RF1LL_Pos                       (7U)
#define FDCAN_ILS_RF1LL_Msk                       (0x1UL<<FDCAN_ILS_RF1LL_Pos)                      /*!< 0x00000080 */
#define FDCAN_ILS_RF1LL                           FDCAN_ILS_RF1LL_Msk                               /*!< Rx FIFO 2 Message Lost Interrupt Line */
#define FDCAN_ILS_HPML_Pos                        (8U)
#define FDCAN_ILS_HPML_Msk                        (0x1UL<<FDCAN_ILS_HPML_Pos)                       /*!< 0x00000100 */
#define FDCAN_ILS_HPML                            FDCAN_ILS_HPML_Msk                                /*!< High Priority Message Interrupt Line */
#define FDCAN_ILS_TCL_Pos                         (9U)
#define FDCAN_ILS_TCL_Msk                         (0x1UL<<FDCAN_ILS_TCL_Pos)                        /*!< 0x00000200 */
#define FDCAN_ILS_TCL                             FDCAN_ILS_TCL_Msk                                 /*!< Transmission Completed Interrupt Line */
#define FDCAN_ILS_TCFL_Pos                        (10U)
#define FDCAN_ILS_TCFL_Msk                        (0x1UL<<FDCAN_ILS_TCFL_Pos)                       /*!< 0x00000400 */
#define FDCAN_ILS_TCFL                            FDCAN_ILS_TCFL_Msk                                /*!< Transmission Cancellation Finished Interrupt Line */
#define FDCAN_ILS_TFEL_Pos                        (11U)
#define FDCAN_ILS_TFEL_Msk                        (0x1UL<<FDCAN_ILS_TFEL_Pos)                       /*!< 0x00000800 */
#define FDCAN_ILS_TFEL                            FDCAN_ILS_TFEL_Msk                                /*!< Tx FIFO Empty Interrupt Line */
#define FDCAN_ILS_TEFNL_Pos                       (12U)
#define FDCAN_ILS_TEFNL_Msk                       (0x1UL<<FDCAN_ILS_TEFNL_Pos)                      /*!< 0x00001000 */
#define FDCAN_ILS_TEFNL                           FDCAN_ILS_TEFNL_Msk                               /*!< Tx Event FIFO New Entry Interrupt Line */
#define FDCAN_ILS_TEFWL_Pos                       (13U)
#define FDCAN_ILS_TEFWL_Msk                       (0x1UL<<FDCAN_ILS_TEFWL_Pos)                      /*!< 0x00002000 */
#define FDCAN_ILS_TEFWL                           FDCAN_ILS_TEFWL_Msk                               /*!< Tx Event FIFO Watermark Reached Interrupt Line */
#define FDCAN_ILS_TEFFL_Pos                       (14U)
#define FDCAN_ILS_TEFFL_Msk                       (0x1UL<<FDCAN_ILS_TEFFL_Pos)                      /*!< 0x00004000 */
#define FDCAN_ILS_TEFFL                           FDCAN_ILS_TEFFL_Msk                               /*!< Tx Event FIFO Full Interrupt Line */
#define FDCAN_ILS_TEFLL_Pos                       (15U)
#define FDCAN_ILS_TEFLL_Msk                       (0x1UL<<FDCAN_ILS_TEFLL_Pos)                      /*!< 0x00008000 */
#define FDCAN_ILS_TEFLL                           FDCAN_ILS_TEFLL_Msk                               /*!< Tx Event FIFO Event Lost Interrupt Line */
#define FDCAN_ILS_TSWL_Pos                        (16U)
#define FDCAN_ILS_TSWL_Msk                        (0x1UL<<FDCAN_ILS_TSWL_Pos)                       /*!< 0x00010000 */
#define FDCAN_ILS_TSWL                            FDCAN_ILS_TSWL_Msk                                /*!< Timestamp Wraparound Interrupt Line */
#define FDCAN_ILS_MRAFL_Pos                       (17U)
#define FDCAN_ILS_MRAFL_Msk                       (0x1UL<<FDCAN_ILS_MRAFL_Pos)                      /*!< 0x00020000 */
#define FDCAN_ILS_MRAFL                           FDCAN_ILS_MRAFL_Msk                               /*!< Message RAM Access Failure Interrupt Line */
#define FDCAN_ILS_TOOL_Pos                        (18U)
#define FDCAN_ILS_TOOL_Msk                        (0x1UL<<FDCAN_ILS_TOOL_Pos)                       /*!< 0x00040000 */
#define FDCAN_ILS_TOOL                            FDCAN_ILS_TOOL_Msk                                /*!< Timeout Occurred Interrupt Line */
#define FDCAN_ILS_DRXL_Pos                        (19U)
#define FDCAN_ILS_DRXL_Msk                        (0x1UL<<FDCAN_ILS_DRXL_Pos)                       /*!< 0x00080000 */
#define FDCAN_ILS_DRXL                            FDCAN_ILS_DRXL_Msk                                /*!< Message stored to Dedicated Rx Buffer Interrupt Line */
#define FDCAN_ILS_BECL_Pos                        (20U)
#define FDCAN_ILS_BECL_Msk                        (0x1UL<<FDCAN_ILS_BECL_Pos)                       /*!< 0x00100000 */
#define FDCAN_ILS_BECL                            FDCAN_ILS_BECL_Msk                                /*!< Bit Error Corrected Interrupt Line */
#define FDCAN_ILS_BEUL_Pos                        (21U)
#define FDCAN_ILS_BEUL_Msk                        (0x1UL<<FDCAN_ILS_BEUL_Pos)                       /*!< 0x00200000 */
#define FDCAN_ILS_BEUL                            FDCAN_ILS_BEUL_Msk                                /*!< Bit Error Uncorrected Interrupt Line */
#define FDCAN_ILS_ELOL_Pos                        (22U)
#define FDCAN_ILS_ELOL_Msk                        (0x1UL<<FDCAN_ILS_ELOL_Pos)                       /*!< 0x00400000 */
#define FDCAN_ILS_ELOL                            FDCAN_ILS_ELOL_Msk                                /*!< Error Logging Overflow Interrupt Line */
#define FDCAN_ILS_EPL_Pos                         (23U)
#define FDCAN_ILS_EPL_Msk                         (0x1UL<<FDCAN_ILS_EPL_Pos)                        /*!< 0x00800000 */
#define FDCAN_ILS_EPL                             FDCAN_ILS_EPL_Msk                                 /*!< Error Passive Interrupt Line */
#define FDCAN_ILS_EWL_Pos                         (24U)
#define FDCAN_ILS_EWL_Msk                         (0x1UL<<FDCAN_ILS_EWL_Pos)                        /*!< 0x01000000 */
#define FDCAN_ILS_EWL                             FDCAN_ILS_EWL_Msk                                 /*!< Warning Status Interrupt Line */
#define FDCAN_ILS_BOL_Pos                         (25U)
#define FDCAN_ILS_BOL_Msk                         (0x1UL<<FDCAN_ILS_BOL_Pos)                        /*!< 0x02000000 */
#define FDCAN_ILS_BOL                             FDCAN_ILS_BOL_Msk                                 /*!< Bus_Off Status Interrupt Line */
#define FDCAN_ILS_WDIL_Pos                        (26U)
#define FDCAN_ILS_WDIL_Msk                        (0x1UL<<FDCAN_ILS_WDIL_Pos)                       /*!< 0x04000000 */
#define FDCAN_ILS_WDIL                            FDCAN_ILS_WDIL_Msk                                /*!< Watchdog Interrupt Line */
#define FDCAN_ILS_PEAL_Pos                        (27U)
#define FDCAN_ILS_PEAL_Msk                        (0x1UL<<FDCAN_ILS_PEAL_Pos)                       /*!< 0x08000000 */
#define FDCAN_ILS_PEAL                            FDCAN_ILS_PEAL_Msk                                /*!< Protocol Error in Arbitration Phase Line */
#define FDCAN_ILS_PEDL_Pos                        (28U)
#define FDCAN_ILS_PEDL_Msk                        (0x1UL<<FDCAN_ILS_PEDL_Pos)                       /*!< 0x10000000 */
#define FDCAN_ILS_PEDL                            FDCAN_ILS_PEDL_Msk                                /*!< Protocol Error in Data Phase Line */
#define FDCAN_ILS_ARAL_Pos                        (29U)
#define FDCAN_ILS_ARAL_Msk                        (0x1UL<<FDCAN_ILS_ARAL_Pos)                       /*!< 0x20000000 */
#define FDCAN_ILS_ARAL                            FDCAN_ILS_ARAL_Msk                                /*!< Access to Reserved Address Line */

/********************************* Bit definition for FDCAN_ILE register ********************************************/
#define FDCAN_ILE_EINT0_Pos                       (0U)
#define FDCAN_ILE_EINT0_Msk                       (0x1UL<<FDCAN_ILE_EINT0_Pos)                      /*!< 0x00000001 */
#define FDCAN_ILE_EINT0                           FDCAN_ILE_EINT0_Msk                               /*!< Enable Interrupt Line 0 */
#define FDCAN_ILE_EINT1_Pos                       (1U)
#define FDCAN_ILE_EINT1_Msk                       (0x1UL<<FDCAN_ILE_EINT1_Pos)                      /*!< 0x00000002 */
#define FDCAN_ILE_EINT1                           FDCAN_ILE_EINT1_Msk                               /*!< Enable Interrupt Line 1 */

/********************************* Bit definition for FDCAN_GFC register ********************************************/
#define FDCAN_GFC_RRFE_Pos                        (0U)
#define FDCAN_GFC_RRFE_Msk                        (0x1UL<<FDCAN_GFC_RRFE_Pos)                       /*!< 0x00000001 */
#define FDCAN_GFC_RRFE                            FDCAN_GFC_RRFE_Msk                                /*!< Reject Remote Frames Extended */
#define FDCAN_GFC_RRFS_Pos                        (1U)
#define FDCAN_GFC_RRFS_Msk                        (0x1UL<<FDCAN_GFC_RRFS_Pos)                       /*!< 0x00000002 */
#define FDCAN_GFC_RRFS                            FDCAN_GFC_RRFS_Msk                                /*!< Reject Remote Frames Standard */
#define FDCAN_GFC_ANFE_Pos                        (2U)
#define FDCAN_GFC_ANFE_Msk                        (0x3UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x0000000C */
#define FDCAN_GFC_ANFE                            FDCAN_GFC_ANFE_Msk                                /*!< Accept Non-matching Frames Extended */
#define FDCAN_GFC_ANFE_0                          (0x1UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x00000004 */
#define FDCAN_GFC_ANFE_1                          (0x2UL<<FDCAN_GFC_ANFE_Pos)                       /*!< 0x00000008 */
#define FDCAN_GFC_ANFS_Pos                        (4U)
#define FDCAN_GFC_ANFS_Msk                        (0x3UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000030 */
#define FDCAN_GFC_ANFS                            FDCAN_GFC_ANFS_Msk                                /*!< Accept Non-matching Frames Standard */
#define FDCAN_GFC_ANFS_0                          (0x1UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000010 */
#define FDCAN_GFC_ANFS_1                          (0x2UL<<FDCAN_GFC_ANFS_Pos)                       /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_SIDFC register ******************************************/
#define FDCAN_SIDFC_FLSSA_Pos                     (2U)
#define FDCAN_SIDFC_FLSSA_Msk                     (0x3FFFUL<<FDCAN_SIDFC_FLSSA_Pos)                 /*!< 0x0000FFFC */
#define FDCAN_SIDFC_FLSSA                         FDCAN_SIDFC_FLSSA_Msk                             /*!< Filter List Standard Start Address */
#define FDCAN_SIDFC_LSS_Pos                       (16U)
#define FDCAN_SIDFC_LSS_Msk                       (0xFFUL<<FDCAN_SIDFC_LSS_Pos)                     /*!< 0x00FF0000 */
#define FDCAN_SIDFC_LSS                           FDCAN_SIDFC_LSS_Msk                               /*!< List Size Standard */

/********************************* Bit definition for FDCAN_XIDFC register ******************************************/
#define FDCAN_XIDFC_FLESA_Pos                     (2U)
#define FDCAN_XIDFC_FLESA_Msk                     (0x3FFFUL<<FDCAN_XIDFC_FLESA_Pos)                 /*!< 0x0000FFFC */
#define FDCAN_XIDFC_FLESA                         FDCAN_XIDFC_FLESA_Msk                             /*!< Filter List Extended Start Address */
#define FDCAN_XIDFC_LSE_Pos                       (16U)
#define FDCAN_XIDFC_LSE_Msk                       (0x7FUL<<FDCAN_XIDFC_LSE_Pos)                     /*!< 0x007F0000 */
#define FDCAN_XIDFC_LSE                           FDCAN_XIDFC_LSE_Msk                               /*!<  List Size Extended */

/********************************* Bit definition for FDCAN_XIDAM register ******************************************/
#define FDCAN_XIDAM_EIDM_Pos                      (0U)
#define FDCAN_XIDAM_EIDM_Msk                      (0x1FFFFFFFUL<<FDCAN_XIDAM_EIDM_Pos)              /*!< 0x1FFFFFFF */
#define FDCAN_XIDAM_EIDM                          FDCAN_XIDAM_EIDM_Msk                              /*!< Extended ID Mask */

/********************************* Bit definition for FDCAN_HPMS register *******************************************/
#define FDCAN_HPMS_BIDX_Pos                       (0U)
#define FDCAN_HPMS_BIDX_Msk                       (0x3FUL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x0000003F */
#define FDCAN_HPMS_BIDX                           FDCAN_HPMS_BIDX_Msk                               /*!< Buffer Index */
#define FDCAN_HPMS_BIDX_0                         (0x1UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000001 */
#define FDCAN_HPMS_BIDX_1                         (0x2UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000002 */
#define FDCAN_HPMS_BIDX_2                         (0x4UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000004 */
#define FDCAN_HPMS_BIDX_3                         (0x8UL<<FDCAN_HPMS_BIDX_Pos)                      /*!< 0x00000008 */
#define FDCAN_HPMS_BIDX_4                         (0x10UL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x00000010 */
#define FDCAN_HPMS_BIDX_5                         (0x20UL<<FDCAN_HPMS_BIDX_Pos)                     /*!< 0x00000020 */
#define FDCAN_HPMS_MSI_Pos                        (6U)
#define FDCAN_HPMS_MSI_Msk                        (0x3UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x000000C0 */
#define FDCAN_HPMS_MSI                            FDCAN_HPMS_MSI_Msk                                /*!<  Message Storage Indicator */
#define FDCAN_HPMS_MSI_0                          (0x1UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x00000040 */
#define FDCAN_HPMS_MSI_1                          (0x2UL<<FDCAN_HPMS_MSI_Pos)                       /*!< 0x00000080 */
#define FDCAN_HPMS_FIDX_Pos                       (8U)
#define FDCAN_HPMS_FIDX_Msk                       (0x7FUL<<FDCAN_HPMS_FIDX_Pos)                     /*!< 0x00007F00 */
#define FDCAN_HPMS_FIDX                           FDCAN_HPMS_FIDX_Msk                               /*!< Filter Index */
#define FDCAN_HPMS_FLST_Pos                       (15U)
#define FDCAN_HPMS_FLST_Msk                       (0x1UL<<FDCAN_HPMS_FLST_Pos)                      /*!< 0x00008000 */
#define FDCAN_HPMS_FLST                           FDCAN_HPMS_FLST_Msk                               /*!< Filter List */

/********************************* Bit definition for FDCAN_NDAT1 register ******************************************/
#define FDCAN_NDAT1_ND0_Pos                       (0U)
#define FDCAN_NDAT1_ND0_Msk                       (0x1UL<<FDCAN_NDAT1_ND0_Pos)                      /*!< 0x00000001 */
#define FDCAN_NDAT1_ND0                           FDCAN_NDAT1_ND0_Msk                               /*!< New Data0 */
#define FDCAN_NDAT1_ND1_Pos                       (1U)
#define FDCAN_NDAT1_ND1_Msk                       (0x1UL<<FDCAN_NDAT1_ND1_Pos)                      /*!< 0x00000002 */
#define FDCAN_NDAT1_ND1                           FDCAN_NDAT1_ND1_Msk                               /*!< New Data1 */
#define FDCAN_NDAT1_ND2_Pos                       (2U)
#define FDCAN_NDAT1_ND2_Msk                       (0x1UL<<FDCAN_NDAT1_ND2_Pos)                      /*!< 0x00000004 */
#define FDCAN_NDAT1_ND2                           FDCAN_NDAT1_ND2_Msk                               /*!< New Data2 */
#define FDCAN_NDAT1_ND3_Pos                       (3U)
#define FDCAN_NDAT1_ND3_Msk                       (0x1UL<<FDCAN_NDAT1_ND3_Pos)                      /*!< 0x00000008 */
#define FDCAN_NDAT1_ND3                           FDCAN_NDAT1_ND3_Msk                               /*!< New Data3 */
#define FDCAN_NDAT1_ND4_Pos                       (4U)
#define FDCAN_NDAT1_ND4_Msk                       (0x1UL<<FDCAN_NDAT1_ND4_Pos)                      /*!< 0x00000010 */
#define FDCAN_NDAT1_ND4                           FDCAN_NDAT1_ND4_Msk                               /*!< New Data4 */
#define FDCAN_NDAT1_ND5_Pos                       (5U)
#define FDCAN_NDAT1_ND5_Msk                       (0x1UL<<FDCAN_NDAT1_ND5_Pos)                      /*!< 0x00000020 */
#define FDCAN_NDAT1_ND5                           FDCAN_NDAT1_ND5_Msk                               /*!< New Data5 */
#define FDCAN_NDAT1_ND6_Pos                       (6U)
#define FDCAN_NDAT1_ND6_Msk                       (0x1UL<<FDCAN_NDAT1_ND6_Pos)                      /*!< 0x00000040 */
#define FDCAN_NDAT1_ND6                           FDCAN_NDAT1_ND6_Msk                               /*!< New Data6 */
#define FDCAN_NDAT1_ND7_Pos                       (7U)
#define FDCAN_NDAT1_ND7_Msk                       (0x1UL<<FDCAN_NDAT1_ND7_Pos)                      /*!< 0x00000080 */
#define FDCAN_NDAT1_ND7                           FDCAN_NDAT1_ND7_Msk                               /*!< New Data7 */
#define FDCAN_NDAT1_ND8_Pos                       (8U)
#define FDCAN_NDAT1_ND8_Msk                       (0x1UL<<FDCAN_NDAT1_ND8_Pos)                      /*!< 0x00000100 */
#define FDCAN_NDAT1_ND8                           FDCAN_NDAT1_ND8_Msk                               /*!< New Data8 */
#define FDCAN_NDAT1_ND9_Pos                       (9U)
#define FDCAN_NDAT1_ND9_Msk                       (0x1UL<<FDCAN_NDAT1_ND9_Pos)                      /*!< 0x00000200 */
#define FDCAN_NDAT1_ND9                           FDCAN_NDAT1_ND9_Msk                               /*!< New Data9 */
#define FDCAN_NDAT1_ND10_Pos                      (10U)
#define FDCAN_NDAT1_ND10_Msk                      (0x1UL<<FDCAN_NDAT1_ND10_Pos)                     /*!< 0x00000400 */
#define FDCAN_NDAT1_ND10                          FDCAN_NDAT1_ND10_Msk                              /*!< New Data10 */
#define FDCAN_NDAT1_ND11_Pos                      (11U)
#define FDCAN_NDAT1_ND11_Msk                      (0x1UL<<FDCAN_NDAT1_ND11_Pos)                     /*!< 0x00000800 */
#define FDCAN_NDAT1_ND11                          FDCAN_NDAT1_ND11_Msk                              /*!< New Data11 */
#define FDCAN_NDAT1_ND12_Pos                      (12U)
#define FDCAN_NDAT1_ND12_Msk                      (0x1UL<<FDCAN_NDAT1_ND12_Pos)                     /*!< 0x00001000 */
#define FDCAN_NDAT1_ND12                          FDCAN_NDAT1_ND12_Msk                              /*!< New Data12 */
#define FDCAN_NDAT1_ND13_Pos                      (13U)
#define FDCAN_NDAT1_ND13_Msk                      (0x1UL<<FDCAN_NDAT1_ND13_Pos)                     /*!< 0x00002000 */
#define FDCAN_NDAT1_ND13                          FDCAN_NDAT1_ND13_Msk                              /*!< New Data13 */
#define FDCAN_NDAT1_ND14_Pos                      (14U)
#define FDCAN_NDAT1_ND14_Msk                      (0x1UL<<FDCAN_NDAT1_ND14_Pos)                     /*!< 0x00004000 */
#define FDCAN_NDAT1_ND14                          FDCAN_NDAT1_ND14_Msk                              /*!< New Data14 */
#define FDCAN_NDAT1_ND15_Pos                      (15U)
#define FDCAN_NDAT1_ND15_Msk                      (0x1UL<<FDCAN_NDAT1_ND15_Pos)                     /*!< 0x00008000 */
#define FDCAN_NDAT1_ND15                          FDCAN_NDAT1_ND15_Msk                              /*!< New Data15 */
#define FDCAN_NDAT1_ND16_Pos                      (16U)
#define FDCAN_NDAT1_ND16_Msk                      (0x1UL<<FDCAN_NDAT1_ND16_Pos)                     /*!< 0x00010000 */
#define FDCAN_NDAT1_ND16                          FDCAN_NDAT1_ND16_Msk                              /*!< New Data16 */
#define FDCAN_NDAT1_ND17_Pos                      (17U)
#define FDCAN_NDAT1_ND17_Msk                      (0x1UL<<FDCAN_NDAT1_ND17_Pos)                     /*!< 0x00020000 */
#define FDCAN_NDAT1_ND17                          FDCAN_NDAT1_ND17_Msk                              /*!< New Data17 */
#define FDCAN_NDAT1_ND18_Pos                      (18U)
#define FDCAN_NDAT1_ND18_Msk                      (0x1UL<<FDCAN_NDAT1_ND18_Pos)                     /*!< 0x00040000 */
#define FDCAN_NDAT1_ND18                          FDCAN_NDAT1_ND18_Msk                              /*!< New Data18 */
#define FDCAN_NDAT1_ND19_Pos                      (19U)
#define FDCAN_NDAT1_ND19_Msk                      (0x1UL<<FDCAN_NDAT1_ND19_Pos)                     /*!< 0x00080000 */
#define FDCAN_NDAT1_ND19                          FDCAN_NDAT1_ND19_Msk                              /*!< New Data19 */
#define FDCAN_NDAT1_ND20_Pos                      (20U)
#define FDCAN_NDAT1_ND20_Msk                      (0x1UL<<FDCAN_NDAT1_ND20_Pos)                     /*!< 0x00100000 */
#define FDCAN_NDAT1_ND20                          FDCAN_NDAT1_ND20_Msk                              /*!< New Data20 */
#define FDCAN_NDAT1_ND21_Pos                      (21U)
#define FDCAN_NDAT1_ND21_Msk                      (0x1UL<<FDCAN_NDAT1_ND21_Pos)                     /*!< 0x00200000 */
#define FDCAN_NDAT1_ND21                          FDCAN_NDAT1_ND21_Msk                              /*!< New Data21 */
#define FDCAN_NDAT1_ND22_Pos                      (22U)
#define FDCAN_NDAT1_ND22_Msk                      (0x1UL<<FDCAN_NDAT1_ND22_Pos)                     /*!< 0x00400000 */
#define FDCAN_NDAT1_ND22                          FDCAN_NDAT1_ND22_Msk                              /*!< New Data22 */
#define FDCAN_NDAT1_ND23_Pos                      (23U)
#define FDCAN_NDAT1_ND23_Msk                      (0x1UL<<FDCAN_NDAT1_ND23_Pos)                     /*!< 0x00800000 */
#define FDCAN_NDAT1_ND23                          FDCAN_NDAT1_ND23_Msk                              /*!< New Data23 */
#define FDCAN_NDAT1_ND24_Pos                      (24U)
#define FDCAN_NDAT1_ND24_Msk                      (0x1UL<<FDCAN_NDAT1_ND24_Pos)                     /*!< 0x01000000 */
#define FDCAN_NDAT1_ND24                          FDCAN_NDAT1_ND24_Msk                              /*!< New Data24 */
#define FDCAN_NDAT1_ND25_Pos                      (25U)
#define FDCAN_NDAT1_ND25_Msk                      (0x1UL<<FDCAN_NDAT1_ND25_Pos)                     /*!< 0x02000000 */
#define FDCAN_NDAT1_ND25                          FDCAN_NDAT1_ND25_Msk                              /*!< New Data25 */
#define FDCAN_NDAT1_ND26_Pos                      (26U)
#define FDCAN_NDAT1_ND26_Msk                      (0x1UL<<FDCAN_NDAT1_ND26_Pos)                     /*!< 0x04000000 */
#define FDCAN_NDAT1_ND26                          FDCAN_NDAT1_ND26_Msk                              /*!< New Data26 */
#define FDCAN_NDAT1_ND27_Pos                      (27U)
#define FDCAN_NDAT1_ND27_Msk                      (0x1UL<<FDCAN_NDAT1_ND27_Pos)                     /*!< 0x08000000 */
#define FDCAN_NDAT1_ND27                          FDCAN_NDAT1_ND27_Msk                              /*!< New Data27 */
#define FDCAN_NDAT1_ND28_Pos                      (28U)
#define FDCAN_NDAT1_ND28_Msk                      (0x1UL<<FDCAN_NDAT1_ND28_Pos)                     /*!< 0x10000000 */
#define FDCAN_NDAT1_ND28                          FDCAN_NDAT1_ND28_Msk                              /*!< New Data28 */
#define FDCAN_NDAT1_ND29_Pos                      (29U)
#define FDCAN_NDAT1_ND29_Msk                      (0x1UL<<FDCAN_NDAT1_ND29_Pos)                     /*!< 0x20000000 */
#define FDCAN_NDAT1_ND29                          FDCAN_NDAT1_ND29_Msk                              /*!< New Data29 */
#define FDCAN_NDAT1_ND30_Pos                      (30U)
#define FDCAN_NDAT1_ND30_Msk                      (0x1UL<<FDCAN_NDAT1_ND30_Pos)                     /*!< 0x40000000 */
#define FDCAN_NDAT1_ND30                          FDCAN_NDAT1_ND30_Msk                              /*!< New Data30 */
#define FDCAN_NDAT1_ND31_Pos                      (31U)
#define FDCAN_NDAT1_ND31_Msk                      (0x1UL<<FDCAN_NDAT1_ND31_Pos)                     /*!< 0x80000000 */
#define FDCAN_NDAT1_ND31                          FDCAN_NDAT1_ND31_Msk                              /*!< New Data31 */

/********************************* Bit definition for FDCAN_NDAT2 register ******************************************/
#define FDCAN_NDAT2_ND32_Pos                      (0U)
#define FDCAN_NDAT2_ND32_Msk                      (0x1UL<<FDCAN_NDAT2_ND32_Pos)                     /*!< 0x00000001 */
#define FDCAN_NDAT2_ND32                          FDCAN_NDAT2_ND32_Msk                              /*!< New Data32 */
#define FDCAN_NDAT2_ND33_Pos                      (1U)
#define FDCAN_NDAT2_ND33_Msk                      (0x1UL<<FDCAN_NDAT2_ND33_Pos)                     /*!< 0x00000002 */
#define FDCAN_NDAT2_ND33                          FDCAN_NDAT2_ND33_Msk                              /*!< New Data33 */
#define FDCAN_NDAT2_ND34_Pos                      (2U)
#define FDCAN_NDAT2_ND34_Msk                      (0x1UL<<FDCAN_NDAT2_ND34_Pos)                     /*!< 0x00000004 */
#define FDCAN_NDAT2_ND34                          FDCAN_NDAT2_ND34_Msk                              /*!< New Data34 */
#define FDCAN_NDAT2_ND35_Pos                      (3U)
#define FDCAN_NDAT2_ND35_Msk                      (0x1UL<<FDCAN_NDAT2_ND35_Pos)                     /*!< 0x00000008 */
#define FDCAN_NDAT2_ND35                          FDCAN_NDAT2_ND35_Msk                              /*!< New Data35 */
#define FDCAN_NDAT2_ND36_Pos                      (4U)
#define FDCAN_NDAT2_ND36_Msk                      (0x1UL<<FDCAN_NDAT2_ND36_Pos)                     /*!< 0x00000010 */
#define FDCAN_NDAT2_ND36                          FDCAN_NDAT2_ND36_Msk                              /*!< New Data36 */
#define FDCAN_NDAT2_ND37_Pos                      (5U)
#define FDCAN_NDAT2_ND37_Msk                      (0x1UL<<FDCAN_NDAT2_ND37_Pos)                     /*!< 0x00000020 */
#define FDCAN_NDAT2_ND37                          FDCAN_NDAT2_ND37_Msk                              /*!< New Data37 */
#define FDCAN_NDAT2_ND38_Pos                      (6U)
#define FDCAN_NDAT2_ND38_Msk                      (0x1UL<<FDCAN_NDAT2_ND38_Pos)                     /*!< 0x00000040 */
#define FDCAN_NDAT2_ND38                          FDCAN_NDAT2_ND38_Msk                              /*!< New Data38 */
#define FDCAN_NDAT2_ND39_Pos                      (7U)
#define FDCAN_NDAT2_ND39_Msk                      (0x1UL<<FDCAN_NDAT2_ND39_Pos)                     /*!< 0x00000080 */
#define FDCAN_NDAT2_ND39                          FDCAN_NDAT2_ND39_Msk                              /*!< New Data39 */
#define FDCAN_NDAT2_ND40_Pos                      (8U)
#define FDCAN_NDAT2_ND40_Msk                      (0x1UL<<FDCAN_NDAT2_ND40_Pos)                     /*!< 0x00000100 */
#define FDCAN_NDAT2_ND40                          FDCAN_NDAT2_ND40_Msk                              /*!< New Data40 */
#define FDCAN_NDAT2_ND41_Pos                      (9U)
#define FDCAN_NDAT2_ND41_Msk                      (0x1UL<<FDCAN_NDAT2_ND41_Pos)                     /*!< 0x00000200 */
#define FDCAN_NDAT2_ND41                          FDCAN_NDAT2_ND41_Msk                              /*!< New Data41 */
#define FDCAN_NDAT2_ND42_Pos                      (10U)
#define FDCAN_NDAT2_ND42_Msk                      (0x1UL<<FDCAN_NDAT2_ND42_Pos)                     /*!< 0x00000400 */
#define FDCAN_NDAT2_ND42                          FDCAN_NDAT2_ND42_Msk                              /*!< New Data42 */
#define FDCAN_NDAT2_ND43_Pos                      (11U)
#define FDCAN_NDAT2_ND43_Msk                      (0x1UL<<FDCAN_NDAT2_ND43_Pos)                     /*!< 0x00000800 */
#define FDCAN_NDAT2_ND43                          FDCAN_NDAT2_ND43_Msk                              /*!< New Data43 */
#define FDCAN_NDAT2_ND44_Pos                      (12U)
#define FDCAN_NDAT2_ND44_Msk                      (0x1UL<<FDCAN_NDAT2_ND44_Pos)                     /*!< 0x00001000 */
#define FDCAN_NDAT2_ND44                          FDCAN_NDAT2_ND44_Msk                              /*!< New Data44 */
#define FDCAN_NDAT2_ND45_Pos                      (13U)
#define FDCAN_NDAT2_ND45_Msk                      (0x1UL<<FDCAN_NDAT2_ND45_Pos)                     /*!< 0x00002000 */
#define FDCAN_NDAT2_ND45                          FDCAN_NDAT2_ND45_Msk                              /*!< New Data45 */
#define FDCAN_NDAT2_ND46_Pos                      (14U)
#define FDCAN_NDAT2_ND46_Msk                      (0x1UL<<FDCAN_NDAT2_ND46_Pos)                     /*!< 0x00004000 */
#define FDCAN_NDAT2_ND46                          FDCAN_NDAT2_ND46_Msk                              /*!< New Data46 */
#define FDCAN_NDAT2_ND47_Pos                      (15U)
#define FDCAN_NDAT2_ND47_Msk                      (0x1UL<<FDCAN_NDAT2_ND47_Pos)                     /*!< 0x00008000 */
#define FDCAN_NDAT2_ND47                          FDCAN_NDAT2_ND47_Msk                              /*!< New Data47 */
#define FDCAN_NDAT2_ND48_Pos                      (16U)
#define FDCAN_NDAT2_ND48_Msk                      (0x1UL<<FDCAN_NDAT2_ND48_Pos)                     /*!< 0x00010000 */
#define FDCAN_NDAT2_ND48                          FDCAN_NDAT2_ND48_Msk                              /*!< New Data48 */
#define FDCAN_NDAT2_ND49_Pos                      (17U)
#define FDCAN_NDAT2_ND49_Msk                      (0x1UL<<FDCAN_NDAT2_ND49_Pos)                     /*!< 0x00020000 */
#define FDCAN_NDAT2_ND49                          FDCAN_NDAT2_ND49_Msk                              /*!< New Data49 */
#define FDCAN_NDAT2_ND50_Pos                      (18U)
#define FDCAN_NDAT2_ND50_Msk                      (0x1UL<<FDCAN_NDAT2_ND50_Pos)                     /*!< 0x00040000 */
#define FDCAN_NDAT2_ND50                          FDCAN_NDAT2_ND50_Msk                              /*!< New Data50 */
#define FDCAN_NDAT2_ND51_Pos                      (19U)
#define FDCAN_NDAT2_ND51_Msk                      (0x1UL<<FDCAN_NDAT2_ND51_Pos)                     /*!< 0x00080000 */
#define FDCAN_NDAT2_ND51                          FDCAN_NDAT2_ND51_Msk                              /*!< New Data51 */
#define FDCAN_NDAT2_ND52_Pos                      (20U)
#define FDCAN_NDAT2_ND52_Msk                      (0x1UL<<FDCAN_NDAT2_ND52_Pos)                     /*!< 0x00100000 */
#define FDCAN_NDAT2_ND52                          FDCAN_NDAT2_ND52_Msk                              /*!< New Data52 */
#define FDCAN_NDAT2_ND53_Pos                      (21U)
#define FDCAN_NDAT2_ND53_Msk                      (0x1UL<<FDCAN_NDAT2_ND53_Pos)                     /*!< 0x00200000 */
#define FDCAN_NDAT2_ND53                          FDCAN_NDAT2_ND53_Msk                              /*!< New Data53 */
#define FDCAN_NDAT2_ND54_Pos                      (22U)
#define FDCAN_NDAT2_ND54_Msk                      (0x1UL<<FDCAN_NDAT2_ND54_Pos)                     /*!< 0x00400000 */
#define FDCAN_NDAT2_ND54                          FDCAN_NDAT2_ND54_Msk                              /*!< New Data54 */
#define FDCAN_NDAT2_ND55_Pos                      (23U)
#define FDCAN_NDAT2_ND55_Msk                      (0x1UL<<FDCAN_NDAT2_ND55_Pos)                     /*!< 0x00800000 */
#define FDCAN_NDAT2_ND55                          FDCAN_NDAT2_ND55_Msk                              /*!< New Data55 */
#define FDCAN_NDAT2_ND56_Pos                      (24U)
#define FDCAN_NDAT2_ND56_Msk                      (0x1UL<<FDCAN_NDAT2_ND56_Pos)                     /*!< 0x01000000 */
#define FDCAN_NDAT2_ND56                          FDCAN_NDAT2_ND56_Msk                              /*!< New Data56 */
#define FDCAN_NDAT2_ND57_Pos                      (25U)
#define FDCAN_NDAT2_ND57_Msk                      (0x1UL<<FDCAN_NDAT2_ND57_Pos)                     /*!< 0x02000000 */
#define FDCAN_NDAT2_ND57                          FDCAN_NDAT2_ND57_Msk                              /*!< New Data57 */
#define FDCAN_NDAT2_ND58_Pos                      (26U)
#define FDCAN_NDAT2_ND58_Msk                      (0x1UL<<FDCAN_NDAT2_ND58_Pos)                     /*!< 0x04000000 */
#define FDCAN_NDAT2_ND58                          FDCAN_NDAT2_ND58_Msk                              /*!< New Data58 */
#define FDCAN_NDAT2_ND59_Pos                      (27U)
#define FDCAN_NDAT2_ND59_Msk                      (0x1UL<<FDCAN_NDAT2_ND59_Pos)                     /*!< 0x08000000 */
#define FDCAN_NDAT2_ND59                          FDCAN_NDAT2_ND59_Msk                              /*!< New Data59 */
#define FDCAN_NDAT2_ND60_Pos                      (28U)
#define FDCAN_NDAT2_ND60_Msk                      (0x1UL<<FDCAN_NDAT2_ND60_Pos)                     /*!< 0x10000000 */
#define FDCAN_NDAT2_ND60                          FDCAN_NDAT2_ND60_Msk                              /*!< New Data60 */
#define FDCAN_NDAT2_ND61_Pos                      (29U)
#define FDCAN_NDAT2_ND61_Msk                      (0x1UL<<FDCAN_NDAT2_ND61_Pos)                     /*!< 0x20000000 */
#define FDCAN_NDAT2_ND61                          FDCAN_NDAT2_ND61_Msk                              /*!< New Data61 */
#define FDCAN_NDAT2_ND62_Pos                      (30U)
#define FDCAN_NDAT2_ND62_Msk                      (0x1UL<<FDCAN_NDAT2_ND62_Pos)                     /*!< 0x40000000 */
#define FDCAN_NDAT2_ND62                          FDCAN_NDAT2_ND62_Msk                              /*!< New Data62 */
#define FDCAN_NDAT2_ND63_Pos                      (31U)
#define FDCAN_NDAT2_ND63_Msk                      (0x1UL<<FDCAN_NDAT2_ND63_Pos)                     /*!< 0x80000000 */
#define FDCAN_NDAT2_ND63                          FDCAN_NDAT2_ND63_Msk                              /*!< New Data63 */

/********************************* Bit definition for FDCAN_RXF0C register ******************************************/
#define FDCAN_RXF0C_F0SA_Pos                      (2U)
#define FDCAN_RXF0C_F0SA_Msk                      (0x3FFFUL<<FDCAN_RXF0C_F0SA_Pos)                  /*!< 0x0000FFFC */
#define FDCAN_RXF0C_F0SA                          FDCAN_RXF0C_F0SA_Msk                              /*!< Rx FIFO 0 Start Address */
#define FDCAN_RXF0C_F0S_Pos                       (16U)
#define FDCAN_RXF0C_F0S_Msk                       (0x7FUL<<FDCAN_RXF0C_F0S_Pos)                     /*!< 0x007F0000 */
#define FDCAN_RXF0C_F0S                           FDCAN_RXF0C_F0S_Msk                               /*!< Rx FIFO 0 Size */
#define FDCAN_RXF0C_F0WM_Pos                      (24U)
#define FDCAN_RXF0C_F0WM_Msk                      (0x7FUL<<FDCAN_RXF0C_F0WM_Pos)                    /*!< 0x7F000000 */
#define FDCAN_RXF0C_F0WM                          FDCAN_RXF0C_F0WM_Msk                              /*!< Rx FIFO 0 Watermark */
#define FDCAN_RXF0C_F0OM_Pos                      (31U)
#define FDCAN_RXF0C_F0OM_Msk                      (0x1UL<<FDCAN_RXF0C_F0OM_Pos)                     /*!< 0x80000000 */
#define FDCAN_RXF0C_F0OM                          FDCAN_RXF0C_F0OM_Msk                              /*!< FIFO 0 Operation Mode */

/********************************* Bit definition for FDCAN_RXF0S register ******************************************/
#define FDCAN_RXF0S_F0FL_Pos                      (0U)
#define FDCAN_RXF0S_F0FL_Msk                      (0x7FUL<<FDCAN_RXF0S_F0FL_Pos)                    /*!< 0x0000007F */
#define FDCAN_RXF0S_F0FL                          FDCAN_RXF0S_F0FL_Msk                              /*!< Rx FIFO 0 Fill Level */
#define FDCAN_RXF0S_F0GI_Pos                      (8U)
#define FDCAN_RXF0S_F0GI_Msk                      (0x3FUL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00003F00 */
#define FDCAN_RXF0S_F0GI                          FDCAN_RXF0S_F0GI_Msk                              /*!< Rx FIFO 0 Get Index */
#define FDCAN_RXF0S_F0GI_0                        (0x1UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXF0S_F0GI_1                        (0x2UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXF0S_F0GI_2                        (0x4UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000400 */
#define FDCAN_RXF0S_F0GI_3                        (0x8UL<<FDCAN_RXF0S_F0GI_Pos)                     /*!< 0x00000800 */
#define FDCAN_RXF0S_F0GI_4                        (0x10UL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00001000 */
#define FDCAN_RXF0S_F0GI_5                        (0x20UL<<FDCAN_RXF0S_F0GI_Pos)                    /*!< 0x00002000 */
#define FDCAN_RXF0S_F0PI_Pos                      (16U)
#define FDCAN_RXF0S_F0PI_Msk                      (0x3FUL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x003F0000 */
#define FDCAN_RXF0S_F0PI                          FDCAN_RXF0S_F0PI_Msk                              /*!< Rx FIFO 0 Put Index */
#define FDCAN_RXF0S_F0PI_0                        (0x1UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00010000 */
#define FDCAN_RXF0S_F0PI_1                        (0x2UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00020000 */
#define FDCAN_RXF0S_F0PI_2                        (0x4UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00040000 */
#define FDCAN_RXF0S_F0PI_3                        (0x8UL<<FDCAN_RXF0S_F0PI_Pos)                     /*!< 0x00080000 */
#define FDCAN_RXF0S_F0PI_4                        (0x10UL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x00100000 */
#define FDCAN_RXF0S_F0PI_5                        (0x20UL<<FDCAN_RXF0S_F0PI_Pos)                    /*!< 0x00200000 */
#define FDCAN_RXF0S_F0F_Pos                       (24U)
#define FDCAN_RXF0S_F0F_Msk                       (0x1UL<<FDCAN_RXF0S_F0F_Pos)                      /*!< 0x01000000 */
#define FDCAN_RXF0S_F0F                           FDCAN_RXF0S_F0F_Msk                               /*!< Rx FIFO 0 Full */
#define FDCAN_RXF0S_RF0L_Pos                      (25U)
#define FDCAN_RXF0S_RF0L_Msk                      (0x1UL<<FDCAN_RXF0S_RF0L_Pos)                     /*!< 0x02000000 */
#define FDCAN_RXF0S_RF0L                          FDCAN_RXF0S_RF0L_Msk                              /*!< Rx FIFO 0 Message Lost */

/********************************* Bit definition for FDCAN_RXF0A register ******************************************/
#define FDCAN_RXF0A_F0AI_Pos                      (0U)
#define FDCAN_RXF0A_F0AI_Msk                      (0x3FUL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x0000003F */
#define FDCAN_RXF0A_F0AI                          FDCAN_RXF0A_F0AI_Msk                              /*!< Rx FIFO 0 Acknowledge Index */
#define FDCAN_RXF0A_F0AI_0                        (0x1UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXF0A_F0AI_1                        (0x2UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXF0A_F0AI_2                        (0x4UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF0A_F0AI_3                        (0x8UL<<FDCAN_RXF0A_F0AI_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF0A_F0AI_4                        (0x10UL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x00000010 */
#define FDCAN_RXF0A_F0AI_5                        (0x20UL<<FDCAN_RXF0A_F0AI_Pos)                    /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_RXBC register *******************************************/
#define FDCAN_RXBC_RBSA_Pos                       (2U)
#define FDCAN_RXBC_RBSA_Msk                       (0x3FFFUL<<FDCAN_RXBC_RBSA_Pos)                   /*!< 0x0000FFFC */
#define FDCAN_RXBC_RBSA                           FDCAN_RXBC_RBSA_Msk                               /*!< Rx Buffer Start Address */

/********************************* Bit definition for FDCAN_RXF1C register ******************************************/
#define FDCAN_RXF1C_F1SA_Pos                      (2U)
#define FDCAN_RXF1C_F1SA_Msk                      (0xFUL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x0000003C */
#define FDCAN_RXF1C_F1SA                          FDCAN_RXF1C_F1SA_Msk                              /*!< Rx FIFO 1 Start Address */
#define FDCAN_RXF1C_F1SA_0                        (0x1UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF1C_F1SA_1                        (0x2UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF1C_F1SA_2                        (0x4UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000010 */
#define FDCAN_RXF1C_F1SA_3                        (0x8UL<<FDCAN_RXF1C_F1SA_Pos)                     /*!< 0x00000020 */
#define FDCAN_RXF1C_F1S_Pos                       (16U)
#define FDCAN_RXF1C_F1S_Msk                       (0x7FUL<<FDCAN_RXF1C_F1S_Pos)                     /*!< 0x007F0000 */
#define FDCAN_RXF1C_F1S                           FDCAN_RXF1C_F1S_Msk                               /*!< Rx FIFO 1 Size */
#define FDCAN_RXF1C_F1WM_Pos                      (24U)
#define FDCAN_RXF1C_F1WM_Msk                      (0x7FUL<<FDCAN_RXF1C_F1WM_Pos)                    /*!< 0x7F000000 */
#define FDCAN_RXF1C_F1WM                          FDCAN_RXF1C_F1WM_Msk                              /*!< Rx FIFO 1 Watermark */
#define FDCAN_RXF1C_F1OM_Pos                      (31U)
#define FDCAN_RXF1C_F1OM_Msk                      (0x1UL<<FDCAN_RXF1C_F1OM_Pos)                     /*!< 0x80000000 */
#define FDCAN_RXF1C_F1OM                          FDCAN_RXF1C_F1OM_Msk                              /*!< FIFO 1 Operation Mode */

/********************************* Bit definition for FDCAN_RXF1S register ******************************************/
#define FDCAN_RXF1S_F1FL_Pos                      (0U)
#define FDCAN_RXF1S_F1FL_Msk                      (0x7FUL<<FDCAN_RXF1S_F1FL_Pos)                    /*!< 0x0000007F */
#define FDCAN_RXF1S_F1FL                          FDCAN_RXF1S_F1FL_Msk                              /*!< Rx FIFO 1 Fill Level */
#define FDCAN_RXF1S_F1GI_Pos                      (8U)
#define FDCAN_RXF1S_F1GI_Msk                      (0x3FUL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00003F00 */
#define FDCAN_RXF1S_F1GI                          FDCAN_RXF1S_F1GI_Msk                              /*!< Rx FIFO 1 Get Index */
#define FDCAN_RXF1S_F1GI_0                        (0x1UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXF1S_F1GI_1                        (0x2UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXF1S_F1GI_2                        (0x4UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000400 */
#define FDCAN_RXF1S_F1GI_3                        (0x8UL<<FDCAN_RXF1S_F1GI_Pos)                     /*!< 0x00000800 */
#define FDCAN_RXF1S_F1GI_4                        (0x10UL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00001000 */
#define FDCAN_RXF1S_F1GI_5                        (0x20UL<<FDCAN_RXF1S_F1GI_Pos)                    /*!< 0x00002000 */
#define FDCAN_RXF1S_F1PI_Pos                      (16U)
#define FDCAN_RXF1S_F1PI_Msk                      (0x3FUL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x003F0000 */
#define FDCAN_RXF1S_F1PI                          FDCAN_RXF1S_F1PI_Msk                              /*!< Rx FIFO 1 Put Index */
#define FDCAN_RXF1S_F1PI_0                        (0x1UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00010000 */
#define FDCAN_RXF1S_F1PI_1                        (0x2UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00020000 */
#define FDCAN_RXF1S_F1PI_2                        (0x4UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00040000 */
#define FDCAN_RXF1S_F1PI_3                        (0x8UL<<FDCAN_RXF1S_F1PI_Pos)                     /*!< 0x00080000 */
#define FDCAN_RXF1S_F1PI_4                        (0x10UL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x00100000 */
#define FDCAN_RXF1S_F1PI_5                        (0x20UL<<FDCAN_RXF1S_F1PI_Pos)                    /*!< 0x00200000 */
#define FDCAN_RXF1S_F1F_Pos                       (24U)
#define FDCAN_RXF1S_F1F_Msk                       (0x1UL<<FDCAN_RXF1S_F1F_Pos)                      /*!< 0x01000000 */
#define FDCAN_RXF1S_F1F                           FDCAN_RXF1S_F1F_Msk                               /*!< Rx FIFO 1 Full */
#define FDCAN_RXF1S_RF1L_Pos                      (25U)
#define FDCAN_RXF1S_RF1L_Msk                      (0x1UL<<FDCAN_RXF1S_RF1L_Pos)                     /*!< 0x02000000 */
#define FDCAN_RXF1S_RF1L                          FDCAN_RXF1S_RF1L_Msk                              /*!< Rx FIFO 1 Message Lost */
#define FDCAN_RXF1S_DMS_Pos                       (30U)
#define FDCAN_RXF1S_DMS_Msk                       (0x3UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0xC0000000 */
#define FDCAN_RXF1S_DMS                           FDCAN_RXF1S_DMS_Msk                               /*!< Debug Message Status */
#define FDCAN_RXF1S_DMS_0                         (0x1UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0x40000000 */
#define FDCAN_RXF1S_DMS_1                         (0x2UL<<FDCAN_RXF1S_DMS_Pos)                      /*!< 0x80000000 */

/********************************* Bit definition for FDCAN_RXF1A register ******************************************/
#define FDCAN_RXF1A_F1AI_Pos                      (0U)
#define FDCAN_RXF1A_F1AI_Msk                      (0x3FUL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x0000003F */
#define FDCAN_RXF1A_F1AI                          FDCAN_RXF1A_F1AI_Msk                              /*!< Rx FIFO 1 Acknowledge Index */
#define FDCAN_RXF1A_F1AI_0                        (0x1UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXF1A_F1AI_1                        (0x2UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXF1A_F1AI_2                        (0x4UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXF1A_F1AI_3                        (0x8UL<<FDCAN_RXF1A_F1AI_Pos)                     /*!< 0x00000008 */
#define FDCAN_RXF1A_F1AI_4                        (0x10UL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x00000010 */
#define FDCAN_RXF1A_F1AI_5                        (0x20UL<<FDCAN_RXF1A_F1AI_Pos)                    /*!< 0x00000020 */

/********************************* Bit definition for FDCAN_RXESC register ******************************************/
#define FDCAN_RXESC_F0DS_Pos                      (0U)
#define FDCAN_RXESC_F0DS_Msk                      (0x7UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000007 */
#define FDCAN_RXESC_F0DS                          FDCAN_RXESC_F0DS_Msk                              /*!< Rx FIFO 0 Data Field Size */
#define FDCAN_RXESC_F0DS_0                        (0x1UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000001 */
#define FDCAN_RXESC_F0DS_1                        (0x2UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000002 */
#define FDCAN_RXESC_F0DS_2                        (0x4UL<<FDCAN_RXESC_F0DS_Pos)                     /*!< 0x00000004 */
#define FDCAN_RXESC_F1DS_Pos                      (4U)
#define FDCAN_RXESC_F1DS_Msk                      (0x7UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000070 */
#define FDCAN_RXESC_F1DS                          FDCAN_RXESC_F1DS_Msk                              /*!< Rx FIFO 1 Data Field Size */
#define FDCAN_RXESC_F1DS_0                        (0x1UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000010 */
#define FDCAN_RXESC_F1DS_1                        (0x2UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000020 */
#define FDCAN_RXESC_F1DS_2                        (0x4UL<<FDCAN_RXESC_F1DS_Pos)                     /*!< 0x00000040 */
#define FDCAN_RXESC_RBDS_Pos                      (8U)
#define FDCAN_RXESC_RBDS_Msk                      (0x7UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000700 */
#define FDCAN_RXESC_RBDS                          FDCAN_RXESC_RBDS_Msk                              /*!< Rx Buffer Data Field Size */
#define FDCAN_RXESC_RBDS_0                        (0x1UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000100 */
#define FDCAN_RXESC_RBDS_1                        (0x2UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000200 */
#define FDCAN_RXESC_RBDS_2                        (0x4UL<<FDCAN_RXESC_RBDS_Pos)                     /*!< 0x00000400 */

/********************************* Bit definition for FDCAN_TXBC register *******************************************/
#define FDCAN_TXBC_TBSA_Pos                       (2U)
#define FDCAN_TXBC_TBSA_Msk                       (0x3FFFUL<<FDCAN_TXBC_TBSA_Pos)                   /*!< 0x0000FFFC */
#define FDCAN_TXBC_TBSA                           FDCAN_TXBC_TBSA_Msk                               /*!< Tx Buffers Start Address */
#define FDCAN_TXBC_NDTB_Pos                       (16U)
#define FDCAN_TXBC_NDTB_Msk                       (0x3FUL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x003F0000 */
#define FDCAN_TXBC_NDTB                           FDCAN_TXBC_NDTB_Msk                               /*!< Number of Dedicated Transmit Buffers */
#define FDCAN_TXBC_NDTB_0                         (0x1UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00010000 */
#define FDCAN_TXBC_NDTB_1                         (0x2UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00020000 */
#define FDCAN_TXBC_NDTB_2                         (0x4UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00040000 */
#define FDCAN_TXBC_NDTB_3                         (0x8UL<<FDCAN_TXBC_NDTB_Pos)                      /*!< 0x00080000 */
#define FDCAN_TXBC_NDTB_4                         (0x10UL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBC_NDTB_5                         (0x20UL<<FDCAN_TXBC_NDTB_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBC_TFQS_Pos                       (24U)
#define FDCAN_TXBC_TFQS_Msk                       (0x3FUL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x3F000000 */
#define FDCAN_TXBC_TFQS                           FDCAN_TXBC_TFQS_Msk                               /*!< Transmit FIFO/Queue Size */
#define FDCAN_TXBC_TFQS_0                         (0x1UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x01000000 */
#define FDCAN_TXBC_TFQS_1                         (0x2UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x02000000 */
#define FDCAN_TXBC_TFQS_2                         (0x4UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x04000000 */
#define FDCAN_TXBC_TFQS_3                         (0x8UL<<FDCAN_TXBC_TFQS_Pos)                      /*!< 0x08000000 */
#define FDCAN_TXBC_TFQS_4                         (0x10UL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBC_TFQS_5                         (0x20UL<<FDCAN_TXBC_TFQS_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBC_TFQM_Pos                       (30U)
#define FDCAN_TXBC_TFQM_Msk                       (0x1UL<<FDCAN_TXBC_TFQM_Pos)                      /*!< 0x40000000 */
#define FDCAN_TXBC_TFQM                           FDCAN_TXBC_TFQM_Msk                               /*!< Tx FIFO/Queue Mode */

/********************************* Bit definition for FDCAN_TXFQS register ******************************************/
#define FDCAN_TXFQS_TFFL_Pos                      (0U)
#define FDCAN_TXFQS_TFFL_Msk                      (0x3FUL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x0000003F */
#define FDCAN_TXFQS_TFFL                          FDCAN_TXFQS_TFFL_Msk                              /*!< Tx FIFO Free Level */
#define FDCAN_TXFQS_TFFL_0                        (0x1UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXFQS_TFFL_1                        (0x2UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXFQS_TFFL_2                        (0x4UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXFQS_TFFL_3                        (0x8UL<<FDCAN_TXFQS_TFFL_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXFQS_TFFL_4                        (0x10UL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXFQS_TFFL_5                        (0x20UL<<FDCAN_TXFQS_TFFL_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXFQS_TFGI_Pos                      (8U)
#define FDCAN_TXFQS_TFGI_Msk                      (0x1FUL<<FDCAN_TXFQS_TFGI_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TXFQS_TFGI                          FDCAN_TXFQS_TFGI_Msk                              /*!< Tx FIFO Get Index */
#define FDCAN_TXFQS_TFGI_0                        (0x1UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXFQS_TFGI_1                        (0x2UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXFQS_TFGI_2                        (0x4UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXFQS_TFGI_3                        (0x8UL<<FDCAN_TXFQS_TFGI_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXFQS_TFGI_4                        (0x10UL<<FDCAN_TXFQS_TFGI_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXFQS_TFQPI_Pos                     (16U)
#define FDCAN_TXFQS_TFQPI_Msk                     (0x1FUL<<FDCAN_TXFQS_TFQPI_Pos)                   /*!< 0x001F0000 */
#define FDCAN_TXFQS_TFQPI                         FDCAN_TXFQS_TFQPI_Msk                             /*!< Tx FIFO/Queue Put Index */
#define FDCAN_TXFQS_TFQPI_0                       (0x1UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00010000 */
#define FDCAN_TXFQS_TFQPI_1                       (0x2UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00020000 */
#define FDCAN_TXFQS_TFQPI_2                       (0x4UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00040000 */
#define FDCAN_TXFQS_TFQPI_3                       (0x8UL<<FDCAN_TXFQS_TFQPI_Pos)                    /*!< 0x00080000 */
#define FDCAN_TXFQS_TFQPI_4                       (0x10UL<<FDCAN_TXFQS_TFQPI_Pos)                   /*!< 0x00100000 */
#define FDCAN_TXFQS_TFQF_Pos                      (21U)
#define FDCAN_TXFQS_TFQF_Msk                      (0x1UL<<FDCAN_TXFQS_TFQF_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXFQS_TFQF                          FDCAN_TXFQS_TFQF_Msk                              /*!< Tx FIFO/Queue Full */

/********************************* Bit definition for FDCAN_TXESC register ******************************************/
#define FDCAN_TXESC_TBDS_Pos                      (0U)
#define FDCAN_TXESC_TBDS_Msk                      (0x7UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000007 */
#define FDCAN_TXESC_TBDS                          FDCAN_TXESC_TBDS_Msk                              /*!< Tx Buffer Data Field Size */
#define FDCAN_TXESC_TBDS_0                        (0x1UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXESC_TBDS_1                        (0x2UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXESC_TBDS_2                        (0x4UL<<FDCAN_TXESC_TBDS_Pos)                     /*!< 0x00000004 */

/********************************* Bit definition for FDCAN_TXBRP register ******************************************/
#define FDCAN_TXBRP_TRP0_Pos                      (0U)
#define FDCAN_TXBRP_TRP0_Msk                      (0x1UL<<FDCAN_TXBRP_TRP0_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXBRP_TRP0                          FDCAN_TXBRP_TRP0_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP1_Pos                      (1U)
#define FDCAN_TXBRP_TRP1_Msk                      (0x1UL<<FDCAN_TXBRP_TRP1_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXBRP_TRP1                          FDCAN_TXBRP_TRP1_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP2_Pos                      (2U)
#define FDCAN_TXBRP_TRP2_Msk                      (0x1UL<<FDCAN_TXBRP_TRP2_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXBRP_TRP2                          FDCAN_TXBRP_TRP2_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP3_Pos                      (3U)
#define FDCAN_TXBRP_TRP3_Msk                      (0x1UL<<FDCAN_TXBRP_TRP3_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXBRP_TRP3                          FDCAN_TXBRP_TRP3_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP4_Pos                      (4U)
#define FDCAN_TXBRP_TRP4_Msk                      (0x1UL<<FDCAN_TXBRP_TRP4_Pos)                     /*!< 0x00000010 */
#define FDCAN_TXBRP_TRP4                          FDCAN_TXBRP_TRP4_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP5_Pos                      (5U)
#define FDCAN_TXBRP_TRP5_Msk                      (0x1UL<<FDCAN_TXBRP_TRP5_Pos)                     /*!< 0x00000020 */
#define FDCAN_TXBRP_TRP5                          FDCAN_TXBRP_TRP5_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP6_Pos                      (6U)
#define FDCAN_TXBRP_TRP6_Msk                      (0x1UL<<FDCAN_TXBRP_TRP6_Pos)                     /*!< 0x00000040 */
#define FDCAN_TXBRP_TRP6                          FDCAN_TXBRP_TRP6_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP7_Pos                      (7U)
#define FDCAN_TXBRP_TRP7_Msk                      (0x1UL<<FDCAN_TXBRP_TRP7_Pos)                     /*!< 0x00000080 */
#define FDCAN_TXBRP_TRP7                          FDCAN_TXBRP_TRP7_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP8_Pos                      (8U)
#define FDCAN_TXBRP_TRP8_Msk                      (0x1UL<<FDCAN_TXBRP_TRP8_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXBRP_TRP8                          FDCAN_TXBRP_TRP8_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP9_Pos                      (9U)
#define FDCAN_TXBRP_TRP9_Msk                      (0x1UL<<FDCAN_TXBRP_TRP9_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXBRP_TRP9                          FDCAN_TXBRP_TRP9_Msk                              /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP10_Pos                     (10U)
#define FDCAN_TXBRP_TRP10_Msk                     (0x1UL<<FDCAN_TXBRP_TRP10_Pos)                    /*!< 0x00000400 */
#define FDCAN_TXBRP_TRP10                         FDCAN_TXBRP_TRP10_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP11_Pos                     (11U)
#define FDCAN_TXBRP_TRP11_Msk                     (0x1UL<<FDCAN_TXBRP_TRP11_Pos)                    /*!< 0x00000800 */
#define FDCAN_TXBRP_TRP11                         FDCAN_TXBRP_TRP11_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP12_Pos                     (12U)
#define FDCAN_TXBRP_TRP12_Msk                     (0x1UL<<FDCAN_TXBRP_TRP12_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXBRP_TRP12                         FDCAN_TXBRP_TRP12_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP13_Pos                     (13U)
#define FDCAN_TXBRP_TRP13_Msk                     (0x1UL<<FDCAN_TXBRP_TRP13_Pos)                    /*!< 0x00002000 */
#define FDCAN_TXBRP_TRP13                         FDCAN_TXBRP_TRP13_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP14_Pos                     (14U)
#define FDCAN_TXBRP_TRP14_Msk                     (0x1UL<<FDCAN_TXBRP_TRP14_Pos)                    /*!< 0x00004000 */
#define FDCAN_TXBRP_TRP14                         FDCAN_TXBRP_TRP14_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP15_Pos                     (15U)
#define FDCAN_TXBRP_TRP15_Msk                     (0x1UL<<FDCAN_TXBRP_TRP15_Pos)                    /*!< 0x00008000 */
#define FDCAN_TXBRP_TRP15                         FDCAN_TXBRP_TRP15_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP16_Pos                     (16U)
#define FDCAN_TXBRP_TRP16_Msk                     (0x1UL<<FDCAN_TXBRP_TRP16_Pos)                    /*!< 0x00010000 */
#define FDCAN_TXBRP_TRP16                         FDCAN_TXBRP_TRP16_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP17_Pos                     (17U)
#define FDCAN_TXBRP_TRP17_Msk                     (0x1UL<<FDCAN_TXBRP_TRP17_Pos)                    /*!< 0x00020000 */
#define FDCAN_TXBRP_TRP17                         FDCAN_TXBRP_TRP17_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP18_Pos                     (18U)
#define FDCAN_TXBRP_TRP18_Msk                     (0x1UL<<FDCAN_TXBRP_TRP18_Pos)                    /*!< 0x00040000 */
#define FDCAN_TXBRP_TRP18                         FDCAN_TXBRP_TRP18_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP19_Pos                     (19U)
#define FDCAN_TXBRP_TRP19_Msk                     (0x1UL<<FDCAN_TXBRP_TRP19_Pos)                    /*!< 0x00080000 */
#define FDCAN_TXBRP_TRP19                         FDCAN_TXBRP_TRP19_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP20_Pos                     (20U)
#define FDCAN_TXBRP_TRP20_Msk                     (0x1UL<<FDCAN_TXBRP_TRP20_Pos)                    /*!< 0x00100000 */
#define FDCAN_TXBRP_TRP20                         FDCAN_TXBRP_TRP20_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP21_Pos                     (21U)
#define FDCAN_TXBRP_TRP21_Msk                     (0x1UL<<FDCAN_TXBRP_TRP21_Pos)                    /*!< 0x00200000 */
#define FDCAN_TXBRP_TRP21                         FDCAN_TXBRP_TRP21_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP22_Pos                     (22U)
#define FDCAN_TXBRP_TRP22_Msk                     (0x1UL<<FDCAN_TXBRP_TRP22_Pos)                    /*!< 0x00400000 */
#define FDCAN_TXBRP_TRP22                         FDCAN_TXBRP_TRP22_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP23_Pos                     (23U)
#define FDCAN_TXBRP_TRP23_Msk                     (0x1UL<<FDCAN_TXBRP_TRP23_Pos)                    /*!< 0x00800000 */
#define FDCAN_TXBRP_TRP23                         FDCAN_TXBRP_TRP23_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP24_Pos                     (24U)
#define FDCAN_TXBRP_TRP24_Msk                     (0x1UL<<FDCAN_TXBRP_TRP24_Pos)                    /*!< 0x01000000 */
#define FDCAN_TXBRP_TRP24                         FDCAN_TXBRP_TRP24_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP25_Pos                     (25U)
#define FDCAN_TXBRP_TRP25_Msk                     (0x1UL<<FDCAN_TXBRP_TRP25_Pos)                    /*!< 0x02000000 */
#define FDCAN_TXBRP_TRP25                         FDCAN_TXBRP_TRP25_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP26_Pos                     (26U)
#define FDCAN_TXBRP_TRP26_Msk                     (0x1UL<<FDCAN_TXBRP_TRP26_Pos)                    /*!< 0x04000000 */
#define FDCAN_TXBRP_TRP26                         FDCAN_TXBRP_TRP26_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP27_Pos                     (27U)
#define FDCAN_TXBRP_TRP27_Msk                     (0x1UL<<FDCAN_TXBRP_TRP27_Pos)                    /*!< 0x08000000 */
#define FDCAN_TXBRP_TRP27                         FDCAN_TXBRP_TRP27_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP28_Pos                     (28U)
#define FDCAN_TXBRP_TRP28_Msk                     (0x1UL<<FDCAN_TXBRP_TRP28_Pos)                    /*!< 0x10000000 */
#define FDCAN_TXBRP_TRP28                         FDCAN_TXBRP_TRP28_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP29_Pos                     (29U)
#define FDCAN_TXBRP_TRP29_Msk                     (0x1UL<<FDCAN_TXBRP_TRP29_Pos)                    /*!< 0x20000000 */
#define FDCAN_TXBRP_TRP29                         FDCAN_TXBRP_TRP29_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP30_Pos                     (30U)
#define FDCAN_TXBRP_TRP30_Msk                     (0x1UL<<FDCAN_TXBRP_TRP30_Pos)                    /*!< 0x40000000 */
#define FDCAN_TXBRP_TRP30                         FDCAN_TXBRP_TRP30_Msk                             /*!< Transmission Request Pending */
#define FDCAN_TXBRP_TRP31_Pos                     (31U)
#define FDCAN_TXBRP_TRP31_Msk                     (0x1UL<<FDCAN_TXBRP_TRP31_Pos)                    /*!< 0x80000000 */
#define FDCAN_TXBRP_TRP31                         FDCAN_TXBRP_TRP31_Msk                             /*!< Transmission Request Pending */

/********************************* Bit definition for FDCAN_TXBAR register ******************************************/
#define FDCAN_TXBAR_AR0_Pos                       (0U)
#define FDCAN_TXBAR_AR0_Msk                       (0x1UL<<FDCAN_TXBAR_AR0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBAR_AR0                           FDCAN_TXBAR_AR0_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR1_Pos                       (1U)
#define FDCAN_TXBAR_AR1_Msk                       (0x1UL<<FDCAN_TXBAR_AR1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBAR_AR1                           FDCAN_TXBAR_AR1_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR2_Pos                       (2U)
#define FDCAN_TXBAR_AR2_Msk                       (0x1UL<<FDCAN_TXBAR_AR2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBAR_AR2                           FDCAN_TXBAR_AR2_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR3_Pos                       (3U)
#define FDCAN_TXBAR_AR3_Msk                       (0x1UL<<FDCAN_TXBAR_AR3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBAR_AR3                           FDCAN_TXBAR_AR3_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR4_Pos                       (4U)
#define FDCAN_TXBAR_AR4_Msk                       (0x1UL<<FDCAN_TXBAR_AR4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBAR_AR4                           FDCAN_TXBAR_AR4_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR5_Pos                       (5U)
#define FDCAN_TXBAR_AR5_Msk                       (0x1UL<<FDCAN_TXBAR_AR5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBAR_AR5                           FDCAN_TXBAR_AR5_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR6_Pos                       (6U)
#define FDCAN_TXBAR_AR6_Msk                       (0x1UL<<FDCAN_TXBAR_AR6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBAR_AR6                           FDCAN_TXBAR_AR6_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR7_Pos                       (7U)
#define FDCAN_TXBAR_AR7_Msk                       (0x1UL<<FDCAN_TXBAR_AR7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBAR_AR7                           FDCAN_TXBAR_AR7_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR8_Pos                       (8U)
#define FDCAN_TXBAR_AR8_Msk                       (0x1UL<<FDCAN_TXBAR_AR8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBAR_AR8                           FDCAN_TXBAR_AR8_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR9_Pos                       (9U)
#define FDCAN_TXBAR_AR9_Msk                       (0x1UL<<FDCAN_TXBAR_AR9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBAR_AR9                           FDCAN_TXBAR_AR9_Msk                               /*!< Add Request */
#define FDCAN_TXBAR_AR10_Pos                      (10U)
#define FDCAN_TXBAR_AR10_Msk                      (0x1UL<<FDCAN_TXBAR_AR10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBAR_AR10                          FDCAN_TXBAR_AR10_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR11_Pos                      (11U)
#define FDCAN_TXBAR_AR11_Msk                      (0x1UL<<FDCAN_TXBAR_AR11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBAR_AR11                          FDCAN_TXBAR_AR11_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR12_Pos                      (12U)
#define FDCAN_TXBAR_AR12_Msk                      (0x1UL<<FDCAN_TXBAR_AR12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBAR_AR12                          FDCAN_TXBAR_AR12_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR13_Pos                      (13U)
#define FDCAN_TXBAR_AR13_Msk                      (0x1UL<<FDCAN_TXBAR_AR13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBAR_AR13                          FDCAN_TXBAR_AR13_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR14_Pos                      (14U)
#define FDCAN_TXBAR_AR14_Msk                      (0x1UL<<FDCAN_TXBAR_AR14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBAR_AR14                          FDCAN_TXBAR_AR14_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR15_Pos                      (15U)
#define FDCAN_TXBAR_AR15_Msk                      (0x1UL<<FDCAN_TXBAR_AR15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBAR_AR15                          FDCAN_TXBAR_AR15_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR16_Pos                      (16U)
#define FDCAN_TXBAR_AR16_Msk                      (0x1UL<<FDCAN_TXBAR_AR16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBAR_AR16                          FDCAN_TXBAR_AR16_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR17_Pos                      (17U)
#define FDCAN_TXBAR_AR17_Msk                      (0x1UL<<FDCAN_TXBAR_AR17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBAR_AR17                          FDCAN_TXBAR_AR17_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR18_Pos                      (18U)
#define FDCAN_TXBAR_AR18_Msk                      (0x1UL<<FDCAN_TXBAR_AR18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBAR_AR18                          FDCAN_TXBAR_AR18_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR19_Pos                      (19U)
#define FDCAN_TXBAR_AR19_Msk                      (0x1UL<<FDCAN_TXBAR_AR19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBAR_AR19                          FDCAN_TXBAR_AR19_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR20_Pos                      (20U)
#define FDCAN_TXBAR_AR20_Msk                      (0x1UL<<FDCAN_TXBAR_AR20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBAR_AR20                          FDCAN_TXBAR_AR20_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR21_Pos                      (21U)
#define FDCAN_TXBAR_AR21_Msk                      (0x1UL<<FDCAN_TXBAR_AR21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBAR_AR21                          FDCAN_TXBAR_AR21_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR22_Pos                      (22U)
#define FDCAN_TXBAR_AR22_Msk                      (0x1UL<<FDCAN_TXBAR_AR22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBAR_AR22                          FDCAN_TXBAR_AR22_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR23_Pos                      (23U)
#define FDCAN_TXBAR_AR23_Msk                      (0x1UL<<FDCAN_TXBAR_AR23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBAR_AR23                          FDCAN_TXBAR_AR23_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR24_Pos                      (24U)
#define FDCAN_TXBAR_AR24_Msk                      (0x1UL<<FDCAN_TXBAR_AR24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBAR_AR24                          FDCAN_TXBAR_AR24_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR25_Pos                      (25U)
#define FDCAN_TXBAR_AR25_Msk                      (0x1UL<<FDCAN_TXBAR_AR25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBAR_AR25                          FDCAN_TXBAR_AR25_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR26_Pos                      (26U)
#define FDCAN_TXBAR_AR26_Msk                      (0x1UL<<FDCAN_TXBAR_AR26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBAR_AR26                          FDCAN_TXBAR_AR26_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR27_Pos                      (27U)
#define FDCAN_TXBAR_AR27_Msk                      (0x1UL<<FDCAN_TXBAR_AR27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBAR_AR27                          FDCAN_TXBAR_AR27_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR28_Pos                      (28U)
#define FDCAN_TXBAR_AR28_Msk                      (0x1UL<<FDCAN_TXBAR_AR28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBAR_AR28                          FDCAN_TXBAR_AR28_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR29_Pos                      (29U)
#define FDCAN_TXBAR_AR29_Msk                      (0x1UL<<FDCAN_TXBAR_AR29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBAR_AR29                          FDCAN_TXBAR_AR29_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR30_Pos                      (30U)
#define FDCAN_TXBAR_AR30_Msk                      (0x1UL<<FDCAN_TXBAR_AR30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBAR_AR30                          FDCAN_TXBAR_AR30_Msk                              /*!< Add Request */
#define FDCAN_TXBAR_AR31_Pos                      (31U)
#define FDCAN_TXBAR_AR31_Msk                      (0x1UL<<FDCAN_TXBAR_AR31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBAR_AR31                          FDCAN_TXBAR_AR31_Msk                              /*!< Add Request */

/********************************* Bit definition for FDCAN_TXBCR register ******************************************/
#define FDCAN_TXBCR_CR0_Pos                       (0U)
#define FDCAN_TXBCR_CR0_Msk                       (0x1UL<<FDCAN_TXBCR_CR0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBCR_CR0                           FDCAN_TXBCR_CR0_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR1_Pos                       (1U)
#define FDCAN_TXBCR_CR1_Msk                       (0x1UL<<FDCAN_TXBCR_CR1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBCR_CR1                           FDCAN_TXBCR_CR1_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR2_Pos                       (2U)
#define FDCAN_TXBCR_CR2_Msk                       (0x1UL<<FDCAN_TXBCR_CR2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBCR_CR2                           FDCAN_TXBCR_CR2_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR3_Pos                       (3U)
#define FDCAN_TXBCR_CR3_Msk                       (0x1UL<<FDCAN_TXBCR_CR3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBCR_CR3                           FDCAN_TXBCR_CR3_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR4_Pos                       (4U)
#define FDCAN_TXBCR_CR4_Msk                       (0x1UL<<FDCAN_TXBCR_CR4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBCR_CR4                           FDCAN_TXBCR_CR4_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR5_Pos                       (5U)
#define FDCAN_TXBCR_CR5_Msk                       (0x1UL<<FDCAN_TXBCR_CR5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBCR_CR5                           FDCAN_TXBCR_CR5_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR6_Pos                       (6U)
#define FDCAN_TXBCR_CR6_Msk                       (0x1UL<<FDCAN_TXBCR_CR6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBCR_CR6                           FDCAN_TXBCR_CR6_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR7_Pos                       (7U)
#define FDCAN_TXBCR_CR7_Msk                       (0x1UL<<FDCAN_TXBCR_CR7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBCR_CR7                           FDCAN_TXBCR_CR7_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR8_Pos                       (8U)
#define FDCAN_TXBCR_CR8_Msk                       (0x1UL<<FDCAN_TXBCR_CR8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBCR_CR8                           FDCAN_TXBCR_CR8_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR9_Pos                       (9U)
#define FDCAN_TXBCR_CR9_Msk                       (0x1UL<<FDCAN_TXBCR_CR9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBCR_CR9                           FDCAN_TXBCR_CR9_Msk                               /*!< Cancellation Request */
#define FDCAN_TXBCR_CR10_Pos                      (10U)
#define FDCAN_TXBCR_CR10_Msk                      (0x1UL<<FDCAN_TXBCR_CR10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBCR_CR10                          FDCAN_TXBCR_CR10_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR11_Pos                      (11U)
#define FDCAN_TXBCR_CR11_Msk                      (0x1UL<<FDCAN_TXBCR_CR11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBCR_CR11                          FDCAN_TXBCR_CR11_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR12_Pos                      (12U)
#define FDCAN_TXBCR_CR12_Msk                      (0x1UL<<FDCAN_TXBCR_CR12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBCR_CR12                          FDCAN_TXBCR_CR12_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR13_Pos                      (13U)
#define FDCAN_TXBCR_CR13_Msk                      (0x1UL<<FDCAN_TXBCR_CR13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBCR_CR13                          FDCAN_TXBCR_CR13_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR14_Pos                      (14U)
#define FDCAN_TXBCR_CR14_Msk                      (0x1UL<<FDCAN_TXBCR_CR14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBCR_CR14                          FDCAN_TXBCR_CR14_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR15_Pos                      (15U)
#define FDCAN_TXBCR_CR15_Msk                      (0x1UL<<FDCAN_TXBCR_CR15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBCR_CR15                          FDCAN_TXBCR_CR15_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR16_Pos                      (16U)
#define FDCAN_TXBCR_CR16_Msk                      (0x1UL<<FDCAN_TXBCR_CR16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBCR_CR16                          FDCAN_TXBCR_CR16_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR17_Pos                      (17U)
#define FDCAN_TXBCR_CR17_Msk                      (0x1UL<<FDCAN_TXBCR_CR17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBCR_CR17                          FDCAN_TXBCR_CR17_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR18_Pos                      (18U)
#define FDCAN_TXBCR_CR18_Msk                      (0x1UL<<FDCAN_TXBCR_CR18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBCR_CR18                          FDCAN_TXBCR_CR18_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR19_Pos                      (19U)
#define FDCAN_TXBCR_CR19_Msk                      (0x1UL<<FDCAN_TXBCR_CR19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBCR_CR19                          FDCAN_TXBCR_CR19_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR20_Pos                      (20U)
#define FDCAN_TXBCR_CR20_Msk                      (0x1UL<<FDCAN_TXBCR_CR20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBCR_CR20                          FDCAN_TXBCR_CR20_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR21_Pos                      (21U)
#define FDCAN_TXBCR_CR21_Msk                      (0x1UL<<FDCAN_TXBCR_CR21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBCR_CR21                          FDCAN_TXBCR_CR21_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR22_Pos                      (22U)
#define FDCAN_TXBCR_CR22_Msk                      (0x1UL<<FDCAN_TXBCR_CR22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBCR_CR22                          FDCAN_TXBCR_CR22_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR23_Pos                      (23U)
#define FDCAN_TXBCR_CR23_Msk                      (0x1UL<<FDCAN_TXBCR_CR23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBCR_CR23                          FDCAN_TXBCR_CR23_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR24_Pos                      (24U)
#define FDCAN_TXBCR_CR24_Msk                      (0x1UL<<FDCAN_TXBCR_CR24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBCR_CR24                          FDCAN_TXBCR_CR24_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR25_Pos                      (25U)
#define FDCAN_TXBCR_CR25_Msk                      (0x1UL<<FDCAN_TXBCR_CR25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBCR_CR25                          FDCAN_TXBCR_CR25_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR26_Pos                      (26U)
#define FDCAN_TXBCR_CR26_Msk                      (0x1UL<<FDCAN_TXBCR_CR26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBCR_CR26                          FDCAN_TXBCR_CR26_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR27_Pos                      (27U)
#define FDCAN_TXBCR_CR27_Msk                      (0x1UL<<FDCAN_TXBCR_CR27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBCR_CR27                          FDCAN_TXBCR_CR27_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR28_Pos                      (28U)
#define FDCAN_TXBCR_CR28_Msk                      (0x1UL<<FDCAN_TXBCR_CR28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBCR_CR28                          FDCAN_TXBCR_CR28_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR29_Pos                      (29U)
#define FDCAN_TXBCR_CR29_Msk                      (0x1UL<<FDCAN_TXBCR_CR29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBCR_CR29                          FDCAN_TXBCR_CR29_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR30_Pos                      (30U)
#define FDCAN_TXBCR_CR30_Msk                      (0x1UL<<FDCAN_TXBCR_CR30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBCR_CR30                          FDCAN_TXBCR_CR30_Msk                              /*!< Cancellation Request */
#define FDCAN_TXBCR_CR31_Pos                      (31U)
#define FDCAN_TXBCR_CR31_Msk                      (0x1UL<<FDCAN_TXBCR_CR31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBCR_CR31                          FDCAN_TXBCR_CR31_Msk                              /*!< Cancellation Request */

/********************************* Bit definition for FDCAN_TXBTO register ******************************************/
#define FDCAN_TXBTO_TO0_Pos                       (0U)
#define FDCAN_TXBTO_TO0_Msk                       (0x1UL<<FDCAN_TXBTO_TO0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBTO_TO0                           FDCAN_TXBTO_TO0_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO1_Pos                       (1U)
#define FDCAN_TXBTO_TO1_Msk                       (0x1UL<<FDCAN_TXBTO_TO1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBTO_TO1                           FDCAN_TXBTO_TO1_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO2_Pos                       (2U)
#define FDCAN_TXBTO_TO2_Msk                       (0x1UL<<FDCAN_TXBTO_TO2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBTO_TO2                           FDCAN_TXBTO_TO2_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO3_Pos                       (3U)
#define FDCAN_TXBTO_TO3_Msk                       (0x1UL<<FDCAN_TXBTO_TO3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBTO_TO3                           FDCAN_TXBTO_TO3_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO4_Pos                       (4U)
#define FDCAN_TXBTO_TO4_Msk                       (0x1UL<<FDCAN_TXBTO_TO4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBTO_TO4                           FDCAN_TXBTO_TO4_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO5_Pos                       (5U)
#define FDCAN_TXBTO_TO5_Msk                       (0x1UL<<FDCAN_TXBTO_TO5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBTO_TO5                           FDCAN_TXBTO_TO5_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO6_Pos                       (6U)
#define FDCAN_TXBTO_TO6_Msk                       (0x1UL<<FDCAN_TXBTO_TO6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBTO_TO6                           FDCAN_TXBTO_TO6_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO7_Pos                       (7U)
#define FDCAN_TXBTO_TO7_Msk                       (0x1UL<<FDCAN_TXBTO_TO7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBTO_TO7                           FDCAN_TXBTO_TO7_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO8_Pos                       (8U)
#define FDCAN_TXBTO_TO8_Msk                       (0x1UL<<FDCAN_TXBTO_TO8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBTO_TO8                           FDCAN_TXBTO_TO8_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO9_Pos                       (9U)
#define FDCAN_TXBTO_TO9_Msk                       (0x1UL<<FDCAN_TXBTO_TO9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBTO_TO9                           FDCAN_TXBTO_TO9_Msk                               /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO10_Pos                      (10U)
#define FDCAN_TXBTO_TO10_Msk                      (0x1UL<<FDCAN_TXBTO_TO10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBTO_TO10                          FDCAN_TXBTO_TO10_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO11_Pos                      (11U)
#define FDCAN_TXBTO_TO11_Msk                      (0x1UL<<FDCAN_TXBTO_TO11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBTO_TO11                          FDCAN_TXBTO_TO11_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO12_Pos                      (12U)
#define FDCAN_TXBTO_TO12_Msk                      (0x1UL<<FDCAN_TXBTO_TO12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBTO_TO12                          FDCAN_TXBTO_TO12_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO13_Pos                      (13U)
#define FDCAN_TXBTO_TO13_Msk                      (0x1UL<<FDCAN_TXBTO_TO13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBTO_TO13                          FDCAN_TXBTO_TO13_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO14_Pos                      (14U)
#define FDCAN_TXBTO_TO14_Msk                      (0x1UL<<FDCAN_TXBTO_TO14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBTO_TO14                          FDCAN_TXBTO_TO14_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO15_Pos                      (15U)
#define FDCAN_TXBTO_TO15_Msk                      (0x1UL<<FDCAN_TXBTO_TO15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBTO_TO15                          FDCAN_TXBTO_TO15_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO16_Pos                      (16U)
#define FDCAN_TXBTO_TO16_Msk                      (0x1UL<<FDCAN_TXBTO_TO16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBTO_TO16                          FDCAN_TXBTO_TO16_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO17_Pos                      (17U)
#define FDCAN_TXBTO_TO17_Msk                      (0x1UL<<FDCAN_TXBTO_TO17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBTO_TO17                          FDCAN_TXBTO_TO17_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO18_Pos                      (18U)
#define FDCAN_TXBTO_TO18_Msk                      (0x1UL<<FDCAN_TXBTO_TO18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBTO_TO18                          FDCAN_TXBTO_TO18_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO19_Pos                      (19U)
#define FDCAN_TXBTO_TO19_Msk                      (0x1UL<<FDCAN_TXBTO_TO19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBTO_TO19                          FDCAN_TXBTO_TO19_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO20_Pos                      (20U)
#define FDCAN_TXBTO_TO20_Msk                      (0x1UL<<FDCAN_TXBTO_TO20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBTO_TO20                          FDCAN_TXBTO_TO20_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO21_Pos                      (21U)
#define FDCAN_TXBTO_TO21_Msk                      (0x1UL<<FDCAN_TXBTO_TO21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBTO_TO21                          FDCAN_TXBTO_TO21_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO22_Pos                      (22U)
#define FDCAN_TXBTO_TO22_Msk                      (0x1UL<<FDCAN_TXBTO_TO22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBTO_TO22                          FDCAN_TXBTO_TO22_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO23_Pos                      (23U)
#define FDCAN_TXBTO_TO23_Msk                      (0x1UL<<FDCAN_TXBTO_TO23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBTO_TO23                          FDCAN_TXBTO_TO23_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO24_Pos                      (24U)
#define FDCAN_TXBTO_TO24_Msk                      (0x1UL<<FDCAN_TXBTO_TO24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBTO_TO24                          FDCAN_TXBTO_TO24_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO25_Pos                      (25U)
#define FDCAN_TXBTO_TO25_Msk                      (0x1UL<<FDCAN_TXBTO_TO25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBTO_TO25                          FDCAN_TXBTO_TO25_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO26_Pos                      (26U)
#define FDCAN_TXBTO_TO26_Msk                      (0x1UL<<FDCAN_TXBTO_TO26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBTO_TO26                          FDCAN_TXBTO_TO26_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO27_Pos                      (27U)
#define FDCAN_TXBTO_TO27_Msk                      (0x1UL<<FDCAN_TXBTO_TO27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBTO_TO27                          FDCAN_TXBTO_TO27_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO28_Pos                      (28U)
#define FDCAN_TXBTO_TO28_Msk                      (0x1UL<<FDCAN_TXBTO_TO28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBTO_TO28                          FDCAN_TXBTO_TO28_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO29_Pos                      (29U)
#define FDCAN_TXBTO_TO29_Msk                      (0x1UL<<FDCAN_TXBTO_TO29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBTO_TO29                          FDCAN_TXBTO_TO29_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO30_Pos                      (30U)
#define FDCAN_TXBTO_TO30_Msk                      (0x1UL<<FDCAN_TXBTO_TO30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBTO_TO30                          FDCAN_TXBTO_TO30_Msk                              /*!< Transmission Occurred */
#define FDCAN_TXBTO_TO31_Pos                      (31U)
#define FDCAN_TXBTO_TO31_Msk                      (0x1UL<<FDCAN_TXBTO_TO31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBTO_TO31                          FDCAN_TXBTO_TO31_Msk                              /*!< Transmission Occurred */

/********************************* Bit definition for FDCAN_TXBCF register ******************************************/
#define FDCAN_TXBCF_CF0_Pos                       (0U)
#define FDCAN_TXBCF_CF0_Msk                       (0x1UL<<FDCAN_TXBCF_CF0_Pos)                      /*!< 0x00000001 */
#define FDCAN_TXBCF_CF0                           FDCAN_TXBCF_CF0_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF1_Pos                       (1U)
#define FDCAN_TXBCF_CF1_Msk                       (0x1UL<<FDCAN_TXBCF_CF1_Pos)                      /*!< 0x00000002 */
#define FDCAN_TXBCF_CF1                           FDCAN_TXBCF_CF1_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF2_Pos                       (2U)
#define FDCAN_TXBCF_CF2_Msk                       (0x1UL<<FDCAN_TXBCF_CF2_Pos)                      /*!< 0x00000004 */
#define FDCAN_TXBCF_CF2                           FDCAN_TXBCF_CF2_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF3_Pos                       (3U)
#define FDCAN_TXBCF_CF3_Msk                       (0x1UL<<FDCAN_TXBCF_CF3_Pos)                      /*!< 0x00000008 */
#define FDCAN_TXBCF_CF3                           FDCAN_TXBCF_CF3_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF4_Pos                       (4U)
#define FDCAN_TXBCF_CF4_Msk                       (0x1UL<<FDCAN_TXBCF_CF4_Pos)                      /*!< 0x00000010 */
#define FDCAN_TXBCF_CF4                           FDCAN_TXBCF_CF4_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF5_Pos                       (5U)
#define FDCAN_TXBCF_CF5_Msk                       (0x1UL<<FDCAN_TXBCF_CF5_Pos)                      /*!< 0x00000020 */
#define FDCAN_TXBCF_CF5                           FDCAN_TXBCF_CF5_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF6_Pos                       (6U)
#define FDCAN_TXBCF_CF6_Msk                       (0x1UL<<FDCAN_TXBCF_CF6_Pos)                      /*!< 0x00000040 */
#define FDCAN_TXBCF_CF6                           FDCAN_TXBCF_CF6_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF7_Pos                       (7U)
#define FDCAN_TXBCF_CF7_Msk                       (0x1UL<<FDCAN_TXBCF_CF7_Pos)                      /*!< 0x00000080 */
#define FDCAN_TXBCF_CF7                           FDCAN_TXBCF_CF7_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF8_Pos                       (8U)
#define FDCAN_TXBCF_CF8_Msk                       (0x1UL<<FDCAN_TXBCF_CF8_Pos)                      /*!< 0x00000100 */
#define FDCAN_TXBCF_CF8                           FDCAN_TXBCF_CF8_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF9_Pos                       (9U)
#define FDCAN_TXBCF_CF9_Msk                       (0x1UL<<FDCAN_TXBCF_CF9_Pos)                      /*!< 0x00000200 */
#define FDCAN_TXBCF_CF9                           FDCAN_TXBCF_CF9_Msk                               /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF10_Pos                      (10U)
#define FDCAN_TXBCF_CF10_Msk                      (0x1UL<<FDCAN_TXBCF_CF10_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXBCF_CF10                          FDCAN_TXBCF_CF10_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF11_Pos                      (11U)
#define FDCAN_TXBCF_CF11_Msk                      (0x1UL<<FDCAN_TXBCF_CF11_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXBCF_CF11                          FDCAN_TXBCF_CF11_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF12_Pos                      (12U)
#define FDCAN_TXBCF_CF12_Msk                      (0x1UL<<FDCAN_TXBCF_CF12_Pos)                     /*!< 0x00001000 */
#define FDCAN_TXBCF_CF12                          FDCAN_TXBCF_CF12_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF13_Pos                      (13U)
#define FDCAN_TXBCF_CF13_Msk                      (0x1UL<<FDCAN_TXBCF_CF13_Pos)                     /*!< 0x00002000 */
#define FDCAN_TXBCF_CF13                          FDCAN_TXBCF_CF13_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF14_Pos                      (14U)
#define FDCAN_TXBCF_CF14_Msk                      (0x1UL<<FDCAN_TXBCF_CF14_Pos)                     /*!< 0x00004000 */
#define FDCAN_TXBCF_CF14                          FDCAN_TXBCF_CF14_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF15_Pos                      (15U)
#define FDCAN_TXBCF_CF15_Msk                      (0x1UL<<FDCAN_TXBCF_CF15_Pos)                     /*!< 0x00008000 */
#define FDCAN_TXBCF_CF15                          FDCAN_TXBCF_CF15_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF16_Pos                      (16U)
#define FDCAN_TXBCF_CF16_Msk                      (0x1UL<<FDCAN_TXBCF_CF16_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXBCF_CF16                          FDCAN_TXBCF_CF16_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF17_Pos                      (17U)
#define FDCAN_TXBCF_CF17_Msk                      (0x1UL<<FDCAN_TXBCF_CF17_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXBCF_CF17                          FDCAN_TXBCF_CF17_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF18_Pos                      (18U)
#define FDCAN_TXBCF_CF18_Msk                      (0x1UL<<FDCAN_TXBCF_CF18_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXBCF_CF18                          FDCAN_TXBCF_CF18_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF19_Pos                      (19U)
#define FDCAN_TXBCF_CF19_Msk                      (0x1UL<<FDCAN_TXBCF_CF19_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXBCF_CF19                          FDCAN_TXBCF_CF19_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF20_Pos                      (20U)
#define FDCAN_TXBCF_CF20_Msk                      (0x1UL<<FDCAN_TXBCF_CF20_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXBCF_CF20                          FDCAN_TXBCF_CF20_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF21_Pos                      (21U)
#define FDCAN_TXBCF_CF21_Msk                      (0x1UL<<FDCAN_TXBCF_CF21_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXBCF_CF21                          FDCAN_TXBCF_CF21_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF22_Pos                      (22U)
#define FDCAN_TXBCF_CF22_Msk                      (0x1UL<<FDCAN_TXBCF_CF22_Pos)                     /*!< 0x00400000 */
#define FDCAN_TXBCF_CF22                          FDCAN_TXBCF_CF22_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF23_Pos                      (23U)
#define FDCAN_TXBCF_CF23_Msk                      (0x1UL<<FDCAN_TXBCF_CF23_Pos)                     /*!< 0x00800000 */
#define FDCAN_TXBCF_CF23                          FDCAN_TXBCF_CF23_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF24_Pos                      (24U)
#define FDCAN_TXBCF_CF24_Msk                      (0x1UL<<FDCAN_TXBCF_CF24_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXBCF_CF24                          FDCAN_TXBCF_CF24_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF25_Pos                      (25U)
#define FDCAN_TXBCF_CF25_Msk                      (0x1UL<<FDCAN_TXBCF_CF25_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXBCF_CF25                          FDCAN_TXBCF_CF25_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF26_Pos                      (26U)
#define FDCAN_TXBCF_CF26_Msk                      (0x1UL<<FDCAN_TXBCF_CF26_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXBCF_CF26                          FDCAN_TXBCF_CF26_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF27_Pos                      (27U)
#define FDCAN_TXBCF_CF27_Msk                      (0x1UL<<FDCAN_TXBCF_CF27_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXBCF_CF27                          FDCAN_TXBCF_CF27_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF28_Pos                      (28U)
#define FDCAN_TXBCF_CF28_Msk                      (0x1UL<<FDCAN_TXBCF_CF28_Pos)                     /*!< 0x10000000 */
#define FDCAN_TXBCF_CF28                          FDCAN_TXBCF_CF28_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF29_Pos                      (29U)
#define FDCAN_TXBCF_CF29_Msk                      (0x1UL<<FDCAN_TXBCF_CF29_Pos)                     /*!< 0x20000000 */
#define FDCAN_TXBCF_CF29                          FDCAN_TXBCF_CF29_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF30_Pos                      (30U)
#define FDCAN_TXBCF_CF30_Msk                      (0x1UL<<FDCAN_TXBCF_CF30_Pos)                     /*!< 0x40000000 */
#define FDCAN_TXBCF_CF30                          FDCAN_TXBCF_CF30_Msk                              /*!< Cancellation Finished */
#define FDCAN_TXBCF_CF31_Pos                      (31U)
#define FDCAN_TXBCF_CF31_Msk                      (0x1UL<<FDCAN_TXBCF_CF31_Pos)                     /*!< 0x80000000 */
#define FDCAN_TXBCF_CF31                          FDCAN_TXBCF_CF31_Msk                              /*!< Cancellation Finished */

/********************************* Bit definition for FDCAN_TXBTIE register *****************************************/
#define FDCAN_TXBTIE_TIE0_Pos                     (0U)
#define FDCAN_TXBTIE_TIE0_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE0_Pos)                    /*!< 0x00000001 */
#define FDCAN_TXBTIE_TIE0                         FDCAN_TXBTIE_TIE0_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE1_Pos                     (1U)
#define FDCAN_TXBTIE_TIE1_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE1_Pos)                    /*!< 0x00000002 */
#define FDCAN_TXBTIE_TIE1                         FDCAN_TXBTIE_TIE1_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE2_Pos                     (2U)
#define FDCAN_TXBTIE_TIE2_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE2_Pos)                    /*!< 0x00000004 */
#define FDCAN_TXBTIE_TIE2                         FDCAN_TXBTIE_TIE2_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE3_Pos                     (3U)
#define FDCAN_TXBTIE_TIE3_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE3_Pos)                    /*!< 0x00000008 */
#define FDCAN_TXBTIE_TIE3                         FDCAN_TXBTIE_TIE3_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE4_Pos                     (4U)
#define FDCAN_TXBTIE_TIE4_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE4_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXBTIE_TIE4                         FDCAN_TXBTIE_TIE4_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE5_Pos                     (5U)
#define FDCAN_TXBTIE_TIE5_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE5_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXBTIE_TIE5                         FDCAN_TXBTIE_TIE5_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE6_Pos                     (6U)
#define FDCAN_TXBTIE_TIE6_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE6_Pos)                    /*!< 0x00000040 */
#define FDCAN_TXBTIE_TIE6                         FDCAN_TXBTIE_TIE6_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE7_Pos                     (7U)
#define FDCAN_TXBTIE_TIE7_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE7_Pos)                    /*!< 0x00000080 */
#define FDCAN_TXBTIE_TIE7                         FDCAN_TXBTIE_TIE7_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE8_Pos                     (8U)
#define FDCAN_TXBTIE_TIE8_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE8_Pos)                    /*!< 0x00000100 */
#define FDCAN_TXBTIE_TIE8                         FDCAN_TXBTIE_TIE8_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE9_Pos                     (9U)
#define FDCAN_TXBTIE_TIE9_Msk                     (0x1UL<<FDCAN_TXBTIE_TIE9_Pos)                    /*!< 0x00000200 */
#define FDCAN_TXBTIE_TIE9                         FDCAN_TXBTIE_TIE9_Msk                             /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE10_Pos                    (10U)
#define FDCAN_TXBTIE_TIE10_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE10_Pos)                   /*!< 0x00000400 */
#define FDCAN_TXBTIE_TIE10                        FDCAN_TXBTIE_TIE10_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE11_Pos                    (11U)
#define FDCAN_TXBTIE_TIE11_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE11_Pos)                   /*!< 0x00000800 */
#define FDCAN_TXBTIE_TIE11                        FDCAN_TXBTIE_TIE11_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE12_Pos                    (12U)
#define FDCAN_TXBTIE_TIE12_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE12_Pos)                   /*!< 0x00001000 */
#define FDCAN_TXBTIE_TIE12                        FDCAN_TXBTIE_TIE12_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE13_Pos                    (13U)
#define FDCAN_TXBTIE_TIE13_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE13_Pos)                   /*!< 0x00002000 */
#define FDCAN_TXBTIE_TIE13                        FDCAN_TXBTIE_TIE13_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE14_Pos                    (14U)
#define FDCAN_TXBTIE_TIE14_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE14_Pos)                   /*!< 0x00004000 */
#define FDCAN_TXBTIE_TIE14                        FDCAN_TXBTIE_TIE14_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE15_Pos                    (15U)
#define FDCAN_TXBTIE_TIE15_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE15_Pos)                   /*!< 0x00008000 */
#define FDCAN_TXBTIE_TIE15                        FDCAN_TXBTIE_TIE15_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE16_Pos                    (16U)
#define FDCAN_TXBTIE_TIE16_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE16_Pos)                   /*!< 0x00010000 */
#define FDCAN_TXBTIE_TIE16                        FDCAN_TXBTIE_TIE16_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE17_Pos                    (17U)
#define FDCAN_TXBTIE_TIE17_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE17_Pos)                   /*!< 0x00020000 */
#define FDCAN_TXBTIE_TIE17                        FDCAN_TXBTIE_TIE17_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE18_Pos                    (18U)
#define FDCAN_TXBTIE_TIE18_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE18_Pos)                   /*!< 0x00040000 */
#define FDCAN_TXBTIE_TIE18                        FDCAN_TXBTIE_TIE18_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE19_Pos                    (19U)
#define FDCAN_TXBTIE_TIE19_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE19_Pos)                   /*!< 0x00080000 */
#define FDCAN_TXBTIE_TIE19                        FDCAN_TXBTIE_TIE19_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE20_Pos                    (20U)
#define FDCAN_TXBTIE_TIE20_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE20_Pos)                   /*!< 0x00100000 */
#define FDCAN_TXBTIE_TIE20                        FDCAN_TXBTIE_TIE20_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE21_Pos                    (21U)
#define FDCAN_TXBTIE_TIE21_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE21_Pos)                   /*!< 0x00200000 */
#define FDCAN_TXBTIE_TIE21                        FDCAN_TXBTIE_TIE21_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE22_Pos                    (22U)
#define FDCAN_TXBTIE_TIE22_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE22_Pos)                   /*!< 0x00400000 */
#define FDCAN_TXBTIE_TIE22                        FDCAN_TXBTIE_TIE22_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE23_Pos                    (23U)
#define FDCAN_TXBTIE_TIE23_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE23_Pos)                   /*!< 0x00800000 */
#define FDCAN_TXBTIE_TIE23                        FDCAN_TXBTIE_TIE23_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE24_Pos                    (24U)
#define FDCAN_TXBTIE_TIE24_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE24_Pos)                   /*!< 0x01000000 */
#define FDCAN_TXBTIE_TIE24                        FDCAN_TXBTIE_TIE24_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE25_Pos                    (25U)
#define FDCAN_TXBTIE_TIE25_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE25_Pos)                   /*!< 0x02000000 */
#define FDCAN_TXBTIE_TIE25                        FDCAN_TXBTIE_TIE25_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE26_Pos                    (26U)
#define FDCAN_TXBTIE_TIE26_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE26_Pos)                   /*!< 0x04000000 */
#define FDCAN_TXBTIE_TIE26                        FDCAN_TXBTIE_TIE26_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE27_Pos                    (27U)
#define FDCAN_TXBTIE_TIE27_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE27_Pos)                   /*!< 0x08000000 */
#define FDCAN_TXBTIE_TIE27                        FDCAN_TXBTIE_TIE27_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE28_Pos                    (28U)
#define FDCAN_TXBTIE_TIE28_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE28_Pos)                   /*!< 0x10000000 */
#define FDCAN_TXBTIE_TIE28                        FDCAN_TXBTIE_TIE28_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE29_Pos                    (29U)
#define FDCAN_TXBTIE_TIE29_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE29_Pos)                   /*!< 0x20000000 */
#define FDCAN_TXBTIE_TIE29                        FDCAN_TXBTIE_TIE29_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE30_Pos                    (30U)
#define FDCAN_TXBTIE_TIE30_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE30_Pos)                   /*!< 0x40000000 */
#define FDCAN_TXBTIE_TIE30                        FDCAN_TXBTIE_TIE30_Msk                            /*!< Transmission Interrupt Enable */
#define FDCAN_TXBTIE_TIE31_Pos                    (31U)
#define FDCAN_TXBTIE_TIE31_Msk                    (0x1UL<<FDCAN_TXBTIE_TIE31_Pos)                   /*!< 0x80000000 */
#define FDCAN_TXBTIE_TIE31                        FDCAN_TXBTIE_TIE31_Msk                            /*!< Transmission Interrupt Enable */

/********************************* Bit definition for FDCAN_TXBCIE register *****************************************/
#define FDCAN_TXBCIE_CFIE0_Pos                    (0U)
#define FDCAN_TXBCIE_CFIE0_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE0_Pos)                   /*!< 0x00000001 */
#define FDCAN_TXBCIE_CFIE0                        FDCAN_TXBCIE_CFIE0_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE1_Pos                    (1U)
#define FDCAN_TXBCIE_CFIE1_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE1_Pos)                   /*!< 0x00000002 */
#define FDCAN_TXBCIE_CFIE1                        FDCAN_TXBCIE_CFIE1_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE2_Pos                    (2U)
#define FDCAN_TXBCIE_CFIE2_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE2_Pos)                   /*!< 0x00000004 */
#define FDCAN_TXBCIE_CFIE2                        FDCAN_TXBCIE_CFIE2_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE3_Pos                    (3U)
#define FDCAN_TXBCIE_CFIE3_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE3_Pos)                   /*!< 0x00000008 */
#define FDCAN_TXBCIE_CFIE3                        FDCAN_TXBCIE_CFIE3_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE4_Pos                    (4U)
#define FDCAN_TXBCIE_CFIE4_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE4_Pos)                   /*!< 0x00000010 */
#define FDCAN_TXBCIE_CFIE4                        FDCAN_TXBCIE_CFIE4_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE5_Pos                    (5U)
#define FDCAN_TXBCIE_CFIE5_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE5_Pos)                   /*!< 0x00000020 */
#define FDCAN_TXBCIE_CFIE5                        FDCAN_TXBCIE_CFIE5_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE6_Pos                    (6U)
#define FDCAN_TXBCIE_CFIE6_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE6_Pos)                   /*!< 0x00000040 */
#define FDCAN_TXBCIE_CFIE6                        FDCAN_TXBCIE_CFIE6_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE7_Pos                    (7U)
#define FDCAN_TXBCIE_CFIE7_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE7_Pos)                   /*!< 0x00000080 */
#define FDCAN_TXBCIE_CFIE7                        FDCAN_TXBCIE_CFIE7_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE8_Pos                    (8U)
#define FDCAN_TXBCIE_CFIE8_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE8_Pos)                   /*!< 0x00000100 */
#define FDCAN_TXBCIE_CFIE8                        FDCAN_TXBCIE_CFIE8_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE9_Pos                    (9U)
#define FDCAN_TXBCIE_CFIE9_Msk                    (0x1UL<<FDCAN_TXBCIE_CFIE9_Pos)                   /*!< 0x00000200 */
#define FDCAN_TXBCIE_CFIE9                        FDCAN_TXBCIE_CFIE9_Msk                            /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE10_Pos                   (10U)
#define FDCAN_TXBCIE_CFIE10_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE10_Pos)                  /*!< 0x00000400 */
#define FDCAN_TXBCIE_CFIE10                       FDCAN_TXBCIE_CFIE10_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE11_Pos                   (11U)
#define FDCAN_TXBCIE_CFIE11_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE11_Pos)                  /*!< 0x00000800 */
#define FDCAN_TXBCIE_CFIE11                       FDCAN_TXBCIE_CFIE11_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE12_Pos                   (12U)
#define FDCAN_TXBCIE_CFIE12_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE12_Pos)                  /*!< 0x00001000 */
#define FDCAN_TXBCIE_CFIE12                       FDCAN_TXBCIE_CFIE12_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE13_Pos                   (13U)
#define FDCAN_TXBCIE_CFIE13_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE13_Pos)                  /*!< 0x00002000 */
#define FDCAN_TXBCIE_CFIE13                       FDCAN_TXBCIE_CFIE13_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE14_Pos                   (14U)
#define FDCAN_TXBCIE_CFIE14_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE14_Pos)                  /*!< 0x00004000 */
#define FDCAN_TXBCIE_CFIE14                       FDCAN_TXBCIE_CFIE14_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE15_Pos                   (15U)
#define FDCAN_TXBCIE_CFIE15_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE15_Pos)                  /*!< 0x00008000 */
#define FDCAN_TXBCIE_CFIE15                       FDCAN_TXBCIE_CFIE15_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE16_Pos                   (16U)
#define FDCAN_TXBCIE_CFIE16_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE16_Pos)                  /*!< 0x00010000 */
#define FDCAN_TXBCIE_CFIE16                       FDCAN_TXBCIE_CFIE16_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE17_Pos                   (17U)
#define FDCAN_TXBCIE_CFIE17_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE17_Pos)                  /*!< 0x00020000 */
#define FDCAN_TXBCIE_CFIE17                       FDCAN_TXBCIE_CFIE17_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE18_Pos                   (18U)
#define FDCAN_TXBCIE_CFIE18_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE18_Pos)                  /*!< 0x00040000 */
#define FDCAN_TXBCIE_CFIE18                       FDCAN_TXBCIE_CFIE18_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE19_Pos                   (19U)
#define FDCAN_TXBCIE_CFIE19_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE19_Pos)                  /*!< 0x00080000 */
#define FDCAN_TXBCIE_CFIE19                       FDCAN_TXBCIE_CFIE19_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE20_Pos                   (20U)
#define FDCAN_TXBCIE_CFIE20_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE20_Pos)                  /*!< 0x00100000 */
#define FDCAN_TXBCIE_CFIE20                       FDCAN_TXBCIE_CFIE20_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE21_Pos                   (21U)
#define FDCAN_TXBCIE_CFIE21_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE21_Pos)                  /*!< 0x00200000 */
#define FDCAN_TXBCIE_CFIE21                       FDCAN_TXBCIE_CFIE21_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE22_Pos                   (22U)
#define FDCAN_TXBCIE_CFIE22_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE22_Pos)                  /*!< 0x00400000 */
#define FDCAN_TXBCIE_CFIE22                       FDCAN_TXBCIE_CFIE22_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE23_Pos                   (23U)
#define FDCAN_TXBCIE_CFIE23_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE23_Pos)                  /*!< 0x00800000 */
#define FDCAN_TXBCIE_CFIE23                       FDCAN_TXBCIE_CFIE23_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE24_Pos                   (24U)
#define FDCAN_TXBCIE_CFIE24_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE24_Pos)                  /*!< 0x01000000 */
#define FDCAN_TXBCIE_CFIE24                       FDCAN_TXBCIE_CFIE24_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE25_Pos                   (25U)
#define FDCAN_TXBCIE_CFIE25_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE25_Pos)                  /*!< 0x02000000 */
#define FDCAN_TXBCIE_CFIE25                       FDCAN_TXBCIE_CFIE25_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE26_Pos                   (26U)
#define FDCAN_TXBCIE_CFIE26_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE26_Pos)                  /*!< 0x04000000 */
#define FDCAN_TXBCIE_CFIE26                       FDCAN_TXBCIE_CFIE26_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE27_Pos                   (27U)
#define FDCAN_TXBCIE_CFIE27_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE27_Pos)                  /*!< 0x08000000 */
#define FDCAN_TXBCIE_CFIE27                       FDCAN_TXBCIE_CFIE27_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE28_Pos                   (28U)
#define FDCAN_TXBCIE_CFIE28_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE28_Pos)                  /*!< 0x10000000 */
#define FDCAN_TXBCIE_CFIE28                       FDCAN_TXBCIE_CFIE28_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE29_Pos                   (29U)
#define FDCAN_TXBCIE_CFIE29_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE29_Pos)                  /*!< 0x20000000 */
#define FDCAN_TXBCIE_CFIE29                       FDCAN_TXBCIE_CFIE29_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE30_Pos                   (30U)
#define FDCAN_TXBCIE_CFIE30_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE30_Pos)                  /*!< 0x40000000 */
#define FDCAN_TXBCIE_CFIE30                       FDCAN_TXBCIE_CFIE30_Msk                           /*!< Cancellation Finished Interrupt Enable */
#define FDCAN_TXBCIE_CFIE31_Pos                   (31U)
#define FDCAN_TXBCIE_CFIE31_Msk                   (0x1UL<<FDCAN_TXBCIE_CFIE31_Pos)                  /*!< 0x80000000 */
#define FDCAN_TXBCIE_CFIE31                       FDCAN_TXBCIE_CFIE31_Msk                           /*!< Cancellation Finished Interrupt Enable */

/********************************* Bit definition for FDCAN_TXEFC register ******************************************/
#define FDCAN_TXEFC_EFSA_Pos                      (2U)
#define FDCAN_TXEFC_EFSA_Msk                      (0x3FFFUL<<FDCAN_TXEFC_EFSA_Pos)                  /*!< 0x0000FFFC */
#define FDCAN_TXEFC_EFSA                          FDCAN_TXEFC_EFSA_Msk                              /*!< Event FIFO Start Address */
#define FDCAN_TXEFC_EFS_Pos                       (16U)
#define FDCAN_TXEFC_EFS_Msk                       (0x3FUL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x003F0000 */
#define FDCAN_TXEFC_EFS                           FDCAN_TXEFC_EFS_Msk                               /*!< Event FIFO Size */
#define FDCAN_TXEFC_EFS_0                         (0x1UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00010000 */
#define FDCAN_TXEFC_EFS_1                         (0x2UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00020000 */
#define FDCAN_TXEFC_EFS_2                         (0x4UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00040000 */
#define FDCAN_TXEFC_EFS_3                         (0x8UL<<FDCAN_TXEFC_EFS_Pos)                      /*!< 0x00080000 */
#define FDCAN_TXEFC_EFS_4                         (0x10UL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x00100000 */
#define FDCAN_TXEFC_EFS_5                         (0x20UL<<FDCAN_TXEFC_EFS_Pos)                     /*!< 0x00200000 */
#define FDCAN_TXEFC_EFWM_Pos                      (24U)
#define FDCAN_TXEFC_EFWM_Msk                      (0x3FUL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x3F000000 */
#define FDCAN_TXEFC_EFWM                          FDCAN_TXEFC_EFWM_Msk                              /*!< Event FIFO Watermark */
#define FDCAN_TXEFC_EFWM_0                        (0x1UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x01000000 */
#define FDCAN_TXEFC_EFWM_1                        (0x2UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXEFC_EFWM_2                        (0x4UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x04000000 */
#define FDCAN_TXEFC_EFWM_3                        (0x8UL<<FDCAN_TXEFC_EFWM_Pos)                     /*!< 0x08000000 */
#define FDCAN_TXEFC_EFWM_4                        (0x10UL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x10000000 */
#define FDCAN_TXEFC_EFWM_5                        (0x20UL<<FDCAN_TXEFC_EFWM_Pos)                    /*!< 0x20000000 */

/********************************* Bit definition for FDCAN_TXEFS register ******************************************/
#define FDCAN_TXEFS_EFFL_Pos                      (0U)
#define FDCAN_TXEFS_EFFL_Msk                      (0x3FUL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x0000003F */
#define FDCAN_TXEFS_EFFL                          FDCAN_TXEFS_EFFL_Msk                              /*!< Event FIFO Fill Level */
#define FDCAN_TXEFS_EFFL_0                        (0x1UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXEFS_EFFL_1                        (0x2UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXEFS_EFFL_2                        (0x4UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXEFS_EFFL_3                        (0x8UL<<FDCAN_TXEFS_EFFL_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXEFS_EFFL_4                        (0x10UL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x00000010 */
#define FDCAN_TXEFS_EFFL_5                        (0x20UL<<FDCAN_TXEFS_EFFL_Pos)                    /*!< 0x00000020 */
#define FDCAN_TXEFS_EFGI_Pos                      (8U)
#define FDCAN_TXEFS_EFGI_Msk                      (0x1FUL<<FDCAN_TXEFS_EFGI_Pos)                    /*!< 0x00001F00 */
#define FDCAN_TXEFS_EFGI                          FDCAN_TXEFS_EFGI_Msk                              /*!< Event FIFO Get Index */
#define FDCAN_TXEFS_EFGI_0                        (0x1UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000100 */
#define FDCAN_TXEFS_EFGI_1                        (0x2UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000200 */
#define FDCAN_TXEFS_EFGI_2                        (0x4UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000400 */
#define FDCAN_TXEFS_EFGI_3                        (0x8UL<<FDCAN_TXEFS_EFGI_Pos)                     /*!< 0x00000800 */
#define FDCAN_TXEFS_EFGI_4                        (0x10UL<<FDCAN_TXEFS_EFGI_Pos)                    /*!< 0x00001000 */
#define FDCAN_TXEFS_EFPI_Pos                      (16U)
#define FDCAN_TXEFS_EFPI_Msk                      (0x1FUL<<FDCAN_TXEFS_EFPI_Pos)                    /*!< 0x001F0000 */
#define FDCAN_TXEFS_EFPI                          FDCAN_TXEFS_EFPI_Msk                              /*!< Event FIFO Put Index */
#define FDCAN_TXEFS_EFPI_0                        (0x1UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00010000 */
#define FDCAN_TXEFS_EFPI_1                        (0x2UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00020000 */
#define FDCAN_TXEFS_EFPI_2                        (0x4UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00040000 */
#define FDCAN_TXEFS_EFPI_3                        (0x8UL<<FDCAN_TXEFS_EFPI_Pos)                     /*!< 0x00080000 */
#define FDCAN_TXEFS_EFPI_4                        (0x10UL<<FDCAN_TXEFS_EFPI_Pos)                    /*!< 0x00100000 */
#define FDCAN_TXEFS_EFF_Pos                       (24U)
#define FDCAN_TXEFS_EFF_Msk                       (0x1UL<<FDCAN_TXEFS_EFF_Pos)                      /*!< 0x01000000 */
#define FDCAN_TXEFS_EFF                           FDCAN_TXEFS_EFF_Msk                               /*!< Event FIFO Full */
#define FDCAN_TXEFS_TEFL_Pos                      (25U)
#define FDCAN_TXEFS_TEFL_Msk                      (0x1UL<<FDCAN_TXEFS_TEFL_Pos)                     /*!< 0x02000000 */
#define FDCAN_TXEFS_TEFL                          FDCAN_TXEFS_TEFL_Msk                              /*!< Tx Event FIFO Element Lost */

/********************************* Bit definition for FDCAN_TXEFA register ******************************************/
#define FDCAN_TXEFA_EFAI_Pos                      (0U)
#define FDCAN_TXEFA_EFAI_Msk                      (0x1FUL<<FDCAN_TXEFA_EFAI_Pos)                    /*!< 0x0000001F */
#define FDCAN_TXEFA_EFAI                          FDCAN_TXEFA_EFAI_Msk                              /*!< Event FIFO Acknowledge Index */
#define FDCAN_TXEFA_EFAI_0                        (0x1UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000001 */
#define FDCAN_TXEFA_EFAI_1                        (0x2UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000002 */
#define FDCAN_TXEFA_EFAI_2                        (0x4UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000004 */
#define FDCAN_TXEFA_EFAI_3                        (0x8UL<<FDCAN_TXEFA_EFAI_Pos)                     /*!< 0x00000008 */
#define FDCAN_TXEFA_EFAI_4                        (0x10UL<<FDCAN_TXEFA_EFAI_Pos)                    /*!< 0x00000010 */


/******************************************************************************/
/*                                                                            */
/*         FLASH and Option Bytes Registers                                   */
/*                                                                            */
/******************************************************************************/
#define GPIO_NRST_CONFIG_SUPPORT         /*!< GPIO feature available only on specific devices: Configure NRST pin */
#define FLASH_SECURABLE_MEMORY_SUPPORT   /*!< Flash feature available only on specific devices: allow to secure memory */
#define FLASH_PCROP_SUPPORT              /*!< Flash feature available only on specific devices: proprietary code read protection areas selected by option */

/******************  FLASH Keys  **********************************************/
#define FLASH_KEY1_Pos                            (0U)
#define FLASH_KEY1_Msk                            (0x45670123UL << FLASH_KEY1_Pos)                  /*!< 0x45670123 */
#define FLASH_KEY1                                FLASH_KEY1_Msk                                    /*!< Flash program erase key1 */
#define FLASH_KEY2_Pos                            (0U)
#define FLASH_KEY2_Msk                            (0xCDEF89ABUL << FLASH_KEY2_Pos)                  /*!< 0xCDEF89AB */
#define FLASH_KEY2                                FLASH_KEY2_Msk                                    /*!< Flash program erase key2: used with FLASH_PEKEY1
                                                                                                         to unlock the write access to the FPEC. */

#define FLASH_OPTKEY1_Pos                         (0U)
#define FLASH_OPTKEY1_Msk                         (0x08192A3BUL << FLASH_OPTKEY1_Pos)               /*!< 0x08192A3B */
#define FLASH_OPTKEY1                             FLASH_OPTKEY1_Msk                                 /*!< Flash option key1 */
#define FLASH_OPTKEY2_Pos                         (0U)
#define FLASH_OPTKEY2_Msk                         (0x4C5D6E7FUL << FLASH_OPTKEY2_Pos)               /*!< 0x4C5D6E7F */
#define FLASH_OPTKEY2                             FLASH_OPTKEY2_Msk                                 /*!< Flash option key2: used with FLASH_OPTKEY1 to
                                                                                                         unlock the write access to the option byte block */

/********************************* Bit definition for FLASH_ACR register ********************************************/
#define FLASH_ACR_LATENCY_Pos                     (0U)
#define FLASH_ACR_LATENCY_Msk                     (0x3UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000003 */
#define FLASH_ACR_LATENCY                         FLASH_ACR_LATENCY_Msk                             /*!< Latency */
#define FLASH_ACR_LATENCY_0                       (0x1UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000001 */
#define FLASH_ACR_LATENCY_1                       (0x2UL<<FLASH_ACR_LATENCY_Pos)                    /*!< 0x00000002 */
#define FLASH_ACR_PRFTEN_Pos                      (8U)
#define FLASH_ACR_PRFTEN_Msk                      (0x1UL<<FLASH_ACR_PRFTEN_Pos)                     /*!< 0x00000100 */
#define FLASH_ACR_PRFTEN                          FLASH_ACR_PRFTEN_Msk                              /*!< Prefect enable */
#define FLASH_ACR_ICEN_Pos                        (9U)
#define FLASH_ACR_ICEN_Msk                        (0x1UL<<FLASH_ACR_ICEN_Pos)                       /*!< 0x00000200 */
#define FLASH_ACR_ICEN                            FLASH_ACR_ICEN_Msk                                /*!< Instruction cache enable */
#define FLASH_ACR_DCEN_Pos                        (10U)
#define FLASH_ACR_DCEN_Msk                        (0x1UL<<FLASH_ACR_DCEN_Pos)                       /*!< 0x00000400 */
#define FLASH_ACR_DCEN                            FLASH_ACR_DCEN_Msk                                /*!< Data cache enable */

/********************************* Bit definition for FLASH_KEYR register *******************************************/
#define FLASH_KEYR_KEY_Pos                        (0U)
#define FLASH_KEYR_KEY_Msk                        (0xFFFFFFFFUL<<FLASH_KEYR_KEY_Pos)                /*!< 0xFFFFFFFF */
#define FLASH_KEYR_KEY                            FLASH_KEYR_KEY_Msk                                /*!< Flash CR key */

/********************************* Bit definition for FLASH_OPTKEYR register ****************************************/
#define FLASH_OPTKEYR_OPTKEY_Pos                  (0U)
#define FLASH_OPTKEYR_OPTKEY_Msk                  (0xFFFFFFFFUL<<FLASH_OPTKEYR_OPTKEY_Pos)          /*!< 0xFFFFFFFF */
#define FLASH_OPTKEYR_OPTKEY                      FLASH_OPTKEYR_OPTKEY_Msk                          /*!< Flash Option key */

/********************************* Bit definition for FLASH_SR register *********************************************/
#define FLASH_SR_EOP_Pos                          (0U)
#define FLASH_SR_EOP_Msk                          (0x1UL<<FLASH_SR_EOP_Pos)                         /*!< 0x00000001 */
#define FLASH_SR_EOP                              FLASH_SR_EOP_Msk                                  /*!< End of operation */
#define FLASH_SR_WRPERR_Pos                       (4U)
#define FLASH_SR_WRPERR_Msk                       (0x1UL<<FLASH_SR_WRPERR_Pos)                      /*!< 0x00000010 */
#define FLASH_SR_WRPERR                           FLASH_SR_WRPERR_Msk                               /*!< Write protection error */
#define FLASH_SR_RDERR0_Pos                       (11U)
#define FLASH_SR_RDERR0_Msk                       (0x1UL<<FLASH_SR_RDERR0_Pos)                      /*!< 0x00000800 */
#define FLASH_SR_RDERR0                           FLASH_SR_RDERR0_Msk                               /*!< Bank0 pcrop read error */
#define FLASH_SR_RDERR1_Pos                       (12U)
#define FLASH_SR_RDERR1_Msk                       (0x1UL<<FLASH_SR_RDERR1_Pos)                      /*!< 0x00001000 */
#define FLASH_SR_RDERR1                           FLASH_SR_RDERR1_Msk                               /*!< Bank1 pcrop read error */
#define FLASH_SR_USRLOCK_Pos                      (13U)
#define FLASH_SR_USRLOCK_Msk                      (0x1UL<<FLASH_SR_USRLOCK_Pos)                     /*!< 0x00002000 */
#define FLASH_SR_USRLOCK                          FLASH_SR_USRLOCK_Msk                              /*!< Userdata Lock  */
#define FLASH_SR_OPTVERR_Pos                      (15U)
#define FLASH_SR_OPTVERR_Msk                      (0x1UL<<FLASH_SR_OPTVERR_Pos)                     /*!< 0x00008000 */
#define FLASH_SR_OPTVERR                          FLASH_SR_OPTVERR_Msk                              /*!< Option and trimming bits loading validity error */
#define FLASH_SR_BSY0_Pos                         (16U)
#define FLASH_SR_BSY0_Msk                         (0x1UL<<FLASH_SR_BSY0_Pos)                        /*!< 0x00010000 */
#define FLASH_SR_BSY0                             FLASH_SR_BSY0_Msk                                 /*!< Bank0 Busy */
#define FLASH_SR_BSY1_Pos                         (17U)
#define FLASH_SR_BSY1_Msk                         (0x1UL<<FLASH_SR_BSY1_Pos)                        /*!< 0x00020000 */
#define FLASH_SR_BSY1                             FLASH_SR_BSY1_Msk                                 /*!< Bank1 Busy */

/********************************* Bit definition for FLASH_CR register *********************************************/
#define FLASH_CR_PG_Pos                           (0U)
#define FLASH_CR_PG_Msk                           (0x1UL<<FLASH_CR_PG_Pos)                          /*!< 0x00000001 */
#define FLASH_CR_PG                               FLASH_CR_PG_Msk                                   /*!< Page Program */
#define FLASH_CR_PER_Pos                          (1U)
#define FLASH_CR_PER_Msk                          (0x1UL<<FLASH_CR_PER_Pos)                         /*!< 0x00000002 */
#define FLASH_CR_PER                              FLASH_CR_PER_Msk                                  /*!< Page Erase */
#define FLASH_CR_MER0_Pos                         (2U)
#define FLASH_CR_MER0_Msk                         (0x1UL<<FLASH_CR_MER0_Pos)                        /*!< 0x00000004 */
#define FLASH_CR_MER0                             FLASH_CR_MER0_Msk                                 /*!< Bank 0 Mass Erase */
#define FLASH_CR_MER1_Pos                         (3U)
#define FLASH_CR_MER1_Msk                         (0x1UL<<FLASH_CR_MER1_Pos)                        /*!< 0x00000008 */
#define FLASH_CR_MER1                             FLASH_CR_MER1_Msk                                 /*!< Bank 1 Mass Erase */
#define FLASH_CR_UPG_Pos                          (4U)
#define FLASH_CR_UPG_Msk                          (0x1UL<<FLASH_CR_UPG_Pos)                         /*!< 0x00000010 */
#define FLASH_CR_UPG                              FLASH_CR_UPG_Msk                                  /*!< Userdata Program */
#define FLASH_CR_UPER_Pos                         (5U)
#define FLASH_CR_UPER_Msk                         (0x1UL<<FLASH_CR_UPER_Pos)                        /*!< 0x00000020 */
#define FLASH_CR_UPER                             FLASH_CR_UPER_Msk                                 /*!< Userdata Page Erase */
#define FLASH_CR_SER_Pos                          (11U)
#define FLASH_CR_SER_Msk                          (0x1UL<<FLASH_CR_SER_Pos)                         /*!< 0x00000800 */
#define FLASH_CR_SER                              FLASH_CR_SER_Msk                                  /*!< Sector Erase */
#define FLASH_CR_OPTSTRT_Pos                      (17U)
#define FLASH_CR_OPTSTRT_Msk                      (0x1UL<<FLASH_CR_OPTSTRT_Pos)                     /*!< 0x00020000 */
#define FLASH_CR_OPTSTRT                          FLASH_CR_OPTSTRT_Msk                              /*!< Option bytes Programming Start */
#define FLASH_CR_UPGSTRT_Pos                      (18U)
#define FLASH_CR_UPGSTRT_Msk                      (0x1UL<<FLASH_CR_UPGSTRT_Pos)                     /*!< 0x00040000 */
#define FLASH_CR_UPGSTRT                          FLASH_CR_UPGSTRT_Msk                              /*!< Userdata Programming Start */
#define FLASH_CR_PGSTRT_Pos                       (19U)
#define FLASH_CR_PGSTRT_Msk                       (0x1UL<<FLASH_CR_PGSTRT_Pos)                      /*!< 0x00080000 */
#define FLASH_CR_PGSTRT                           FLASH_CR_PGSTRT_Msk                               /*!< Programming Start */
#define FLASH_CR_RDERR0IE_Pos                     (22U)
#define FLASH_CR_RDERR0IE_Msk                     (0x1UL<<FLASH_CR_RDERR0IE_Pos)                    /*!< 0x00400000 */
#define FLASH_CR_RDERR0IE                         FLASH_CR_RDERR0IE_Msk                             /*!< Bank0 pcrop read error interrupt enable */
#define FLASH_CR_RDERR1IE_Pos                     (23U)
#define FLASH_CR_RDERR1IE_Msk                     (0x1UL<<FLASH_CR_RDERR1IE_Pos)                    /*!< 0x00800000 */
#define FLASH_CR_RDERR1IE                         FLASH_CR_RDERR1IE_Msk                             /*!< Bank1 pcrop read error interrupt enable */
#define FLASH_CR_EOPIE_Pos                        (24U)
#define FLASH_CR_EOPIE_Msk                        (0x1UL<<FLASH_CR_EOPIE_Pos)                       /*!< 0x01000000 */
#define FLASH_CR_EOPIE                            FLASH_CR_EOPIE_Msk                                /*!< End of operation interrupt enable */
#define FLASH_CR_ERRIE_Pos                        (25U)
#define FLASH_CR_ERRIE_Msk                        (0x1UL<<FLASH_CR_ERRIE_Pos)                       /*!< 0x02000000 */
#define FLASH_CR_ERRIE                            FLASH_CR_ERRIE_Msk                                /*!< Error interrupt enable */
#define FLASH_CR_OBL_LAUNCH_Pos                   (27U)
#define FLASH_CR_OBL_LAUNCH_Msk                   (0x1UL<<FLASH_CR_OBL_LAUNCH_Pos)                  /*!< 0x08000000 */
#define FLASH_CR_OBL_LAUNCH                       FLASH_CR_OBL_LAUNCH_Msk                           /*!< Force the option bytes loading */
#define FLASH_CR_OPTLOCK_Pos                      (30U)
#define FLASH_CR_OPTLOCK_Msk                      (0x1UL<<FLASH_CR_OPTLOCK_Pos)                     /*!< 0x40000000 */
#define FLASH_CR_OPTLOCK                          FLASH_CR_OPTLOCK_Msk                              /*!< Option Lock */
#define FLASH_CR_LOCK_Pos                         (31U)
#define FLASH_CR_LOCK_Msk                         (0x1UL<<FLASH_CR_LOCK_Pos)                        /*!< 0x80000000 */
#define FLASH_CR_LOCK                             FLASH_CR_LOCK_Msk                                 /*!< Lock */

/********************************* Bit definition for FLASH_ECCR register *******************************************/
#define FLASH_ECCR_ADDR_ECC_Pos                   (0U)
#define FLASH_ECCR_ADDR_ECC_Msk                   (0x1FFFFUL<<FLASH_ECCR_ADDR_ECC_Pos)              /*!< 0x0001FFFF */
#define FLASH_ECCR_ADDR_ECC                       FLASH_ECCR_ADDR_ECC_Msk                           /*!< ECC fail address */
#define FLASH_ECCR_BK_ECC_Pos                     (21U)
#define FLASH_ECCR_BK_ECC_Msk                     (0x1UL<<FLASH_ECCR_BK_ECC_Pos)                    /*!< 0x00200000 */
#define FLASH_ECCR_BK_ECC                         FLASH_ECCR_BK_ECC_Msk                             /*!< ECC fail bank */
#define FLASH_ECCR_SYSF_ECC_Pos                   (22U)
#define FLASH_ECCR_SYSF_ECC_Msk                   (0x1UL<<FLASH_ECCR_SYSF_ECC_Pos)                  /*!< 0x00400000 */
#define FLASH_ECCR_SYSF_ECC                       FLASH_ECCR_SYSF_ECC_Msk                           /*!< System Flash ECC fail */
#define FLASH_ECCR_ECCCIE_Pos                     (24U)
#define FLASH_ECCR_ECCCIE_Msk                     (0x1UL<<FLASH_ECCR_ECCCIE_Pos)                    /*!< 0x01000000 */
#define FLASH_ECCR_ECCCIE                         FLASH_ECCR_ECCCIE_Msk                             /*!< ECC correction interrupt enable */
#define FLASH_ECCR_ECCC_Pos                       (30U)
#define FLASH_ECCR_ECCC_Msk                       (0x1UL<<FLASH_ECCR_ECCC_Pos)                      /*!< 0x40000000 */
#define FLASH_ECCR_ECCC                           FLASH_ECCR_ECCC_Msk                               /*!< ECC correction */
#define FLASH_ECCR_ECCD_Pos                       (31U)
#define FLASH_ECCR_ECCD_Msk                       (0x1UL<<FLASH_ECCR_ECCD_Pos)                      /*!< 0x80000000 */
#define FLASH_ECCR_ECCD                           FLASH_ECCR_ECCD_Msk                               /*!< ECC detection */

/********************************* Bit definition for FLASH_OPTR register *******************************************/
#define FLASH_OPTR_RDP_Pos                        (0U)
#define FLASH_OPTR_RDP_Msk                        (0xFFUL<<FLASH_OPTR_RDP_Pos)                      /*!< 0x000000FF */
#define FLASH_OPTR_RDP                            FLASH_OPTR_RDP_Msk                                /*!< Read Protection */
#define FLASH_OPTR_NRST_STANDBY_Pos               (8U)
#define FLASH_OPTR_NRST_STANDBY_Msk               (0x1UL<<FLASH_OPTR_NRST_STANDBY_Pos)              /*!< 0x00000100 */
#define FLASH_OPTR_NRST_STANDBY                   FLASH_OPTR_NRST_STANDBY_Msk                       /*!< nrst standby */
#define FLASH_OPTR_NRST_STOP_Pos                  (9U)
#define FLASH_OPTR_NRST_STOP_Msk                  (0x1UL<<FLASH_OPTR_NRST_STOP_Pos)                 /*!< 0x00000200 */
#define FLASH_OPTR_NRST_STOP                      FLASH_OPTR_NRST_STOP_Msk                          /*!< nrst stop */
#define FLASH_OPTR_IWDG_STDBY_Pos                 (10U)
#define FLASH_OPTR_IWDG_STDBY_Msk                 (0x1UL<<FLASH_OPTR_IWDG_STDBY_Pos)                /*!< 0x00000400 */
#define FLASH_OPTR_IWDG_STDBY                     FLASH_OPTR_IWDG_STDBY_Msk                         /*!< IWDG Standy Count Control */
#define FLASH_OPTR_IWDG_STOP_Pos                  (11U)
#define FLASH_OPTR_IWDG_STOP_Msk                  (0x1UL<<FLASH_OPTR_IWDG_STOP_Pos)                 /*!< 0x00000800 */
#define FLASH_OPTR_IWDG_STOP                      FLASH_OPTR_IWDG_STOP_Msk                          /*!< IWDG Stop Count Control */
#define FLASH_OPTR_IWDG_SW_Pos                    (12U)
#define FLASH_OPTR_IWDG_SW_Msk                    (0x1UL<<FLASH_OPTR_IWDG_SW_Pos)                   /*!< 0x00001000 */
#define FLASH_OPTR_IWDG_SW                        FLASH_OPTR_IWDG_SW_Msk                            /*!< IWDG Software Enable */
#define FLASH_OPTR_WWDG_SW_Pos                    (13U)
#define FLASH_OPTR_WWDG_SW_Msk                    (0x1UL<<FLASH_OPTR_WWDG_SW_Pos)                   /*!< 0x00002000 */
#define FLASH_OPTR_WWDG_SW                        FLASH_OPTR_WWDG_SW_Msk                            /*!< WWDG Software Enable */
#define FLASH_OPTR_NRST_MODE_Pos                  (14U)
#define FLASH_OPTR_NRST_MODE_Msk                  (0x1UL<<FLASH_OPTR_NRST_MODE_Pos)                 /*!< 0x00004000 */
#define FLASH_OPTR_NRST_MODE                      FLASH_OPTR_NRST_MODE_Msk                          /*!< nrst mode */
#define FLASH_OPTR_NBOOT1_Pos                     (15U)
#define FLASH_OPTR_NBOOT1_Msk                     (0x1UL<<FLASH_OPTR_NBOOT1_Pos)                    /*!< 0x00008000 */
#define FLASH_OPTR_NBOOT1                         FLASH_OPTR_NBOOT1_Msk                             /*!< nboot1 */
#define FLASH_OPTR_BOR_EN_Pos                     (16U)
#define FLASH_OPTR_BOR_EN_Msk                     (0x1UL<<FLASH_OPTR_BOR_EN_Pos)                    /*!< 0x00010000 */
#define FLASH_OPTR_BOR_EN                         FLASH_OPTR_BOR_EN_Msk                             /*!< BOR enable */
#define FLASH_OPTR_BOR_LEV_Pos                    (17U)
#define FLASH_OPTR_BOR_LEV_Msk                    (0x7UL<<FLASH_OPTR_BOR_LEV_Pos)                   /*!< 0x000E0000 */
#define FLASH_OPTR_BOR_LEV                        FLASH_OPTR_BOR_LEV_Msk                            /*!< BOR Level */
#define FLASH_OPTR_BOR_LEV_0                      (0x1UL<<FLASH_OPTR_BOR_LEV_Pos)                   /*!< 0x00020000 */
#define FLASH_OPTR_BOR_LEV_1                      (0x2UL<<FLASH_OPTR_BOR_LEV_Pos)                   /*!< 0x00040000 */
#define FLASH_OPTR_BOR_LEV_2                      (0x4UL<<FLASH_OPTR_BOR_LEV_Pos)                   /*!< 0x00080000 */
#define FLASH_OPTR_SRAM_PE_Pos                    (20U)
#define FLASH_OPTR_SRAM_PE_Msk                    (0x1UL<<FLASH_OPTR_SRAM_PE_Pos)                   /*!< 0x00100000 */
#define FLASH_OPTR_SRAM_PE                        FLASH_OPTR_SRAM_PE_Msk                            /*!< sram parity enable */
#define FLASH_OPTR_SWD_MODE_Pos                   (21U)
#define FLASH_OPTR_SWD_MODE_Msk                   (0x3UL<<FLASH_OPTR_SWD_MODE_Pos)                  /*!< 0x00600000 */
#define FLASH_OPTR_SWD_MODE                       FLASH_OPTR_SWD_MODE_Msk                           /*!< SWD mode */
#define FLASH_OPTR_SWD_MODE_0                     (0x1UL<<FLASH_OPTR_SWD_MODE_Pos)                  /*!< 0x00200000 */
#define FLASH_OPTR_SWD_MODE_1                     (0x2UL<<FLASH_OPTR_SWD_MODE_Pos)                  /*!< 0x00400000 */
#define FLASH_OPTR_BFB_Pos                        (23U)
#define FLASH_OPTR_BFB_Msk                        (0x1UL<<FLASH_OPTR_BFB_Pos)                       /*!< 0x00800000 */
#define FLASH_OPTR_BFB                            FLASH_OPTR_BFB_Msk                                /*!< Boot Flash Bank */
#define FLASH_OPTR_BOOT_LOCK_Pos                  (24U)
#define FLASH_OPTR_BOOT_LOCK_Msk                  (0x1UL<<FLASH_OPTR_BOOT_LOCK_Pos)                 /*!< 0x01000000 */
#define FLASH_OPTR_BOOT_LOCK                      FLASH_OPTR_BOOT_LOCK_Msk                          /*!< Boot Lock */
#define FLASH_OPTR_ECC_EN_Pos                     (25U)
#define FLASH_OPTR_ECC_EN_Msk                     (0x1UL<<FLASH_OPTR_ECC_EN_Pos)                    /*!< 0x02000000 */
#define FLASH_OPTR_ECC_EN                         FLASH_OPTR_ECC_EN_Msk                             /*!< ECC enable */

/********************************* Bit definition for FLASH_WRPR register *******************************************/
#define FLASH_WRPR_BK0_WRPX_Pos                   (0U)
#define FLASH_WRPR_BK0_WRPX_Msk                   (0xFFFFUL<<FLASH_WRPR_BK0_WRPX_Pos)               /*!< 0x0000FFFF */
#define FLASH_WRPR_BK0_WRPX                       FLASH_WRPR_BK0_WRPX_Msk                           /*!< Bank0 Sector Write Protection */
#define FLASH_WRPR_BK1_WRPX_Pos                   (16U)
#define FLASH_WRPR_BK1_WRPX_Msk                   (0xFFFFUL<<FLASH_WRPR_BK1_WRPX_Pos)               /*!< 0xFFFF0000 */
#define FLASH_WRPR_BK1_WRPX                       FLASH_WRPR_BK1_WRPX_Msk                           /*!< Bank1 Sector Write Protection */

/********************************* Bit definition for FLASH_PCROPR0 register ****************************************/
#define FLASH_PCROPR0_PCROP0SR_Pos                (0U)
#define FLASH_PCROPR0_PCROP0SR_Msk                (0x1FFUL<<FLASH_PCROPR0_PCROP0SR_Pos)             /*!< 0x000001FF */
#define FLASH_PCROPR0_PCROP0SR                    FLASH_PCROPR0_PCROP0SR_Msk                        /*!< Bank0 pcrop start address */
#define FLASH_PCROPR0_PCROP0ER_Pos                (16U)
#define FLASH_PCROPR0_PCROP0ER_Msk                (0x1FFUL<<FLASH_PCROPR0_PCROP0ER_Pos)             /*!< 0x01FF0000 */
#define FLASH_PCROPR0_PCROP0ER                    FLASH_PCROPR0_PCROP0ER_Msk                        /*!< Bank0 pcrop end address */

/********************************* Bit definition for FLASH_PCROPR1 register ****************************************/
#define FLASH_PCROPR1_PCROP1SR_Pos                (0U)
#define FLASH_PCROPR1_PCROP1SR_Msk                (0x1FFUL<<FLASH_PCROPR1_PCROP1SR_Pos)             /*!< 0x000001FF */
#define FLASH_PCROPR1_PCROP1SR                    FLASH_PCROPR1_PCROP1SR_Msk                        /*!< Bank1 pcrop start address */
#define FLASH_PCROPR1_PCROP1ER_Pos                (16U)
#define FLASH_PCROPR1_PCROP1ER_Msk                (0x1FFUL<<FLASH_PCROPR1_PCROP1ER_Pos)             /*!< 0x01FF0000 */
#define FLASH_PCROPR1_PCROP1ER                    FLASH_PCROPR1_PCROP1ER_Msk                        /*!< Bank1 pcrop end address */

/********************************* Bit definition for FLASH_LPCR register *******************************************/
#define FLASH_LPCR_SLEEP_EN_Pos                   (0U)
#define FLASH_LPCR_SLEEP_EN_Msk                   (0x1UL<<FLASH_LPCR_SLEEP_EN_Pos)                  /*!< 0x00000001 */
#define FLASH_LPCR_SLEEP_EN                       FLASH_LPCR_SLEEP_EN_Msk                           /*!< Sleep enable */
#define FLASH_LPCR_SLEEP_TIME_Pos                 (8U)
#define FLASH_LPCR_SLEEP_TIME_Msk                 (0xFFUL<<FLASH_LPCR_SLEEP_TIME_Pos)               /*!< 0x0000FF00 */
#define FLASH_LPCR_SLEEP_TIME                     FLASH_LPCR_SLEEP_TIME_Msk                         /*!< Sleep time */
#define FLASH_LPCR_LPRUN_ENTERT_Pos               (16U)
#define FLASH_LPCR_LPRUN_ENTERT_Msk               (0x7UL<<FLASH_LPCR_LPRUN_ENTERT_Pos)              /*!< 0x00070000 */
#define FLASH_LPCR_LPRUN_ENTERT                   FLASH_LPCR_LPRUN_ENTERT_Msk                       /*!< Low power run mode enter time */
#define FLASH_LPCR_LPRUN_ENTERT_0                 (0x1UL<<FLASH_LPCR_LPRUN_ENTERT_Pos)              /*!< 0x00010000 */
#define FLASH_LPCR_LPRUN_ENTERT_1                 (0x2UL<<FLASH_LPCR_LPRUN_ENTERT_Pos)              /*!< 0x00020000 */
#define FLASH_LPCR_LPRUN_ENTERT_2                 (0x4UL<<FLASH_LPCR_LPRUN_ENTERT_Pos)              /*!< 0x00040000 */
#define FLASH_LPCR_LPRUN_EXITT_Pos                (20U)
#define FLASH_LPCR_LPRUN_EXITT_Msk                (0x7UL<<FLASH_LPCR_LPRUN_EXITT_Pos)               /*!< 0x00700000 */
#define FLASH_LPCR_LPRUN_EXITT                    FLASH_LPCR_LPRUN_EXITT_Msk                        /*!< Low power run mode exit time */
#define FLASH_LPCR_LPRUN_EXITT_0                  (0x1UL<<FLASH_LPCR_LPRUN_EXITT_Pos)               /*!< 0x00100000 */
#define FLASH_LPCR_LPRUN_EXITT_1                  (0x2UL<<FLASH_LPCR_LPRUN_EXITT_Pos)               /*!< 0x00200000 */
#define FLASH_LPCR_LPRUN_EXITT_2                  (0x4UL<<FLASH_LPCR_LPRUN_EXITT_Pos)               /*!< 0x00400000 */

/********************************* Bit definition for FLASH_TS0 register ********************************************/
#define FLASH_TS0_TS0_Pos                         (0U)
#define FLASH_TS0_TS0_Msk                         (0x1FFUL<<FLASH_TS0_TS0_Pos)                      /*!< 0x000001FF */
#define FLASH_TS0_TS0                             FLASH_TS0_TS0_Msk                                 /*!< TS0 */

/********************************* Bit definition for FLASH_TS1 register ********************************************/
#define FLASH_TS1_TS1_Pos                         (0U)
#define FLASH_TS1_TS1_Msk                         (0x7FFUL<<FLASH_TS1_TS1_Pos)                      /*!< 0x000007FF */
#define FLASH_TS1_TS1                             FLASH_TS1_TS1_Msk                                 /*!< TS1 */

/********************************* Bit definition for FLASH_TS2P register *******************************************/
#define FLASH_TS2P_TS2P_Pos                       (0U)
#define FLASH_TS2P_TS2P_Msk                       (0x1FFUL<<FLASH_TS2P_TS2P_Pos)                    /*!< 0x000001FF */
#define FLASH_TS2P_TS2P                           FLASH_TS2P_TS2P_Msk                               /*!< TS2P */

/********************************* Bit definition for FLASH_TPS3 register *******************************************/
#define FLASH_TPS3_TPS3_Pos                       (0U)
#define FLASH_TPS3_TPS3_Msk                       (0x1FFFUL<<FLASH_TPS3_TPS3_Pos)                   /*!< 0x00001FFF */
#define FLASH_TPS3_TPS3                           FLASH_TPS3_TPS3_Msk                               /*!< TPS3 */

/********************************* Bit definition for FLASH_TS3 register ********************************************/
#define FLASH_TS3_TS3_Pos                         (0U)
#define FLASH_TS3_TS3_Msk                         (0x1FFUL<<FLASH_TS3_TS3_Pos)                      /*!< 0x000001FF */
#define FLASH_TS3_TS3                             FLASH_TS3_TS3_Msk                                 /*!< TS3 */

/********************************* Bit definition for FLASH_PERTPE register *****************************************/
#define FLASH_PERTPE_PERTPE_Pos                   (0U)
#define FLASH_PERTPE_PERTPE_Msk                   (0x3FFFFUL<<FLASH_PERTPE_PERTPE_Pos)              /*!< 0x0003FFFF */
#define FLASH_PERTPE_PERTPE                       FLASH_PERTPE_PERTPE_Msk                           /*!< PERTPE */

/********************************* Bit definition for FLASH_SMERTPE register ****************************************/
#define FLASH_SMERTPE_SMERTPE_Pos                 (0U)
#define FLASH_SMERTPE_SMERTPE_Msk                 (0x3FFFFUL<<FLASH_SMERTPE_SMERTPE_Pos)            /*!< 0x0003FFFF */
#define FLASH_SMERTPE_SMERTPE                     FLASH_SMERTPE_SMERTPE_Msk                         /*!< SMERTPE */

/********************************* Bit definition for FLASH_PRGTPE register *****************************************/
#define FLASH_PRGTPE_PRGTPE_Pos                   (0U)
#define FLASH_PRGTPE_PRGTPE_Msk                   (0xFFFFUL<<FLASH_PRGTPE_PRGTPE_Pos)               /*!< 0x0000FFFF */
#define FLASH_PRGTPE_PRGTPE                       FLASH_PRGTPE_PRGTPE_Msk                           /*!< PRGTPE */

/********************************* Bit definition for FLASH_PRETPE register *****************************************/
#define FLASH_PRETPE_PRETPE_Pos                   (0U)
#define FLASH_PRETPE_PRETPE_Msk                   (0x3FFFUL<<FLASH_PRETPE_PRETPE_Pos)               /*!< 0x00003FFF */
#define FLASH_PRETPE_PRETPE                       FLASH_PRETPE_PRETPE_Msk                           /*!< PRETPE */

/********************************* Bit definition for FLASH_TRD2WPL register ****************************************/
#define FLASH_TRD2WPL_PRETPE_Pos                  (0U)
#define FLASH_TRD2WPL_PRETPE_Msk                  (0xFFUL<<FLASH_TRD2WPL_PRETPE_Pos)                /*!< 0x000000FF */
#define FLASH_TRD2WPL_PRETPE                      FLASH_TRD2WPL_PRETPE_Msk                          /*!< TRD2WPL */

/******************************************************************************/
/*                                                                            */
/*                            General Purpose I/O (GPIO)                      */
/*                                                                            */
/******************************************************************************/
/******************  Bits definition for GPIO_MODER register  *****************/
#define GPIO_MODER_MODE0_Pos           (0U)
#define GPIO_MODER_MODE0_Msk           (0x3UL << GPIO_MODER_MODE0_Pos)          /*!< 0x00000003 */
#define GPIO_MODER_MODE0               GPIO_MODER_MODE0_Msk
#define GPIO_MODER_MODE0_0             (0x1UL << GPIO_MODER_MODE0_Pos)          /*!< 0x00000001 */
#define GPIO_MODER_MODE0_1             (0x2UL << GPIO_MODER_MODE0_Pos)          /*!< 0x00000002 */
#define GPIO_MODER_MODE1_Pos           (2U)
#define GPIO_MODER_MODE1_Msk           (0x3UL << GPIO_MODER_MODE1_Pos)          /*!< 0x0000000C */
#define GPIO_MODER_MODE1               GPIO_MODER_MODE1_Msk
#define GPIO_MODER_MODE1_0             (0x1UL << GPIO_MODER_MODE1_Pos)          /*!< 0x00000004 */
#define GPIO_MODER_MODE1_1             (0x2UL << GPIO_MODER_MODE1_Pos)          /*!< 0x00000008 */
#define GPIO_MODER_MODE2_Pos           (4U)
#define GPIO_MODER_MODE2_Msk           (0x3UL << GPIO_MODER_MODE2_Pos)          /*!< 0x00000030 */
#define GPIO_MODER_MODE2               GPIO_MODER_MODE2_Msk
#define GPIO_MODER_MODE2_0             (0x1UL << GPIO_MODER_MODE2_Pos)          /*!< 0x00000010 */
#define GPIO_MODER_MODE2_1             (0x2UL << GPIO_MODER_MODE2_Pos)          /*!< 0x00000020 */
#define GPIO_MODER_MODE3_Pos           (6U)
#define GPIO_MODER_MODE3_Msk           (0x3UL << GPIO_MODER_MODE3_Pos)          /*!< 0x000000C0 */
#define GPIO_MODER_MODE3               GPIO_MODER_MODE3_Msk
#define GPIO_MODER_MODE3_0             (0x1UL << GPIO_MODER_MODE3_Pos)          /*!< 0x00000040 */
#define GPIO_MODER_MODE3_1             (0x2UL << GPIO_MODER_MODE3_Pos)          /*!< 0x00000080 */
#define GPIO_MODER_MODE4_Pos           (8U)
#define GPIO_MODER_MODE4_Msk           (0x3UL << GPIO_MODER_MODE4_Pos)          /*!< 0x00000300 */
#define GPIO_MODER_MODE4               GPIO_MODER_MODE4_Msk
#define GPIO_MODER_MODE4_0             (0x1UL << GPIO_MODER_MODE4_Pos)          /*!< 0x00000100 */
#define GPIO_MODER_MODE4_1             (0x2UL << GPIO_MODER_MODE4_Pos)          /*!< 0x00000200 */
#define GPIO_MODER_MODE5_Pos           (10U)
#define GPIO_MODER_MODE5_Msk           (0x3UL << GPIO_MODER_MODE5_Pos)          /*!< 0x00000C00 */
#define GPIO_MODER_MODE5               GPIO_MODER_MODE5_Msk
#define GPIO_MODER_MODE5_0             (0x1UL << GPIO_MODER_MODE5_Pos)          /*!< 0x00000400 */
#define GPIO_MODER_MODE5_1             (0x2UL << GPIO_MODER_MODE5_Pos)          /*!< 0x00000800 */
#define GPIO_MODER_MODE6_Pos           (12U)
#define GPIO_MODER_MODE6_Msk           (0x3UL << GPIO_MODER_MODE6_Pos)          /*!< 0x00003000 */
#define GPIO_MODER_MODE6               GPIO_MODER_MODE6_Msk
#define GPIO_MODER_MODE6_0             (0x1UL << GPIO_MODER_MODE6_Pos)          /*!< 0x00001000 */
#define GPIO_MODER_MODE6_1             (0x2UL << GPIO_MODER_MODE6_Pos)          /*!< 0x00002000 */
#define GPIO_MODER_MODE7_Pos           (14U)
#define GPIO_MODER_MODE7_Msk           (0x3UL << GPIO_MODER_MODE7_Pos)          /*!< 0x0000C000 */
#define GPIO_MODER_MODE7               GPIO_MODER_MODE7_Msk
#define GPIO_MODER_MODE7_0             (0x1UL << GPIO_MODER_MODE7_Pos)          /*!< 0x00004000 */
#define GPIO_MODER_MODE7_1             (0x2UL << GPIO_MODER_MODE7_Pos)          /*!< 0x00008000 */
#define GPIO_MODER_MODE8_Pos           (16U)
#define GPIO_MODER_MODE8_Msk           (0x3UL << GPIO_MODER_MODE8_Pos)          /*!< 0x00030000 */
#define GPIO_MODER_MODE8               GPIO_MODER_MODE8_Msk
#define GPIO_MODER_MODE8_0             (0x1UL << GPIO_MODER_MODE8_Pos)          /*!< 0x00010000 */
#define GPIO_MODER_MODE8_1             (0x2UL << GPIO_MODER_MODE8_Pos)          /*!< 0x00020000 */
#define GPIO_MODER_MODE9_Pos           (18U)
#define GPIO_MODER_MODE9_Msk           (0x3UL << GPIO_MODER_MODE9_Pos)          /*!< 0x000C0000 */
#define GPIO_MODER_MODE9               GPIO_MODER_MODE9_Msk
#define GPIO_MODER_MODE9_0             (0x1UL << GPIO_MODER_MODE9_Pos)          /*!< 0x00040000 */
#define GPIO_MODER_MODE9_1             (0x2UL << GPIO_MODER_MODE9_Pos)          /*!< 0x00080000 */
#define GPIO_MODER_MODE10_Pos          (20U)
#define GPIO_MODER_MODE10_Msk          (0x3UL << GPIO_MODER_MODE10_Pos)         /*!< 0x00300000 */
#define GPIO_MODER_MODE10              GPIO_MODER_MODE10_Msk
#define GPIO_MODER_MODE10_0            (0x1UL << GPIO_MODER_MODE10_Pos)         /*!< 0x00100000 */
#define GPIO_MODER_MODE10_1            (0x2UL << GPIO_MODER_MODE10_Pos)         /*!< 0x00200000 */
#define GPIO_MODER_MODE11_Pos          (22U)
#define GPIO_MODER_MODE11_Msk          (0x3UL << GPIO_MODER_MODE11_Pos)         /*!< 0x00C00000 */
#define GPIO_MODER_MODE11              GPIO_MODER_MODE11_Msk
#define GPIO_MODER_MODE11_0            (0x1UL << GPIO_MODER_MODE11_Pos)         /*!< 0x00400000 */
#define GPIO_MODER_MODE11_1            (0x2UL << GPIO_MODER_MODE11_Pos)         /*!< 0x00800000 */
#define GPIO_MODER_MODE12_Pos          (24U)
#define GPIO_MODER_MODE12_Msk          (0x3UL << GPIO_MODER_MODE12_Pos)         /*!< 0x03000000 */
#define GPIO_MODER_MODE12              GPIO_MODER_MODE12_Msk
#define GPIO_MODER_MODE12_0            (0x1UL << GPIO_MODER_MODE12_Pos)         /*!< 0x01000000 */
#define GPIO_MODER_MODE12_1            (0x2UL << GPIO_MODER_MODE12_Pos)         /*!< 0x02000000 */
#define GPIO_MODER_MODE13_Pos          (26U)
#define GPIO_MODER_MODE13_Msk          (0x3UL << GPIO_MODER_MODE13_Pos)         /*!< 0x0C000000 */
#define GPIO_MODER_MODE13              GPIO_MODER_MODE13_Msk
#define GPIO_MODER_MODE13_0            (0x1UL << GPIO_MODER_MODE13_Pos)         /*!< 0x04000000 */
#define GPIO_MODER_MODE13_1            (0x2UL << GPIO_MODER_MODE13_Pos)         /*!< 0x08000000 */
#define GPIO_MODER_MODE14_Pos          (28U)
#define GPIO_MODER_MODE14_Msk          (0x3UL << GPIO_MODER_MODE14_Pos)         /*!< 0x30000000 */
#define GPIO_MODER_MODE14              GPIO_MODER_MODE14_Msk
#define GPIO_MODER_MODE14_0            (0x1UL << GPIO_MODER_MODE14_Pos)         /*!< 0x10000000 */
#define GPIO_MODER_MODE14_1            (0x2UL << GPIO_MODER_MODE14_Pos)         /*!< 0x20000000 */
#define GPIO_MODER_MODE15_Pos          (30U)
#define GPIO_MODER_MODE15_Msk          (0x3UL << GPIO_MODER_MODE15_Pos)         /*!< 0xC0000000 */
#define GPIO_MODER_MODE15              GPIO_MODER_MODE15_Msk
#define GPIO_MODER_MODE15_0            (0x1UL << GPIO_MODER_MODE15_Pos)         /*!< 0x40000000 */
#define GPIO_MODER_MODE15_1            (0x2UL << GPIO_MODER_MODE15_Pos)         /*!< 0x80000000 */

/******************  Bits definition for GPIO_OTYPER register  ****************/
#define GPIO_OTYPER_OT0_Pos            (0U)
#define GPIO_OTYPER_OT0_Msk            (0x1UL << GPIO_OTYPER_OT0_Pos)           /*!< 0x00000001 */
#define GPIO_OTYPER_OT0                GPIO_OTYPER_OT0_Msk
#define GPIO_OTYPER_OT1_Pos            (1U)
#define GPIO_OTYPER_OT1_Msk            (0x1UL << GPIO_OTYPER_OT1_Pos)           /*!< 0x00000002 */
#define GPIO_OTYPER_OT1                GPIO_OTYPER_OT1_Msk
#define GPIO_OTYPER_OT2_Pos            (2U)
#define GPIO_OTYPER_OT2_Msk            (0x1UL << GPIO_OTYPER_OT2_Pos)           /*!< 0x00000004 */
#define GPIO_OTYPER_OT2                GPIO_OTYPER_OT2_Msk
#define GPIO_OTYPER_OT3_Pos            (3U)
#define GPIO_OTYPER_OT3_Msk            (0x1UL << GPIO_OTYPER_OT3_Pos)           /*!< 0x00000008 */
#define GPIO_OTYPER_OT3                GPIO_OTYPER_OT3_Msk
#define GPIO_OTYPER_OT4_Pos            (4U)
#define GPIO_OTYPER_OT4_Msk            (0x1UL << GPIO_OTYPER_OT4_Pos)           /*!< 0x00000010 */
#define GPIO_OTYPER_OT4                GPIO_OTYPER_OT4_Msk
#define GPIO_OTYPER_OT5_Pos            (5U)
#define GPIO_OTYPER_OT5_Msk            (0x1UL << GPIO_OTYPER_OT5_Pos)           /*!< 0x00000020 */
#define GPIO_OTYPER_OT5                GPIO_OTYPER_OT5_Msk
#define GPIO_OTYPER_OT6_Pos            (6U)
#define GPIO_OTYPER_OT6_Msk            (0x1UL << GPIO_OTYPER_OT6_Pos)           /*!< 0x00000040 */
#define GPIO_OTYPER_OT6                GPIO_OTYPER_OT6_Msk
#define GPIO_OTYPER_OT7_Pos            (7U)
#define GPIO_OTYPER_OT7_Msk            (0x1UL << GPIO_OTYPER_OT7_Pos)           /*!< 0x00000080 */
#define GPIO_OTYPER_OT7                GPIO_OTYPER_OT7_Msk
#define GPIO_OTYPER_OT8_Pos            (8U)
#define GPIO_OTYPER_OT8_Msk            (0x1UL << GPIO_OTYPER_OT8_Pos)           /*!< 0x00000100 */
#define GPIO_OTYPER_OT8                GPIO_OTYPER_OT8_Msk
#define GPIO_OTYPER_OT9_Pos            (9U)
#define GPIO_OTYPER_OT9_Msk            (0x1UL << GPIO_OTYPER_OT9_Pos)           /*!< 0x00000200 */
#define GPIO_OTYPER_OT9                GPIO_OTYPER_OT9_Msk
#define GPIO_OTYPER_OT10_Pos           (10U)
#define GPIO_OTYPER_OT10_Msk           (0x1UL << GPIO_OTYPER_OT10_Pos)          /*!< 0x00000400 */
#define GPIO_OTYPER_OT10               GPIO_OTYPER_OT10_Msk
#define GPIO_OTYPER_OT11_Pos           (11U)
#define GPIO_OTYPER_OT11_Msk           (0x1UL << GPIO_OTYPER_OT11_Pos)          /*!< 0x00000800 */
#define GPIO_OTYPER_OT11               GPIO_OTYPER_OT11_Msk
#define GPIO_OTYPER_OT12_Pos           (12U)
#define GPIO_OTYPER_OT12_Msk           (0x1UL << GPIO_OTYPER_OT12_Pos)          /*!< 0x00001000 */
#define GPIO_OTYPER_OT12               GPIO_OTYPER_OT12_Msk
#define GPIO_OTYPER_OT13_Pos           (13U)
#define GPIO_OTYPER_OT13_Msk           (0x1UL << GPIO_OTYPER_OT13_Pos)          /*!< 0x00002000 */
#define GPIO_OTYPER_OT13               GPIO_OTYPER_OT13_Msk
#define GPIO_OTYPER_OT14_Pos           (14U)
#define GPIO_OTYPER_OT14_Msk           (0x1UL << GPIO_OTYPER_OT14_Pos)          /*!< 0x00004000 */
#define GPIO_OTYPER_OT14               GPIO_OTYPER_OT14_Msk
#define GPIO_OTYPER_OT15_Pos           (15U)
#define GPIO_OTYPER_OT15_Msk           (0x1UL << GPIO_OTYPER_OT15_Pos)          /*!< 0x00008000 */
#define GPIO_OTYPER_OT15               GPIO_OTYPER_OT15_Msk

/******************  Bits definition for GPIO_OSPEEDR register  ***************/
#define GPIO_OSPEEDR_OSPEED0_Pos       (0U)
#define GPIO_OSPEEDR_OSPEED0_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED0_Pos)      /*!< 0x00000003 */
#define GPIO_OSPEEDR_OSPEED0           GPIO_OSPEEDR_OSPEED0_Msk
#define GPIO_OSPEEDR_OSPEED0_0         (0x1UL << GPIO_OSPEEDR_OSPEED0_Pos)      /*!< 0x00000001 */
#define GPIO_OSPEEDR_OSPEED0_1         (0x2UL << GPIO_OSPEEDR_OSPEED0_Pos)      /*!< 0x00000002 */
#define GPIO_OSPEEDR_OSPEED1_Pos       (2U)
#define GPIO_OSPEEDR_OSPEED1_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED1_Pos)      /*!< 0x0000000C */
#define GPIO_OSPEEDR_OSPEED1           GPIO_OSPEEDR_OSPEED1_Msk
#define GPIO_OSPEEDR_OSPEED1_0         (0x1UL << GPIO_OSPEEDR_OSPEED1_Pos)      /*!< 0x00000004 */
#define GPIO_OSPEEDR_OSPEED1_1         (0x2UL << GPIO_OSPEEDR_OSPEED1_Pos)      /*!< 0x00000008 */
#define GPIO_OSPEEDR_OSPEED2_Pos       (4U)
#define GPIO_OSPEEDR_OSPEED2_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED2_Pos)      /*!< 0x00000030 */
#define GPIO_OSPEEDR_OSPEED2           GPIO_OSPEEDR_OSPEED2_Msk
#define GPIO_OSPEEDR_OSPEED2_0         (0x1UL << GPIO_OSPEEDR_OSPEED2_Pos)      /*!< 0x00000010 */
#define GPIO_OSPEEDR_OSPEED2_1         (0x2UL << GPIO_OSPEEDR_OSPEED2_Pos)      /*!< 0x00000020 */
#define GPIO_OSPEEDR_OSPEED3_Pos       (6U)
#define GPIO_OSPEEDR_OSPEED3_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED3_Pos)      /*!< 0x000000C0 */
#define GPIO_OSPEEDR_OSPEED3           GPIO_OSPEEDR_OSPEED3_Msk
#define GPIO_OSPEEDR_OSPEED3_0         (0x1UL << GPIO_OSPEEDR_OSPEED3_Pos)      /*!< 0x00000040 */
#define GPIO_OSPEEDR_OSPEED3_1         (0x2UL << GPIO_OSPEEDR_OSPEED3_Pos)      /*!< 0x00000080 */
#define GPIO_OSPEEDR_OSPEED4_Pos       (8U)
#define GPIO_OSPEEDR_OSPEED4_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED4_Pos)      /*!< 0x00000300 */
#define GPIO_OSPEEDR_OSPEED4           GPIO_OSPEEDR_OSPEED4_Msk
#define GPIO_OSPEEDR_OSPEED4_0         (0x1UL << GPIO_OSPEEDR_OSPEED4_Pos)      /*!< 0x00000100 */
#define GPIO_OSPEEDR_OSPEED4_1         (0x2UL << GPIO_OSPEEDR_OSPEED4_Pos)      /*!< 0x00000200 */
#define GPIO_OSPEEDR_OSPEED5_Pos       (10U)
#define GPIO_OSPEEDR_OSPEED5_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED5_Pos)      /*!< 0x00000C00 */
#define GPIO_OSPEEDR_OSPEED5           GPIO_OSPEEDR_OSPEED5_Msk
#define GPIO_OSPEEDR_OSPEED5_0         (0x1UL << GPIO_OSPEEDR_OSPEED5_Pos)      /*!< 0x00000400 */
#define GPIO_OSPEEDR_OSPEED5_1         (0x2UL << GPIO_OSPEEDR_OSPEED5_Pos)      /*!< 0x00000800 */
#define GPIO_OSPEEDR_OSPEED6_Pos       (12U)
#define GPIO_OSPEEDR_OSPEED6_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED6_Pos)      /*!< 0x00003000 */
#define GPIO_OSPEEDR_OSPEED6           GPIO_OSPEEDR_OSPEED6_Msk
#define GPIO_OSPEEDR_OSPEED6_0         (0x1UL << GPIO_OSPEEDR_OSPEED6_Pos)      /*!< 0x00001000 */
#define GPIO_OSPEEDR_OSPEED6_1         (0x2UL << GPIO_OSPEEDR_OSPEED6_Pos)      /*!< 0x00002000 */
#define GPIO_OSPEEDR_OSPEED7_Pos       (14U)
#define GPIO_OSPEEDR_OSPEED7_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED7_Pos)      /*!< 0x0000C000 */
#define GPIO_OSPEEDR_OSPEED7           GPIO_OSPEEDR_OSPEED7_Msk
#define GPIO_OSPEEDR_OSPEED7_0         (0x1UL << GPIO_OSPEEDR_OSPEED7_Pos)      /*!< 0x00004000 */
#define GPIO_OSPEEDR_OSPEED7_1         (0x2UL << GPIO_OSPEEDR_OSPEED7_Pos)      /*!< 0x00008000 */
#define GPIO_OSPEEDR_OSPEED8_Pos       (16U)
#define GPIO_OSPEEDR_OSPEED8_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED8_Pos)      /*!< 0x00030000 */
#define GPIO_OSPEEDR_OSPEED8           GPIO_OSPEEDR_OSPEED8_Msk
#define GPIO_OSPEEDR_OSPEED8_0         (0x1UL << GPIO_OSPEEDR_OSPEED8_Pos)      /*!< 0x00010000 */
#define GPIO_OSPEEDR_OSPEED8_1         (0x2UL << GPIO_OSPEEDR_OSPEED8_Pos)      /*!< 0x00020000 */
#define GPIO_OSPEEDR_OSPEED9_Pos       (18U)
#define GPIO_OSPEEDR_OSPEED9_Msk       (0x3UL << GPIO_OSPEEDR_OSPEED9_Pos)      /*!< 0x000C0000 */
#define GPIO_OSPEEDR_OSPEED9           GPIO_OSPEEDR_OSPEED9_Msk
#define GPIO_OSPEEDR_OSPEED9_0         (0x1UL << GPIO_OSPEEDR_OSPEED9_Pos)      /*!< 0x00040000 */
#define GPIO_OSPEEDR_OSPEED9_1         (0x2UL << GPIO_OSPEEDR_OSPEED9_Pos)      /*!< 0x00080000 */
#define GPIO_OSPEEDR_OSPEED10_Pos      (20U)
#define GPIO_OSPEEDR_OSPEED10_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED10_Pos)     /*!< 0x00300000 */
#define GPIO_OSPEEDR_OSPEED10          GPIO_OSPEEDR_OSPEED10_Msk
#define GPIO_OSPEEDR_OSPEED10_0        (0x1UL << GPIO_OSPEEDR_OSPEED10_Pos)     /*!< 0x00100000 */
#define GPIO_OSPEEDR_OSPEED10_1        (0x2UL << GPIO_OSPEEDR_OSPEED10_Pos)     /*!< 0x00200000 */
#define GPIO_OSPEEDR_OSPEED11_Pos      (22U)
#define GPIO_OSPEEDR_OSPEED11_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED11_Pos)     /*!< 0x00C00000 */
#define GPIO_OSPEEDR_OSPEED11          GPIO_OSPEEDR_OSPEED11_Msk
#define GPIO_OSPEEDR_OSPEED11_0        (0x1UL << GPIO_OSPEEDR_OSPEED11_Pos)     /*!< 0x00400000 */
#define GPIO_OSPEEDR_OSPEED11_1        (0x2UL << GPIO_OSPEEDR_OSPEED11_Pos)     /*!< 0x00800000 */
#define GPIO_OSPEEDR_OSPEED12_Pos      (24U)
#define GPIO_OSPEEDR_OSPEED12_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED12_Pos)     /*!< 0x03000000 */
#define GPIO_OSPEEDR_OSPEED12          GPIO_OSPEEDR_OSPEED12_Msk
#define GPIO_OSPEEDR_OSPEED12_0        (0x1UL << GPIO_OSPEEDR_OSPEED12_Pos)     /*!< 0x01000000 */
#define GPIO_OSPEEDR_OSPEED12_1        (0x2UL << GPIO_OSPEEDR_OSPEED12_Pos)     /*!< 0x02000000 */
#define GPIO_OSPEEDR_OSPEED13_Pos      (26U)
#define GPIO_OSPEEDR_OSPEED13_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED13_Pos)     /*!< 0x0C000000 */
#define GPIO_OSPEEDR_OSPEED13          GPIO_OSPEEDR_OSPEED13_Msk
#define GPIO_OSPEEDR_OSPEED13_0        (0x1UL << GPIO_OSPEEDR_OSPEED13_Pos)     /*!< 0x04000000 */
#define GPIO_OSPEEDR_OSPEED13_1        (0x2UL << GPIO_OSPEEDR_OSPEED13_Pos)     /*!< 0x08000000 */
#define GPIO_OSPEEDR_OSPEED14_Pos      (28U)
#define GPIO_OSPEEDR_OSPEED14_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED14_Pos)     /*!< 0x30000000 */
#define GPIO_OSPEEDR_OSPEED14          GPIO_OSPEEDR_OSPEED14_Msk
#define GPIO_OSPEEDR_OSPEED14_0        (0x1UL << GPIO_OSPEEDR_OSPEED14_Pos)     /*!< 0x10000000 */
#define GPIO_OSPEEDR_OSPEED14_1        (0x2UL << GPIO_OSPEEDR_OSPEED14_Pos)     /*!< 0x20000000 */
#define GPIO_OSPEEDR_OSPEED15_Pos      (30U)
#define GPIO_OSPEEDR_OSPEED15_Msk      (0x3UL << GPIO_OSPEEDR_OSPEED15_Pos)     /*!< 0xC0000000 */
#define GPIO_OSPEEDR_OSPEED15          GPIO_OSPEEDR_OSPEED15_Msk
#define GPIO_OSPEEDR_OSPEED15_0        (0x1UL << GPIO_OSPEEDR_OSPEED15_Pos)     /*!< 0x40000000 */
#define GPIO_OSPEEDR_OSPEED15_1        (0x2UL << GPIO_OSPEEDR_OSPEED15_Pos)     /*!< 0x80000000 */

/******************  Bits definition for GPIO_PUPDR register  *****************/
#define GPIO_PUPDR_PUPD0_Pos           (0U)
#define GPIO_PUPDR_PUPD0_Msk           (0x3UL << GPIO_PUPDR_PUPD0_Pos)          /*!< 0x00000003 */
#define GPIO_PUPDR_PUPD0               GPIO_PUPDR_PUPD0_Msk
#define GPIO_PUPDR_PUPD0_0             (0x1UL << GPIO_PUPDR_PUPD0_Pos)          /*!< 0x00000001 */
#define GPIO_PUPDR_PUPD0_1             (0x2UL << GPIO_PUPDR_PUPD0_Pos)          /*!< 0x00000002 */
#define GPIO_PUPDR_PUPD1_Pos           (2U)
#define GPIO_PUPDR_PUPD1_Msk           (0x3UL << GPIO_PUPDR_PUPD1_Pos)          /*!< 0x0000000C */
#define GPIO_PUPDR_PUPD1               GPIO_PUPDR_PUPD1_Msk
#define GPIO_PUPDR_PUPD1_0             (0x1UL << GPIO_PUPDR_PUPD1_Pos)          /*!< 0x00000004 */
#define GPIO_PUPDR_PUPD1_1             (0x2UL << GPIO_PUPDR_PUPD1_Pos)          /*!< 0x00000008 */
#define GPIO_PUPDR_PUPD2_Pos           (4U)
#define GPIO_PUPDR_PUPD2_Msk           (0x3UL << GPIO_PUPDR_PUPD2_Pos)          /*!< 0x00000030 */
#define GPIO_PUPDR_PUPD2               GPIO_PUPDR_PUPD2_Msk
#define GPIO_PUPDR_PUPD2_0             (0x1UL << GPIO_PUPDR_PUPD2_Pos)          /*!< 0x00000010 */
#define GPIO_PUPDR_PUPD2_1             (0x2UL << GPIO_PUPDR_PUPD2_Pos)          /*!< 0x00000020 */
#define GPIO_PUPDR_PUPD3_Pos           (6U)
#define GPIO_PUPDR_PUPD3_Msk           (0x3UL << GPIO_PUPDR_PUPD3_Pos)          /*!< 0x000000C0 */
#define GPIO_PUPDR_PUPD3               GPIO_PUPDR_PUPD3_Msk
#define GPIO_PUPDR_PUPD3_0             (0x1UL << GPIO_PUPDR_PUPD3_Pos)          /*!< 0x00000040 */
#define GPIO_PUPDR_PUPD3_1             (0x2UL << GPIO_PUPDR_PUPD3_Pos)          /*!< 0x00000080 */
#define GPIO_PUPDR_PUPD4_Pos           (8U)
#define GPIO_PUPDR_PUPD4_Msk           (0x3UL << GPIO_PUPDR_PUPD4_Pos)          /*!< 0x00000300 */
#define GPIO_PUPDR_PUPD4               GPIO_PUPDR_PUPD4_Msk
#define GPIO_PUPDR_PUPD4_0             (0x1UL << GPIO_PUPDR_PUPD4_Pos)          /*!< 0x00000100 */
#define GPIO_PUPDR_PUPD4_1             (0x2UL << GPIO_PUPDR_PUPD4_Pos)          /*!< 0x00000200 */
#define GPIO_PUPDR_PUPD5_Pos           (10U)
#define GPIO_PUPDR_PUPD5_Msk           (0x3UL << GPIO_PUPDR_PUPD5_Pos)          /*!< 0x00000C00 */
#define GPIO_PUPDR_PUPD5               GPIO_PUPDR_PUPD5_Msk
#define GPIO_PUPDR_PUPD5_0             (0x1UL << GPIO_PUPDR_PUPD5_Pos)          /*!< 0x00000400 */
#define GPIO_PUPDR_PUPD5_1             (0x2UL << GPIO_PUPDR_PUPD5_Pos)          /*!< 0x00000800 */
#define GPIO_PUPDR_PUPD6_Pos           (12U)
#define GPIO_PUPDR_PUPD6_Msk           (0x3UL << GPIO_PUPDR_PUPD6_Pos)          /*!< 0x00003000 */
#define GPIO_PUPDR_PUPD6               GPIO_PUPDR_PUPD6_Msk
#define GPIO_PUPDR_PUPD6_0             (0x1UL << GPIO_PUPDR_PUPD6_Pos)          /*!< 0x00001000 */
#define GPIO_PUPDR_PUPD6_1             (0x2UL << GPIO_PUPDR_PUPD6_Pos)          /*!< 0x00002000 */
#define GPIO_PUPDR_PUPD7_Pos           (14U)
#define GPIO_PUPDR_PUPD7_Msk           (0x3UL << GPIO_PUPDR_PUPD7_Pos)          /*!< 0x0000C000 */
#define GPIO_PUPDR_PUPD7               GPIO_PUPDR_PUPD7_Msk
#define GPIO_PUPDR_PUPD7_0             (0x1UL << GPIO_PUPDR_PUPD7_Pos)          /*!< 0x00004000 */
#define GPIO_PUPDR_PUPD7_1             (0x2UL << GPIO_PUPDR_PUPD7_Pos)          /*!< 0x00008000 */
#define GPIO_PUPDR_PUPD8_Pos           (16U)
#define GPIO_PUPDR_PUPD8_Msk           (0x3UL << GPIO_PUPDR_PUPD8_Pos)          /*!< 0x00030000 */
#define GPIO_PUPDR_PUPD8               GPIO_PUPDR_PUPD8_Msk
#define GPIO_PUPDR_PUPD8_0             (0x1UL << GPIO_PUPDR_PUPD8_Pos)          /*!< 0x00010000 */
#define GPIO_PUPDR_PUPD8_1             (0x2UL << GPIO_PUPDR_PUPD8_Pos)          /*!< 0x00020000 */
#define GPIO_PUPDR_PUPD9_Pos           (18U)
#define GPIO_PUPDR_PUPD9_Msk           (0x3UL << GPIO_PUPDR_PUPD9_Pos)          /*!< 0x000C0000 */
#define GPIO_PUPDR_PUPD9               GPIO_PUPDR_PUPD9_Msk
#define GPIO_PUPDR_PUPD9_0             (0x1UL << GPIO_PUPDR_PUPD9_Pos)          /*!< 0x00040000 */
#define GPIO_PUPDR_PUPD9_1             (0x2UL << GPIO_PUPDR_PUPD9_Pos)          /*!< 0x00080000 */
#define GPIO_PUPDR_PUPD10_Pos          (20U)
#define GPIO_PUPDR_PUPD10_Msk          (0x3UL << GPIO_PUPDR_PUPD10_Pos)         /*!< 0x00300000 */
#define GPIO_PUPDR_PUPD10              GPIO_PUPDR_PUPD10_Msk
#define GPIO_PUPDR_PUPD10_0            (0x1UL << GPIO_PUPDR_PUPD10_Pos)         /*!< 0x00100000 */
#define GPIO_PUPDR_PUPD10_1            (0x2UL << GPIO_PUPDR_PUPD10_Pos)         /*!< 0x00200000 */
#define GPIO_PUPDR_PUPD11_Pos          (22U)
#define GPIO_PUPDR_PUPD11_Msk          (0x3UL << GPIO_PUPDR_PUPD11_Pos)         /*!< 0x00C00000 */
#define GPIO_PUPDR_PUPD11              GPIO_PUPDR_PUPD11_Msk
#define GPIO_PUPDR_PUPD11_0            (0x1UL << GPIO_PUPDR_PUPD11_Pos)         /*!< 0x00400000 */
#define GPIO_PUPDR_PUPD11_1            (0x2UL << GPIO_PUPDR_PUPD11_Pos)         /*!< 0x00800000 */
#define GPIO_PUPDR_PUPD12_Pos          (24U)
#define GPIO_PUPDR_PUPD12_Msk          (0x3UL << GPIO_PUPDR_PUPD12_Pos)         /*!< 0x03000000 */
#define GPIO_PUPDR_PUPD12              GPIO_PUPDR_PUPD12_Msk
#define GPIO_PUPDR_PUPD12_0            (0x1UL << GPIO_PUPDR_PUPD12_Pos)         /*!< 0x01000000 */
#define GPIO_PUPDR_PUPD12_1            (0x2UL << GPIO_PUPDR_PUPD12_Pos)         /*!< 0x02000000 */
#define GPIO_PUPDR_PUPD13_Pos          (26U)
#define GPIO_PUPDR_PUPD13_Msk          (0x3UL << GPIO_PUPDR_PUPD13_Pos)         /*!< 0x0C000000 */
#define GPIO_PUPDR_PUPD13              GPIO_PUPDR_PUPD13_Msk
#define GPIO_PUPDR_PUPD13_0            (0x1UL << GPIO_PUPDR_PUPD13_Pos)         /*!< 0x04000000 */
#define GPIO_PUPDR_PUPD13_1            (0x2UL << GPIO_PUPDR_PUPD13_Pos)         /*!< 0x08000000 */
#define GPIO_PUPDR_PUPD14_Pos          (28U)
#define GPIO_PUPDR_PUPD14_Msk          (0x3UL << GPIO_PUPDR_PUPD14_Pos)         /*!< 0x30000000 */
#define GPIO_PUPDR_PUPD14              GPIO_PUPDR_PUPD14_Msk
#define GPIO_PUPDR_PUPD14_0            (0x1UL << GPIO_PUPDR_PUPD14_Pos)         /*!< 0x10000000 */
#define GPIO_PUPDR_PUPD14_1            (0x2UL << GPIO_PUPDR_PUPD14_Pos)         /*!< 0x20000000 */
#define GPIO_PUPDR_PUPD15_Pos          (30U)
#define GPIO_PUPDR_PUPD15_Msk          (0x3UL << GPIO_PUPDR_PUPD15_Pos)         /*!< 0xC0000000 */
#define GPIO_PUPDR_PUPD15              GPIO_PUPDR_PUPD15_Msk
#define GPIO_PUPDR_PUPD15_0            (0x1UL << GPIO_PUPDR_PUPD15_Pos)         /*!< 0x40000000 */
#define GPIO_PUPDR_PUPD15_1            (0x2UL << GPIO_PUPDR_PUPD15_Pos)         /*!< 0x80000000 */

/******************  Bits definition for GPIO_IDR register  *******************/
#define GPIO_IDR_ID0_Pos               (0U)
#define GPIO_IDR_ID0_Msk               (0x1UL << GPIO_IDR_ID0_Pos)              /*!< 0x00000001 */
#define GPIO_IDR_ID0                   GPIO_IDR_ID0_Msk
#define GPIO_IDR_ID1_Pos               (1U)
#define GPIO_IDR_ID1_Msk               (0x1UL << GPIO_IDR_ID1_Pos)              /*!< 0x00000002 */
#define GPIO_IDR_ID1                   GPIO_IDR_ID1_Msk
#define GPIO_IDR_ID2_Pos               (2U)
#define GPIO_IDR_ID2_Msk               (0x1UL << GPIO_IDR_ID2_Pos)              /*!< 0x00000004 */
#define GPIO_IDR_ID2                   GPIO_IDR_ID2_Msk
#define GPIO_IDR_ID3_Pos               (3U)
#define GPIO_IDR_ID3_Msk               (0x1UL << GPIO_IDR_ID3_Pos)              /*!< 0x00000008 */
#define GPIO_IDR_ID3                   GPIO_IDR_ID3_Msk
#define GPIO_IDR_ID4_Pos               (4U)
#define GPIO_IDR_ID4_Msk               (0x1UL << GPIO_IDR_ID4_Pos)              /*!< 0x00000010 */
#define GPIO_IDR_ID4                   GPIO_IDR_ID4_Msk
#define GPIO_IDR_ID5_Pos               (5U)
#define GPIO_IDR_ID5_Msk               (0x1UL << GPIO_IDR_ID5_Pos)              /*!< 0x00000020 */
#define GPIO_IDR_ID5                   GPIO_IDR_ID5_Msk
#define GPIO_IDR_ID6_Pos               (6U)
#define GPIO_IDR_ID6_Msk               (0x1UL << GPIO_IDR_ID6_Pos)              /*!< 0x00000040 */
#define GPIO_IDR_ID6                   GPIO_IDR_ID6_Msk
#define GPIO_IDR_ID7_Pos               (7U)
#define GPIO_IDR_ID7_Msk               (0x1UL << GPIO_IDR_ID7_Pos)              /*!< 0x00000080 */
#define GPIO_IDR_ID7                   GPIO_IDR_ID7_Msk
#define GPIO_IDR_ID8_Pos               (8U)
#define GPIO_IDR_ID8_Msk               (0x1UL << GPIO_IDR_ID8_Pos)              /*!< 0x00000100 */
#define GPIO_IDR_ID8                   GPIO_IDR_ID8_Msk
#define GPIO_IDR_ID9_Pos               (9U)
#define GPIO_IDR_ID9_Msk               (0x1UL << GPIO_IDR_ID9_Pos)              /*!< 0x00000200 */
#define GPIO_IDR_ID9                   GPIO_IDR_ID9_Msk
#define GPIO_IDR_ID10_Pos              (10U)
#define GPIO_IDR_ID10_Msk              (0x1UL << GPIO_IDR_ID10_Pos)             /*!< 0x00000400 */
#define GPIO_IDR_ID10                  GPIO_IDR_ID10_Msk
#define GPIO_IDR_ID11_Pos              (11U)
#define GPIO_IDR_ID11_Msk              (0x1UL << GPIO_IDR_ID11_Pos)             /*!< 0x00000800 */
#define GPIO_IDR_ID11                  GPIO_IDR_ID11_Msk
#define GPIO_IDR_ID12_Pos              (12U)
#define GPIO_IDR_ID12_Msk              (0x1UL << GPIO_IDR_ID12_Pos)             /*!< 0x00001000 */
#define GPIO_IDR_ID12                  GPIO_IDR_ID12_Msk
#define GPIO_IDR_ID13_Pos              (13U)
#define GPIO_IDR_ID13_Msk              (0x1UL << GPIO_IDR_ID13_Pos)             /*!< 0x00002000 */
#define GPIO_IDR_ID13                  GPIO_IDR_ID13_Msk
#define GPIO_IDR_ID14_Pos              (14U)
#define GPIO_IDR_ID14_Msk              (0x1UL << GPIO_IDR_ID14_Pos)             /*!< 0x00004000 */
#define GPIO_IDR_ID14                  GPIO_IDR_ID14_Msk
#define GPIO_IDR_ID15_Pos              (15U)
#define GPIO_IDR_ID15_Msk              (0x1UL << GPIO_IDR_ID15_Pos)             /*!< 0x00008000 */
#define GPIO_IDR_ID15                  GPIO_IDR_ID15_Msk

/******************  Bits definition for GPIO_ODR register  *******************/
#define GPIO_ODR_OD0_Pos               (0U)
#define GPIO_ODR_OD0_Msk               (0x1UL << GPIO_ODR_OD0_Pos)              /*!< 0x00000001 */
#define GPIO_ODR_OD0                   GPIO_ODR_OD0_Msk
#define GPIO_ODR_OD1_Pos               (1U)
#define GPIO_ODR_OD1_Msk               (0x1UL << GPIO_ODR_OD1_Pos)              /*!< 0x00000002 */
#define GPIO_ODR_OD1                   GPIO_ODR_OD1_Msk
#define GPIO_ODR_OD2_Pos               (2U)
#define GPIO_ODR_OD2_Msk               (0x1UL << GPIO_ODR_OD2_Pos)              /*!< 0x00000004 */
#define GPIO_ODR_OD2                   GPIO_ODR_OD2_Msk
#define GPIO_ODR_OD3_Pos               (3U)
#define GPIO_ODR_OD3_Msk               (0x1UL << GPIO_ODR_OD3_Pos)              /*!< 0x00000008 */
#define GPIO_ODR_OD3                   GPIO_ODR_OD3_Msk
#define GPIO_ODR_OD4_Pos               (4U)
#define GPIO_ODR_OD4_Msk               (0x1UL << GPIO_ODR_OD4_Pos)              /*!< 0x00000010 */
#define GPIO_ODR_OD4                   GPIO_ODR_OD4_Msk
#define GPIO_ODR_OD5_Pos               (5U)
#define GPIO_ODR_OD5_Msk               (0x1UL << GPIO_ODR_OD5_Pos)              /*!< 0x00000020 */
#define GPIO_ODR_OD5                   GPIO_ODR_OD5_Msk
#define GPIO_ODR_OD6_Pos               (6U)
#define GPIO_ODR_OD6_Msk               (0x1UL << GPIO_ODR_OD6_Pos)              /*!< 0x00000040 */
#define GPIO_ODR_OD6                   GPIO_ODR_OD6_Msk
#define GPIO_ODR_OD7_Pos               (7U)
#define GPIO_ODR_OD7_Msk               (0x1UL << GPIO_ODR_OD7_Pos)              /*!< 0x00000080 */
#define GPIO_ODR_OD7                   GPIO_ODR_OD7_Msk
#define GPIO_ODR_OD8_Pos               (8U)
#define GPIO_ODR_OD8_Msk               (0x1UL << GPIO_ODR_OD8_Pos)              /*!< 0x00000100 */
#define GPIO_ODR_OD8                   GPIO_ODR_OD8_Msk
#define GPIO_ODR_OD9_Pos               (9U)
#define GPIO_ODR_OD9_Msk               (0x1UL << GPIO_ODR_OD9_Pos)              /*!< 0x00000200 */
#define GPIO_ODR_OD9                   GPIO_ODR_OD9_Msk
#define GPIO_ODR_OD10_Pos              (10U)
#define GPIO_ODR_OD10_Msk              (0x1UL << GPIO_ODR_OD10_Pos)             /*!< 0x00000400 */
#define GPIO_ODR_OD10                  GPIO_ODR_OD10_Msk
#define GPIO_ODR_OD11_Pos              (11U)
#define GPIO_ODR_OD11_Msk              (0x1UL << GPIO_ODR_OD11_Pos)             /*!< 0x00000800 */
#define GPIO_ODR_OD11                  GPIO_ODR_OD11_Msk
#define GPIO_ODR_OD12_Pos              (12U)
#define GPIO_ODR_OD12_Msk              (0x1UL << GPIO_ODR_OD12_Pos)             /*!< 0x00001000 */
#define GPIO_ODR_OD12                  GPIO_ODR_OD12_Msk
#define GPIO_ODR_OD13_Pos              (13U)
#define GPIO_ODR_OD13_Msk              (0x1UL << GPIO_ODR_OD13_Pos)             /*!< 0x00002000 */
#define GPIO_ODR_OD13                  GPIO_ODR_OD13_Msk
#define GPIO_ODR_OD14_Pos              (14U)
#define GPIO_ODR_OD14_Msk              (0x1UL << GPIO_ODR_OD14_Pos)             /*!< 0x00004000 */
#define GPIO_ODR_OD14                  GPIO_ODR_OD14_Msk
#define GPIO_ODR_OD15_Pos              (15U)
#define GPIO_ODR_OD15_Msk              (0x1UL << GPIO_ODR_OD15_Pos)             /*!< 0x00008000 */
#define GPIO_ODR_OD15                  GPIO_ODR_OD15_Msk

/******************  Bits definition for GPIO_BSRR register  ******************/
#define GPIO_BSRR_BS0_Pos              (0U)
#define GPIO_BSRR_BS0_Msk              (0x1UL << GPIO_BSRR_BS0_Pos)             /*!< 0x00000001 */
#define GPIO_BSRR_BS0                  GPIO_BSRR_BS0_Msk
#define GPIO_BSRR_BS1_Pos              (1U)
#define GPIO_BSRR_BS1_Msk              (0x1UL << GPIO_BSRR_BS1_Pos)             /*!< 0x00000002 */
#define GPIO_BSRR_BS1                  GPIO_BSRR_BS1_Msk
#define GPIO_BSRR_BS2_Pos              (2U)
#define GPIO_BSRR_BS2_Msk              (0x1UL << GPIO_BSRR_BS2_Pos)             /*!< 0x00000004 */
#define GPIO_BSRR_BS2                  GPIO_BSRR_BS2_Msk
#define GPIO_BSRR_BS3_Pos              (3U)
#define GPIO_BSRR_BS3_Msk              (0x1UL << GPIO_BSRR_BS3_Pos)             /*!< 0x00000008 */
#define GPIO_BSRR_BS3                  GPIO_BSRR_BS3_Msk
#define GPIO_BSRR_BS4_Pos              (4U)
#define GPIO_BSRR_BS4_Msk              (0x1UL << GPIO_BSRR_BS4_Pos)             /*!< 0x00000010 */
#define GPIO_BSRR_BS4                  GPIO_BSRR_BS4_Msk
#define GPIO_BSRR_BS5_Pos              (5U)
#define GPIO_BSRR_BS5_Msk              (0x1UL << GPIO_BSRR_BS5_Pos)             /*!< 0x00000020 */
#define GPIO_BSRR_BS5                  GPIO_BSRR_BS5_Msk
#define GPIO_BSRR_BS6_Pos              (6U)
#define GPIO_BSRR_BS6_Msk              (0x1UL << GPIO_BSRR_BS6_Pos)             /*!< 0x00000040 */
#define GPIO_BSRR_BS6                  GPIO_BSRR_BS6_Msk
#define GPIO_BSRR_BS7_Pos              (7U)
#define GPIO_BSRR_BS7_Msk              (0x1UL << GPIO_BSRR_BS7_Pos)             /*!< 0x00000080 */
#define GPIO_BSRR_BS7                  GPIO_BSRR_BS7_Msk
#define GPIO_BSRR_BS8_Pos              (8U)
#define GPIO_BSRR_BS8_Msk              (0x1UL << GPIO_BSRR_BS8_Pos)             /*!< 0x00000100 */
#define GPIO_BSRR_BS8                  GPIO_BSRR_BS8_Msk
#define GPIO_BSRR_BS9_Pos              (9U)
#define GPIO_BSRR_BS9_Msk              (0x1UL << GPIO_BSRR_BS9_Pos)             /*!< 0x00000200 */
#define GPIO_BSRR_BS9                  GPIO_BSRR_BS9_Msk
#define GPIO_BSRR_BS10_Pos             (10U)
#define GPIO_BSRR_BS10_Msk             (0x1UL << GPIO_BSRR_BS10_Pos)            /*!< 0x00000400 */
#define GPIO_BSRR_BS10                 GPIO_BSRR_BS10_Msk
#define GPIO_BSRR_BS11_Pos             (11U)
#define GPIO_BSRR_BS11_Msk             (0x1UL << GPIO_BSRR_BS11_Pos)            /*!< 0x00000800 */
#define GPIO_BSRR_BS11                 GPIO_BSRR_BS11_Msk
#define GPIO_BSRR_BS12_Pos             (12U)
#define GPIO_BSRR_BS12_Msk             (0x1UL << GPIO_BSRR_BS12_Pos)            /*!< 0x00001000 */
#define GPIO_BSRR_BS12                 GPIO_BSRR_BS12_Msk
#define GPIO_BSRR_BS13_Pos             (13U)
#define GPIO_BSRR_BS13_Msk             (0x1UL << GPIO_BSRR_BS13_Pos)            /*!< 0x00002000 */
#define GPIO_BSRR_BS13                 GPIO_BSRR_BS13_Msk
#define GPIO_BSRR_BS14_Pos             (14U)
#define GPIO_BSRR_BS14_Msk             (0x1UL << GPIO_BSRR_BS14_Pos)            /*!< 0x00004000 */
#define GPIO_BSRR_BS14                 GPIO_BSRR_BS14_Msk
#define GPIO_BSRR_BS15_Pos             (15U)
#define GPIO_BSRR_BS15_Msk             (0x1UL << GPIO_BSRR_BS15_Pos)            /*!< 0x00008000 */
#define GPIO_BSRR_BS15                 GPIO_BSRR_BS15_Msk
#define GPIO_BSRR_BR0_Pos              (16U)
#define GPIO_BSRR_BR0_Msk              (0x1UL << GPIO_BSRR_BR0_Pos)             /*!< 0x00010000 */
#define GPIO_BSRR_BR0                  GPIO_BSRR_BR0_Msk
#define GPIO_BSRR_BR1_Pos              (17U)
#define GPIO_BSRR_BR1_Msk              (0x1UL << GPIO_BSRR_BR1_Pos)             /*!< 0x00020000 */
#define GPIO_BSRR_BR1                  GPIO_BSRR_BR1_Msk
#define GPIO_BSRR_BR2_Pos              (18U)
#define GPIO_BSRR_BR2_Msk              (0x1UL << GPIO_BSRR_BR2_Pos)             /*!< 0x00040000 */
#define GPIO_BSRR_BR2                  GPIO_BSRR_BR2_Msk
#define GPIO_BSRR_BR3_Pos              (19U)
#define GPIO_BSRR_BR3_Msk              (0x1UL << GPIO_BSRR_BR3_Pos)             /*!< 0x00080000 */
#define GPIO_BSRR_BR3                  GPIO_BSRR_BR3_Msk
#define GPIO_BSRR_BR4_Pos              (20U)
#define GPIO_BSRR_BR4_Msk              (0x1UL << GPIO_BSRR_BR4_Pos)             /*!< 0x00100000 */
#define GPIO_BSRR_BR4                  GPIO_BSRR_BR4_Msk
#define GPIO_BSRR_BR5_Pos              (21U)
#define GPIO_BSRR_BR5_Msk              (0x1UL << GPIO_BSRR_BR5_Pos)             /*!< 0x00200000 */
#define GPIO_BSRR_BR5                  GPIO_BSRR_BR5_Msk
#define GPIO_BSRR_BR6_Pos              (22U)
#define GPIO_BSRR_BR6_Msk              (0x1UL << GPIO_BSRR_BR6_Pos)             /*!< 0x00400000 */
#define GPIO_BSRR_BR6                  GPIO_BSRR_BR6_Msk
#define GPIO_BSRR_BR7_Pos              (23U)
#define GPIO_BSRR_BR7_Msk              (0x1UL << GPIO_BSRR_BR7_Pos)             /*!< 0x00800000 */
#define GPIO_BSRR_BR7                  GPIO_BSRR_BR7_Msk
#define GPIO_BSRR_BR8_Pos              (24U)
#define GPIO_BSRR_BR8_Msk              (0x1UL << GPIO_BSRR_BR8_Pos)             /*!< 0x01000000 */
#define GPIO_BSRR_BR8                  GPIO_BSRR_BR8_Msk
#define GPIO_BSRR_BR9_Pos              (25U)
#define GPIO_BSRR_BR9_Msk              (0x1UL << GPIO_BSRR_BR9_Pos)             /*!< 0x02000000 */
#define GPIO_BSRR_BR9                  GPIO_BSRR_BR9_Msk
#define GPIO_BSRR_BR10_Pos             (26U)
#define GPIO_BSRR_BR10_Msk             (0x1UL << GPIO_BSRR_BR10_Pos)            /*!< 0x04000000 */
#define GPIO_BSRR_BR10                 GPIO_BSRR_BR10_Msk
#define GPIO_BSRR_BR11_Pos             (27U)
#define GPIO_BSRR_BR11_Msk             (0x1UL << GPIO_BSRR_BR11_Pos)            /*!< 0x08000000 */
#define GPIO_BSRR_BR11                 GPIO_BSRR_BR11_Msk
#define GPIO_BSRR_BR12_Pos             (28U)
#define GPIO_BSRR_BR12_Msk             (0x1UL << GPIO_BSRR_BR12_Pos)            /*!< 0x10000000 */
#define GPIO_BSRR_BR12                 GPIO_BSRR_BR12_Msk
#define GPIO_BSRR_BR13_Pos             (29U)
#define GPIO_BSRR_BR13_Msk             (0x1UL << GPIO_BSRR_BR13_Pos)            /*!< 0x20000000 */
#define GPIO_BSRR_BR13                 GPIO_BSRR_BR13_Msk
#define GPIO_BSRR_BR14_Pos             (30U)
#define GPIO_BSRR_BR14_Msk             (0x1UL << GPIO_BSRR_BR14_Pos)            /*!< 0x40000000 */
#define GPIO_BSRR_BR14                 GPIO_BSRR_BR14_Msk
#define GPIO_BSRR_BR15_Pos             (31U)
#define GPIO_BSRR_BR15_Msk             (0x1UL << GPIO_BSRR_BR15_Pos)            /*!< 0x80000000 */
#define GPIO_BSRR_BR15                 GPIO_BSRR_BR15_Msk

/****************** Bit definition for GPIO_LCKR register *********************/
#define GPIO_LCKR_LCK0_Pos             (0U)
#define GPIO_LCKR_LCK0_Msk             (0x1UL << GPIO_LCKR_LCK0_Pos)            /*!< 0x00000001 */
#define GPIO_LCKR_LCK0                 GPIO_LCKR_LCK0_Msk
#define GPIO_LCKR_LCK1_Pos             (1U)
#define GPIO_LCKR_LCK1_Msk             (0x1UL << GPIO_LCKR_LCK1_Pos)            /*!< 0x00000002 */
#define GPIO_LCKR_LCK1                 GPIO_LCKR_LCK1_Msk
#define GPIO_LCKR_LCK2_Pos             (2U)
#define GPIO_LCKR_LCK2_Msk             (0x1UL << GPIO_LCKR_LCK2_Pos)            /*!< 0x00000004 */
#define GPIO_LCKR_LCK2                 GPIO_LCKR_LCK2_Msk
#define GPIO_LCKR_LCK3_Pos             (3U)
#define GPIO_LCKR_LCK3_Msk             (0x1UL << GPIO_LCKR_LCK3_Pos)            /*!< 0x00000008 */
#define GPIO_LCKR_LCK3                 GPIO_LCKR_LCK3_Msk
#define GPIO_LCKR_LCK4_Pos             (4U)
#define GPIO_LCKR_LCK4_Msk             (0x1UL << GPIO_LCKR_LCK4_Pos)            /*!< 0x00000010 */
#define GPIO_LCKR_LCK4                 GPIO_LCKR_LCK4_Msk
#define GPIO_LCKR_LCK5_Pos             (5U)
#define GPIO_LCKR_LCK5_Msk             (0x1UL << GPIO_LCKR_LCK5_Pos)            /*!< 0x00000020 */
#define GPIO_LCKR_LCK5                 GPIO_LCKR_LCK5_Msk
#define GPIO_LCKR_LCK6_Pos             (6U)
#define GPIO_LCKR_LCK6_Msk             (0x1UL << GPIO_LCKR_LCK6_Pos)            /*!< 0x00000040 */
#define GPIO_LCKR_LCK6                 GPIO_LCKR_LCK6_Msk
#define GPIO_LCKR_LCK7_Pos             (7U)
#define GPIO_LCKR_LCK7_Msk             (0x1UL << GPIO_LCKR_LCK7_Pos)            /*!< 0x00000080 */
#define GPIO_LCKR_LCK7                 GPIO_LCKR_LCK7_Msk
#define GPIO_LCKR_LCK8_Pos             (8U)
#define GPIO_LCKR_LCK8_Msk             (0x1UL << GPIO_LCKR_LCK8_Pos)            /*!< 0x00000100 */
#define GPIO_LCKR_LCK8                 GPIO_LCKR_LCK8_Msk
#define GPIO_LCKR_LCK9_Pos             (9U)
#define GPIO_LCKR_LCK9_Msk             (0x1UL << GPIO_LCKR_LCK9_Pos)            /*!< 0x00000200 */
#define GPIO_LCKR_LCK9                 GPIO_LCKR_LCK9_Msk
#define GPIO_LCKR_LCK10_Pos            (10U)
#define GPIO_LCKR_LCK10_Msk            (0x1UL << GPIO_LCKR_LCK10_Pos)           /*!< 0x00000400 */
#define GPIO_LCKR_LCK10                GPIO_LCKR_LCK10_Msk
#define GPIO_LCKR_LCK11_Pos            (11U)
#define GPIO_LCKR_LCK11_Msk            (0x1UL << GPIO_LCKR_LCK11_Pos)           /*!< 0x00000800 */
#define GPIO_LCKR_LCK11                GPIO_LCKR_LCK11_Msk
#define GPIO_LCKR_LCK12_Pos            (12U)
#define GPIO_LCKR_LCK12_Msk            (0x1UL << GPIO_LCKR_LCK12_Pos)           /*!< 0x00001000 */
#define GPIO_LCKR_LCK12                GPIO_LCKR_LCK12_Msk
#define GPIO_LCKR_LCK13_Pos            (13U)
#define GPIO_LCKR_LCK13_Msk            (0x1UL << GPIO_LCKR_LCK13_Pos)           /*!< 0x00002000 */
#define GPIO_LCKR_LCK13                GPIO_LCKR_LCK13_Msk
#define GPIO_LCKR_LCK14_Pos            (14U)
#define GPIO_LCKR_LCK14_Msk            (0x1UL << GPIO_LCKR_LCK14_Pos)           /*!< 0x00004000 */
#define GPIO_LCKR_LCK14                GPIO_LCKR_LCK14_Msk
#define GPIO_LCKR_LCK15_Pos            (15U)
#define GPIO_LCKR_LCK15_Msk            (0x1UL << GPIO_LCKR_LCK15_Pos)           /*!< 0x00008000 */
#define GPIO_LCKR_LCK15                GPIO_LCKR_LCK15_Msk
#define GPIO_LCKR_LCKK_Pos             (16U)
#define GPIO_LCKR_LCKK_Msk             (0x1UL << GPIO_LCKR_LCKK_Pos)            /*!< 0x00010000 */
#define GPIO_LCKR_LCKK                 GPIO_LCKR_LCKK_Msk

/****************** Bit definition for GPIO_AFRL register *********************/
#define GPIO_AFRL_AFSEL0_Pos           (0U)
#define GPIO_AFRL_AFSEL0_Msk           (0xFUL << GPIO_AFRL_AFSEL0_Pos)          /*!< 0x0000000F */
#define GPIO_AFRL_AFSEL0               GPIO_AFRL_AFSEL0_Msk
#define GPIO_AFRL_AFSEL0_0             (0x1UL << GPIO_AFRL_AFSEL0_Pos)          /*!< 0x00000001 */
#define GPIO_AFRL_AFSEL0_1             (0x2UL << GPIO_AFRL_AFSEL0_Pos)          /*!< 0x00000002 */
#define GPIO_AFRL_AFSEL0_2             (0x4UL << GPIO_AFRL_AFSEL0_Pos)          /*!< 0x00000004 */
#define GPIO_AFRL_AFSEL0_3             (0x8UL << GPIO_AFRL_AFSEL0_Pos)          /*!< 0x00000008 */
#define GPIO_AFRL_AFSEL1_Pos           (4U)
#define GPIO_AFRL_AFSEL1_Msk           (0xFUL << GPIO_AFRL_AFSEL1_Pos)          /*!< 0x000000F0 */
#define GPIO_AFRL_AFSEL1               GPIO_AFRL_AFSEL1_Msk
#define GPIO_AFRL_AFSEL1_0             (0x1UL << GPIO_AFRL_AFSEL1_Pos)          /*!< 0x00000010 */
#define GPIO_AFRL_AFSEL1_1             (0x2UL << GPIO_AFRL_AFSEL1_Pos)          /*!< 0x00000020 */
#define GPIO_AFRL_AFSEL1_2             (0x4UL << GPIO_AFRL_AFSEL1_Pos)          /*!< 0x00000040 */
#define GPIO_AFRL_AFSEL1_3             (0x8UL << GPIO_AFRL_AFSEL1_Pos)          /*!< 0x00000080 */
#define GPIO_AFRL_AFSEL2_Pos           (8U)
#define GPIO_AFRL_AFSEL2_Msk           (0xFUL << GPIO_AFRL_AFSEL2_Pos)          /*!< 0x00000F00 */
#define GPIO_AFRL_AFSEL2               GPIO_AFRL_AFSEL2_Msk
#define GPIO_AFRL_AFSEL2_0             (0x1UL << GPIO_AFRL_AFSEL2_Pos)          /*!< 0x00000100 */
#define GPIO_AFRL_AFSEL2_1             (0x2UL << GPIO_AFRL_AFSEL2_Pos)          /*!< 0x00000200 */
#define GPIO_AFRL_AFSEL2_2             (0x4UL << GPIO_AFRL_AFSEL2_Pos)          /*!< 0x00000400 */
#define GPIO_AFRL_AFSEL2_3             (0x8UL << GPIO_AFRL_AFSEL2_Pos)          /*!< 0x00000800 */
#define GPIO_AFRL_AFSEL3_Pos           (12U)
#define GPIO_AFRL_AFSEL3_Msk           (0xFUL << GPIO_AFRL_AFSEL3_Pos)          /*!< 0x0000F000 */
#define GPIO_AFRL_AFSEL3               GPIO_AFRL_AFSEL3_Msk
#define GPIO_AFRL_AFSEL3_0             (0x1UL << GPIO_AFRL_AFSEL3_Pos)          /*!< 0x00001000 */
#define GPIO_AFRL_AFSEL3_1             (0x2UL << GPIO_AFRL_AFSEL3_Pos)          /*!< 0x00002000 */
#define GPIO_AFRL_AFSEL3_2             (0x4UL << GPIO_AFRL_AFSEL3_Pos)          /*!< 0x00004000 */
#define GPIO_AFRL_AFSEL3_3             (0x8UL << GPIO_AFRL_AFSEL3_Pos)          /*!< 0x00008000 */
#define GPIO_AFRL_AFSEL4_Pos           (16U)
#define GPIO_AFRL_AFSEL4_Msk           (0xFUL << GPIO_AFRL_AFSEL4_Pos)          /*!< 0x000F0000 */
#define GPIO_AFRL_AFSEL4               GPIO_AFRL_AFSEL4_Msk
#define GPIO_AFRL_AFSEL4_0             (0x1UL << GPIO_AFRL_AFSEL4_Pos)          /*!< 0x00010000 */
#define GPIO_AFRL_AFSEL4_1             (0x2UL << GPIO_AFRL_AFSEL4_Pos)          /*!< 0x00020000 */
#define GPIO_AFRL_AFSEL4_2             (0x4UL << GPIO_AFRL_AFSEL4_Pos)          /*!< 0x00040000 */
#define GPIO_AFRL_AFSEL4_3             (0x8UL << GPIO_AFRL_AFSEL4_Pos)          /*!< 0x00080000 */
#define GPIO_AFRL_AFSEL5_Pos           (20U)
#define GPIO_AFRL_AFSEL5_Msk           (0xFUL << GPIO_AFRL_AFSEL5_Pos)          /*!< 0x00F00000 */
#define GPIO_AFRL_AFSEL5               GPIO_AFRL_AFSEL5_Msk
#define GPIO_AFRL_AFSEL5_0             (0x1UL << GPIO_AFRL_AFSEL5_Pos)          /*!< 0x00100000 */
#define GPIO_AFRL_AFSEL5_1             (0x2UL << GPIO_AFRL_AFSEL5_Pos)          /*!< 0x00200000 */
#define GPIO_AFRL_AFSEL5_2             (0x4UL << GPIO_AFRL_AFSEL5_Pos)          /*!< 0x00400000 */
#define GPIO_AFRL_AFSEL5_3             (0x8UL << GPIO_AFRL_AFSEL5_Pos)          /*!< 0x00800000 */
#define GPIO_AFRL_AFSEL6_Pos           (24U)
#define GPIO_AFRL_AFSEL6_Msk           (0xFUL << GPIO_AFRL_AFSEL6_Pos)          /*!< 0x0F000000 */
#define GPIO_AFRL_AFSEL6               GPIO_AFRL_AFSEL6_Msk
#define GPIO_AFRL_AFSEL6_0             (0x1UL << GPIO_AFRL_AFSEL6_Pos)          /*!< 0x01000000 */
#define GPIO_AFRL_AFSEL6_1             (0x2UL << GPIO_AFRL_AFSEL6_Pos)          /*!< 0x02000000 */
#define GPIO_AFRL_AFSEL6_2             (0x4UL << GPIO_AFRL_AFSEL6_Pos)          /*!< 0x04000000 */
#define GPIO_AFRL_AFSEL6_3             (0x8UL << GPIO_AFRL_AFSEL6_Pos)          /*!< 0x08000000 */
#define GPIO_AFRL_AFSEL7_Pos           (28U)
#define GPIO_AFRL_AFSEL7_Msk           (0xFUL << GPIO_AFRL_AFSEL7_Pos)          /*!< 0xF0000000 */
#define GPIO_AFRL_AFSEL7               GPIO_AFRL_AFSEL7_Msk
#define GPIO_AFRL_AFSEL7_0             (0x1UL << GPIO_AFRL_AFSEL7_Pos)          /*!< 0x10000000 */
#define GPIO_AFRL_AFSEL7_1             (0x2UL << GPIO_AFRL_AFSEL7_Pos)          /*!< 0x20000000 */
#define GPIO_AFRL_AFSEL7_2             (0x4UL << GPIO_AFRL_AFSEL7_Pos)          /*!< 0x40000000 */
#define GPIO_AFRL_AFSEL7_3             (0x8UL << GPIO_AFRL_AFSEL7_Pos)          /*!< 0x80000000 */

/****************** Bit definition for GPIO_AFRH register *********************/
#define GPIO_AFRH_AFSEL8_Pos           (0U)
#define GPIO_AFRH_AFSEL8_Msk           (0xFUL << GPIO_AFRH_AFSEL8_Pos)          /*!< 0x0000000F */
#define GPIO_AFRH_AFSEL8               GPIO_AFRH_AFSEL8_Msk
#define GPIO_AFRH_AFSEL8_0             (0x1UL << GPIO_AFRH_AFSEL8_Pos)          /*!< 0x00000001 */
#define GPIO_AFRH_AFSEL8_1             (0x2UL << GPIO_AFRH_AFSEL8_Pos)          /*!< 0x00000002 */
#define GPIO_AFRH_AFSEL8_2             (0x4UL << GPIO_AFRH_AFSEL8_Pos)          /*!< 0x00000004 */
#define GPIO_AFRH_AFSEL8_3             (0x8UL << GPIO_AFRH_AFSEL8_Pos)          /*!< 0x00000008 */
#define GPIO_AFRH_AFSEL9_Pos           (4U)
#define GPIO_AFRH_AFSEL9_Msk           (0xFUL << GPIO_AFRH_AFSEL9_Pos)          /*!< 0x000000F0 */
#define GPIO_AFRH_AFSEL9               GPIO_AFRH_AFSEL9_Msk
#define GPIO_AFRH_AFSEL9_0             (0x1UL << GPIO_AFRH_AFSEL9_Pos)          /*!< 0x00000010 */
#define GPIO_AFRH_AFSEL9_1             (0x2UL << GPIO_AFRH_AFSEL9_Pos)          /*!< 0x00000020 */
#define GPIO_AFRH_AFSEL9_2             (0x4UL << GPIO_AFRH_AFSEL9_Pos)          /*!< 0x00000040 */
#define GPIO_AFRH_AFSEL9_3             (0x8UL << GPIO_AFRH_AFSEL9_Pos)          /*!< 0x00000080 */
#define GPIO_AFRH_AFSEL10_Pos          (8U)
#define GPIO_AFRH_AFSEL10_Msk          (0xFUL << GPIO_AFRH_AFSEL10_Pos)         /*!< 0x00000F00 */
#define GPIO_AFRH_AFSEL10              GPIO_AFRH_AFSEL10_Msk
#define GPIO_AFRH_AFSEL10_0            (0x1UL << GPIO_AFRH_AFSEL10_Pos)         /*!< 0x00000100 */
#define GPIO_AFRH_AFSEL10_1            (0x2UL << GPIO_AFRH_AFSEL10_Pos)         /*!< 0x00000200 */
#define GPIO_AFRH_AFSEL10_2            (0x4UL << GPIO_AFRH_AFSEL10_Pos)         /*!< 0x00000400 */
#define GPIO_AFRH_AFSEL10_3            (0x8UL << GPIO_AFRH_AFSEL10_Pos)         /*!< 0x00000800 */
#define GPIO_AFRH_AFSEL11_Pos          (12U)
#define GPIO_AFRH_AFSEL11_Msk          (0xFUL << GPIO_AFRH_AFSEL11_Pos)         /*!< 0x0000F000 */
#define GPIO_AFRH_AFSEL11              GPIO_AFRH_AFSEL11_Msk
#define GPIO_AFRH_AFSEL11_0            (0x1UL << GPIO_AFRH_AFSEL11_Pos)         /*!< 0x00001000 */
#define GPIO_AFRH_AFSEL11_1            (0x2UL << GPIO_AFRH_AFSEL11_Pos)         /*!< 0x00002000 */
#define GPIO_AFRH_AFSEL11_2            (0x4UL << GPIO_AFRH_AFSEL11_Pos)         /*!< 0x00004000 */
#define GPIO_AFRH_AFSEL11_3            (0x8UL << GPIO_AFRH_AFSEL11_Pos)         /*!< 0x00008000 */
#define GPIO_AFRH_AFSEL12_Pos          (16U)
#define GPIO_AFRH_AFSEL12_Msk          (0xFUL << GPIO_AFRH_AFSEL12_Pos)         /*!< 0x000F0000 */
#define GPIO_AFRH_AFSEL12              GPIO_AFRH_AFSEL12_Msk
#define GPIO_AFRH_AFSEL12_0            (0x1UL << GPIO_AFRH_AFSEL12_Pos)         /*!< 0x00010000 */
#define GPIO_AFRH_AFSEL12_1            (0x2UL << GPIO_AFRH_AFSEL12_Pos)         /*!< 0x00020000 */
#define GPIO_AFRH_AFSEL12_2            (0x4UL << GPIO_AFRH_AFSEL12_Pos)         /*!< 0x00040000 */
#define GPIO_AFRH_AFSEL12_3            (0x8UL << GPIO_AFRH_AFSEL12_Pos)         /*!< 0x00080000 */
#define GPIO_AFRH_AFSEL13_Pos          (20U)
#define GPIO_AFRH_AFSEL13_Msk          (0xFUL << GPIO_AFRH_AFSEL13_Pos)         /*!< 0x00F00000 */
#define GPIO_AFRH_AFSEL13              GPIO_AFRH_AFSEL13_Msk
#define GPIO_AFRH_AFSEL13_0            (0x1UL << GPIO_AFRH_AFSEL13_Pos)         /*!< 0x00100000 */
#define GPIO_AFRH_AFSEL13_1            (0x2UL << GPIO_AFRH_AFSEL13_Pos)         /*!< 0x00200000 */
#define GPIO_AFRH_AFSEL13_2            (0x4UL << GPIO_AFRH_AFSEL13_Pos)         /*!< 0x00400000 */
#define GPIO_AFRH_AFSEL13_3            (0x8UL << GPIO_AFRH_AFSEL13_Pos)         /*!< 0x00800000 */
#define GPIO_AFRH_AFSEL14_Pos          (24U)
#define GPIO_AFRH_AFSEL14_Msk          (0xFUL << GPIO_AFRH_AFSEL14_Pos)         /*!< 0x0F000000 */
#define GPIO_AFRH_AFSEL14              GPIO_AFRH_AFSEL14_Msk
#define GPIO_AFRH_AFSEL14_0            (0x1UL << GPIO_AFRH_AFSEL14_Pos)         /*!< 0x01000000 */
#define GPIO_AFRH_AFSEL14_1            (0x2UL << GPIO_AFRH_AFSEL14_Pos)         /*!< 0x02000000 */
#define GPIO_AFRH_AFSEL14_2            (0x4UL << GPIO_AFRH_AFSEL14_Pos)         /*!< 0x04000000 */
#define GPIO_AFRH_AFSEL14_3            (0x8UL << GPIO_AFRH_AFSEL14_Pos)         /*!< 0x08000000 */
#define GPIO_AFRH_AFSEL15_Pos          (28U)
#define GPIO_AFRH_AFSEL15_Msk          (0xFUL << GPIO_AFRH_AFSEL15_Pos)         /*!< 0xF0000000 */
#define GPIO_AFRH_AFSEL15              GPIO_AFRH_AFSEL15_Msk
#define GPIO_AFRH_AFSEL15_0            (0x1UL << GPIO_AFRH_AFSEL15_Pos)         /*!< 0x10000000 */
#define GPIO_AFRH_AFSEL15_1            (0x2UL << GPIO_AFRH_AFSEL15_Pos)         /*!< 0x20000000 */
#define GPIO_AFRH_AFSEL15_2            (0x4UL << GPIO_AFRH_AFSEL15_Pos)         /*!< 0x40000000 */
#define GPIO_AFRH_AFSEL15_3            (0x8UL << GPIO_AFRH_AFSEL15_Pos)         /*!< 0x80000000 */

/******************  Bits definition for GPIO_BRR register  ******************/
#define GPIO_BRR_BR0_Pos               (0U)
#define GPIO_BRR_BR0_Msk               (0x1UL << GPIO_BRR_BR0_Pos)              /*!< 0x00000001 */
#define GPIO_BRR_BR0                   GPIO_BRR_BR0_Msk
#define GPIO_BRR_BR1_Pos               (1U)
#define GPIO_BRR_BR1_Msk               (0x1UL << GPIO_BRR_BR1_Pos)              /*!< 0x00000002 */
#define GPIO_BRR_BR1                   GPIO_BRR_BR1_Msk
#define GPIO_BRR_BR2_Pos               (2U)
#define GPIO_BRR_BR2_Msk               (0x1UL << GPIO_BRR_BR2_Pos)              /*!< 0x00000004 */
#define GPIO_BRR_BR2                   GPIO_BRR_BR2_Msk
#define GPIO_BRR_BR3_Pos               (3U)
#define GPIO_BRR_BR3_Msk               (0x1UL << GPIO_BRR_BR3_Pos)              /*!< 0x00000008 */
#define GPIO_BRR_BR3                   GPIO_BRR_BR3_Msk
#define GPIO_BRR_BR4_Pos               (4U)
#define GPIO_BRR_BR4_Msk               (0x1UL << GPIO_BRR_BR4_Pos)              /*!< 0x00000010 */
#define GPIO_BRR_BR4                   GPIO_BRR_BR4_Msk
#define GPIO_BRR_BR5_Pos               (5U)
#define GPIO_BRR_BR5_Msk               (0x1UL << GPIO_BRR_BR5_Pos)              /*!< 0x00000020 */
#define GPIO_BRR_BR5                   GPIO_BRR_BR5_Msk
#define GPIO_BRR_BR6_Pos               (6U)
#define GPIO_BRR_BR6_Msk               (0x1UL << GPIO_BRR_BR6_Pos)              /*!< 0x00000040 */
#define GPIO_BRR_BR6                   GPIO_BRR_BR6_Msk
#define GPIO_BRR_BR7_Pos               (7U)
#define GPIO_BRR_BR7_Msk               (0x1UL << GPIO_BRR_BR7_Pos)              /*!< 0x00000080 */
#define GPIO_BRR_BR7                   GPIO_BRR_BR7_Msk
#define GPIO_BRR_BR8_Pos               (8U)
#define GPIO_BRR_BR8_Msk               (0x1UL << GPIO_BRR_BR8_Pos)              /*!< 0x00000100 */
#define GPIO_BRR_BR8                   GPIO_BRR_BR8_Msk
#define GPIO_BRR_BR9_Pos               (9U)
#define GPIO_BRR_BR9_Msk               (0x1UL << GPIO_BRR_BR9_Pos)              /*!< 0x00000200 */
#define GPIO_BRR_BR9                   GPIO_BRR_BR9_Msk
#define GPIO_BRR_BR10_Pos              (10U)
#define GPIO_BRR_BR10_Msk              (0x1UL << GPIO_BRR_BR10_Pos)             /*!< 0x00000400 */
#define GPIO_BRR_BR10                  GPIO_BRR_BR10_Msk
#define GPIO_BRR_BR11_Pos              (11U)
#define GPIO_BRR_BR11_Msk              (0x1UL << GPIO_BRR_BR11_Pos)             /*!< 0x00000800 */
#define GPIO_BRR_BR11                  GPIO_BRR_BR11_Msk
#define GPIO_BRR_BR12_Pos              (12U)
#define GPIO_BRR_BR12_Msk              (0x1UL << GPIO_BRR_BR12_Pos)             /*!< 0x00001000 */
#define GPIO_BRR_BR12                  GPIO_BRR_BR12_Msk
#define GPIO_BRR_BR13_Pos              (13U)
#define GPIO_BRR_BR13_Msk              (0x1UL << GPIO_BRR_BR13_Pos)             /*!< 0x00002000 */
#define GPIO_BRR_BR13                  GPIO_BRR_BR13_Msk
#define GPIO_BRR_BR14_Pos              (14U)
#define GPIO_BRR_BR14_Msk              (0x1UL << GPIO_BRR_BR14_Pos)             /*!< 0x00004000 */
#define GPIO_BRR_BR14                  GPIO_BRR_BR14_Msk
#define GPIO_BRR_BR15_Pos              (15U)
#define GPIO_BRR_BR15_Msk              (0x1UL << GPIO_BRR_BR15_Pos)             /*!< 0x00008000 */
#define GPIO_BRR_BR15                  GPIO_BRR_BR15_Msk


/********************************************************************************************************************/
/********************************* I2C ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for I2C_CR1 register **********************************************/
#define I2C_CR1_PE_Pos                            (0U)
#define I2C_CR1_PE_Msk                            (0x1UL<<I2C_CR1_PE_Pos)                           /*!< 0x00000001 */
#define I2C_CR1_PE                                I2C_CR1_PE_Msk                                    /*!< Peripheral enable */
#define I2C_CR1_SMBUS_Pos                         (1U)
#define I2C_CR1_SMBUS_Msk                         (0x1UL<<I2C_CR1_SMBUS_Pos)                        /*!< 0x00000002 */
#define I2C_CR1_SMBUS                             I2C_CR1_SMBUS_Msk                                 /*!< SMBus mode */
#define I2C_CR1_WUPEN_Pos                         (2U)
#define I2C_CR1_WUPEN_Msk                         (0x1UL<<I2C_CR1_WUPEN_Pos)                        /*!< 0x00000004 */
#define I2C_CR1_WUPEN                             I2C_CR1_WUPEN_Msk                                 /*!< Wakeup from Stop mode enable */
#define I2C_CR1_SMBTYPE_Pos                       (3U)
#define I2C_CR1_SMBTYPE_Msk                       (0x1UL<<I2C_CR1_SMBTYPE_Pos)                      /*!< 0x00000008 */
#define I2C_CR1_SMBTYPE                           I2C_CR1_SMBTYPE_Msk                               /*!< SMBus type */
#define I2C_CR1_ENARP_Pos                         (4U)
#define I2C_CR1_ENARP_Msk                         (0x1UL<<I2C_CR1_ENARP_Pos)                        /*!< 0x00000010 */
#define I2C_CR1_ENARP                             I2C_CR1_ENARP_Msk                                 /*!< ARP enable */
#define I2C_CR1_ENPEC_Pos                         (5U)
#define I2C_CR1_ENPEC_Msk                         (0x1UL<<I2C_CR1_ENPEC_Pos)                        /*!< 0x00000020 */
#define I2C_CR1_ENPEC                             I2C_CR1_ENPEC_Msk                                 /*!< PEC enable */
#define I2C_CR1_ENGC_Pos                          (6U)
#define I2C_CR1_ENGC_Msk                          (0x1UL<<I2C_CR1_ENGC_Pos)                         /*!< 0x00000040 */
#define I2C_CR1_ENGC                              I2C_CR1_ENGC_Msk                                  /*!< General call enable */
#define I2C_CR1_NOSTRETCH_Pos                     (7U)
#define I2C_CR1_NOSTRETCH_Msk                     (0x1UL<<I2C_CR1_NOSTRETCH_Pos)                    /*!< 0x00000080 */
#define I2C_CR1_NOSTRETCH                         I2C_CR1_NOSTRETCH_Msk                             /*!< Clock stretching disable (Slave mode) */
#define I2C_CR1_START_Pos                         (8U)
#define I2C_CR1_START_Msk                         (0x1UL<<I2C_CR1_START_Pos)                        /*!< 0x00000100 */
#define I2C_CR1_START                             I2C_CR1_START_Msk                                 /*!< Start generation */
#define I2C_CR1_STOP_Pos                          (9U)
#define I2C_CR1_STOP_Msk                          (0x1UL<<I2C_CR1_STOP_Pos)                         /*!< 0x00000200 */
#define I2C_CR1_STOP                              I2C_CR1_STOP_Msk                                  /*!< Stop generation */
#define I2C_CR1_ACK_Pos                           (10U)
#define I2C_CR1_ACK_Msk                           (0x1UL<<I2C_CR1_ACK_Pos)                          /*!< 0x00000400 */
#define I2C_CR1_ACK                               I2C_CR1_ACK_Msk                                   /*!< Acknowledge enable */
#define I2C_CR1_POS_Pos                           (11U)
#define I2C_CR1_POS_Msk                           (0x1UL<<I2C_CR1_POS_Pos)                          /*!< 0x00000800 */
#define I2C_CR1_POS                               I2C_CR1_POS_Msk                                   /*!< Acknowledge(for data reception) */
#define I2C_CR1_PEC_Pos                           (12U)
#define I2C_CR1_PEC_Msk                           (0x1UL<<I2C_CR1_PEC_Pos)                          /*!< 0x00001000 */
#define I2C_CR1_PEC                               I2C_CR1_PEC_Msk                                   /*!< Packet error checking */
#define I2C_CR1_ALERT_Pos                         (13U)
#define I2C_CR1_ALERT_Msk                         (0x1UL<<I2C_CR1_ALERT_Pos)                        /*!< 0x00002000 */
#define I2C_CR1_ALERT                             I2C_CR1_ALERT_Msk                                 /*!< SMBus alert */
#define I2C_CR1_SWRST_Pos                         (15U)
#define I2C_CR1_SWRST_Msk                         (0x1UL<<I2C_CR1_SWRST_Pos)                        /*!< 0x00008000 */
#define I2C_CR1_SWRST                             I2C_CR1_SWRST_Msk                                 /*!< Software reset */
#define I2C_CR1_WKUP_DIV_Pos                      (16U)
#define I2C_CR1_WKUP_DIV_Msk                      (0x3UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00030000 */
#define I2C_CR1_WKUP_DIV                          I2C_CR1_WKUP_DIV_Msk                              /*!< PCLK division */
#define I2C_CR1_WKUP_DIV_0                        (0x1UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00010000 */
#define I2C_CR1_WKUP_DIV_1                        (0x2UL<<I2C_CR1_WKUP_DIV_Pos)                     /*!< 0x00020000 */
#define I2C_CR1_WKUP_CNT_Pos                      (18U)
#define I2C_CR1_WKUP_CNT_Msk                      (0x3UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x000C0000 */
#define I2C_CR1_WKUP_CNT                          I2C_CR1_WKUP_CNT_Msk                              /*!< Timeout count number */
#define I2C_CR1_WKUP_CNT_0                        (0x1UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x00040000 */
#define I2C_CR1_WKUP_CNT_1                        (0x2UL<<I2C_CR1_WKUP_CNT_Pos)                     /*!< 0x00080000 */

/********************************* Bit definition for I2C_CR2 register **********************************************/
#define I2C_CR2_FREQ_Pos                          (0U)
#define I2C_CR2_FREQ_Msk                          (0x7FUL<<I2C_CR2_FREQ_Pos)                        /*!< 0x0000007F */
#define I2C_CR2_FREQ                              I2C_CR2_FREQ_Msk                                  /*!< Peripheral clock frequency */
#define I2C_CR2_ITERREN_Pos                       (8U)
#define I2C_CR2_ITERREN_Msk                       (0x1UL<<I2C_CR2_ITERREN_Pos)                      /*!< 0x00000100 */
#define I2C_CR2_ITERREN                           I2C_CR2_ITERREN_Msk                               /*!< Error interrupt enable */
#define I2C_CR2_ITEVTEN_Pos                       (9U)
#define I2C_CR2_ITEVTEN_Msk                       (0x1UL<<I2C_CR2_ITEVTEN_Pos)                      /*!< 0x00000200 */
#define I2C_CR2_ITEVTEN                           I2C_CR2_ITEVTEN_Msk                               /*!< Event interrupt enable */
#define I2C_CR2_ITBUFEN_Pos                       (10U)
#define I2C_CR2_ITBUFEN_Msk                       (0x1UL<<I2C_CR2_ITBUFEN_Pos)                      /*!< 0x00000400 */
#define I2C_CR2_ITBUFEN                           I2C_CR2_ITBUFEN_Msk                               /*!< Buffer interrupt enable */
#define I2C_CR2_DMAEN_Pos                         (11U)
#define I2C_CR2_DMAEN_Msk                         (0x1UL<<I2C_CR2_DMAEN_Pos)                        /*!< 0x00000800 */
#define I2C_CR2_DMAEN                             I2C_CR2_DMAEN_Msk                                 /*!< DMA requests enable */
#define I2C_CR2_LAST_Pos                          (12U)
#define I2C_CR2_LAST_Msk                          (0x1UL<<I2C_CR2_LAST_Pos)                         /*!< 0x00001000 */
#define I2C_CR2_LAST                              I2C_CR2_LAST_Msk                                  /*!< DMA last transfer */

/********************************* Bit definition for I2C_OAR1 register *********************************************/
#define I2C_OAR1_ADD_Pos                          (0U)
#define I2C_OAR1_ADD_Msk                          (0x3FFUL<<I2C_OAR1_ADD_Pos)                       /*!< 0x000003FF */
#define I2C_OAR1_ADD                              I2C_OAR1_ADD_Msk                                  /*!< Interface address */
#define I2C_OAR1_ADDMODE_Pos                      (15U)
#define I2C_OAR1_ADDMODE_Msk                      (0x1UL<<I2C_OAR1_ADDMODE_Pos)                     /*!< 0x00008000 */
#define I2C_OAR1_ADDMODE                          I2C_OAR1_ADDMODE_Msk                              /*!< Addressing mode (slave mode) */

/********************************* Bit definition for I2C_OAR2 register *********************************************/
#define I2C_OAR2_ENDUAL_Pos                       (0U)
#define I2C_OAR2_ENDUAL_Msk                       (0x1UL<<I2C_OAR2_ENDUAL_Pos)                      /*!< 0x00000001 */
#define I2C_OAR2_ENDUAL                           I2C_OAR2_ENDUAL_Msk                               /*!< Dual addressing mode enable */
#define I2C_OAR2_ADD2_Pos                         (1U)
#define I2C_OAR2_ADD2_Msk                         (0x7FUL<<I2C_OAR2_ADD2_Pos)                       /*!< 0x000000FE */
#define I2C_OAR2_ADD2                             I2C_OAR2_ADD2_Msk                                 /*!< Interface address */
#define I2C_OAR2_OA2MSK_Pos                       (8U)
#define I2C_OAR2_OA2MSK_Msk                       (0x7UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000700 */
#define I2C_OAR2_OA2MSK                           I2C_OAR2_OA2MSK_Msk                               /*!< Own Address 2 masks */
#define I2C_OAR2_OA2MSK_0                         (0x1UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000100 */
#define I2C_OAR2_OA2MSK_1                         (0x2UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000200 */
#define I2C_OAR2_OA2MSK_2                         (0x4UL<<I2C_OAR2_OA2MSK_Pos)                      /*!< 0x00000400 */

/********************************* Bit definition for I2C_DR register ***********************************************/
#define I2C_DR_DR_Pos                             (0U)
#define I2C_DR_DR_Msk                             (0xFFUL<<I2C_DR_DR_Pos)                           /*!< 0x000000FF */
#define I2C_DR_DR                                 I2C_DR_DR_Msk                                     /*!< 8-bit data register */

/********************************* Bit definition for I2C_SR1 register **********************************************/
#define I2C_SR1_SB_Pos                            (0U)
#define I2C_SR1_SB_Msk                            (0x1UL<<I2C_SR1_SB_Pos)                           /*!< 0x00000001 */
#define I2C_SR1_SB                                I2C_SR1_SB_Msk                                    /*!< Start bit (Master mode) */
#define I2C_SR1_ADDR_Pos                          (1U)
#define I2C_SR1_ADDR_Msk                          (0x1UL<<I2C_SR1_ADDR_Pos)                         /*!< 0x00000002 */
#define I2C_SR1_ADDR                              I2C_SR1_ADDR_Msk                                  /*!< Address sent (master mode)/matched (slave mode) */
#define I2C_SR1_BTF_Pos                           (2U)
#define I2C_SR1_BTF_Msk                           (0x1UL<<I2C_SR1_BTF_Pos)                          /*!< 0x00000004 */
#define I2C_SR1_BTF                               I2C_SR1_BTF_Msk                                   /*!< Byte transfer finished */
#define I2C_SR1_ADD10_Pos                         (3U)
#define I2C_SR1_ADD10_Msk                         (0x1UL<<I2C_SR1_ADD10_Pos)                        /*!< 0x00000008 */
#define I2C_SR1_ADD10                             I2C_SR1_ADD10_Msk                                 /*!< 10-bit header sent (Master mode) */
#define I2C_SR1_STOPF_Pos                         (4U)
#define I2C_SR1_STOPF_Msk                         (0x1UL<<I2C_SR1_STOPF_Pos)                        /*!< 0x00000010 */
#define I2C_SR1_STOPF                             I2C_SR1_STOPF_Msk                                 /*!< Stop detection (slave mode) */
#define I2C_SR1_RXNE_Pos                          (6U)
#define I2C_SR1_RXNE_Msk                          (0x1UL<<I2C_SR1_RXNE_Pos)                         /*!< 0x00000040 */
#define I2C_SR1_RXNE                              I2C_SR1_RXNE_Msk                                  /*!< Data register not empty (receivers) */
#define I2C_SR1_TXE_Pos                           (7U)
#define I2C_SR1_TXE_Msk                           (0x1UL<<I2C_SR1_TXE_Pos)                          /*!< 0x00000080 */
#define I2C_SR1_TXE                               I2C_SR1_TXE_Msk                                   /*!< Data register empty (transmitters) */
#define I2C_SR1_BERR_Pos                          (8U)
#define I2C_SR1_BERR_Msk                          (0x1UL<<I2C_SR1_BERR_Pos)                         /*!< 0x00000100 */
#define I2C_SR1_BERR                              I2C_SR1_BERR_Msk                                  /*!< Bus error */
#define I2C_SR1_ARLO_Pos                          (9U)
#define I2C_SR1_ARLO_Msk                          (0x1UL<<I2C_SR1_ARLO_Pos)                         /*!< 0x00000200 */
#define I2C_SR1_ARLO                              I2C_SR1_ARLO_Msk                                  /*!< Arbitration lost (master mode) */
#define I2C_SR1_AF_Pos                            (10U)
#define I2C_SR1_AF_Msk                            (0x1UL<<I2C_SR1_AF_Pos)                           /*!< 0x00000400 */
#define I2C_SR1_AF                                I2C_SR1_AF_Msk                                    /*!< Acknowledge failure */
#define I2C_SR1_OVR_Pos                           (11U)
#define I2C_SR1_OVR_Msk                           (0x1UL<<I2C_SR1_OVR_Pos)                          /*!< 0x00000800 */
#define I2C_SR1_OVR                               I2C_SR1_OVR_Msk                                   /*!< Overrun/Underrun */
#define I2C_SR1_PECERR_Pos                        (12U)
#define I2C_SR1_PECERR_Msk                        (0x1UL<<I2C_SR1_PECERR_Pos)                       /*!< 0x00001000 */
#define I2C_SR1_PECERR                            I2C_SR1_PECERR_Msk                                /*!< PEC Error in reception */
#define I2C_SR1_TIMEOUT_Pos                       (14U)
#define I2C_SR1_TIMEOUT_Msk                       (0x1UL<<I2C_SR1_TIMEOUT_Pos)                      /*!< 0x00004000 */
#define I2C_SR1_TIMEOUT                           I2C_SR1_TIMEOUT_Msk                               /*!< Timeout or Tlow error */
#define I2C_SR1_SMBALERT_Pos                      (15U)
#define I2C_SR1_SMBALERT_Msk                      (0x1UL<<I2C_SR1_SMBALERT_Pos)                     /*!< 0x00008000 */
#define I2C_SR1_SMBALERT                          I2C_SR1_SMBALERT_Msk                              /*!< SMBus alert */

/********************************* Bit definition for I2C_SR2 register **********************************************/
#define I2C_SR2_MSL_Pos                           (0U)
#define I2C_SR2_MSL_Msk                           (0x1UL<<I2C_SR2_MSL_Pos)                          /*!< 0x00000001 */
#define I2C_SR2_MSL                               I2C_SR2_MSL_Msk                                   /*!< Master/slave */
#define I2C_SR2_BUSY_Pos                          (1U)
#define I2C_SR2_BUSY_Msk                          (0x1UL<<I2C_SR2_BUSY_Pos)                         /*!< 0x00000002 */
#define I2C_SR2_BUSY                              I2C_SR2_BUSY_Msk                                  /*!< Bus busy */
#define I2C_SR2_TRA_Pos                           (2U)
#define I2C_SR2_TRA_Msk                           (0x1UL<<I2C_SR2_TRA_Pos)                          /*!< 0x00000004 */
#define I2C_SR2_TRA                               I2C_SR2_TRA_Msk                                   /*!< Transmitter/receiver */
#define I2C_SR2_GENCALL_Pos                       (4U)
#define I2C_SR2_GENCALL_Msk                       (0x1UL<<I2C_SR2_GENCALL_Pos)                      /*!< 0x00000010 */
#define I2C_SR2_GENCALL                           I2C_SR2_GENCALL_Msk                               /*!< General call address (Slave mode) */
#define I2C_SR2_SMBDEFAULT_Pos                    (5U)
#define I2C_SR2_SMBDEFAULT_Msk                    (0x1UL<<I2C_SR2_SMBDEFAULT_Pos)                   /*!< 0x00000020 */
#define I2C_SR2_SMBDEFAULT                        I2C_SR2_SMBDEFAULT_Msk                            /*!< SMBus device default address (Slave mode) */
#define I2C_SR2_SMBHOST_Pos                       (6U)
#define I2C_SR2_SMBHOST_Msk                       (0x1UL<<I2C_SR2_SMBHOST_Pos)                      /*!< 0x00000040 */
#define I2C_SR2_SMBHOST                           I2C_SR2_SMBHOST_Msk                               /*!< SMBus host header (Slave mode) */
#define I2C_SR2_DUALF_Pos                         (7U)
#define I2C_SR2_DUALF_Msk                         (0x1UL<<I2C_SR2_DUALF_Pos)                        /*!< 0x00000080 */
#define I2C_SR2_DUALF                             I2C_SR2_DUALF_Msk                                 /*!< Dual flag (Slave mode) */
#define I2C_SR2_PEC_Pos                           (8U)
#define I2C_SR2_PEC_Msk                           (0xFFUL<<I2C_SR2_PEC_Pos)                         /*!< 0x0000FF00 */
#define I2C_SR2_PEC                               I2C_SR2_PEC_Msk                                   /*!< Packet error checking register */

/********************************* Bit definition for I2C_CCR register **********************************************/
#define I2C_CCR_CCR_Pos                           (0U)
#define I2C_CCR_CCR_Msk                           (0xFFFUL<<I2C_CCR_CCR_Pos)                        /*!< 0x00000FFF */
#define I2C_CCR_CCR                               I2C_CCR_CCR_Msk                                   /*!< Clock control register in Fm/Sm mode (Master mode) */
#define I2C_CCR_FP_Pos                            (13U)
#define I2C_CCR_FP_Msk                            (0x1UL<<I2C_CCR_FP_Pos)                           /*!< 0x00002000 */
#define I2C_CCR_FP                                I2C_CCR_FP_Msk                                    /*!< I2C master mode selection */
#define I2C_CCR_DUTY_Pos                          (14U)
#define I2C_CCR_DUTY_Msk                          (0x1UL<<I2C_CCR_DUTY_Pos)                         /*!< 0x00004000 */
#define I2C_CCR_DUTY                              I2C_CCR_DUTY_Msk                                  /*!< Fm mode duty cycle */
#define I2C_CCR_FS_Pos                            (15U)
#define I2C_CCR_FS_Msk                            (0x1UL<<I2C_CCR_FS_Pos)                           /*!< 0x00008000 */
#define I2C_CCR_FS                                I2C_CCR_FS_Msk                                    /*!< I2C master mode selection */

/********************************* Bit definition for I2C_TRISE register ********************************************/
#define I2C_TRISE_TRISE_Pos                       (0U)
#define I2C_TRISE_TRISE_Msk                       (0x7FUL<<I2C_TRISE_TRISE_Pos)                     /*!< 0x0000007F */
#define I2C_TRISE_TRISE                           I2C_TRISE_TRISE_Msk                               /*!< Maximum rise time in Fm+/Fm/Sm mode (Master mode) */
#define I2C_TRISE_TDATAHOLD_Pos                   (7U)
#define I2C_TRISE_TDATAHOLD_Msk                   (0x1FUL<<I2C_TRISE_TDATAHOLD_Pos)                 /*!< 0x00000F80 */
#define I2C_TRISE_TDATAHOLD                       I2C_TRISE_TDATAHOLD_Msk                           /*!< Data hold time */
#define I2C_TRISE_TDATAHOLD_0                     (0x1UL<<I2C_TRISE_TDATAHOLD_Pos)                  /*!< 0x00000080 */
#define I2C_TRISE_TDATAHOLD_1                     (0x2UL<<I2C_TRISE_TDATAHOLD_Pos)                  /*!< 0x00000100 */
#define I2C_TRISE_TDATAHOLD_2                     (0x4UL<<I2C_TRISE_TDATAHOLD_Pos)                  /*!< 0x00000200 */
#define I2C_TRISE_TDATAHOLD_3                     (0x8UL<<I2C_TRISE_TDATAHOLD_Pos)                  /*!< 0x00000400 */
#define I2C_TRISE_TDATAHOLD_4                     (0x10UL<<I2C_TRISE_TDATAHOLD_Pos)                 /*!< 0x00000800 */
#define I2C_TRISE_TDATAHOLD_SEL_Pos               (12U)
#define I2C_TRISE_TDATAHOLD_SEL_Msk               (0x1UL<<I2C_TRISE_TDATAHOLD_SEL_Pos)              /*!< 0x00001000 */
#define I2C_TRISE_TDATAHOLD_SEL                   I2C_TRISE_TDATAHOLD_SEL_Msk                       /*!< Data hold time select */

/********************************* Bit definition for I2C_TIMEOUTR register *****************************************/
#define I2C_TIMEOUTR_TIMEOUTA_Pos                 (0U)
#define I2C_TIMEOUTR_TIMEOUTA_Msk                 (0xFFFUL<<I2C_TIMEOUTR_TIMEOUTA_Pos)              /*!< 0x00000FFF */
#define I2C_TIMEOUTR_TIMEOUTA                     I2C_TIMEOUTR_TIMEOUTA_Msk                         /*!< Bus Timeout A */
#define I2C_TIMEOUTR_TIDLE_Pos                    (12U)
#define I2C_TIMEOUTR_TIDLE_Msk                    (0x1UL<<I2C_TIMEOUTR_TIDLE_Pos)                   /*!< 0x00001000 */
#define I2C_TIMEOUTR_TIDLE                        I2C_TIMEOUTR_TIDLE_Msk                            /*!< Idle clock timeout detection */
#define I2C_TIMEOUTR_TIMEOUTEN_Pos                (15U)
#define I2C_TIMEOUTR_TIMEOUTEN_Msk                (0x1UL<<I2C_TIMEOUTR_TIMEOUTEN_Pos)               /*!< 0x00008000 */
#define I2C_TIMEOUTR_TIMEOUTEN                    I2C_TIMEOUTR_TIMEOUTEN_Msk                        /*!< Clock timeout enable */
#define I2C_TIMEOUTR_TIMEOUTB_Pos                 (16U)
#define I2C_TIMEOUTR_TIMEOUTB_Msk                 (0xFFFUL<<I2C_TIMEOUTR_TIMEOUTB_Pos)              /*!< 0x0FFF0000 */
#define I2C_TIMEOUTR_TIMEOUTB                     I2C_TIMEOUTR_TIMEOUTB_Msk                         /*!< Bus timeout B */
#define I2C_TIMEOUTR_TEXTEN_Pos                   (31U)
#define I2C_TIMEOUTR_TEXTEN_Msk                   (0x1UL<<I2C_TIMEOUTR_TEXTEN_Pos)                  /*!< 0x80000000 */
#define I2C_TIMEOUTR_TEXTEN                       I2C_TIMEOUTR_TEXTEN_Msk                           /*!< Extended clock timeout enable */


/********************************************************************************************************************/
/********************************* IWDG *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for IWDG_KR register **********************************************/
#define IWDG_KR_KEY_Pos                           (0U)
#define IWDG_KR_KEY_Msk                           (0xFFFFUL<<IWDG_KR_KEY_Pos)                       /*!< 0x0000FFFF */
#define IWDG_KR_KEY                               IWDG_KR_KEY_Msk                                   /*!< IWDG KEY */

/********************************* Bit definition for IWDG_PR register **********************************************/
#define IWDG_PR_PR_Pos                            (0U)
#define IWDG_PR_PR_Msk                            (0x7UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000007 */
#define IWDG_PR_PR                                IWDG_PR_PR_Msk                                    /*!< IWDG preload */
#define IWDG_PR_PR_0                              (0x1UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000001 */
#define IWDG_PR_PR_1                              (0x2UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000002 */
#define IWDG_PR_PR_2                              (0x4UL<<IWDG_PR_PR_Pos)                           /*!< 0x00000004 */

/********************************* Bit definition for IWDG_RLR register *********************************************/
#define IWDG_RLR_RL_Pos                           (0U)
#define IWDG_RLR_RL_Msk                           (0xFFFUL<<IWDG_RLR_RL_Pos)                        /*!< 0x00000FFF */
#define IWDG_RLR_RL                               IWDG_RLR_RL_Msk                                   /*!< IWDG reload */

/********************************* Bit definition for IWDG_SR register **********************************************/
#define IWDG_SR_PVU_Pos                           (0U)
#define IWDG_SR_PVU_Msk                           (0x1UL<<IWDG_SR_PVU_Pos)                          /*!< 0x00000001 */
#define IWDG_SR_PVU                               IWDG_SR_PVU_Msk                                   /*!< Watchdog prescaler value update */
#define IWDG_SR_RVU_Pos                           (1U)
#define IWDG_SR_RVU_Msk                           (0x1UL<<IWDG_SR_RVU_Pos)                          /*!< 0x00000002 */
#define IWDG_SR_RVU                               IWDG_SR_RVU_Msk                                   /*!< Watchdog counter reload value update */


/********************************************************************************************************************/
/********************************* LCD ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LCD_CR register ***********************************************/
#define LCD_CR_EN_Pos                             (0U)
#define LCD_CR_EN_Msk                             (0x1UL<<LCD_CR_EN_Pos)                            /*!< 0x00000001 */
#define LCD_CR_EN                                 LCD_CR_EN_Msk                                     
#define LCD_CR_LCDCLK_Pos                         (1U)
#define LCD_CR_LCDCLK_Msk                         (0x3UL<<LCD_CR_LCDCLK_Pos)                        /*!< 0x00000006 */
#define LCD_CR_LCDCLK                             LCD_CR_LCDCLK_Msk
#define LCD_CR_LCDCLK_0                           (0x1UL<<LCD_CR_LCDCLK_Pos)                        /*!< 0x00000002 */
#define LCD_CR_LCDCLK_1                           (0x2UL<<LCD_CR_LCDCLK_Pos)                        /*!< 0x00000004 */
#define LCD_CR_BIAS_Pos                           (4U)
#define LCD_CR_BIAS_Msk                           (0x3UL<<LCD_CR_BIAS_Pos)                          /*!< 0x00000030 */
#define LCD_CR_BIAS                               LCD_CR_BIAS_Msk
#define LCD_CR_BIAS_0                             (0x1UL<<LCD_CR_BIAS_Pos)                          /*!< 0x00000010 */
#define LCD_CR_BIAS_1                             (0x2UL<<LCD_CR_BIAS_Pos)                          /*!< 0x00000020 */
#define LCD_CR_DUTY_Pos                           (6U)
#define LCD_CR_DUTY_Msk                           (0x7UL<<LCD_CR_DUTY_Pos)                          /*!< 0x000001C0 */
#define LCD_CR_DUTY                               LCD_CR_DUTY_Msk
#define LCD_CR_DUTY_0                             (0x1UL<<LCD_CR_DUTY_Pos)                          /*!< 0x00000040 */
#define LCD_CR_DUTY_1                             (0x2UL<<LCD_CR_DUTY_Pos)                          /*!< 0x00000080 */
#define LCD_CR_DUTY_2                             (0x4UL<<LCD_CR_DUTY_Pos)                          /*!< 0x00000100 */
#define LCD_CR_BSEL_Pos                           (9U)
#define LCD_CR_BSEL_Msk                           (0x3UL<<LCD_CR_BSEL_Pos)                          /*!< 0x00000600 */
#define LCD_CR_BSEL                               LCD_CR_BSEL_Msk
#define LCD_CR_BSEL_0                             (0x1UL<<LCD_CR_BSEL_Pos)                          /*!< 0x00000200 */
#define LCD_CR_BSEL_1                             (0x2UL<<LCD_CR_BSEL_Pos)                          /*!< 0x00000400 */
#define LCD_CR_CONTRAST_Pos                       (12U)
#define LCD_CR_CONTRAST_Msk                       (0xFUL<<LCD_CR_CONTRAST_Pos)                      /*!< 0x0000F000 */
#define LCD_CR_CONTRAST                           LCD_CR_CONTRAST_Msk
#define LCD_CR_CONTRAST_0                         (0x1UL<<LCD_CR_CONTRAST_Pos)                      /*!< 0x00001000 */
#define LCD_CR_CONTRAST_1                         (0x2UL<<LCD_CR_CONTRAST_Pos)                      /*!< 0x00002000 */
#define LCD_CR_CONTRAST_2                         (0x4UL<<LCD_CR_CONTRAST_Pos)                      /*!< 0x00004000 */
#define LCD_CR_CONTRAST_3                         (0x8UL<<LCD_CR_CONTRAST_Pos)                      /*!< 0x00008000 */
#define LCD_CR_VSEL_Pos                           (16U)
#define LCD_CR_VSEL_Msk                           (0x1UL<<LCD_CR_VSEL_Pos)                          /*!< 0x00010000 */
#define LCD_CR_VSEL                               LCD_CR_VSEL_Msk                                   
#define LCD_CR_VCP_SEL_Pos                        (17U)
#define LCD_CR_VCP_SEL_Msk                        (0x7UL<<LCD_CR_VCP_SEL_Pos)                       /*!< 0x000E0000 */
#define LCD_CR_VCP_SEL                            LCD_CR_VCP_SEL_Msk
#define LCD_CR_VCP_SEL_0                          (0x1UL<<LCD_CR_VCP_SEL_Pos)                       /*!< 0x00020000 */
#define LCD_CR_VCP_SEL_1                          (0x2UL<<LCD_CR_VCP_SEL_Pos)                       /*!< 0x00040000 */
#define LCD_CR_VCP_SEL_2                          (0x4UL<<LCD_CR_VCP_SEL_Pos)                       /*!< 0x00080000 */
#define LCD_CR_FCCTL_Pos                          (22U)
#define LCD_CR_FCCTL_Msk                          (0x7UL<<LCD_CR_FCCTL_Pos)                         /*!< 0x01C00000 */
#define LCD_CR_FCCTL                              LCD_CR_FCCTL_Msk
#define LCD_CR_FCCTL_0                            (0x1UL<<LCD_CR_FCCTL_Pos)                         /*!< 0x00400000 */
#define LCD_CR_FCCTL_1                            (0x2UL<<LCD_CR_FCCTL_Pos)                         /*!< 0x00800000 */
#define LCD_CR_FCCTL_2                            (0x4UL<<LCD_CR_FCCTL_Pos)                         /*!< 0x01000000 */
#define LCD_CR_DRV_MOD_Pos                        (25U)
#define LCD_CR_DRV_MOD_Msk                        (0x3UL<<LCD_CR_DRV_MOD_Pos)                       /*!< 0x06000000 */
#define LCD_CR_DRV_MOD                            LCD_CR_DRV_MOD_Msk
#define LCD_CR_DRV_MOD_0                          (0x1UL<<LCD_CR_DRV_MOD_Pos)                       /*!< 0x02000000 */
#define LCD_CR_DRV_MOD_1                          (0x2UL<<LCD_CR_DRV_MOD_Pos)                       /*!< 0x04000000 */
#define LCD_CR_CP_PUSLE_Pos                       (27U)
#define LCD_CR_CP_PUSLE_Msk                       (0x7UL<<LCD_CR_CP_PUSLE_Pos)                      /*!< 0x38000000 */
#define LCD_CR_CP_PUSLE                           LCD_CR_CP_PUSLE_Msk
#define LCD_CR_CP_PUSLE_0                         (0x1UL<<LCD_CR_CP_PUSLE_Pos)                      /*!< 0x08000000 */
#define LCD_CR_CP_PUSLE_1                         (0x2UL<<LCD_CR_CP_PUSLE_Pos)                      /*!< 0x10000000 */
#define LCD_CR_CP_PUSLE_2                         (0x4UL<<LCD_CR_CP_PUSLE_Pos)                      /*!< 0x20000000 */

/********************************* Bit definition for LCD_CSR register **********************************************/
#define LCD_CSR_BLINKCNT_Pos                      (0U)
#define LCD_CSR_BLINKCNT_Msk                      (0x3FUL<<LCD_CSR_BLINKCNT_Pos)                    /*!< 0x0000003F */
#define LCD_CSR_BLINKCNT                          LCD_CSR_BLINKCNT_Msk
#define LCD_CSR_BLINKCNT_0                        (0x1UL<<LCD_CSR_BLINKCNT_Pos)                     /*!< 0x00000001 */
#define LCD_CSR_BLINKCNT_1                        (0x2UL<<LCD_CSR_BLINKCNT_Pos)                     /*!< 0x00000002 */
#define LCD_CSR_BLINKCNT_2                        (0x4UL<<LCD_CSR_BLINKCNT_Pos)                     /*!< 0x00000004 */
#define LCD_CSR_BLINKCNT_3                        (0x8UL<<LCD_CSR_BLINKCNT_Pos)                     /*!< 0x00000008 */
#define LCD_CSR_BLINKCNT_4                        (0x10UL<<LCD_CSR_BLINKCNT_Pos)                    /*!< 0x00000010 */
#define LCD_CSR_BLINKCNT_5                        (0x20UL<<LCD_CSR_BLINKCNT_Pos)                    /*!< 0x00000020 */
#define LCD_CSR_BLINKEN_Pos                       (6U)
#define LCD_CSR_BLINKEN_Msk                       (0x1UL<<LCD_CSR_BLINKEN_Pos)                      /*!< 0x00000040 */
#define LCD_CSR_BLINKEN                           LCD_CSR_BLINKEN_Msk                               
#define LCD_CSR_MODE_Pos                          (8U)
#define LCD_CSR_MODE_Msk                          (0x1UL<<LCD_CSR_MODE_Pos)                         /*!< 0x00000100 */
#define LCD_CSR_MODE                              LCD_CSR_MODE_Msk                                  
#define LCD_CSR_DMAEN_Pos                         (9U)
#define LCD_CSR_DMAEN_Msk                         (0x1UL<<LCD_CSR_DMAEN_Pos)                        /*!< 0x00000200 */
#define LCD_CSR_DMAEN                             LCD_CSR_DMAEN_Msk                                 
#define LCD_CSR_EOFIE_Pos                         (10U)
#define LCD_CSR_EOFIE_Msk                         (0x1UL<<LCD_CSR_EOFIE_Pos)                        /*!< 0x00000400 */
#define LCD_CSR_EOFIE                             LCD_CSR_EOFIE_Msk                                 
#define LCD_CSR_UDDIE_Pos                         (11U)
#define LCD_CSR_UDDIE_Msk                         (0x1UL<<LCD_CSR_UDDIE_Pos)                        /*!< 0x00000800 */
#define LCD_CSR_UDDIE                             LCD_CSR_UDDIE_Msk                                 
#define LCD_CSR_EOF_Pos                           (12U)
#define LCD_CSR_EOF_Msk                           (0x1UL<<LCD_CSR_EOF_Pos)                          /*!< 0x00001000 */
#define LCD_CSR_EOF                               LCD_CSR_EOF_Msk                                   
#define LCD_CSR_UDR_Pos                           (13U)
#define LCD_CSR_UDR_Msk                           (0x1UL<<LCD_CSR_UDR_Pos)                          /*!< 0x00002000 */
#define LCD_CSR_UDR                               LCD_CSR_UDR_Msk                                   
#define LCD_CSR_UDD_Pos                           (14U)
#define LCD_CSR_UDD_Msk                           (0x1UL<<LCD_CSR_UDD_Pos)                          /*!< 0x00004000 */
#define LCD_CSR_UDD                               LCD_CSR_UDD_Msk                                   
#define LCD_CSR_SWCAP_DRV_Pos                     (16U)
#define LCD_CSR_SWCAP_DRV_Msk                     (0x7UL<<LCD_CSR_SWCAP_DRV_Pos)                    /*!< 0x00070000 */
#define LCD_CSR_SWCAP_DRV                         LCD_CSR_SWCAP_DRV_Msk
#define LCD_CSR_SWCAP_DRV_0                       (0x1UL<<LCD_CSR_SWCAP_DRV_Pos)                    /*!< 0x00010000 */
#define LCD_CSR_SWCAP_DRV_1                       (0x2UL<<LCD_CSR_SWCAP_DRV_Pos)                    /*!< 0x00020000 */
#define LCD_CSR_SWCAP_DRV_2                       (0x4UL<<LCD_CSR_SWCAP_DRV_Pos)                    /*!< 0x00040000 */
#define LCD_CSR_SWCAP_CNT_Pos                     (19U)
#define LCD_CSR_SWCAP_CNT_Msk                     (0x7UL<<LCD_CSR_SWCAP_CNT_Pos)                    /*!< 0x00380000 */
#define LCD_CSR_SWCAP_CNT                         LCD_CSR_SWCAP_CNT_Msk
#define LCD_CSR_SWCAP_CNT_0                       (0x1UL<<LCD_CSR_SWCAP_CNT_Pos)                    /*!< 0x00080000 */
#define LCD_CSR_SWCAP_CNT_1                       (0x2UL<<LCD_CSR_SWCAP_CNT_Pos)                    /*!< 0x00100000 */
#define LCD_CSR_SWCAP_CNT_2                       (0x4UL<<LCD_CSR_SWCAP_CNT_Pos)                    /*!< 0x00200000 */

/********************************* Bit definition for LCD_INTCLR register *******************************************/
#define LCD_INTCLR_EOFC_Pos                       (10U)
#define LCD_INTCLR_EOFC_Msk                       (0x1UL<<LCD_INTCLR_EOFC_Pos)                      /*!< 0x00000400 */
#define LCD_INTCLR_EOFC                           LCD_INTCLR_EOFC_Msk                               
#define LCD_INTCLR_UDDFC_Pos                      (11U)
#define LCD_INTCLR_UDDFC_Msk                      (0x1UL<<LCD_INTCLR_UDDFC_Pos)                     /*!< 0x00000800 */
#define LCD_INTCLR_UDDFC                          LCD_INTCLR_UDDFC_Msk                              

/********************************* Bit definition for LCD_POEN0 register ********************************************/
#define LCD_POEN0_SX_Pos                          (0U)
#define LCD_POEN0_SX_Msk                          (0xFFFFFFFFUL<<LCD_POEN0_SX_Pos)                  /*!< 0xFFFFFFFF */
#define LCD_POEN0_SX                              LCD_POEN0_SX_Msk

/********************************* Bit definition for LCD_POEN1 register ********************************************/
#define LCD_POEN1_SX_Pos                          (0U)
#define LCD_POEN1_SX_Msk                          (0xFUL<<LCD_POEN1_SX_Pos)                         /*!< 0x0000000F */
#define LCD_POEN1_SX                              LCD_POEN1_SX_Msk
#define LCD_POEN1_SX_0                            (0x1UL<<LCD_POEN1_SX_Pos)                         /*!< 0x00000001 */
#define LCD_POEN1_SX_1                            (0x2UL<<LCD_POEN1_SX_Pos)                         /*!< 0x00000002 */
#define LCD_POEN1_SX_2                            (0x4UL<<LCD_POEN1_SX_Pos)                         /*!< 0x00000004 */
#define LCD_POEN1_SX_3                            (0x8UL<<LCD_POEN1_SX_Pos)                         /*!< 0x00000008 */
#define LCD_POEN1_SXCY_Pos                        (4U)
#define LCD_POEN1_SXCY_Msk                        (0xFUL<<LCD_POEN1_SXCY_Pos)                       /*!< 0x000000F0 */
#define LCD_POEN1_SXCY                            LCD_POEN1_SXCY_Msk
#define LCD_POEN1_SXCY_0                          (0x1UL<<LCD_POEN1_SXCY_Pos)                       /*!< 0x00000010 */
#define LCD_POEN1_SXCY_1                          (0x2UL<<LCD_POEN1_SXCY_Pos)                       /*!< 0x00000020 */
#define LCD_POEN1_SXCY_2                          (0x4UL<<LCD_POEN1_SXCY_Pos)                       /*!< 0x00000040 */
#define LCD_POEN1_SXCY_3                          (0x8UL<<LCD_POEN1_SXCY_Pos)                       /*!< 0x00000080 */
#define LCD_POEN1_CX_Pos                          (8U)
#define LCD_POEN1_CX_Msk                          (0xFUL<<LCD_POEN1_CX_Pos)                         /*!< 0x00000F00 */
#define LCD_POEN1_CX                              LCD_POEN1_CX_Msk
#define LCD_POEN1_CX_0                            (0x1UL<<LCD_POEN1_CX_Pos)                         /*!< 0x00000100 */
#define LCD_POEN1_CX_1                            (0x2UL<<LCD_POEN1_CX_Pos)                         /*!< 0x00000200 */
#define LCD_POEN1_CX_2                            (0x4UL<<LCD_POEN1_CX_Pos)                         /*!< 0x00000400 */
#define LCD_POEN1_CX_3                            (0x8UL<<LCD_POEN1_CX_Pos)                         /*!< 0x00000800 */

/********************  Bit definition for LCD_RAM0~15 register  ********************/
#define LCD_RAM_D0_Pos              (0U)
#define LCD_RAM_D0_Msk              (0x1UL << LCD_RAM_D0_Pos)                /*!< 0x00000001 */
#define LCD_RAM_D0                  LCD_RAM_D0_Msk
#define LCD_RAM_D1_Pos              (1U)
#define LCD_RAM_D1_Msk              (0x1UL << LCD_RAM_D1_Pos)                /*!< 0x00000002 */
#define LCD_RAM_D1                  LCD_RAM_D1_Msk
#define LCD_RAM_D2_Pos              (2U)
#define LCD_RAM_D2_Msk              (0x1UL << LCD_RAM_D2_Pos)                /*!< 0x00000004 */
#define LCD_RAM_D2                  LCD_RAM_D2_Msk
#define LCD_RAM_D3_Pos              (3U)
#define LCD_RAM_D3_Msk              (0x1UL << LCD_RAM_D3_Pos)                /*!< 0x00000008 */
#define LCD_RAM_D3                  LCD_RAM_D3_Msk
#define LCD_RAM_D4_Pos              (4U)
#define LCD_RAM_D4_Msk              (0x1UL << LCD_RAM_D4_Pos)                /*!< 0x00000010 */
#define LCD_RAM_D4                  LCD_RAM_D4_Msk
#define LCD_RAM_D5_Pos              (5U)
#define LCD_RAM_D5_Msk              (0x1UL << LCD_RAM_D5_Pos)                /*!< 0x00000020 */
#define LCD_RAM_D5                  LCD_RAM_D5_Msk
#define LCD_RAM_D6_Pos              (6U)
#define LCD_RAM_D6_Msk              (0x1UL << LCD_RAM_D6_Pos)                /*!< 0x00000040 */
#define LCD_RAM_D6                  LCD_RAM_D6_Msk
#define LCD_RAM_D7_Pos              (7U)
#define LCD_RAM_D7_Msk              (0x1UL << LCD_RAM_D7_Pos)                /*!< 0x00000080 */
#define LCD_RAM_D7                  LCD_RAM_D7_Msk
#define LCD_RAM_D8_Pos              (8U)
#define LCD_RAM_D8_Msk              (0x1UL << LCD_RAM_D8_Pos)                /*!< 0x00000100 */
#define LCD_RAM_D8                  LCD_RAM_D8_Msk
#define LCD_RAM_D9_Pos              (9U)
#define LCD_RAM_D9_Msk              (0x1UL << LCD_RAM_D9_Pos)                /*!< 0x00000200 */
#define LCD_RAM_D9                  LCD_RAM_D9_Msk
#define LCD_RAM_D10_Pos             (10U)
#define LCD_RAM_D10_Msk             (0x1UL << LCD_RAM_D10_Pos)               /*!< 0x00000400 */
#define LCD_RAM_D10                 LCD_RAM_D10_Msk
#define LCD_RAM_D11_Pos             (11U)
#define LCD_RAM_D11_Msk             (0x1UL << LCD_RAM_D11_Pos)               /*!< 0x00000800 */
#define LCD_RAM_D11                 LCD_RAM_D11_Msk
#define LCD_RAM_D12_Pos             (12U)
#define LCD_RAM_D12_Msk             (0x1UL << LCD_RAM_D12_Pos)               /*!< 0x00001000 */
#define LCD_RAM_D12                 LCD_RAM_D12_Msk
#define LCD_RAM_D13_Pos             (13U)
#define LCD_RAM_D13_Msk             (0x1UL << LCD_RAM_D13_Pos)               /*!< 0x00002000 */
#define LCD_RAM_D13                 LCD_RAM_D13_Msk
#define LCD_RAM_D14_Pos             (14U)
#define LCD_RAM_D14_Msk             (0x1UL << LCD_RAM_D14_Pos)               /*!< 0x00004000 */
#define LCD_RAM_D14                 LCD_RAM_D14_Msk
#define LCD_RAM_D15_Pos             (15U)
#define LCD_RAM_D15_Msk             (0x1UL << LCD_RAM_D15_Pos)               /*!< 0x00008000 */
#define LCD_RAM_D15                 LCD_RAM_D15_Msk
#define LCD_RAM_D16_Pos             (16U)
#define LCD_RAM_D16_Msk             (0x1UL << LCD_RAM_D16_Pos)               /*!< 0x00010000 */
#define LCD_RAM_D16                 LCD_RAM_D16_Msk
#define LCD_RAM_D17_Pos             (17U)
#define LCD_RAM_D17_Msk             (0x1UL << LCD_RAM_D17_Pos)               /*!< 0x00020000 */
#define LCD_RAM_D17                 LCD_RAM_D17_Msk
#define LCD_RAM_D18_Pos             (18U)
#define LCD_RAM_D18_Msk             (0x1UL << LCD_RAM_D18_Pos)               /*!< 0x00040000 */
#define LCD_RAM_D18                 LCD_RAM_D18_Msk
#define LCD_RAM_D19_Pos             (19U)
#define LCD_RAM_D19_Msk             (0x1UL << LCD_RAM_D19_Pos)               /*!< 0x00080000 */
#define LCD_RAM_D19                 LCD_RAM_D19_Msk
#define LCD_RAM_D20_Pos             (20U)
#define LCD_RAM_D20_Msk             (0x1UL << LCD_RAM_D20_Pos)               /*!< 0x00100000 */
#define LCD_RAM_D20                 LCD_RAM_D20_Msk
#define LCD_RAM_D21_Pos             (21U)
#define LCD_RAM_D21_Msk             (0x1UL << LCD_RAM_D21_Pos)               /*!< 0x00200000 */
#define LCD_RAM_D21                 LCD_RAM_D21_Msk
#define LCD_RAM_D22_Pos             (22U)
#define LCD_RAM_D22_Msk             (0x1UL << LCD_RAM_D22_Pos)               /*!< 0x00400000 */
#define LCD_RAM_D22                 LCD_RAM_D22_Msk
#define LCD_RAM_D23_Pos             (23U)
#define LCD_RAM_D23_Msk             (0x1UL << LCD_RAM_D23_Pos)               /*!< 0x00800000 */
#define LCD_RAM_D23                 LCD_RAM_D23_Msk
#define LCD_RAM_D24_Pos             (24U)
#define LCD_RAM_D24_Msk             (0x1UL << LCD_RAM_D24_Pos)               /*!< 0x01000000 */
#define LCD_RAM_D24                 LCD_RAM_D24_Msk
#define LCD_RAM_D25_Pos             (25U)
#define LCD_RAM_D25_Msk             (0x1UL << LCD_RAM_D25_Pos)               /*!< 0x02000000 */
#define LCD_RAM_D25                 LCD_RAM_D25_Msk
#define LCD_RAM_D26_Pos             (26U)
#define LCD_RAM_D26_Msk             (0x1UL << LCD_RAM_D26_Pos)               /*!< 0x04000000 */
#define LCD_RAM_D26                 LCD_RAM_D26_Msk
#define LCD_RAM_D27_Pos             (27U)
#define LCD_RAM_D27_Msk             (0x1UL << LCD_RAM_D27_Pos)               /*!< 0x08000000 */
#define LCD_RAM_D27                 LCD_RAM_D27_Msk
#define LCD_RAM_D28_Pos             (28U)
#define LCD_RAM_D28_Msk             (0x1UL << LCD_RAM_D28_Pos)               /*!< 0x10000000 */
#define LCD_RAM_D28                 LCD_RAM_D28_Msk
#define LCD_RAM_D29_Pos             (29U)
#define LCD_RAM_D29_Msk             (0x1UL << LCD_RAM_D29_Pos)               /*!< 0x20000000 */
#define LCD_RAM_D29                 LCD_RAM_D29_Msk
#define LCD_RAM_D30_Pos             (30U)
#define LCD_RAM_D30_Msk             (0x1UL << LCD_RAM_D30_Pos)               /*!< 0x40000000 */
#define LCD_RAM_D30                 LCD_RAM_D30_Msk
#define LCD_RAM_D31_Pos             (31U)
#define LCD_RAM_D31_Msk             (0x1UL << LCD_RAM_D31_Pos)               /*!< 0x80000000 */
#define LCD_RAM_D31                 LCD_RAM_D31_Msk

/********************************************************************************************************************/
/********************************* LPUART ***************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LPUART_CR1 register *******************************************/
#define LPUART_CR1_UE_Pos                         (0U)
#define LPUART_CR1_UE_Msk                         (0x1UL<<LPUART_CR1_UE_Pos)                        /*!< 0x00000001 */
#define LPUART_CR1_UE                             LPUART_CR1_UE_Msk                                 /*!< LPUART enabled */
#define LPUART_CR1_UESM_Pos                       (1U)
#define LPUART_CR1_UESM_Msk                       (0x1UL<<LPUART_CR1_UESM_Pos)                      /*!< 0x00000002 */
#define LPUART_CR1_UESM                           LPUART_CR1_UESM_Msk                               /*!< LPUART enabled in Stop mode */
#define LPUART_CR1_RE_Pos                         (2U)
#define LPUART_CR1_RE_Msk                         (0x1UL<<LPUART_CR1_RE_Pos)                        /*!< 0x00000004 */
#define LPUART_CR1_RE                             LPUART_CR1_RE_Msk                                 /*!< Receive enable */
#define LPUART_CR1_TE_Pos                         (3U)
#define LPUART_CR1_TE_Msk                         (0x1UL<<LPUART_CR1_TE_Pos)                        /*!< 0x00000008 */
#define LPUART_CR1_TE                             LPUART_CR1_TE_Msk                                 /*!< Transfer enable */
#define LPUART_CR1_IDLEIE_Pos                     (4U)
#define LPUART_CR1_IDLEIE_Msk                     (0x1UL<<LPUART_CR1_IDLEIE_Pos)                    /*!< 0x00000010 */
#define LPUART_CR1_IDLEIE                         LPUART_CR1_IDLEIE_Msk                             /*!< IDLE interrupt enable */
#define LPUART_CR1_TCIE_Pos                       (6U)
#define LPUART_CR1_TCIE_Msk                       (0x1UL<<LPUART_CR1_TCIE_Pos)                      /*!< 0x00000040 */
#define LPUART_CR1_TCIE                           LPUART_CR1_TCIE_Msk                               /*!< End of Transfer interrupt enabled */
#define LPUART_CR1_PEIE_Pos                       (8U)
#define LPUART_CR1_PEIE_Msk                       (0x1UL<<LPUART_CR1_PEIE_Pos)                      /*!< 0x00000100 */
#define LPUART_CR1_PEIE                           LPUART_CR1_PEIE_Msk                               /*!< PE interrupt enable */
#define LPUART_CR1_PS_Pos                         (9U)
#define LPUART_CR1_PS_Msk                         (0x1UL<<LPUART_CR1_PS_Pos)                        /*!< 0x00000200 */
#define LPUART_CR1_PS                             LPUART_CR1_PS_Msk                                 /*!< Parity selection */
#define LPUART_CR1_PCE_Pos                        (10U)
#define LPUART_CR1_PCE_Msk                        (0x1UL<<LPUART_CR1_PCE_Pos)                       /*!< 0x00000400 */
#define LPUART_CR1_PCE                            LPUART_CR1_PCE_Msk                                /*!< Parity control */
#define LPUART_CR1_WAKE_Pos                       (11U)
#define LPUART_CR1_WAKE_Msk                       (0x1UL<<LPUART_CR1_WAKE_Pos)                      /*!< 0x00000800 */
#define LPUART_CR1_WAKE                           LPUART_CR1_WAKE_Msk                               /*!< Receive wake-up mode */
#define LPUART_CR1_M0_Pos                         (12U)
#define LPUART_CR1_M0_Msk                         (0x1UL<<LPUART_CR1_M0_Pos)                        /*!< 0x00001000 */
#define LPUART_CR1_M0                             LPUART_CR1_M0_Msk                                 /*!< Configure the word length in combination with M1 */
#define LPUART_CR1_MME_Pos                        (13U)
#define LPUART_CR1_MME_Msk                        (0x1UL<<LPUART_CR1_MME_Pos)                       /*!< 0x00002000 */
#define LPUART_CR1_MME                            LPUART_CR1_MME_Msk                                /*!< Silent mode enabled */
#define LPUART_CR1_CMIE_Pos                       (14U)
#define LPUART_CR1_CMIE_Msk                       (0x1UL<<LPUART_CR1_CMIE_Pos)                      /*!< 0x00004000 */
#define LPUART_CR1_CMIE                           LPUART_CR1_CMIE_Msk                               /*!< Character match interrupt enabled */
#define LPUART_CR1_DEDT_Pos                       (16U)
#define LPUART_CR1_DEDT_Msk                       (0x1FUL<<LPUART_CR1_DEDT_Pos)                     /*!< 0x001F0000 */
#define LPUART_CR1_DEDT                           LPUART_CR1_DEDT_Msk                               /*!< The time between the transmit frame stop bit and the invalid DE (driver enable) signal */
#define LPUART_CR1_DEDT_0                         (0x1UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00010000 */
#define LPUART_CR1_DEDT_1                         (0x2UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00020000 */
#define LPUART_CR1_DEDT_2                         (0x4UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00040000 */
#define LPUART_CR1_DEDT_3                         (0x8UL<<LPUART_CR1_DEDT_Pos)                      /*!< 0x00080000 */
#define LPUART_CR1_DEDT_4                         (0x10UL<<LPUART_CR1_DEDT_Pos)                     /*!< 0x00100000 */
#define LPUART_CR1_DEAT_Pos                       (21U)
#define LPUART_CR1_DEAT_Msk                       (0x1FUL<<LPUART_CR1_DEAT_Pos)                     /*!< 0x03E00000 */
#define LPUART_CR1_DEAT                           LPUART_CR1_DEAT_Msk                               /*!< The time between the DE (driver enable) signal validity and the start bit */
#define LPUART_CR1_DEAT_0                         (0x1UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00200000 */
#define LPUART_CR1_DEAT_1                         (0x2UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00400000 */
#define LPUART_CR1_DEAT_2                         (0x4UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x00800000 */
#define LPUART_CR1_DEAT_3                         (0x8UL<<LPUART_CR1_DEAT_Pos)                      /*!< 0x01000000 */
#define LPUART_CR1_DEAT_4                         (0x10UL<<LPUART_CR1_DEAT_Pos)                     /*!< 0x02000000 */
#define LPUART_CR1_M1_Pos                         (28U)
#define LPUART_CR1_M1_Msk                         (0x1UL<<LPUART_CR1_M1_Pos)                        /*!< 0x10000000 */
#define LPUART_CR1_M1                             LPUART_CR1_M1_Msk                                 /*!< The value of {M1,M0} the configurate length */

/********************************* Bit definition for LPUART_CR2 register *******************************************/
#define LPUART_CR2_ADDM7_Pos                      (4U)
#define LPUART_CR2_ADDM7_Msk                      (0x1UL<<LPUART_CR2_ADDM7_Pos)                     /*!< 0x00000010 */
#define LPUART_CR2_ADDM7                          LPUART_CR2_ADDM7_Msk                              /*!< 7bit/4bit address detection configuration */
#define LPUART_CR2_STOP_Pos                       (12U)
#define LPUART_CR2_STOP_Msk                       (0x3UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00003000 */
#define LPUART_CR2_STOP                           LPUART_CR2_STOP_Msk                               /*!< Stop bit configuration */
#define LPUART_CR2_STOP_0                         (0x1UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00001000 */
#define LPUART_CR2_STOP_1                         (0x2UL<<LPUART_CR2_STOP_Pos)                      /*!< 0x00002000 */
#define LPUART_CR2_SWAP_Pos                       (15U)
#define LPUART_CR2_SWAP_Msk                       (0x1UL<<LPUART_CR2_SWAP_Pos)                      /*!< 0x00008000 */
#define LPUART_CR2_SWAP                           LPUART_CR2_SWAP_Msk                               /*!< TX/RX pin interchangeable */
#define LPUART_CR2_RXINV_Pos                      (16U)
#define LPUART_CR2_RXINV_Msk                      (0x1UL<<LPUART_CR2_RXINV_Pos)                     /*!< 0x00010000 */
#define LPUART_CR2_RXINV                          LPUART_CR2_RXINV_Msk                              /*!< The RX pin is active level inverting */
#define LPUART_CR2_TXINV_Pos                      (17U)
#define LPUART_CR2_TXINV_Msk                      (0x1UL<<LPUART_CR2_TXINV_Pos)                     /*!< 0x00020000 */
#define LPUART_CR2_TXINV                          LPUART_CR2_TXINV_Msk                              /*!< TX pin active level inverted */
#define LPUART_CR2_DATAINV_Pos                    (18U)
#define LPUART_CR2_DATAINV_Msk                    (0x1UL<<LPUART_CR2_DATAINV_Pos)                   /*!< 0x00040000 */
#define LPUART_CR2_DATAINV                        LPUART_CR2_DATAINV_Msk                            /*!< Inverse binary data processing */
#define LPUART_CR2_MSBFIRST_Pos                   (19U)
#define LPUART_CR2_MSBFIRST_Msk                   (0x1UL<<LPUART_CR2_MSBFIRST_Pos)                  /*!< 0x00080000 */
#define LPUART_CR2_MSBFIRST                       LPUART_CR2_MSBFIRST_Msk                           /*!< MSB first */
#define LPUART_CR2_TXOE_ALWAYS_ON_Pos             (20U)
#define LPUART_CR2_TXOE_ALWAYS_ON_Msk             (0x1UL<<LPUART_CR2_TXOE_ALWAYS_ON_Pos)            /*!< 0x00100000 */
#define LPUART_CR2_TXOE_ALWAYS_ON                 LPUART_CR2_TXOE_ALWAYS_ON_Msk                     
#define LPUART_CR2_ADD_Pos                        (24U)
#define LPUART_CR2_ADD_Msk                        (0xFFUL<<LPUART_CR2_ADD_Pos)                      /*!< 0xFF000000 */
#define LPUART_CR2_ADD                            LPUART_CR2_ADD_Msk                                /*!< LPUART address */

/********************************* Bit definition for LPUART_CR3 register *******************************************/
#define LPUART_CR3_EIE_Pos                        (0U)
#define LPUART_CR3_EIE_Msk                        (0x1UL<<LPUART_CR3_EIE_Pos)                       /*!< 0x00000001 */
#define LPUART_CR3_EIE                            LPUART_CR3_EIE_Msk                                /*!< Error interrupts are enabled */
#define LPUART_CR3_HDSEL_Pos                      (3U)
#define LPUART_CR3_HDSEL_Msk                      (0x1UL<<LPUART_CR3_HDSEL_Pos)                     /*!< 0x00000008 */
#define LPUART_CR3_HDSEL                          LPUART_CR3_HDSEL_Msk                              /*!< Half-duplex selection */
#define LPUART_CR3_DMAR_Pos                       (6U)
#define LPUART_CR3_DMAR_Msk                       (0x1UL<<LPUART_CR3_DMAR_Pos)                      /*!< 0x00000040 */
#define LPUART_CR3_DMAR                           LPUART_CR3_DMAR_Msk                               /*!< Enable DMA on reception */
#define LPUART_CR3_DMAT_Pos                       (7U)
#define LPUART_CR3_DMAT_Msk                       (0x1UL<<LPUART_CR3_DMAT_Pos)                      /*!< 0x00000080 */
#define LPUART_CR3_DMAT                           LPUART_CR3_DMAT_Msk                               /*!< Enable DMA while transmitting */
#define LPUART_CR3_RTSE_Pos                       (8U)
#define LPUART_CR3_RTSE_Msk                       (0x1UL<<LPUART_CR3_RTSE_Pos)                      /*!< 0x00000100 */
#define LPUART_CR3_RTSE                           LPUART_CR3_RTSE_Msk                               /*!< RTS enabled */
#define LPUART_CR3_CTSE_Pos                       (9U)
#define LPUART_CR3_CTSE_Msk                       (0x1UL<<LPUART_CR3_CTSE_Pos)                      /*!< 0x00000200 */
#define LPUART_CR3_CTSE                           LPUART_CR3_CTSE_Msk                               /*!< CTS enabled */
#define LPUART_CR3_CTSIE_Pos                      (10U)
#define LPUART_CR3_CTSIE_Msk                      (0x1UL<<LPUART_CR3_CTSIE_Pos)                     /*!< 0x00000400 */
#define LPUART_CR3_CTSIE                          LPUART_CR3_CTSIE_Msk                              /*!< CTS interrupt enable */
#define LPUART_CR3_OVRDIS_Pos                     (12U)
#define LPUART_CR3_OVRDIS_Msk                     (0x1UL<<LPUART_CR3_OVRDIS_Pos)                    /*!< 0x00001000 */
#define LPUART_CR3_OVRDIS                         LPUART_CR3_OVRDIS_Msk                             /*!< Overrun is prohibited */
#define LPUART_CR3_DDRE_Pos                       (13U)
#define LPUART_CR3_DDRE_Msk                       (0x1UL<<LPUART_CR3_DDRE_Pos)                      /*!< 0x00002000 */
#define LPUART_CR3_DDRE                           LPUART_CR3_DDRE_Msk                               /*!< Whether DMA is disabled when receiving an error */
#define LPUART_CR3_DEM_Pos                        (14U)
#define LPUART_CR3_DEM_Msk                        (0x1UL<<LPUART_CR3_DEM_Pos)                       /*!< 0x00004000 */
#define LPUART_CR3_DEM                            LPUART_CR3_DEM_Msk                                /*!< DE mode enabled */
#define LPUART_CR3_DEP_Pos                        (15U)
#define LPUART_CR3_DEP_Msk                        (0x1UL<<LPUART_CR3_DEP_Pos)                       /*!< 0x00008000 */
#define LPUART_CR3_DEP                            LPUART_CR3_DEP_Msk                                /*!< DE polarity selection */
#define LPUART_CR3_WUS_Pos                        (20U)
#define LPUART_CR3_WUS_Msk                        (0x3UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00300000 */
#define LPUART_CR3_WUS                            LPUART_CR3_WUS_Msk                                /*!< Low-power wake-up options */
#define LPUART_CR3_WUS_0                          (0x1UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00100000 */
#define LPUART_CR3_WUS_1                          (0x2UL<<LPUART_CR3_WUS_Pos)                       /*!< 0x00200000 */
#define LPUART_CR3_WUFIE_Pos                      (22U)
#define LPUART_CR3_WUFIE_Msk                      (0x1UL<<LPUART_CR3_WUFIE_Pos)                     /*!< 0x00400000 */
#define LPUART_CR3_WUFIE                          LPUART_CR3_WUFIE_Msk                              /*!< Low-power wake-up interrupt enable */

/********************************* Bit definition for LPUART_BRR register *******************************************/
#define LPUART_BRR_BBR_Pos                        (0U)
#define LPUART_BRR_BBR_Msk                        (0xFFFFFUL<<LPUART_BRR_BBR_Pos)                   /*!< 0x000FFFFF */
#define LPUART_BRR_BBR                            LPUART_BRR_BBR_Msk                                /*!< LPUART baud rate */

/********************************* Bit definition for LPUART_LPWKUP_BRR register ************************************/
#define LPUART_LPWKUP_BRR_LPWKUP_BBR_Pos          (0U)
#define LPUART_LPWKUP_BRR_LPWKUP_BBR_Msk          (0xFFFFFUL<<LPUART_LPWKUP_BRR_LPWKUP_BBR_Pos)     /*!< 0x000FFFFF */
#define LPUART_LPWKUP_BRR_LPWKUP_BBR              LPUART_LPWKUP_BRR_LPWKUP_BBR_Msk                  /*!< LPUART lpwkup baud rate */
#define LPUART_LPWKUP_BRR_HW_BRR_UPEN_Pos         (20U)
#define LPUART_LPWKUP_BRR_HW_BRR_UPEN_Msk         (0x1UL<<LPUART_LPWKUP_BRR_HW_BRR_UPEN_Pos)        /*!< 0x00100000 */
#define LPUART_LPWKUP_BRR_HW_BRR_UPEN             LPUART_LPWKUP_BRR_HW_BRR_UPEN_Msk                 /*!< LPUART wkup brr update enable */

/********************************* Bit definition for LPUART_RQR register *******************************************/
#define LPUART_RQR_SBKRQ_Pos                      (1U)
#define LPUART_RQR_SBKRQ_Msk                      (0x1UL<<LPUART_RQR_SBKRQ_Pos)                     /*!< 0x00000002 */
#define LPUART_RQR_SBKRQ                          LPUART_RQR_SBKRQ_Msk                              /*!< Send an abort request */
#define LPUART_RQR_MMRQ_Pos                       (2U)
#define LPUART_RQR_MMRQ_Msk                       (0x1UL<<LPUART_RQR_MMRQ_Pos)                      /*!< 0x00000004 */
#define LPUART_RQR_MMRQ                           LPUART_RQR_MMRQ_Msk                               /*!< Silent mode request */

/********************************* Bit definition for LPUART_ISR register *******************************************/
#define LPUART_ISR_PE_Pos                         (0U)
#define LPUART_ISR_PE_Msk                         (0x1UL<<LPUART_ISR_PE_Pos)                        /*!< 0x00000001 */
#define LPUART_ISR_PE                             LPUART_ISR_PE_Msk                                 /*!< Parity error */
#define LPUART_ISR_FE_Pos                         (1U)
#define LPUART_ISR_FE_Msk                         (0x1UL<<LPUART_ISR_FE_Pos)                        /*!< 0x00000002 */
#define LPUART_ISR_FE                             LPUART_ISR_FE_Msk                                 /*!< Frame error flag */
#define LPUART_ISR_NE_Pos                         (2U)
#define LPUART_ISR_NE_Msk                         (0x1UL<<LPUART_ISR_NE_Pos)                        /*!< 0x00000004 */
#define LPUART_ISR_NE                             LPUART_ISR_NE_Msk                                 /*!< START bit noise flag */
#define LPUART_ISR_ORE_Pos                        (3U)
#define LPUART_ISR_ORE_Msk                        (0x1UL<<LPUART_ISR_ORE_Pos)                       /*!< 0x00000008 */
#define LPUART_ISR_ORE                            LPUART_ISR_ORE_Msk                                /*!< Overrun error flag */
#define LPUART_ISR_IDLE_Pos                       (4U)
#define LPUART_ISR_IDLE_Msk                       (0x1UL<<LPUART_ISR_IDLE_Pos)                      /*!< 0x00000010 */
#define LPUART_ISR_IDLE                           LPUART_ISR_IDLE_Msk                               /*!< idle flag */
#define LPUART_ISR_RXFNE_Pos                      (5U)
#define LPUART_ISR_RXFNE_Msk                      (0x1UL<<LPUART_ISR_RXFNE_Pos)                     /*!< 0x00000020 */
#define LPUART_ISR_RXFNE                          LPUART_ISR_RXFNE_Msk                              /*!< RX register not empty */
#define LPUART_ISR_TC_Pos                         (6U)
#define LPUART_ISR_TC_Msk                         (0x1UL<<LPUART_ISR_TC_Pos)                        /*!< 0x00000040 */
#define LPUART_ISR_TC                             LPUART_ISR_TC_Msk                                 /*!< Transmission completion flag */
#define LPUART_ISR_TXFNF_Pos                      (7U)
#define LPUART_ISR_TXFNF_Msk                      (0x1UL<<LPUART_ISR_TXFNF_Pos)                     /*!< 0x00000080 */
#define LPUART_ISR_TXFNF                          LPUART_ISR_TXFNF_Msk                              /*!< TX register not full */
#define LPUART_ISR_CTSIF_Pos                      (9U)
#define LPUART_ISR_CTSIF_Msk                      (0x1UL<<LPUART_ISR_CTSIF_Pos)                     /*!< 0x00000200 */
#define LPUART_ISR_CTSIF                          LPUART_ISR_CTSIF_Msk                              /*!< CTS interrupt flag */
#define LPUART_ISR_CTS_Pos                        (10U)
#define LPUART_ISR_CTS_Msk                        (0x1UL<<LPUART_ISR_CTS_Pos)                       /*!< 0x00000400 */
#define LPUART_ISR_CTS                            LPUART_ISR_CTS_Msk                                /*!< CTS flag */
#define LPUART_ISR_BUSY_Pos                       (16U)
#define LPUART_ISR_BUSY_Msk                       (0x1UL<<LPUART_ISR_BUSY_Pos)                      /*!< 0x00010000 */
#define LPUART_ISR_BUSY                           LPUART_ISR_BUSY_Msk                               /*!< Busy flag */
#define LPUART_ISR_CMF_Pos                        (17U)
#define LPUART_ISR_CMF_Msk                        (0x1UL<<LPUART_ISR_CMF_Pos)                       /*!< 0x00020000 */
#define LPUART_ISR_CMF                            LPUART_ISR_CMF_Msk                                /*!< Address match flag */
#define LPUART_ISR_SBKF_Pos                       (18U)
#define LPUART_ISR_SBKF_Msk                       (0x1UL<<LPUART_ISR_SBKF_Pos)                      /*!< 0x00040000 */
#define LPUART_ISR_SBKF                           LPUART_ISR_SBKF_Msk                               /*!< Send abort flag */
#define LPUART_ISR_RWU_Pos                        (19U)
#define LPUART_ISR_RWU_Msk                        (0x1UL<<LPUART_ISR_RWU_Pos)                       /*!< 0x00080000 */
#define LPUART_ISR_RWU                            LPUART_ISR_RWU_Msk                                /*!< Receive Mute mode wake-up flag */
#define LPUART_ISR_WUF_Pos                        (20U)
#define LPUART_ISR_WUF_Msk                        (0x1UL<<LPUART_ISR_WUF_Pos)                       /*!< 0x00100000 */
#define LPUART_ISR_WUF                            LPUART_ISR_WUF_Msk                                /*!< Wake flag from low-power mode */
#define LPUART_ISR_TEACK_Pos                      (21U)
#define LPUART_ISR_TEACK_Msk                      (0x1UL<<LPUART_ISR_TEACK_Pos)                     /*!< 0x00200000 */
#define LPUART_ISR_TEACK                          LPUART_ISR_TEACK_Msk                              /*!< Transport enable acknowledgement flag */
#define LPUART_ISR_REACK_Pos                      (22U)
#define LPUART_ISR_REACK_Msk                      (0x1UL<<LPUART_ISR_REACK_Pos)                     /*!< 0x00400000 */
#define LPUART_ISR_REACK                          LPUART_ISR_REACK_Msk                              /*!< Receive the enable acknowledgment flag */

/********************************* Bit definition for LPUART_ICR register *******************************************/
#define LPUART_ICR_PECF_Pos                       (0U)
#define LPUART_ICR_PECF_Msk                       (0x1UL<<LPUART_ICR_PECF_Pos)                      /*!< 0x00000001 */
#define LPUART_ICR_PECF                           LPUART_ICR_PECF_Msk                               /*!< Check value error flag reset */
#define LPUART_ICR_FECF_Pos                       (1U)
#define LPUART_ICR_FECF_Msk                       (0x1UL<<LPUART_ICR_FECF_Pos)                      /*!< 0x00000002 */
#define LPUART_ICR_FECF                           LPUART_ICR_FECF_Msk                               /*!< Frame error flag reset */
#define LPUART_ICR_NECF_Pos                       (2U)
#define LPUART_ICR_NECF_Msk                       (0x1UL<<LPUART_ICR_NECF_Pos)                      /*!< 0x00000004 */
#define LPUART_ICR_NECF                           LPUART_ICR_NECF_Msk                               /*!< Clear the START bit noise flag */
#define LPUART_ICR_ORECF_Pos                      (3U)
#define LPUART_ICR_ORECF_Msk                      (0x1UL<<LPUART_ICR_ORECF_Pos)                     /*!< 0x00000008 */
#define LPUART_ICR_ORECF                          LPUART_ICR_ORECF_Msk                              /*!< Overrun error flag reset */
#define LPUART_ICR_IDLECF_Pos                     (4U)
#define LPUART_ICR_IDLECF_Msk                     (0x1UL<<LPUART_ICR_IDLECF_Pos)                    /*!< 0x00000010 */
#define LPUART_ICR_IDLECF                         LPUART_ICR_IDLECF_Msk                             /*!< Clear idle flag  */
#define LPUART_ICR_TCCF_Pos                       (6U)
#define LPUART_ICR_TCCF_Msk                       (0x1UL<<LPUART_ICR_TCCF_Pos)                      /*!< 0x00000040 */
#define LPUART_ICR_TCCF                           LPUART_ICR_TCCF_Msk                               /*!< Transfer completion flag reset */
#define LPUART_ICR_CTSCF_Pos                      (9U)
#define LPUART_ICR_CTSCF_Msk                      (0x1UL<<LPUART_ICR_CTSCF_Pos)                     /*!< 0x00000200 */
#define LPUART_ICR_CTSCF                          LPUART_ICR_CTSCF_Msk                              /*!< CTS flag reset */
#define LPUART_ICR_CMCF_Pos                       (17U)
#define LPUART_ICR_CMCF_Msk                       (0x1UL<<LPUART_ICR_CMCF_Pos)                      /*!< 0x00020000 */
#define LPUART_ICR_CMCF                           LPUART_ICR_CMCF_Msk                               /*!< Address matching flag reset */
#define LPUART_ICR_WUCF_Pos                       (20U)
#define LPUART_ICR_WUCF_Msk                       (0x1UL<<LPUART_ICR_WUCF_Pos)                      /*!< 0x00100000 */
#define LPUART_ICR_WUCF                           LPUART_ICR_WUCF_Msk                               /*!< Low power wake-up flag reset */

/********************************* Bit definition for LPUART_RDR register *******************************************/
#define LPUART_RDR_RDR_Pos                        (0U)
#define LPUART_RDR_RDR_Msk                        (0x1FFUL<<LPUART_RDR_RDR_Pos)                     /*!< 0x000001FF */
#define LPUART_RDR_RDR                            LPUART_RDR_RDR_Msk                                /*!< Receive Data Register */

/********************************* Bit definition for LPUART_TDR register *******************************************/
#define LPUART_TDR_TDR_Pos                        (0U)
#define LPUART_TDR_TDR_Msk                        (0x1FFUL<<LPUART_TDR_TDR_Pos)                     /*!< 0x000001FF */
#define LPUART_TDR_TDR                            LPUART_TDR_TDR_Msk                                /*!< Transmit Data Register */

/********************************* Bit definition for LPUART_PRESC register *****************************************/
#define LPUART_PRESC_PRESCALER_Pos                (0U)
#define LPUART_PRESC_PRESCALER_Msk                (0xFUL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x0000000F */
#define LPUART_PRESC_PRESCALER                    LPUART_PRESC_PRESCALER_Msk                        /*!< Input clock prescaler register */
#define LPUART_PRESC_PRESCALER_0                  (0x1UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000001 */
#define LPUART_PRESC_PRESCALER_1                  (0x2UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000002 */
#define LPUART_PRESC_PRESCALER_2                  (0x4UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000004 */
#define LPUART_PRESC_PRESCALER_3                  (0x8UL<<LPUART_PRESC_PRESCALER_Pos)               /*!< 0x00000008 */

/********************************* Bit definition for LPUART_DBG register *******************************************/
#define LPUART_DBG_DBGSEL_Pos                     (0U)
#define LPUART_DBG_DBGSEL_Msk                     (0x3UL<<LPUART_DBG_DBGSEL_Pos)                    /*!< 0x00000003 */
#define LPUART_DBG_DBGSEL                         LPUART_DBG_DBGSEL_Msk                             /*!< debug select */
#define LPUART_DBG_DBGSEL_0                       (0x1UL<<LPUART_DBG_DBGSEL_Pos)                    /*!< 0x00000001 */
#define LPUART_DBG_DBGSEL_1                       (0x2UL<<LPUART_DBG_DBGSEL_Pos)                    /*!< 0x00000002 */
#define LPUART_DBG_OFFSET_ADD_Pos                 (24U)
#define LPUART_DBG_OFFSET_ADD_Msk                 (0xFFUL<<LPUART_DBG_OFFSET_ADD_Pos)               /*!< 0xFF000000 */
#define LPUART_DBG_OFFSET_ADD                     LPUART_DBG_OFFSET_ADD_Msk                         /*!< offset add */

/********************************************************************************************************************/
/********************************* OPA ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for OPA_OCR register **********************************************/
#define OPA_OCR_OPA_EXTOEN_Pos                    (0U)
#define OPA_OCR_OPA_EXTOEN_Msk                    (0x1UL<<OPA_OCR_OPA_EXTOEN_Pos)                   /*!< 0x00000001 */
#define OPA_OCR_OPA_EXTOEN                        OPA_OCR_OPA_EXTOEN_Msk                            
#define OPA_OCR_OPA_EXTOSEL_Pos                   (1U)
#define OPA_OCR_OPA_EXTOSEL_Msk                   (0x1UL<<OPA_OCR_OPA_EXTOSEL_Pos)                  /*!< 0x00000002 */
#define OPA_OCR_OPA_EXTOSEL                       OPA_OCR_OPA_EXTOSEL_Msk                           

/********************************* Bit definition for OPA_CR register ***********************************************/
#define OPA_CR_OPA_VINPSEL_Pos                    (0U)
#define OPA_CR_OPA_VINPSEL_Msk                    (0x1UL<<OPA_CR_OPA_VINPSEL_Pos)                   /*!< 0x00000001 */
#define OPA_CR_OPA_VINPSEL                        OPA_CR_OPA_VINPSEL_Msk                            
#define OPA_CR_OPA_VINMSEL_Pos                    (2U)
#define OPA_CR_OPA_VINMSEL_Msk                    (0x1UL<<OPA_CR_OPA_VINMSEL_Pos)                   /*!< 0x00000004 */
#define OPA_CR_OPA_VINMSEL                        OPA_CR_OPA_VINMSEL_Msk                            
#define OPA_CR_OPAEN_Pos                          (5U)
#define OPA_CR_OPAEN_Msk                          (0x1UL<<OPA_CR_OPAEN_Pos)                         /*!< 0x00000020 */
#define OPA_CR_OPAEN                              OPA_CR_OPAEN_Msk                                  

/********************************* Bit definition for OPA_TMR register **********************************************/
#define OPA_TMR_OPA_TM_Pos                        (0U)
#define OPA_TMR_OPA_TM_Msk                        (0x1UL<<OPA_TMR_OPA_TM_Pos)                       /*!< 0x00000001 */
#define OPA_TMR_OPA_TM                            OPA_TMR_OPA_TM_Msk                                


/******************************************************************************/
/*                                                                            */
/*                        Power Control (PWR)                                 */
/*                                                                            */
/******************************************************************************/
#define PWR_PVD_SUPPORT                       /*!< PWR feature available only on specific devices: Power Voltage Detection feature */
#define PWR_BOR_SUPPORT                       /*!< PWR feature available only on specific devices: Brown-Out Reset feature         */
#define PWR_SHDW_SUPPORT                      /*!< PWR feature available only on specific devices: Shutdown mode */

/********************  Bit definition for PWR_CR1 register  ********************/
#define PWR_CR1_DBP_Pos              (8U)
#define PWR_CR1_DBP_Msk              (0x1UL << PWR_CR1_DBP_Pos)                   /*!< 0x00000100 */
#define PWR_CR1_DBP                  PWR_CR1_DBP_Msk                              /*!< Disable Backup Domain write protection */
#define PWR_CR1_FLS_SLPTIME_Pos      (12U)
#define PWR_CR1_FLS_SLPTIME_Msk      (0x3UL << PWR_CR1_FLS_SLPTIME_Pos)           /*!< 0x00003000 */
#define PWR_CR1_FLS_SLPTIME          PWR_CR1_FLS_SLPTIME_Msk
#define PWR_CR1_FLS_SLPTIME_0        (0x1UL << PWR_CR1_FLS_SLPTIME_Pos)
#define PWR_CR1_FLS_SLPTIME_1        (0x2UL << PWR_CR1_FLS_SLPTIME_Pos)
#define PWR_CR1_LPR_Pos              (14U)
#define PWR_CR1_LPR_Msk              (0x1UL << PWR_CR1_LPR_Pos)                   /*!< 0x00004000 */
#define PWR_CR1_LPR                  PWR_CR1_LPR_Msk                              /*!< Regulator Low-Power Run mode */

#define PWR_CR1_SRAM_RETV_CTRL_Pos   (18U)
#define PWR_CR1_SRAM_RETV_CTRL_Msk   (0x1UL << PWR_CR1_SRAM_RETV_CTRL_Pos)
#define PWR_CR1_SRAM_RETV_CTRL       PWR_CR1_SRAM_RETV_CTRL_Msk


/********************  Bit definition for PWR_CR2 register  ********************/
#define PWR_CR2_PVDE_Pos             (0U)
#define PWR_CR2_PVDE_Msk             (0x1UL << PWR_CR2_PVDE_Pos)                  /*!< 0x00000001 */
#define PWR_CR2_PVDE                 PWR_CR2_PVDE_Msk                             /*!< Programmable Voltage Detector Enable */
#define PWR_CR2_SRCSEL_Pos           (2U)
#define PWR_CR2_SRCSEL_Msk           (0x1UL << PWR_CR2_SRCSEL_Pos)                /*!< 0x00000004*/
#define PWR_CR2_SRCSEL               PWR_CR2_SRCSEL_Msk                           /*!<  Selection bit field */
#define PWR_CR2_PVDT_Pos             (4U)
#define PWR_CR2_PVDT_Msk             (0x7UL << PWR_CR2_PVDT_Pos)                  /*!< 0x00000070 */
#define PWR_CR2_PVDT                 PWR_CR2_PVDT_Msk                             /*!< PVD Rising Threshold Selection bit field */
#define PWR_CR2_PVDT_0               (0x1UL << PWR_CR2_PVDT_Pos)                  /*!< 0x00000010 */
#define PWR_CR2_PVDT_1               (0x2UL << PWR_CR2_PVDT_Pos)                  /*!< 0x00000020 */
#define PWR_CR2_PVDT_2               (0x4UL << PWR_CR2_PVDT_Pos)                  /*!< 0x00000040 */
#define PWR_CR2_FLTEN_Pos            (8U)
#define PWR_CR2_FLTEN_Msk            (0x1UL << PWR_CR2_FLTEN_Pos)                 /*!< 0x00000100 */
#define PWR_CR2_FLTEN                PWR_CR2_FLTEN_Msk                            /*!< filter enable control bit field */
#define PWR_CR2_FLT_TIME_Pos         (9U)
#define PWR_CR2_FLT_TIME_Msk         (0x7UL << PWR_CR2_FLT_TIME_Pos)              /*!< 0x00000E00*/
#define PWR_CR2_FLT_TIME             PWR_CR2_FLT_TIME_Msk                         /*!<  Selection bit field */
#define PWR_CR2_FLT_TIME_0           (0x1UL << PWR_CR2_FLT_TIME_Pos)              /*!< 0x00000200 */
#define PWR_CR2_FLT_TIME_1           (0x2UL << PWR_CR2_FLT_TIME_Pos)              /*!< 0x00000400 */
#define PWR_CR2_FLT_TIME_2           (0x4UL << PWR_CR2_FLT_TIME_Pos)              /*!< 0x00000800 */

/********************  Bit definition for PWR_SR register  ********************/
#define PWR_SR_PVDO_Pos              (11U)
#define PWR_SR_PVDO_Msk              (0x1UL << PWR_SR_PVDO_Pos)                   /*!< 0x00000800 */
#define PWR_SR_PVDO                  PWR_SR_PVDO_Msk                              /*!< Power voltage detector output */



/******************************************************************************/
/*                                                                            */
/*                           Reset and Clock Control (RCC)                    */
/*                                                                            */
/******************************************************************************/
/*
* @brief Specific device feature definitions
*/
#define RCC_LSE_SUPPORT
#define RCC_PLL_SUPPORT

/********************************* Bit definition for RCC_CR register ***********************************************/
#define RCC_CR_MSIDIV_Pos                         (0U)
#define RCC_CR_MSIDIV_Msk                         (0x7UL<<RCC_CR_MSIDIV_Pos)                        /*!< 0x00000007 */
#define RCC_CR_MSIDIV                             RCC_CR_MSIDIV_Msk
#define RCC_CR_MSIDIV_0                           (0x1UL<<RCC_CR_MSIDIV_Pos)                        /*!< 0x00000001 */
#define RCC_CR_MSIDIV_1                           (0x2UL<<RCC_CR_MSIDIV_Pos)                        /*!< 0x00000002 */
#define RCC_CR_MSIDIV_2                           (0x4UL<<RCC_CR_MSIDIV_Pos)                        /*!< 0x00000004 */
#define RCC_CR_HSION_Pos                          (8U)
#define RCC_CR_HSION_Msk                          (0x1UL<<RCC_CR_HSION_Pos)                         /*!< 0x00000100 */
#define RCC_CR_HSION                              RCC_CR_HSION_Msk                                  
#define RCC_CR_MSIRDY_Pos                         (9U)
#define RCC_CR_MSIRDY_Msk                         (0x1UL<<RCC_CR_MSIRDY_Pos)                        /*!< 0x00000200 */
#define RCC_CR_MSIRDY                             RCC_CR_MSIRDY_Msk                                 
#define RCC_CR_HSIRDY_Pos                         (10U)
#define RCC_CR_HSIRDY_Msk                         (0x1UL<<RCC_CR_HSIRDY_Pos)                         /*!< 0x00000100 */
#define RCC_CR_HSIRDY                             RCC_CR_HSIRDY_Msk                                  
#define RCC_CR_HSIDIV_Pos                         (11U)
#define RCC_CR_HSIDIV_Msk                         (0x7UL<<RCC_CR_HSIDIV_Pos)                        /*!< 0x00003800 */
#define RCC_CR_HSIDIV                             RCC_CR_HSIDIV_Msk
#define RCC_CR_HSIDIV_0                           (0x1UL<<RCC_CR_HSIDIV_Pos)                        /*!< 0x00000800 */
#define RCC_CR_HSIDIV_1                           (0x2UL<<RCC_CR_HSIDIV_Pos)                        /*!< 0x00001000 */
#define RCC_CR_HSIDIV_2                           (0x4UL<<RCC_CR_HSIDIV_Pos)                        /*!< 0x00002000 */
#define RCC_CR_HSEON_Pos                          (16U)
#define RCC_CR_HSEON_Msk                          (0x1UL<<RCC_CR_HSEON_Pos)                         /*!< 0x00010000 */
#define RCC_CR_HSEON                              RCC_CR_HSEON_Msk                                  
#define RCC_CR_HSERDY_Pos                         (17U)
#define RCC_CR_HSERDY_Msk                         (0x1UL<<RCC_CR_HSERDY_Pos)                        /*!< 0x00020000 */
#define RCC_CR_HSERDY                             RCC_CR_HSERDY_Msk                                 
#define RCC_CR_HSEBYP_Pos                         (18U)
#define RCC_CR_HSEBYP_Msk                         (0x1UL<<RCC_CR_HSEBYP_Pos)                        /*!< 0x00040000 */
#define RCC_CR_HSEBYP                             RCC_CR_HSEBYP_Msk                                 
#define RCC_CR_HSE_CSSON_Pos                      (19U)
#define RCC_CR_HSE_CSSON_Msk                      (0x1UL<<RCC_CR_HSE_CSSON_Pos)                     /*!< 0x00080000 */
#define RCC_CR_HSE_CSSON                          RCC_CR_HSE_CSSON_Msk                              
#define RCC_CR_PLLON_Pos                          (24U)
#define RCC_CR_PLLON_Msk                          (0x1UL<<RCC_CR_PLLON_Pos)                         /*!< 0x01000000 */
#define RCC_CR_PLLON                              RCC_CR_PLLON_Msk                                  
#define RCC_CR_PLLRDY_Pos                         (25U)
#define RCC_CR_PLLRDY_Msk                         (0x1UL<<RCC_CR_PLLRDY_Pos)                        /*!< 0x02000000 */
#define RCC_CR_PLLRDY                             RCC_CR_PLLRDY_Msk                                 

/********************************* Bit definition for RCC_ICSCR register ********************************************/
#define RCC_ICSCR_HSI_TRIMCR_Pos                  (0U)
#define RCC_ICSCR_HSI_TRIMCR_Msk                  (0x1FFFUL<<RCC_ICSCR_HSI_TRIMCR_Pos)              /*!< 0x00001FFF */
#define RCC_ICSCR_HSI_TRIMCR                      RCC_ICSCR_HSI_TRIMCR_Msk
#define RCC_ICSCR_HSI_TRIM_Pos            (0U)
#define RCC_ICSCR_HSI_TRIM_Msk            (0x1FFFUL << RCC_ICSCR_HSI_TRIM_Pos)   /*!< 0x00001FFF */
#define RCC_ICSCR_HSI_TRIM                RCC_ICSCR_HSI_TRIM_Msk                 /*!< HSITRIM[14:8] bits */
#define RCC_ICSCR_HSI_TRIM_0              (0x01UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000001 */
#define RCC_ICSCR_HSI_TRIM_1              (0x02UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000002 */
#define RCC_ICSCR_HSI_TRIM_2              (0x04UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000004 */
#define RCC_ICSCR_HSI_TRIM_3              (0x08UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000008 */
#define RCC_ICSCR_HSI_TRIM_4              (0x10UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000010 */
#define RCC_ICSCR_HSI_TRIM_5              (0x20UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000020 */
#define RCC_ICSCR_HSI_TRIM_6              (0x40UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000040 */
#define RCC_ICSCR_HSI_TRIM_7              (0x80UL << RCC_ICSCR_HSI_TRIM_Pos)     /*!< 0x00000080 */
#define RCC_ICSCR_HSI_TRIM_8              (0x100UL << RCC_ICSCR_HSI_TRIM_Pos)    /*!< 0x00000100 */
#define RCC_ICSCR_HSI_TRIM_9              (0x200UL << RCC_ICSCR_HSI_TRIM_Pos)    /*!< 0x00000200 */
#define RCC_ICSCR_HSI_TRIM_10             (0x400UL << RCC_ICSCR_HSI_TRIM_Pos)    /*!< 0x00000400 */
#define RCC_ICSCR_HSI_TRIM_11             (0x800UL << RCC_ICSCR_HSI_TRIM_Pos)    /*!< 0x00000800 */
#define RCC_ICSCR_HSI_TRIM_12             (0x1000UL << RCC_ICSCR_HSI_TRIM_Pos)   /*!< 0x00001000 */
#define RCC_ICSCR_HSI_FS_OP_Pos                   (13U)
#define RCC_ICSCR_HSI_FS_OP_Msk                   (0x7UL<<RCC_ICSCR_HSI_FS_OP_Pos)                  /*!< 0x0000E000 */
#define RCC_ICSCR_HSI_FS_OP                       RCC_ICSCR_HSI_FS_OP_Msk
#define RCC_ICSCR_HSI_FS_OP_0                     (0x1UL<<RCC_ICSCR_HSI_FS_OP_Pos)                  /*!< 0x00002000 */
#define RCC_ICSCR_HSI_FS_OP_1                     (0x2UL<<RCC_ICSCR_HSI_FS_OP_Pos)                  /*!< 0x00004000 */
#define RCC_ICSCR_HSI_FS_OP_2                     (0x4UL<<RCC_ICSCR_HSI_FS_OP_Pos)                  /*!< 0x00008000 */
#define RCC_ICSCR_HSI_FS_Pos              (13U)
#define RCC_ICSCR_HSI_FS_Msk              (0x7UL << RCC_ICSCR_HSI_FS_Pos)       /*!< 0x0000E000 */
#define RCC_ICSCR_HSI_FS                  RCC_ICSCR_HSI_FS_Msk                  /*!< HSIFS[15:13] bits */
#define RCC_ICSCR_HSI_FS_0                (0x01UL << RCC_ICSCR_HSI_FS_Pos)      /*!< 0x00002000 */
#define RCC_ICSCR_HSI_FS_1                (0x02UL << RCC_ICSCR_HSI_FS_Pos)      /*!< 0x00004000 */
#define RCC_ICSCR_HSI_FS_2                (0x04UL << RCC_ICSCR_HSI_FS_Pos)      /*!< 0x00008000 */
/********************************* Bit definition for RCC_CFGR register *********************************************/
#define RCC_CFGR_SW_Pos                           (0U)
#define RCC_CFGR_SW_Msk                           (0x7UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000007 */
#define RCC_CFGR_SW                               RCC_CFGR_SW_Msk
#define RCC_CFGR_SW_0                             (0x1UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000001 */
#define RCC_CFGR_SW_1                             (0x2UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000002 */
#define RCC_CFGR_SW_2                             (0x4UL<<RCC_CFGR_SW_Pos)                          /*!< 0x00000004 */
#define RCC_CFGR_SW_HSISYS                        (0x00000000U)                                      /*!< HSI selected as system clock */
#define RCC_CFGR_SW_HSE                           (0x00000001U)                                      /*!< HSE selected as system clock */
#define RCC_CFGR_SW_PLL                           (0x00000002U)
#define RCC_CFGR_SW_LSI                           (0x00000003U)
#define RCC_CFGR_SW_LSE                           (0x00000004U)
#define RCC_CFGR_SW_MSI                           (0x00000005U)
#define RCC_CFGR_SW_FUNCTCLK                      (0x00000006U)
#define RCC_CFGR_SWS_Pos                          (3U)
#define RCC_CFGR_SWS_Msk                          (0x7UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000038 */
#define RCC_CFGR_SWS                              RCC_CFGR_SWS_Msk
#define RCC_CFGR_SWS_0                            (0x1UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000008 */
#define RCC_CFGR_SWS_1                            (0x2UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000010 */
#define RCC_CFGR_SWS_2                            (0x4UL<<RCC_CFGR_SWS_Pos)                         /*!< 0x00000020 */
#define RCC_CFGR_SWS_HSISYS                       (0x00000000U)                                      /*!< HSI selected as system clock */
#define RCC_CFGR_SWS_HSE                          (0x00000008U)                                      /*!< HSE selected as system clock */
#define RCC_CFGR_SWS_PLL                          (0x00000010U)
#define RCC_CFGR_SWS_LSI                          (0x00000018U)
#define RCC_CFGR_SWS_LSE                          (0x00000020U)
#define RCC_CFGR_SWS_MSI                          (0x00000028U)
#define RCC_CFGR_SWS_FUNCTCLK                     (0x00000030U)
#define RCC_CFGR_HPRE_Pos                         (8U)
#define RCC_CFGR_HPRE_Msk                         (0xFUL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000F00 */
#define RCC_CFGR_HPRE                             RCC_CFGR_HPRE_Msk
#define RCC_CFGR_HPRE_0                           (0x1UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000100 */
#define RCC_CFGR_HPRE_1                           (0x2UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000200 */
#define RCC_CFGR_HPRE_2                           (0x4UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000400 */
#define RCC_CFGR_HPRE_3                           (0x8UL<<RCC_CFGR_HPRE_Pos)                        /*!< 0x00000800 */
#define RCC_CFGR_PPRE_Pos                         (12U)
#define RCC_CFGR_PPRE_Msk                         (0x7UL<<RCC_CFGR_PPRE_Pos)                        /*!< 0x00007000 */
#define RCC_CFGR_PPRE                             RCC_CFGR_PPRE_Msk
#define RCC_CFGR_PPRE_0                           (0x1UL<<RCC_CFGR_PPRE_Pos)                        /*!< 0x00001000 */
#define RCC_CFGR_PPRE_1                           (0x2UL<<RCC_CFGR_PPRE_Pos)                        /*!< 0x00002000 */
#define RCC_CFGR_PPRE_2                           (0x4UL<<RCC_CFGR_PPRE_Pos)                        /*!< 0x00004000 */
#define RCC_CFGR_MCOSEL_Pos                       (24U)
#define RCC_CFGR_MCOSEL_Msk                       (0xFUL<<RCC_CFGR_MCOSEL_Pos)                      /*!< 0x0F000000 */
#define RCC_CFGR_MCOSEL                           RCC_CFGR_MCOSEL_Msk
#define RCC_CFGR_MCOSEL_0                         (0x1UL<<RCC_CFGR_MCOSEL_Pos)                      /*!< 0x01000000 */
#define RCC_CFGR_MCOSEL_1                         (0x2UL<<RCC_CFGR_MCOSEL_Pos)                      /*!< 0x02000000 */
#define RCC_CFGR_MCOSEL_2                         (0x4UL<<RCC_CFGR_MCOSEL_Pos)                      /*!< 0x04000000 */
#define RCC_CFGR_MCOSEL_3                         (0x8UL<<RCC_CFGR_MCOSEL_Pos)                      /*!< 0x08000000 */
#define RCC_CFGR_MCOPRE_Pos                       (28U)
#define RCC_CFGR_MCOPRE_Msk                       (0x7UL<<RCC_CFGR_MCOPRE_Pos)                      /*!< 0x70000000 */
#define RCC_CFGR_MCOPRE                           RCC_CFGR_MCOPRE_Msk
#define RCC_CFGR_MCOPRE_0                         (0x1UL<<RCC_CFGR_MCOPRE_Pos)                      /*!< 0x10000000 */
#define RCC_CFGR_MCOPRE_1                         (0x2UL<<RCC_CFGR_MCOPRE_Pos)                      /*!< 0x20000000 */
#define RCC_CFGR_MCOPRE_2                         (0x4UL<<RCC_CFGR_MCOPRE_Pos)                      /*!< 0x40000000 */

/********************************* Bit definition for RCC_PLLCFGR register ******************************************/
#define RCC_PLLCFGR_PLLSRC_Pos                    (0U)
#define RCC_PLLCFGR_PLLSRC_Msk                    (0x1UL<<RCC_PLLCFGR_PLLSRC_Pos)                   /*!< 0x00000001 */
#define RCC_PLLCFGR_PLLSRC                        RCC_PLLCFGR_PLLSRC_Msk                            
#define RCC_PLLCFGR_PLLSRC_HSI_Pos                (0U)
#define RCC_PLLCFGR_PLLSRC_HSI_Msk                (0x0UL << RCC_PLLCFGR_PLLSRC_HSI_Pos) /*!< 0x00000002 */
#define RCC_PLLCFGR_PLLSRC_HSI                    RCC_PLLCFGR_PLLSRC_HSI_Msk            /*!< HSI source clock selected */
#define RCC_PLLCFGR_PLLSRC_HSE_Pos                (0U)
#define RCC_PLLCFGR_PLLSRC_HSE_Msk                (0x1UL << RCC_PLLCFGR_PLLSRC_HSE_Pos) /*!< 0x00000003 */
#define RCC_PLLCFGR_PLLSRC_HSE                    RCC_PLLCFGR_PLLSRC_HSE_Msk            /*!< HSE source clock selected */
#define RCC_PLLCFGR_PLLMUL_Pos                    (2U)
#define RCC_PLLCFGR_PLLMUL_Msk                    (0x1FUL<<RCC_PLLCFGR_PLLMUL_Pos)                  /*!< 0x0000007C */
#define RCC_PLLCFGR_PLLMUL                        RCC_PLLCFGR_PLLMUL_Msk
#define RCC_PLLCFGR_PLLMUL_0                      (0x1UL<<RCC_PLLCFGR_PLLMUL_Pos)                   /*!< 0x00000004 */
#define RCC_PLLCFGR_PLLMUL_1                      (0x2UL<<RCC_PLLCFGR_PLLMUL_Pos)                   /*!< 0x00000008 */
#define RCC_PLLCFGR_PLLMUL_2                      (0x4UL<<RCC_PLLCFGR_PLLMUL_Pos)                   /*!< 0x00000010 */
#define RCC_PLLCFGR_PLLMUL_3                      (0x8UL<<RCC_PLLCFGR_PLLMUL_Pos)                   /*!< 0x00000020 */
#define RCC_PLLCFGR_PLLMUL_4                      (0x10UL<<RCC_PLLCFGR_PLLMUL_Pos)                  /*!< 0x00000040 */

/********************************* Bit definition for RCC_ECSCR register ********************************************/
#define RCC_ECSCR_HSE_DRV_Pos                     (0U)
#define RCC_ECSCR_HSE_DRV_Msk                     (0x3UL<<RCC_ECSCR_HSE_DRV_Pos)                    /*!< 0x00000003 */
#define RCC_ECSCR_HSE_DRV                         RCC_ECSCR_HSE_DRV_Msk
#define RCC_ECSCR_HSE_DRV_0                       (0x1UL<<RCC_ECSCR_HSE_DRV_Pos)                    /*!< 0x00000001 */
#define RCC_ECSCR_HSE_DRV_1                       (0x2UL<<RCC_ECSCR_HSE_DRV_Pos)                    /*!< 0x00000002 */
#define RCC_ECSCR_HSE_RDYSEL_Pos                  (2U)
#define RCC_ECSCR_HSE_RDYSEL_Msk                  (0x3UL<<RCC_ECSCR_HSE_RDYSEL_Pos)                 /*!< 0x0000000C */
#define RCC_ECSCR_HSE_RDYSEL                      RCC_ECSCR_HSE_RDYSEL_Msk
#define RCC_ECSCR_HSE_RDYSEL_0                    (0x1UL<<RCC_ECSCR_HSE_RDYSEL_Pos)                 /*!< 0x00000004 */
#define RCC_ECSCR_HSE_RDYSEL_1                    (0x2UL<<RCC_ECSCR_HSE_RDYSEL_Pos)                 /*!< 0x00000008 */
#define RCC_ECSCR_HSE_FILT_EN_Pos                 (4U)
#define RCC_ECSCR_HSE_FILT_EN_Msk                 (0x1UL<<RCC_ECSCR_HSE_FILT_EN_Pos)                /*!< 0x00000010 */
#define RCC_ECSCR_HSE_FILT_EN                     RCC_ECSCR_HSE_FILT_EN_Msk                         

/********************************* Bit definition for RCC_CIER register *********************************************/
#define RCC_CIER_LSIRDYIE_Pos                     (0U)
#define RCC_CIER_LSIRDYIE_Msk                     (0x1UL<<RCC_CIER_LSIRDYIE_Pos)                    /*!< 0x00000001 */
#define RCC_CIER_LSIRDYIE                         RCC_CIER_LSIRDYIE_Msk                             
#define RCC_CIER_LSERDYIE_Pos                     (1U)
#define RCC_CIER_LSERDYIE_Msk                     (0x1UL<<RCC_CIER_LSERDYIE_Pos)                    /*!< 0x00000002 */
#define RCC_CIER_LSERDYIE                         RCC_CIER_LSERDYIE_Msk                             
#define RCC_CIER_HSIRDYIE_Pos                     (3U)
#define RCC_CIER_HSIRDYIE_Msk                     (0x1UL<<RCC_CIER_HSIRDYIE_Pos)                    /*!< 0x00000008 */
#define RCC_CIER_HSIRDYIE                         RCC_CIER_HSIRDYIE_Msk                             
#define RCC_CIER_HSERDYIE_Pos                     (4U)
#define RCC_CIER_HSERDYIE_Msk                     (0x1UL<<RCC_CIER_HSERDYIE_Pos)                    /*!< 0x00000010 */
#define RCC_CIER_HSERDYIE                         RCC_CIER_HSERDYIE_Msk                             
#define RCC_CIER_PLLRDYIE_Pos                     (5U)
#define RCC_CIER_PLLRDYIE_Msk                     (0x1UL<<RCC_CIER_PLLRDYIE_Pos)                    /*!< 0x00000020 */
#define RCC_CIER_PLLRDYIE                         RCC_CIER_PLLRDYIE_Msk                             

/********************************* Bit definition for RCC_CIFR register *********************************************/
#define RCC_CIFR_LSIRDYF_Pos                      (0U)
#define RCC_CIFR_LSIRDYF_Msk                      (0x1UL<<RCC_CIFR_LSIRDYF_Pos)                     /*!< 0x00000001 */
#define RCC_CIFR_LSIRDYF                          RCC_CIFR_LSIRDYF_Msk                              
#define RCC_CIFR_LSERDYF_Pos                      (1U)
#define RCC_CIFR_LSERDYF_Msk                      (0x1UL<<RCC_CIFR_LSERDYF_Pos)                     /*!< 0x00000002 */
#define RCC_CIFR_LSERDYF                          RCC_CIFR_LSERDYF_Msk                              
#define RCC_CIFR_HSIRDYF_Pos                      (3U)
#define RCC_CIFR_HSIRDYF_Msk                      (0x1UL<<RCC_CIFR_HSIRDYF_Pos)                     /*!< 0x00000008 */
#define RCC_CIFR_HSIRDYF                          RCC_CIFR_HSIRDYF_Msk                              
#define RCC_CIFR_HSERDYF_Pos                      (4U)
#define RCC_CIFR_HSERDYF_Msk                      (0x1UL<<RCC_CIFR_HSERDYF_Pos)                     /*!< 0x00000010 */
#define RCC_CIFR_HSERDYF                          RCC_CIFR_HSERDYF_Msk                              
#define RCC_CIFR_PLLRDYF_Pos                      (5U)
#define RCC_CIFR_PLLRDYF_Msk                      (0x1UL<<RCC_CIFR_PLLRDYF_Pos)                     /*!< 0x00000020 */
#define RCC_CIFR_PLLRDYF                          RCC_CIFR_PLLRDYF_Msk                              
#define RCC_CIFR_CSSF_Pos                         (8U)
#define RCC_CIFR_CSSF_Msk                         (0x1UL<<RCC_CIFR_CSSF_Pos)                        /*!< 0x00000100 */
#define RCC_CIFR_CSSF                             RCC_CIFR_CSSF_Msk                                 
#define RCC_CIFR_LSECSSF_Pos                      (9U)
#define RCC_CIFR_LSECSSF_Msk                      (0x1UL<<RCC_CIFR_LSECSSF_Pos)                     /*!< 0x00000200 */
#define RCC_CIFR_LSECSSF                          RCC_CIFR_LSECSSF_Msk                              

/********************************* Bit definition for RCC_CICR register *********************************************/
#define RCC_CICR_LSIRDYC_Pos                      (0U)
#define RCC_CICR_LSIRDYC_Msk                      (0x1UL<<RCC_CICR_LSIRDYC_Pos)                     /*!< 0x00000001 */
#define RCC_CICR_LSIRDYC                          RCC_CICR_LSIRDYC_Msk                              
#define RCC_CICR_LSERDYC_Pos                      (1U)
#define RCC_CICR_LSERDYC_Msk                      (0x1UL<<RCC_CICR_LSERDYC_Pos)                     /*!< 0x00000002 */
#define RCC_CICR_LSERDYC                          RCC_CICR_LSERDYC_Msk                              
#define RCC_CICR_HSIRDYC_Pos                      (3U)
#define RCC_CICR_HSIRDYC_Msk                      (0x1UL<<RCC_CICR_HSIRDYC_Pos)                     /*!< 0x00000008 */
#define RCC_CICR_HSIRDYC                          RCC_CICR_HSIRDYC_Msk                              
#define RCC_CICR_HSERDYC_Pos                      (4U)
#define RCC_CICR_HSERDYC_Msk                      (0x1UL<<RCC_CICR_HSERDYC_Pos)                     /*!< 0x00000010 */
#define RCC_CICR_HSERDYC                          RCC_CICR_HSERDYC_Msk                              
#define RCC_CICR_PLLRDYC_Pos                      (5U)
#define RCC_CICR_PLLRDYC_Msk                      (0x1UL<<RCC_CICR_PLLRDYC_Pos)                     /*!< 0x00000020 */
#define RCC_CICR_PLLRDYC                          RCC_CICR_PLLRDYC_Msk                              
#define RCC_CICR_CSSC_Pos                         (8U)
#define RCC_CICR_CSSC_Msk                         (0x1UL<<RCC_CICR_CSSC_Pos)                        /*!< 0x00000100 */
#define RCC_CICR_CSSC                             RCC_CICR_CSSC_Msk                                 
#define RCC_CICR_LSECSSC_Pos                      (9U)
#define RCC_CICR_LSECSSC_Msk                      (0x1UL<<RCC_CICR_LSECSSC_Pos)                     /*!< 0x00000200 */
#define RCC_CICR_LSECSSC                          RCC_CICR_LSECSSC_Msk                              

/********************************* Bit definition for RCC_IOPRSTR register ******************************************/
#define RCC_IOPRSTR_GPIOARST_Pos                  (0U)
#define RCC_IOPRSTR_GPIOARST_Msk                  (0x1UL<<RCC_IOPRSTR_GPIOARST_Pos)                 /*!< 0x00000001 */
#define RCC_IOPRSTR_GPIOARST                      RCC_IOPRSTR_GPIOARST_Msk                          
#define RCC_IOPRSTR_GPIOBRST_Pos                  (1U)
#define RCC_IOPRSTR_GPIOBRST_Msk                  (0x1UL<<RCC_IOPRSTR_GPIOBRST_Pos)                 /*!< 0x00000002 */
#define RCC_IOPRSTR_GPIOBRST                      RCC_IOPRSTR_GPIOBRST_Msk                          
#define RCC_IOPRSTR_GPIOCRST_Pos                  (2U)
#define RCC_IOPRSTR_GPIOCRST_Msk                  (0x1UL<<RCC_IOPRSTR_GPIOCRST_Pos)                 /*!< 0x00000004 */
#define RCC_IOPRSTR_GPIOCRST                      RCC_IOPRSTR_GPIOCRST_Msk                          
#define RCC_IOPRSTR_GPIODRST_Pos                  (3U)
#define RCC_IOPRSTR_GPIODRST_Msk                  (0x1UL<<RCC_IOPRSTR_GPIODRST_Pos)                 /*!< 0x00000008 */
#define RCC_IOPRSTR_GPIODRST                      RCC_IOPRSTR_GPIODRST_Msk                          

/********************************* Bit definition for RCC_AHBRSTR register ******************************************/
#define RCC_AHBRSTR_DMARST_Pos                    (0U)
#define RCC_AHBRSTR_DMARST_Msk                    (0x1UL<<RCC_AHBRSTR_DMARST_Pos)                   /*!< 0x00000001 */
#define RCC_AHBRSTR_DMARST                        RCC_AHBRSTR_DMARST_Msk                            
#define RCC_AHBRSTR_CRCRST_Pos                    (12U)
#define RCC_AHBRSTR_CRCRST_Msk                    (0x1UL<<RCC_AHBRSTR_CRCRST_Pos)                   /*!< 0x00001000 */
#define RCC_AHBRSTR_CRCRST                        RCC_AHBRSTR_CRCRST_Msk                            

/********************************* Bit definition for RCC_APBRSTR1 register *****************************************/
#define RCC_APBRSTR1_TIM2RST_Pos                  (0U)
#define RCC_APBRSTR1_TIM2RST_Msk                  (0x1UL<<RCC_APBRSTR1_TIM2RST_Pos)                 /*!< 0x00000001 */
#define RCC_APBRSTR1_TIM2RST                      RCC_APBRSTR1_TIM2RST_Msk                          
#define RCC_APBRSTR1_TIM3RST_Pos                  (1U)
#define RCC_APBRSTR1_TIM3RST_Msk                  (0x1UL<<RCC_APBRSTR1_TIM3RST_Pos)                 /*!< 0x00000002 */
#define RCC_APBRSTR1_TIM3RST                      RCC_APBRSTR1_TIM3RST_Msk                          
#define RCC_APBRSTR1_TIM6RST_Pos                  (4U)
#define RCC_APBRSTR1_TIM6RST_Msk                  (0x1UL<<RCC_APBRSTR1_TIM6RST_Pos)                 /*!< 0x00000010 */
#define RCC_APBRSTR1_TIM6RST                      RCC_APBRSTR1_TIM6RST_Msk                          
#define RCC_APBRSTR1_TIM7RST_Pos                  (5U)
#define RCC_APBRSTR1_TIM7RST_Msk                  (0x1UL<<RCC_APBRSTR1_TIM7RST_Pos)                 /*!< 0x00000020 */
#define RCC_APBRSTR1_TIM7RST                      RCC_APBRSTR1_TIM7RST_Msk                          
#define RCC_APBRSTR1_PWMRST_Pos                   (6U)
#define RCC_APBRSTR1_PWMRST_Msk                   (0x1UL<<RCC_APBRSTR1_PWMRST_Pos)                  /*!< 0x00000040 */
#define RCC_APBRSTR1_PWMRST                       RCC_APBRSTR1_PWMRST_Msk                           
#define RCC_APBRSTR1_RTCAPBRST_Pos                (10U)
#define RCC_APBRSTR1_RTCAPBRST_Msk                (0x1UL<<RCC_APBRSTR1_RTCAPBRST_Pos)               /*!< 0x00000400 */
#define RCC_APBRSTR1_RTCAPBRST                    RCC_APBRSTR1_RTCAPBRST_Msk                        
#define RCC_APBRSTR1_WWDGRST_Pos                  (11U)
#define RCC_APBRSTR1_WWDGRST_Msk                  (0x1UL<<RCC_APBRSTR1_WWDGRST_Pos)                 /*!< 0x00000800 */
#define RCC_APBRSTR1_WWDGRST                      RCC_APBRSTR1_WWDGRST_Msk                          
#define RCC_APBRSTR1_SPI2RST_Pos                  (14U)
#define RCC_APBRSTR1_SPI2RST_Msk                  (0x1UL<<RCC_APBRSTR1_SPI2RST_Pos)                 /*!< 0x00004000 */
#define RCC_APBRSTR1_SPI2RST                      RCC_APBRSTR1_SPI2RST_Msk                          
#define RCC_APBRSTR1_LPUART2RST_Pos               (16U)
#define RCC_APBRSTR1_LPUART2RST_Msk               (0x1UL<<RCC_APBRSTR1_LPUART2RST_Pos)              /*!< 0x00010000 */
#define RCC_APBRSTR1_LPUART2RST                   RCC_APBRSTR1_LPUART2RST_Msk                       
#define RCC_APBRSTR1_USART2RST_Pos                (17U)
#define RCC_APBRSTR1_USART2RST_Msk                (0x1UL<<RCC_APBRSTR1_USART2RST_Pos)               /*!< 0x00020000 */
#define RCC_APBRSTR1_USART2RST                    RCC_APBRSTR1_USART2RST_Msk                        
#define RCC_APBRSTR1_UART1RST_Pos                 (18U)
#define RCC_APBRSTR1_UART1RST_Msk                 (0x1UL<<RCC_APBRSTR1_UART1RST_Pos)                /*!< 0x00040000 */
#define RCC_APBRSTR1_UART1RST                     RCC_APBRSTR1_UART1RST_Msk                         
#define RCC_APBRSTR1_UART2RST_Pos                 (19U)
#define RCC_APBRSTR1_UART2RST_Msk                 (0x1UL<<RCC_APBRSTR1_UART2RST_Pos)                /*!< 0x00080000 */
#define RCC_APBRSTR1_UART2RST                     RCC_APBRSTR1_UART2RST_Msk                         
#define RCC_APBRSTR1_LPUART1RST_Pos               (20U)
#define RCC_APBRSTR1_LPUART1RST_Msk               (0x1UL<<RCC_APBRSTR1_LPUART1RST_Pos)              /*!< 0x00100000 */
#define RCC_APBRSTR1_LPUART1RST                   RCC_APBRSTR1_LPUART1RST_Msk                       
#define RCC_APBRSTR1_I2C1RST_Pos                  (21U)
#define RCC_APBRSTR1_I2C1RST_Msk                  (0x1UL<<RCC_APBRSTR1_I2C1RST_Pos)                 /*!< 0x00200000 */
#define RCC_APBRSTR1_I2C1RST                      RCC_APBRSTR1_I2C1RST_Msk                          
#define RCC_APBRSTR1_I2C2RST_Pos                  (22U)
#define RCC_APBRSTR1_I2C2RST_Msk                  (0x1UL<<RCC_APBRSTR1_I2C2RST_Pos)                 /*!< 0x00400000 */
#define RCC_APBRSTR1_I2C2RST                      RCC_APBRSTR1_I2C2RST_Msk                          
#define RCC_APBRSTR1_OPARST_Pos                   (23U)
#define RCC_APBRSTR1_OPARST_Msk                   (0x1UL<<RCC_APBRSTR1_OPARST_Pos)                  /*!< 0x00800000 */
#define RCC_APBRSTR1_OPARST                       RCC_APBRSTR1_OPARST_Msk                           
#define RCC_APBRSTR1_CANRST_Pos                   (25U)
#define RCC_APBRSTR1_CANRST_Msk                   (0x1UL<<RCC_APBRSTR1_CANRST_Pos)                  /*!< 0x02000000 */
#define RCC_APBRSTR1_CANRST                       RCC_APBRSTR1_CANRST_Msk                           
#define RCC_APBRSTR1_DBGRST_Pos                   (27U)
#define RCC_APBRSTR1_DBGRST_Msk                   (0x1UL<<RCC_APBRSTR1_DBGRST_Pos)                  /*!< 0x08000000 */
#define RCC_APBRSTR1_DBGRST                       RCC_APBRSTR1_DBGRST_Msk                           
#define RCC_APBRSTR1_PWRRST_Pos                   (28U)
#define RCC_APBRSTR1_PWRRST_Msk                   (0x1UL<<RCC_APBRSTR1_PWRRST_Pos)                  /*!< 0x10000000 */
#define RCC_APBRSTR1_PWRRST                       RCC_APBRSTR1_PWRRST_Msk                           
#define RCC_APBRSTR1_DAC1RST_Pos                  (29U)
#define RCC_APBRSTR1_DAC1RST_Msk                  (0x1UL<<RCC_APBRSTR1_DAC1RST_Pos)                 /*!< 0x20000000 */
#define RCC_APBRSTR1_DAC1RST                      RCC_APBRSTR1_DAC1RST_Msk                          
#define RCC_APBRSTR1_LPTIM2RST_Pos                (30U)
#define RCC_APBRSTR1_LPTIM2RST_Msk                (0x1UL<<RCC_APBRSTR1_LPTIM2RST_Pos)               /*!< 0x40000000 */
#define RCC_APBRSTR1_LPTIM2RST                    RCC_APBRSTR1_LPTIM2RST_Msk                        
#define RCC_APBRSTR1_LPTIM1RST_Pos                (31U)
#define RCC_APBRSTR1_LPTIM1RST_Msk                (0x1UL<<RCC_APBRSTR1_LPTIM1RST_Pos)               /*!< 0x80000000 */
#define RCC_APBRSTR1_LPTIM1RST                    RCC_APBRSTR1_LPTIM1RST_Msk                        

/********************************* Bit definition for RCC_APBRSTR2 register *****************************************/
#define RCC_APBRSTR2_SYSCFGRST_Pos                (0U)
#define RCC_APBRSTR2_SYSCFGRST_Msk                (0x1UL<<RCC_APBRSTR2_SYSCFGRST_Pos)               /*!< 0x00000001 */
#define RCC_APBRSTR2_SYSCFGRST                    RCC_APBRSTR2_SYSCFGRST_Msk                        
#define RCC_APBRSTR2_TIM1RST_Pos                  (11U)
#define RCC_APBRSTR2_TIM1RST_Msk                  (0x1UL<<RCC_APBRSTR2_TIM1RST_Pos)                 /*!< 0x00000800 */
#define RCC_APBRSTR2_TIM1RST                      RCC_APBRSTR2_TIM1RST_Msk                          
#define RCC_APBRSTR2_SPI1RST_Pos                  (12U)
#define RCC_APBRSTR2_SPI1RST_Msk                  (0x1UL<<RCC_APBRSTR2_SPI1RST_Pos)                 /*!< 0x00001000 */
#define RCC_APBRSTR2_SPI1RST                      RCC_APBRSTR2_SPI1RST_Msk                          
#define RCC_APBRSTR2_USART1RST_Pos                (14U)
#define RCC_APBRSTR2_USART1RST_Msk                (0x1UL<<RCC_APBRSTR2_USART1RST_Pos)               /*!< 0x00004000 */
#define RCC_APBRSTR2_USART1RST                    RCC_APBRSTR2_USART1RST_Msk                        
#define RCC_APBRSTR2_TIM15RST_Pos                 (16U)
#define RCC_APBRSTR2_TIM15RST_Msk                 (0x1UL<<RCC_APBRSTR2_TIM15RST_Pos)                /*!< 0x00010000 */
#define RCC_APBRSTR2_TIM15RST                     RCC_APBRSTR2_TIM15RST_Msk                         
#define RCC_APBRSTR2_TIM16RST_Pos                 (17U)
#define RCC_APBRSTR2_TIM16RST_Msk                 (0x1UL<<RCC_APBRSTR2_TIM16RST_Pos)                /*!< 0x00020000 */
#define RCC_APBRSTR2_TIM16RST                     RCC_APBRSTR2_TIM16RST_Msk                         
#define RCC_APBRSTR2_TIM17RST_Pos                 (18U)
#define RCC_APBRSTR2_TIM17RST_Msk                 (0x1UL<<RCC_APBRSTR2_TIM17RST_Pos)                /*!< 0x00040000 */
#define RCC_APBRSTR2_TIM17RST                     RCC_APBRSTR2_TIM17RST_Msk                         
#define RCC_APBRSTR2_ADCRST_Pos                   (20U)
#define RCC_APBRSTR2_ADCRST_Msk                   (0x1UL<<RCC_APBRSTR2_ADCRST_Pos)                  /*!< 0x00100000 */
#define RCC_APBRSTR2_ADCRST                       RCC_APBRSTR2_ADCRST_Msk                           
#define RCC_APBRSTR2_COMP1RST_Pos                 (21U)
#define RCC_APBRSTR2_COMP1RST_Msk                 (0x1UL<<RCC_APBRSTR2_COMP1RST_Pos)                /*!< 0x00200000 */
#define RCC_APBRSTR2_COMP1RST                     RCC_APBRSTR2_COMP1RST_Msk                         
#define RCC_APBRSTR2_COMP2RST_Pos                 (22U)
#define RCC_APBRSTR2_COMP2RST_Msk                 (0x1UL<<RCC_APBRSTR2_COMP2RST_Pos)                /*!< 0x00400000 */
#define RCC_APBRSTR2_COMP2RST                     RCC_APBRSTR2_COMP2RST_Msk                         
#define RCC_APBRSTR2_LCDRST_Pos                   (24U)
#define RCC_APBRSTR2_LCDRST_Msk                   (0x1UL<<RCC_APBRSTR2_LCDRST_Pos)                  /*!< 0x01000000 */
#define RCC_APBRSTR2_LCDRST                       RCC_APBRSTR2_LCDRST_Msk                           
#define RCC_APBRSTR2_TKRST_Pos                    (25U)
#define RCC_APBRSTR2_TKRST_Msk                    (0x1UL<<RCC_APBRSTR2_TKRST_Pos)                   /*!< 0x02000000 */
#define RCC_APBRSTR2_TKRST                        RCC_APBRSTR2_TKRST_Msk                            

/********************************* Bit definition for RCC_IOPENR register *******************************************/
#define RCC_IOPENR_GPIOAEN_Pos                    (0U)
#define RCC_IOPENR_GPIOAEN_Msk                    (0x1UL<<RCC_IOPENR_GPIOAEN_Pos)                   /*!< 0x00000001 */
#define RCC_IOPENR_GPIOAEN                        RCC_IOPENR_GPIOAEN_Msk                            
#define RCC_IOPENR_GPIOBEN_Pos                    (1U)
#define RCC_IOPENR_GPIOBEN_Msk                    (0x1UL<<RCC_IOPENR_GPIOBEN_Pos)                   /*!< 0x00000002 */
#define RCC_IOPENR_GPIOBEN                        RCC_IOPENR_GPIOBEN_Msk                            
#define RCC_IOPENR_GPIOCEN_Pos                    (2U)
#define RCC_IOPENR_GPIOCEN_Msk                    (0x1UL<<RCC_IOPENR_GPIOCEN_Pos)                   /*!< 0x00000004 */
#define RCC_IOPENR_GPIOCEN                        RCC_IOPENR_GPIOCEN_Msk                            
#define RCC_IOPENR_GPIODEN_Pos                    (3U)
#define RCC_IOPENR_GPIODEN_Msk                    (0x1UL<<RCC_IOPENR_GPIODEN_Pos)                   /*!< 0x00000008 */
#define RCC_IOPENR_GPIODEN                        RCC_IOPENR_GPIODEN_Msk                            

/********************************* Bit definition for RCC_AHBENR register *******************************************/
#define RCC_AHBENR_DMAEN_Pos                      (0U)
#define RCC_AHBENR_DMAEN_Msk                      (0x1UL<<RCC_AHBENR_DMAEN_Pos)                     /*!< 0x00000001 */
#define RCC_AHBENR_DMAEN                          RCC_AHBENR_DMAEN_Msk                              
#define RCC_AHBENR_FLASHEN_Pos                    (8U)
#define RCC_AHBENR_FLASHEN_Msk                    (0x1UL<<RCC_AHBENR_FLASHEN_Pos)                   /*!< 0x00000100 */
#define RCC_AHBENR_FLASHEN                        RCC_AHBENR_FLASHEN_Msk                            
#define RCC_AHBENR_SRAMEN_Pos                     (9U)
#define RCC_AHBENR_SRAMEN_Msk                     (0x1UL<<RCC_AHBENR_SRAMEN_Pos)                    /*!< 0x00000200 */
#define RCC_AHBENR_SRAMEN                         RCC_AHBENR_SRAMEN_Msk                             
#define RCC_AHBENR_CRCEN_Pos                      (12U)
#define RCC_AHBENR_CRCEN_Msk                      (0x1UL<<RCC_AHBENR_CRCEN_Pos)                     /*!< 0x00001000 */
#define RCC_AHBENR_CRCEN                          RCC_AHBENR_CRCEN_Msk                              

/********************************* Bit definition for RCC_APBENR1 register ******************************************/
#define RCC_APBENR1_TIM2EN_Pos                    (0U)
#define RCC_APBENR1_TIM2EN_Msk                    (0x1UL<<RCC_APBENR1_TIM2EN_Pos)                   /*!< 0x00000001 */
#define RCC_APBENR1_TIM2EN                        RCC_APBENR1_TIM2EN_Msk                            
#define RCC_APBENR1_TIM3EN_Pos                    (1U)
#define RCC_APBENR1_TIM3EN_Msk                    (0x1UL<<RCC_APBENR1_TIM3EN_Pos)                   /*!< 0x00000002 */
#define RCC_APBENR1_TIM3EN                        RCC_APBENR1_TIM3EN_Msk                            
#define RCC_APBENR1_TIM6EN_Pos                    (4U)
#define RCC_APBENR1_TIM6EN_Msk                    (0x1UL<<RCC_APBENR1_TIM6EN_Pos)                   /*!< 0x00000010 */
#define RCC_APBENR1_TIM6EN                        RCC_APBENR1_TIM6EN_Msk                            
#define RCC_APBENR1_TIM7EN_Pos                    (5U)
#define RCC_APBENR1_TIM7EN_Msk                    (0x1UL<<RCC_APBENR1_TIM7EN_Pos)                   /*!< 0x00000020 */
#define RCC_APBENR1_TIM7EN                        RCC_APBENR1_TIM7EN_Msk                            
#define RCC_APBENR1_PWMEN_Pos                     (6U)
#define RCC_APBENR1_PWMEN_Msk                     (0x1UL<<RCC_APBENR1_PWMEN_Pos)                    /*!< 0x00000040 */
#define RCC_APBENR1_PWMEN                         RCC_APBENR1_PWMEN_Msk                             
#define RCC_APBENR1_RTCAPBEN_Pos                  (10U)
#define RCC_APBENR1_RTCAPBEN_Msk                  (0x1UL<<RCC_APBENR1_RTCAPBEN_Pos)                 /*!< 0x00000400 */
#define RCC_APBENR1_RTCAPBEN                      RCC_APBENR1_RTCAPBEN_Msk                          
#define RCC_APBENR1_WWDGEN_Pos                    (11U)
#define RCC_APBENR1_WWDGEN_Msk                    (0x1UL<<RCC_APBENR1_WWDGEN_Pos)                   /*!< 0x00000800 */
#define RCC_APBENR1_WWDGEN                        RCC_APBENR1_WWDGEN_Msk                            
#define RCC_APBENR1_SPI2EN_Pos                    (14U)
#define RCC_APBENR1_SPI2EN_Msk                    (0x1UL<<RCC_APBENR1_SPI2EN_Pos)                   /*!< 0x00004000 */
#define RCC_APBENR1_SPI2EN                        RCC_APBENR1_SPI2EN_Msk                            
#define RCC_APBENR1_LPUART2EN_Pos                 (16U)
#define RCC_APBENR1_LPUART2EN_Msk                 (0x1UL<<RCC_APBENR1_LPUART2EN_Pos)                /*!< 0x00010000 */
#define RCC_APBENR1_LPUART2EN                     RCC_APBENR1_LPUART2EN_Msk                         
#define RCC_APBENR1_USART2EN_Pos                  (17U)
#define RCC_APBENR1_USART2EN_Msk                  (0x1UL<<RCC_APBENR1_USART2EN_Pos)                 /*!< 0x00020000 */
#define RCC_APBENR1_USART2EN                      RCC_APBENR1_USART2EN_Msk                          
#define RCC_APBENR1_UART1EN_Pos                   (18U)
#define RCC_APBENR1_UART1EN_Msk                   (0x1UL<<RCC_APBENR1_UART1EN_Pos)                  /*!< 0x00040000 */
#define RCC_APBENR1_UART1EN                       RCC_APBENR1_UART1EN_Msk                           
#define RCC_APBENR1_UART2EN_Pos                   (19U)
#define RCC_APBENR1_UART2EN_Msk                   (0x1UL<<RCC_APBENR1_UART2EN_Pos)                  /*!< 0x00080000 */
#define RCC_APBENR1_UART2EN                       RCC_APBENR1_UART2EN_Msk                           
#define RCC_APBENR1_LPUART1EN_Pos                 (20U)
#define RCC_APBENR1_LPUART1EN_Msk                 (0x1UL<<RCC_APBENR1_LPUART1EN_Pos)                /*!< 0x00100000 */
#define RCC_APBENR1_LPUART1EN                     RCC_APBENR1_LPUART1EN_Msk                         
#define RCC_APBENR1_I2C1EN_Pos                    (21U)
#define RCC_APBENR1_I2C1EN_Msk                    (0x1UL<<RCC_APBENR1_I2C1EN_Pos)                   /*!< 0x00200000 */
#define RCC_APBENR1_I2C1EN                        RCC_APBENR1_I2C1EN_Msk                            
#define RCC_APBENR1_I2C2EN_Pos                    (22U)
#define RCC_APBENR1_I2C2EN_Msk                    (0x1UL<<RCC_APBENR1_I2C2EN_Pos)                   /*!< 0x00400000 */
#define RCC_APBENR1_I2C2EN                        RCC_APBENR1_I2C2EN_Msk                            
#define RCC_APBENR1_OPAEN_Pos                     (23U)
#define RCC_APBENR1_OPAEN_Msk                     (0x1UL<<RCC_APBENR1_OPAEN_Pos)                    /*!< 0x00800000 */
#define RCC_APBENR1_OPAEN                         RCC_APBENR1_OPAEN_Msk                             
#define RCC_APBENR1_CANEN_Pos                     (25U)
#define RCC_APBENR1_CANEN_Msk                     (0x1UL<<RCC_APBENR1_CANEN_Pos)                    /*!< 0x02000000 */
#define RCC_APBENR1_CANEN                         RCC_APBENR1_CANEN_Msk                             
#define RCC_APBENR1_DBGEN_Pos                     (27U)
#define RCC_APBENR1_DBGEN_Msk                     (0x1UL<<RCC_APBENR1_DBGEN_Pos)                    /*!< 0x08000000 */
#define RCC_APBENR1_DBGEN                         RCC_APBENR1_DBGEN_Msk                             
#define RCC_APBENR1_PWREN_Pos                     (28U)
#define RCC_APBENR1_PWREN_Msk                     (0x1UL<<RCC_APBENR1_PWREN_Pos)                    /*!< 0x10000000 */
#define RCC_APBENR1_PWREN                         RCC_APBENR1_PWREN_Msk                             
#define RCC_APBENR1_DACEN_Pos                     (29U)
#define RCC_APBENR1_DACEN_Msk                     (0x1UL<<RCC_APBENR1_DACEN_Pos)                    /*!< 0x20000000 */
#define RCC_APBENR1_DACEN                         RCC_APBENR1_DACEN_Msk                             
#define RCC_APBENR1_LPTIM2EN_Pos                  (30U)
#define RCC_APBENR1_LPTIM2EN_Msk                  (0x1UL<<RCC_APBENR1_LPTIM2EN_Pos)                 /*!< 0x40000000 */
#define RCC_APBENR1_LPTIM2EN                      RCC_APBENR1_LPTIM2EN_Msk                          
#define RCC_APBENR1_LPTIM1EN_Pos                  (31U)
#define RCC_APBENR1_LPTIM1EN_Msk                  (0x1UL<<RCC_APBENR1_LPTIM1EN_Pos)                 /*!< 0x80000000 */
#define RCC_APBENR1_LPTIM1EN                      RCC_APBENR1_LPTIM1EN_Msk                          

/********************************* Bit definition for RCC_APBENR2 register ******************************************/
#define RCC_APBENR2_SYSCFGEN_Pos                  (0U)
#define RCC_APBENR2_SYSCFGEN_Msk                  (0x1UL<<RCC_APBENR2_SYSCFGEN_Pos)                 /*!< 0x00000001 */
#define RCC_APBENR2_SYSCFGEN                      RCC_APBENR2_SYSCFGEN_Msk                          
#define RCC_APBENR2_TIM1EN_Pos                    (11U)
#define RCC_APBENR2_TIM1EN_Msk                    (0x1UL<<RCC_APBENR2_TIM1EN_Pos)                   /*!< 0x00000800 */
#define RCC_APBENR2_TIM1EN                        RCC_APBENR2_TIM1EN_Msk                            
#define RCC_APBENR2_SPI1EN_Pos                    (12U)
#define RCC_APBENR2_SPI1EN_Msk                    (0x1UL<<RCC_APBENR2_SPI1EN_Pos)                   /*!< 0x00001000 */
#define RCC_APBENR2_SPI1EN                        RCC_APBENR2_SPI1EN_Msk                            
#define RCC_APBENR2_USART1EN_Pos                  (14U)
#define RCC_APBENR2_USART1EN_Msk                  (0x1UL<<RCC_APBENR2_USART1EN_Pos)                 /*!< 0x00004000 */
#define RCC_APBENR2_USART1EN                      RCC_APBENR2_USART1EN_Msk                          
#define RCC_APBENR2_TIM15EN_Pos                   (16U)
#define RCC_APBENR2_TIM15EN_Msk                   (0x1UL<<RCC_APBENR2_TIM15EN_Pos)                  /*!< 0x00010000 */
#define RCC_APBENR2_TIM15EN                       RCC_APBENR2_TIM15EN_Msk                           
#define RCC_APBENR2_TIM16EN_Pos                   (17U)
#define RCC_APBENR2_TIM16EN_Msk                   (0x1UL<<RCC_APBENR2_TIM16EN_Pos)                  /*!< 0x00020000 */
#define RCC_APBENR2_TIM16EN                       RCC_APBENR2_TIM16EN_Msk                           
#define RCC_APBENR2_TIM17EN_Pos                   (18U)
#define RCC_APBENR2_TIM17EN_Msk                   (0x1UL<<RCC_APBENR2_TIM17EN_Pos)                  /*!< 0x00040000 */
#define RCC_APBENR2_TIM17EN                       RCC_APBENR2_TIM17EN_Msk                           
#define RCC_APBENR2_ADCEN_Pos                     (20U)
#define RCC_APBENR2_ADCEN_Msk                     (0x1UL<<RCC_APBENR2_ADCEN_Pos)                    /*!< 0x00100000 */
#define RCC_APBENR2_ADCEN                         RCC_APBENR2_ADCEN_Msk                             
#define RCC_APBENR2_COMP1EN_Pos                   (21U)
#define RCC_APBENR2_COMP1EN_Msk                   (0x1UL<<RCC_APBENR2_COMP1EN_Pos)                  /*!< 0x00200000 */
#define RCC_APBENR2_COMP1EN                       RCC_APBENR2_COMP1EN_Msk                           
#define RCC_APBENR2_COMP2EN_Pos                   (22U)
#define RCC_APBENR2_COMP2EN_Msk                   (0x1UL<<RCC_APBENR2_COMP2EN_Pos)                  /*!< 0x00400000 */
#define RCC_APBENR2_COMP2EN                       RCC_APBENR2_COMP2EN_Msk                           
#define RCC_APBENR2_COMP3EN_Pos                   (23U)
#define RCC_APBENR2_COMP3EN_Msk                   (0x1UL<<RCC_APBENR2_COMP3EN_Pos)                  /*!< 0x00800000 */
#define RCC_APBENR2_COMP3EN                       RCC_APBENR2_COMP3EN_Msk                           
#define RCC_APBENR2_LCDEN_Pos                     (24U)
#define RCC_APBENR2_LCDEN_Msk                     (0x1UL<<RCC_APBENR2_LCDEN_Pos)                    /*!< 0x01000000 */
#define RCC_APBENR2_LCDEN                         RCC_APBENR2_LCDEN_Msk                             
#define RCC_APBENR2_TKEN_Pos                      (25U)
#define RCC_APBENR2_TKEN_Msk                      (0x1UL<<RCC_APBENR2_TKEN_Pos)                     /*!< 0x02000000 */
#define RCC_APBENR2_TKEN                          RCC_APBENR2_TKEN_Msk                              

/********************************* Bit definition for RCC_CCIPR register ********************************************/
#define RCC_CCIPR_TIMCLKCTRL_Pos                  (0U)
#define RCC_CCIPR_TIMCLKCTRL_Msk                  (0x1UL<<RCC_CCIPR_TIMCLKCTRL_Pos)                 /*!< 0x00000001 */
#define RCC_CCIPR_TIMCLKCTRL                      RCC_CCIPR_TIMCLKCTRL_Msk                          
#define RCC_CCIPR_CANSEL_Pos                      (5U)
#define RCC_CCIPR_CANSEL_Msk                      (0x3UL<<RCC_CCIPR_CANSEL_Pos)                     /*!< 0x00000060 */
#define RCC_CCIPR_CANSEL                          RCC_CCIPR_CANSEL_Msk
#define RCC_CCIPR_CANSEL_0                        (0x1UL<<RCC_CCIPR_CANSEL_Pos)                     /*!< 0x00000020 */
#define RCC_CCIPR_CANSEL_1                        (0x2UL<<RCC_CCIPR_CANSEL_Pos)                     /*!< 0x00000040 */
#define RCC_CCIPR_PVDSEL_Pos                      (7U)
#define RCC_CCIPR_PVDSEL_Msk                      (0x1UL<<RCC_CCIPR_PVDSEL_Pos)                     /*!< 0x00000080 */
#define RCC_CCIPR_PVDSEL                          RCC_CCIPR_PVDSEL_Msk                              
#define RCC_CCIPR_COMP1SEL_Pos                    (8U)
#define RCC_CCIPR_COMP1SEL_Msk                    (0x1UL<<RCC_CCIPR_COMP1SEL_Pos)                   /*!< 0x00000100 */
#define RCC_CCIPR_COMP1SEL                        RCC_CCIPR_COMP1SEL_Msk                            
#define RCC_CCIPR_COMP2SEL_Pos                    (9U)
#define RCC_CCIPR_COMP2SEL_Msk                    (0x1UL<<RCC_CCIPR_COMP2SEL_Pos)                   /*!< 0x00000200 */
#define RCC_CCIPR_COMP2SEL                        RCC_CCIPR_COMP2SEL_Msk                            
#define RCC_CCIPR_COMP3SEL_Pos                    (10U)
#define RCC_CCIPR_COMP3SEL_Msk                    (0x1UL<<RCC_CCIPR_COMP3SEL_Pos)                   /*!< 0x00000400 */
#define RCC_CCIPR_COMP3SEL                        RCC_CCIPR_COMP3SEL_Msk                            
#define RCC_CCIPR_LPUART1SEL_Pos                  (11U)
#define RCC_CCIPR_LPUART1SEL_Msk                  (0x3UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00001800 */
#define RCC_CCIPR_LPUART1SEL                      RCC_CCIPR_LPUART1SEL_Msk
#define RCC_CCIPR_LPUART1SEL_0                    (0x1UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00000800 */
#define RCC_CCIPR_LPUART1SEL_1                    (0x2UL<<RCC_CCIPR_LPUART1SEL_Pos)                 /*!< 0x00001000 */
#define RCC_CCIPR_LPUART2SEL_Pos                  (13U)
#define RCC_CCIPR_LPUART2SEL_Msk                  (0x3UL<<RCC_CCIPR_LPUART2SEL_Pos)                 /*!< 0x00006000 */
#define RCC_CCIPR_LPUART2SEL                      RCC_CCIPR_LPUART2SEL_Msk
#define RCC_CCIPR_LPUART2SEL_0                    (0x1UL<<RCC_CCIPR_LPUART2SEL_Pos)                 /*!< 0x00002000 */
#define RCC_CCIPR_LPUART2SEL_1                    (0x2UL<<RCC_CCIPR_LPUART2SEL_Pos)                 /*!< 0x00004000 */
#define RCC_CCIPR_IWDGSEL_Pos                     (15U)
#define RCC_CCIPR_IWDGSEL_Msk                     (0x1UL<<RCC_CCIPR_IWDGSEL_Pos)                    /*!< 0x00008000 */
#define RCC_CCIPR_IWDGSEL                         RCC_CCIPR_IWDGSEL_Msk                             
#define RCC_CCIPR_LPTIM1SEL_Pos                   (16U)
#define RCC_CCIPR_LPTIM1SEL_Msk                   (0x3UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x00030000 */
#define RCC_CCIPR_LPTIM1SEL                       RCC_CCIPR_LPTIM1SEL_Msk
#define RCC_CCIPR_LPTIM1SEL_0                     (0x1UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x00010000 */
#define RCC_CCIPR_LPTIM1SEL_1                     (0x2UL<<RCC_CCIPR_LPTIM1SEL_Pos)                  /*!< 0x00020000 */
#define RCC_CCIPR_LPTIM2SEL_Pos                   (18U)
#define RCC_CCIPR_LPTIM2SEL_Msk                   (0x3UL<<RCC_CCIPR_LPTIM2SEL_Pos)                  /*!< 0x000C0000 */
#define RCC_CCIPR_LPTIM2SEL                       RCC_CCIPR_LPTIM2SEL_Msk
#define RCC_CCIPR_LPTIM2SEL_0                     (0x1UL<<RCC_CCIPR_LPTIM2SEL_Pos)                  /*!< 0x00040000 */
#define RCC_CCIPR_LPTIM2SEL_1                     (0x2UL<<RCC_CCIPR_LPTIM2SEL_Pos)                  /*!< 0x00080000 */
#define RCC_CCIPR_PWMSEL_Pos                      (21U)
#define RCC_CCIPR_PWMSEL_Msk                      (0x1UL<<RCC_CCIPR_PWMSEL_Pos)                     /*!< 0x00200000 */
#define RCC_CCIPR_PWMSEL                          RCC_CCIPR_PWMSEL_Msk                              
#define RCC_CCIPR_ADCSEL_Pos                      (30U)
#define RCC_CCIPR_ADCSEL_Msk                      (0x3UL<<RCC_CCIPR_ADCSEL_Pos)                     /*!< 0xC0000000 */
#define RCC_CCIPR_ADCSEL                          RCC_CCIPR_ADCSEL_Msk
#define RCC_CCIPR_ADCSEL_0                        (0x1UL<<RCC_CCIPR_ADCSEL_Pos)                     /*!< 0x40000000 */
#define RCC_CCIPR_ADCSEL_1                        (0x2UL<<RCC_CCIPR_ADCSEL_Pos)                     /*!< 0x80000000 */

/********************************* Bit definition for RCC_BDCR register *********************************************/
#define RCC_BDCR_LSERDY_Pos                       (1U)
#define RCC_BDCR_LSERDY_Msk                       (0x1UL<<RCC_BDCR_LSERDY_Pos)                      /*!< 0x00000002 */
#define RCC_BDCR_LSERDY                           RCC_BDCR_LSERDY_Msk                               
#define RCC_BDCR_LSEBYP_Pos                       (2U)
#define RCC_BDCR_LSEBYP_Msk                       (0x1UL<<RCC_BDCR_LSEBYP_Pos)                      /*!< 0x00000004 */
#define RCC_BDCR_LSEBYP                           RCC_BDCR_LSEBYP_Msk                               
#define RCC_BDCR_LSEDRV_Pos                       (3U)
#define RCC_BDCR_LSEDRV_Msk                       (0x3UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000018 */
#define RCC_BDCR_LSEDRV                           RCC_BDCR_LSEDRV_Msk
#define RCC_BDCR_LSEDRV_0                         (0x1UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000008 */
#define RCC_BDCR_LSEDRV_1                         (0x2UL<<RCC_BDCR_LSEDRV_Pos)                      /*!< 0x00000010 */
#define RCC_BDCR_LSECSSON_Pos                     (5U)
#define RCC_BDCR_LSECSSON_Msk                     (0x1UL<<RCC_BDCR_LSECSSON_Pos)                    /*!< 0x00000020 */
#define RCC_BDCR_LSECSSON                         RCC_BDCR_LSECSSON_Msk                             
#define RCC_BDCR_LSECSSD_Pos                      (6U)
#define RCC_BDCR_LSECSSD_Msk                      (0x1UL<<RCC_BDCR_LSECSSD_Pos)                     /*!< 0x00000040 */
#define RCC_BDCR_LSECSSD                          RCC_BDCR_LSECSSD_Msk                              
#define RCC_BDCR_RTCSEL_Pos                       (8U)
#define RCC_BDCR_RTCSEL_Msk                       (0x3UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000300 */
#define RCC_BDCR_RTCSEL                           RCC_BDCR_RTCSEL_Msk
#define RCC_BDCR_RTCSEL_0                         (0x1UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000100 */
#define RCC_BDCR_RTCSEL_1                         (0x2UL<<RCC_BDCR_RTCSEL_Pos)                      /*!< 0x00000200 */
#define RCC_BDCR_LSEEN_Pos                        (10U)
#define RCC_BDCR_LSEEN_Msk                        (0xFUL<<RCC_BDCR_LSEEN_Pos)                       /*!< 0x00003C00 */
#define RCC_BDCR_LSEEN                            RCC_BDCR_LSEEN_Msk
#define RCC_BDCR_LSEEN_0                          (0x1UL<<RCC_BDCR_LSEEN_Pos)                       /*!< 0x00000400 */
#define RCC_BDCR_LSEEN_1                          (0x2UL<<RCC_BDCR_LSEEN_Pos)                       /*!< 0x00000800 */
#define RCC_BDCR_LSEEN_2                          (0x4UL<<RCC_BDCR_LSEEN_Pos)                       /*!< 0x00001000 */
#define RCC_BDCR_LSEEN_3                          (0x8UL<<RCC_BDCR_LSEEN_Pos)                       /*!< 0x00002000 */
#define RCC_BDCR_LSCSEL_Pos                       (14U)
#define RCC_BDCR_LSCSEL_Msk                       (0x1UL<<RCC_BDCR_LSCSEL_Pos)                      /*!< 0x00004000 */
#define RCC_BDCR_LSCSEL                           RCC_BDCR_LSCSEL_Msk                               
#define RCC_BDCR_RTCEN_Pos                        (15U)
#define RCC_BDCR_RTCEN_Msk                        (0x1UL<<RCC_BDCR_RTCEN_Pos)                       /*!< 0x00008000 */
#define RCC_BDCR_RTCEN                            RCC_BDCR_RTCEN_Msk                                
#define RCC_BDCR_BDRST_Pos                        (16U)
#define RCC_BDCR_BDRST_Msk                        (0x1UL<<RCC_BDCR_BDRST_Pos)                       /*!< 0x00010000 */
#define RCC_BDCR_BDRST                            RCC_BDCR_BDRST_Msk                                
#define RCC_BDCR_LSI_TRIMCR_Pos                   (17U)
#define RCC_BDCR_LSI_TRIMCR_Msk                   (0x1FFUL<<RCC_BDCR_LSI_TRIMCR_Pos)                /*!< 0x03FE0000 */
#define RCC_BDCR_LSI_TRIMCR                       RCC_BDCR_LSI_TRIMCR_Msk
#define RCC_BDCR_LSION_Pos                        (26U)
#define RCC_BDCR_LSION_Msk                        (0x1UL<<RCC_BDCR_LSION_Pos)                       /*!< 0x04000000 */
#define RCC_BDCR_LSION                            RCC_BDCR_LSION_Msk                                
#define RCC_BDCR_LSIRDY_Pos                       (27U)
#define RCC_BDCR_LSIRDY_Msk                       (0x1UL<<RCC_BDCR_LSIRDY_Pos)                      /*!< 0x08000000 */
#define RCC_BDCR_LSIRDY                           RCC_BDCR_LSIRDY_Msk                               
#define RCC_BDCR_LSE_RDYSEL_Pos                   (28U)
#define RCC_BDCR_LSE_RDYSEL_Msk                   (0x3UL<<RCC_BDCR_LSE_RDYSEL_Pos)                  /*!< 0x30000000 */
#define RCC_BDCR_LSE_RDYSEL                       RCC_BDCR_LSE_RDYSEL_Msk
#define RCC_BDCR_LSE_RDYSEL_0                     (0x1UL<<RCC_BDCR_LSE_RDYSEL_Pos)                  /*!< 0x10000000 */
#define RCC_BDCR_LSE_RDYSEL_1                     (0x2UL<<RCC_BDCR_LSE_RDYSEL_Pos)                  /*!< 0x20000000 */

/********************************* Bit definition for RCC_CSR register **********************************************/
#define RCC_CSR_NRST_FLTDIS_Pos                   (8U)
#define RCC_CSR_NRST_FLTDIS_Msk                   (0x1UL<<RCC_CSR_NRST_FLTDIS_Pos)                  /*!< 0x00000100 */
#define RCC_CSR_NRST_FLTDIS                       RCC_CSR_NRST_FLTDIS_Msk                           
#define RCC_CSR_RMVF_Pos                          (23U)
#define RCC_CSR_RMVF_Msk                          (0x1UL<<RCC_CSR_RMVF_Pos)                         /*!< 0x00800000 */
#define RCC_CSR_RMVF                              RCC_CSR_RMVF_Msk                                  
#define RCC_CSR_OBLRSTF_Pos                       (25U)
#define RCC_CSR_OBLRSTF_Msk                       (0x1UL<<RCC_CSR_OBLRSTF_Pos)                      /*!< 0x02000000 */
#define RCC_CSR_OBLRSTF                           RCC_CSR_OBLRSTF_Msk                               
#define RCC_CSR_PINRSTF_Pos                       (26U)
#define RCC_CSR_PINRSTF_Msk                       (0x1UL<<RCC_CSR_PINRSTF_Pos)                      /*!< 0x04000000 */
#define RCC_CSR_PINRSTF                           RCC_CSR_PINRSTF_Msk                               
#define RCC_CSR_PWRRSTF_Pos                       (27U)
#define RCC_CSR_PWRRSTF_Msk                       (0x1UL<<RCC_CSR_PWRRSTF_Pos)                      /*!< 0x08000000 */
#define RCC_CSR_PWRRSTF                           RCC_CSR_PWRRSTF_Msk                               
#define RCC_CSR_SFTRSTF_Pos                       (28U)
#define RCC_CSR_SFTRSTF_Msk                       (0x1UL<<RCC_CSR_SFTRSTF_Pos)                      /*!< 0x10000000 */
#define RCC_CSR_SFTRSTF                           RCC_CSR_SFTRSTF_Msk                               
#define RCC_CSR_IWDGRSTF_Pos                      (29U)
#define RCC_CSR_IWDGRSTF_Msk                      (0x1UL<<RCC_CSR_IWDGRSTF_Pos)                     /*!< 0x20000000 */
#define RCC_CSR_IWDGRSTF                          RCC_CSR_IWDGRSTF_Msk                              
#define RCC_CSR_WWDGRSTF_Pos                      (30U)
#define RCC_CSR_WWDGRSTF_Msk                      (0x1UL<<RCC_CSR_WWDGRSTF_Pos)                     /*!< 0x40000000 */
#define RCC_CSR_WWDGRSTF                          RCC_CSR_WWDGRSTF_Msk                              
#define RCC_CSR_LPWRRSTF_Pos                      (31U)
#define RCC_CSR_LPWRRSTF_Msk                      (0x1UL<<RCC_CSR_LPWRRSTF_Pos)                     /*!< 0x80000000 */
#define RCC_CSR_LPWRRSTF                          RCC_CSR_LPWRRSTF_Msk                              

/********************************************************************************************************************/
/********************************* RTC ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for RTC_TR register ***********************************************/
#define RTC_TR_SU_Pos                             (0U)
#define RTC_TR_SU_Msk                             (0xFUL<<RTC_TR_SU_Pos)                            /*!< 0x0000000F */
#define RTC_TR_SU                                 RTC_TR_SU_Msk
#define RTC_TR_SU_0                               (0x1UL<<RTC_TR_SU_Pos)                            /*!< 0x00000001 */
#define RTC_TR_SU_1                               (0x2UL<<RTC_TR_SU_Pos)                            /*!< 0x00000002 */
#define RTC_TR_SU_2                               (0x4UL<<RTC_TR_SU_Pos)                            /*!< 0x00000004 */
#define RTC_TR_SU_3                               (0x8UL<<RTC_TR_SU_Pos)                            /*!< 0x00000008 */
#define RTC_TR_ST_Pos                             (4U)
#define RTC_TR_ST_Msk                             (0x7UL<<RTC_TR_ST_Pos)                            /*!< 0x00000070 */
#define RTC_TR_ST                                 RTC_TR_ST_Msk
#define RTC_TR_ST_0                               (0x1UL<<RTC_TR_ST_Pos)                            /*!< 0x00000010 */
#define RTC_TR_ST_1                               (0x2UL<<RTC_TR_ST_Pos)                            /*!< 0x00000020 */
#define RTC_TR_ST_2                               (0x4UL<<RTC_TR_ST_Pos)                            /*!< 0x00000040 */
#define RTC_TR_MNU_Pos                            (8U)
#define RTC_TR_MNU_Msk                            (0xFUL<<RTC_TR_MNU_Pos)                           /*!< 0x00000F00 */
#define RTC_TR_MNU                                RTC_TR_MNU_Msk
#define RTC_TR_MNU_0                              (0x1UL<<RTC_TR_MNU_Pos)                           /*!< 0x00000100 */
#define RTC_TR_MNU_1                              (0x2UL<<RTC_TR_MNU_Pos)                           /*!< 0x00000200 */
#define RTC_TR_MNU_2                              (0x4UL<<RTC_TR_MNU_Pos)                           /*!< 0x00000400 */
#define RTC_TR_MNU_3                              (0x8UL<<RTC_TR_MNU_Pos)                           /*!< 0x00000800 */
#define RTC_TR_MNT_Pos                            (12U)
#define RTC_TR_MNT_Msk                            (0x7UL<<RTC_TR_MNT_Pos)                           /*!< 0x00007000 */
#define RTC_TR_MNT                                RTC_TR_MNT_Msk
#define RTC_TR_MNT_0                              (0x1UL<<RTC_TR_MNT_Pos)                           /*!< 0x00001000 */
#define RTC_TR_MNT_1                              (0x2UL<<RTC_TR_MNT_Pos)                           /*!< 0x00002000 */
#define RTC_TR_MNT_2                              (0x4UL<<RTC_TR_MNT_Pos)                           /*!< 0x00004000 */
#define RTC_TR_HU_Pos                             (16U)
#define RTC_TR_HU_Msk                             (0xFUL<<RTC_TR_HU_Pos)                            /*!< 0x000F0000 */
#define RTC_TR_HU                                 RTC_TR_HU_Msk
#define RTC_TR_HU_0                               (0x1UL<<RTC_TR_HU_Pos)                            /*!< 0x00010000 */
#define RTC_TR_HU_1                               (0x2UL<<RTC_TR_HU_Pos)                            /*!< 0x00020000 */
#define RTC_TR_HU_2                               (0x4UL<<RTC_TR_HU_Pos)                            /*!< 0x00040000 */
#define RTC_TR_HU_3                               (0x8UL<<RTC_TR_HU_Pos)                            /*!< 0x00080000 */
#define RTC_TR_HT_Pos                             (20U)
#define RTC_TR_HT_Msk                             (0x3UL<<RTC_TR_HT_Pos)                            /*!< 0x00300000 */
#define RTC_TR_HT                                 RTC_TR_HT_Msk
#define RTC_TR_HT_0                               (0x1UL<<RTC_TR_HT_Pos)                            /*!< 0x00100000 */
#define RTC_TR_HT_1                               (0x2UL<<RTC_TR_HT_Pos)                            /*!< 0x00200000 */
#define RTC_TR_PM_Pos                             (22U)
#define RTC_TR_PM_Msk                             (0x1UL<<RTC_TR_PM_Pos)                            /*!< 0x00400000 */
#define RTC_TR_PM                                 RTC_TR_PM_Msk                                     

/********************************* Bit definition for RTC_DR register ***********************************************/
#define RTC_DR_DU_Pos                             (0U)
#define RTC_DR_DU_Msk                             (0xFUL<<RTC_DR_DU_Pos)                            /*!< 0x0000000F */
#define RTC_DR_DU                                 RTC_DR_DU_Msk
#define RTC_DR_DU_0                               (0x1UL<<RTC_DR_DU_Pos)                            /*!< 0x00000001 */
#define RTC_DR_DU_1                               (0x2UL<<RTC_DR_DU_Pos)                            /*!< 0x00000002 */
#define RTC_DR_DU_2                               (0x4UL<<RTC_DR_DU_Pos)                            /*!< 0x00000004 */
#define RTC_DR_DU_3                               (0x8UL<<RTC_DR_DU_Pos)                            /*!< 0x00000008 */
#define RTC_DR_DT_Pos                             (4U)
#define RTC_DR_DT_Msk                             (0x3UL<<RTC_DR_DT_Pos)                            /*!< 0x00000030 */
#define RTC_DR_DT                                 RTC_DR_DT_Msk
#define RTC_DR_DT_0                               (0x1UL<<RTC_DR_DT_Pos)                            /*!< 0x00000010 */
#define RTC_DR_DT_1                               (0x2UL<<RTC_DR_DT_Pos)                            /*!< 0x00000020 */
#define RTC_DR_MU_Pos                             (8U)
#define RTC_DR_MU_Msk                             (0xFUL<<RTC_DR_MU_Pos)                            /*!< 0x00000F00 */
#define RTC_DR_MU                                 RTC_DR_MU_Msk
#define RTC_DR_MU_0                               (0x1UL<<RTC_DR_MU_Pos)                            /*!< 0x00000100 */
#define RTC_DR_MU_1                               (0x2UL<<RTC_DR_MU_Pos)                            /*!< 0x00000200 */
#define RTC_DR_MU_2                               (0x4UL<<RTC_DR_MU_Pos)                            /*!< 0x00000400 */
#define RTC_DR_MU_3                               (0x8UL<<RTC_DR_MU_Pos)                            /*!< 0x00000800 */
#define RTC_DR_MT_Pos                             (12U)
#define RTC_DR_MT_Msk                             (0x1UL<<RTC_DR_MT_Pos)                            /*!< 0x00001000 */
#define RTC_DR_MT                                 RTC_DR_MT_Msk                                     
#define RTC_DR_WDU_Pos                            (13U)
#define RTC_DR_WDU_Msk                            (0x7UL<<RTC_DR_WDU_Pos)                           /*!< 0x0000E000 */
#define RTC_DR_WDU                                RTC_DR_WDU_Msk
#define RTC_DR_WDU_0                              (0x1UL<<RTC_DR_WDU_Pos)                           /*!< 0x00002000 */
#define RTC_DR_WDU_1                              (0x2UL<<RTC_DR_WDU_Pos)                           /*!< 0x00004000 */
#define RTC_DR_WDU_2                              (0x4UL<<RTC_DR_WDU_Pos)                           /*!< 0x00008000 */
#define RTC_DR_YU_Pos                             (16U)
#define RTC_DR_YU_Msk                             (0xFUL<<RTC_DR_YU_Pos)                            /*!< 0x000F0000 */
#define RTC_DR_YU                                 RTC_DR_YU_Msk
#define RTC_DR_YU_0                               (0x1UL<<RTC_DR_YU_Pos)                            /*!< 0x00010000 */
#define RTC_DR_YU_1                               (0x2UL<<RTC_DR_YU_Pos)                            /*!< 0x00020000 */
#define RTC_DR_YU_2                               (0x4UL<<RTC_DR_YU_Pos)                            /*!< 0x00040000 */
#define RTC_DR_YU_3                               (0x8UL<<RTC_DR_YU_Pos)                            /*!< 0x00080000 */
#define RTC_DR_YT_Pos                             (20U)
#define RTC_DR_YT_Msk                             (0xFUL<<RTC_DR_YT_Pos)                            /*!< 0x00F00000 */
#define RTC_DR_YT                                 RTC_DR_YT_Msk
#define RTC_DR_YT_0                               (0x1UL<<RTC_DR_YT_Pos)                            /*!< 0x00100000 */
#define RTC_DR_YT_1                               (0x2UL<<RTC_DR_YT_Pos)                            /*!< 0x00200000 */
#define RTC_DR_YT_2                               (0x4UL<<RTC_DR_YT_Pos)                            /*!< 0x00400000 */
#define RTC_DR_YT_3                               (0x8UL<<RTC_DR_YT_Pos)                            /*!< 0x00800000 */

/********************************* Bit definition for RTC_SSR register **********************************************/
#define RTC_SSR_SS_Pos                            (0U)
#define RTC_SSR_SS_Msk                            (0xFFFFUL<<RTC_SSR_SS_Pos)                        /*!< 0x0000FFFF */
#define RTC_SSR_SS                                RTC_SSR_SS_Msk

/********************************* Bit definition for RTC_ICSR register *********************************************/
#define RTC_ICSR_ALRAWF_Pos                       (0U)
#define RTC_ICSR_ALRAWF_Msk                       (0x1UL<<RTC_ICSR_ALRAWF_Pos)                      /*!< 0x00000001 */
#define RTC_ICSR_ALRAWF                           RTC_ICSR_ALRAWF_Msk                               
#define RTC_ICSR_ALRBWF_Pos                       (1U)
#define RTC_ICSR_ALRBWF_Msk                       (0x1UL<<RTC_ICSR_ALRBWF_Pos)                      /*!< 0x00000002 */
#define RTC_ICSR_ALRBWF                           RTC_ICSR_ALRBWF_Msk                               
#define RTC_ICSR_WUTWF_Pos                        (2U)
#define RTC_ICSR_WUTWF_Msk                        (0x1UL<<RTC_ICSR_WUTWF_Pos)                       /*!< 0x00000004 */
#define RTC_ICSR_WUTWF                            RTC_ICSR_WUTWF_Msk                                
#define RTC_ICSR_SHPF_Pos                         (3U)
#define RTC_ICSR_SHPF_Msk                         (0x1UL<<RTC_ICSR_SHPF_Pos)                        /*!< 0x00000008 */
#define RTC_ICSR_SHPF                             RTC_ICSR_SHPF_Msk                                 
#define RTC_ICSR_INITS_Pos                        (4U)
#define RTC_ICSR_INITS_Msk                        (0x1UL<<RTC_ICSR_INITS_Pos)                       /*!< 0x00000010 */
#define RTC_ICSR_INITS                            RTC_ICSR_INITS_Msk                                
#define RTC_ICSR_RSF_Pos                          (5U)
#define RTC_ICSR_RSF_Msk                          (0x1UL<<RTC_ICSR_RSF_Pos)                         /*!< 0x00000020 */
#define RTC_ICSR_RSF                              RTC_ICSR_RSF_Msk                                  
#define RTC_ICSR_INITF_Pos                        (6U)
#define RTC_ICSR_INITF_Msk                        (0x1UL<<RTC_ICSR_INITF_Pos)                       /*!< 0x00000040 */
#define RTC_ICSR_INITF                            RTC_ICSR_INITF_Msk                                
#define RTC_ICSR_INIT_Pos                         (7U)
#define RTC_ICSR_INIT_Msk                         (0x1UL<<RTC_ICSR_INIT_Pos)                        /*!< 0x00000080 */
#define RTC_ICSR_INIT                             RTC_ICSR_INIT_Msk                                 
#define RTC_ICSR_RECALPF_Pos                      (16U)
#define RTC_ICSR_RECALPF_Msk                      (0x1UL<<RTC_ICSR_RECALPF_Pos)                     /*!< 0x00010000 */
#define RTC_ICSR_RECALPF                          RTC_ICSR_RECALPF_Msk                              

/********************************* Bit definition for RTC_PRER register *********************************************/
#define RTC_PRER_PREDIV_S_Pos                     (0U)
#define RTC_PRER_PREDIV_S_Msk                     (0x7FFFUL<<RTC_PRER_PREDIV_S_Pos)                 /*!< 0x00007FFF */
#define RTC_PRER_PREDIV_S                         RTC_PRER_PREDIV_S_Msk
#define RTC_PRER_PREDIV_A_Pos                     (16U)
#define RTC_PRER_PREDIV_A_Msk                     (0x7FUL<<RTC_PRER_PREDIV_A_Pos)                   /*!< 0x007F0000 */
#define RTC_PRER_PREDIV_A                         RTC_PRER_PREDIV_A_Msk

/********************************* Bit definition for RTC_WUTR register *********************************************/
#define RTC_WUTR_WUT_Pos                          (0U)
#define RTC_WUTR_WUT_Msk                          (0xFFFFUL<<RTC_WUTR_WUT_Pos)                      /*!< 0x0000FFFF */
#define RTC_WUTR_WUT                              RTC_WUTR_WUT_Msk

/********************************* Bit definition for RTC_CR register ***********************************************/
#define RTC_CR_WUCKSEL_Pos                        (0U)
#define RTC_CR_WUCKSEL_Msk                        (0x7UL<<RTC_CR_WUCKSEL_Pos)                       /*!< 0x00000007 */
#define RTC_CR_WUCKSEL                            RTC_CR_WUCKSEL_Msk
#define RTC_CR_WUCKSEL_0                          (0x1UL<<RTC_CR_WUCKSEL_Pos)                       /*!< 0x00000001 */
#define RTC_CR_WUCKSEL_1                          (0x2UL<<RTC_CR_WUCKSEL_Pos)                       /*!< 0x00000002 */
#define RTC_CR_WUCKSEL_2                          (0x4UL<<RTC_CR_WUCKSEL_Pos)                       /*!< 0x00000004 */
#define RTC_CR_TSEDGE_Pos                         (3U)
#define RTC_CR_TSEDGE_Msk                         (0x1UL<<RTC_CR_TSEDGE_Pos)                        /*!< 0x00000008 */
#define RTC_CR_TSEDGE                             RTC_CR_TSEDGE_Msk                                 
#define RTC_CR_REFCKON_Pos                        (4U)
#define RTC_CR_REFCKON_Msk                        (0x1UL<<RTC_CR_REFCKON_Pos)                       /*!< 0x00000010 */
#define RTC_CR_REFCKON                            RTC_CR_REFCKON_Msk                                
#define RTC_CR_BYPSHAD_Pos                        (5U)
#define RTC_CR_BYPSHAD_Msk                        (0x1UL<<RTC_CR_BYPSHAD_Pos)                       /*!< 0x00000020 */
#define RTC_CR_BYPSHAD                            RTC_CR_BYPSHAD_Msk                                
#define RTC_CR_FMT_Pos                            (6U)
#define RTC_CR_FMT_Msk                            (0x1UL<<RTC_CR_FMT_Pos)                           /*!< 0x00000040 */
#define RTC_CR_FMT                                RTC_CR_FMT_Msk                                    
#define RTC_CR_ALRAE_Pos                          (8U)
#define RTC_CR_ALRAE_Msk                          (0x1UL<<RTC_CR_ALRAE_Pos)                         /*!< 0x00000100 */
#define RTC_CR_ALRAE                              RTC_CR_ALRAE_Msk                                  
#define RTC_CR_ALRBE_Pos                          (9U)
#define RTC_CR_ALRBE_Msk                          (0x1UL<<RTC_CR_ALRBE_Pos)                         /*!< 0x00000200 */
#define RTC_CR_ALRBE                              RTC_CR_ALRBE_Msk                                  
#define RTC_CR_WUTE_Pos                           (10U)
#define RTC_CR_WUTE_Msk                           (0x1UL<<RTC_CR_WUTE_Pos)                          /*!< 0x00000400 */
#define RTC_CR_WUTE                               RTC_CR_WUTE_Msk                                   
#define RTC_CR_TSE_Pos                            (11U)
#define RTC_CR_TSE_Msk                            (0x1UL<<RTC_CR_TSE_Pos)                           /*!< 0x00000800 */
#define RTC_CR_TSE                                RTC_CR_TSE_Msk                                    
#define RTC_CR_ALRAIE_Pos                         (12U)
#define RTC_CR_ALRAIE_Msk                         (0x1UL<<RTC_CR_ALRAIE_Pos)                        /*!< 0x00001000 */
#define RTC_CR_ALRAIE                             RTC_CR_ALRAIE_Msk                                 
#define RTC_CR_ALRBIE_Pos                         (13U)
#define RTC_CR_ALRBIE_Msk                         (0x1UL<<RTC_CR_ALRBIE_Pos)                        /*!< 0x00002000 */
#define RTC_CR_ALRBIE                             RTC_CR_ALRBIE_Msk                                 
#define RTC_CR_WUTIE_Pos                          (14U)
#define RTC_CR_WUTIE_Msk                          (0x1UL<<RTC_CR_WUTIE_Pos)                         /*!< 0x00004000 */
#define RTC_CR_WUTIE                              RTC_CR_WUTIE_Msk                                  
#define RTC_CR_TSIE_Pos                           (15U)
#define RTC_CR_TSIE_Msk                           (0x1UL<<RTC_CR_TSIE_Pos)                          /*!< 0x00008000 */
#define RTC_CR_TSIE                               RTC_CR_TSIE_Msk                                   
#define RTC_CR_ADD1H_Pos                          (16U)
#define RTC_CR_ADD1H_Msk                          (0x1UL<<RTC_CR_ADD1H_Pos)                         /*!< 0x00010000 */
#define RTC_CR_ADD1H                              RTC_CR_ADD1H_Msk                                  
#define RTC_CR_SUB1H_Pos                          (17U)
#define RTC_CR_SUB1H_Msk                          (0x1UL<<RTC_CR_SUB1H_Pos)                         /*!< 0x00020000 */
#define RTC_CR_SUB1H                              RTC_CR_SUB1H_Msk                                  
#define RTC_CR_BKP_Pos                            (18U)
#define RTC_CR_BKP_Msk                            (0x1UL<<RTC_CR_BKP_Pos)                           /*!< 0x00040000 */
#define RTC_CR_BKP                                RTC_CR_BKP_Msk                                    
#define RTC_CR_COSEL_Pos                          (19U)
#define RTC_CR_COSEL_Msk                          (0x1UL<<RTC_CR_COSEL_Pos)                         /*!< 0x00080000 */
#define RTC_CR_COSEL                              RTC_CR_COSEL_Msk                                  
#define RTC_CR_POL_Pos                            (20U)
#define RTC_CR_POL_Msk                            (0x1UL<<RTC_CR_POL_Pos)                           /*!< 0x00100000 */
#define RTC_CR_POL                                RTC_CR_POL_Msk                                    
#define RTC_CR_OSEL_Pos                           (21U)
#define RTC_CR_OSEL_Msk                           (0x3UL<<RTC_CR_OSEL_Pos)                          /*!< 0x00600000 */
#define RTC_CR_OSEL                               RTC_CR_OSEL_Msk
#define RTC_CR_OSEL_0                             (0x1UL<<RTC_CR_OSEL_Pos)                          /*!< 0x00200000 */
#define RTC_CR_OSEL_1                             (0x2UL<<RTC_CR_OSEL_Pos)                          /*!< 0x00400000 */
#define RTC_CR_COE_Pos                            (23U)
#define RTC_CR_COE_Msk                            (0x1UL<<RTC_CR_COE_Pos)                           /*!< 0x00800000 */
#define RTC_CR_COE                                RTC_CR_COE_Msk                                    
#define RTC_CR_ITSE_Pos                           (24U)
#define RTC_CR_ITSE_Msk                           (0x1UL<<RTC_CR_ITSE_Pos)                          /*!< 0x01000000 */
#define RTC_CR_ITSE                               RTC_CR_ITSE_Msk                                   
#define RTC_CR_TAMPTS_Pos                         (25U)
#define RTC_CR_TAMPTS_Msk                         (0x1UL<<RTC_CR_TAMPTS_Pos)                        /*!< 0x02000000 */
#define RTC_CR_TAMPTS                             RTC_CR_TAMPTS_Msk                                 
#define RTC_CR_TAMPOE_Pos                         (26U)
#define RTC_CR_TAMPOE_Msk                         (0x1UL<<RTC_CR_TAMPOE_Pos)                        /*!< 0x04000000 */
#define RTC_CR_TAMPOE                             RTC_CR_TAMPOE_Msk                                 
#define RTC_CR_TAMPALRM_PU_Pos                    (29U)
#define RTC_CR_TAMPALRM_PU_Msk                    (0x1UL<<RTC_CR_TAMPALRM_PU_Pos)                   /*!< 0x20000000 */
#define RTC_CR_TAMPALRM_PU                        RTC_CR_TAMPALRM_PU_Msk                            
#define RTC_CR_TAMPALRM_TYPE_Pos                  (30U)
#define RTC_CR_TAMPALRM_TYPE_Msk                  (0x1UL<<RTC_CR_TAMPALRM_TYPE_Pos)                 /*!< 0x40000000 */
#define RTC_CR_TAMPALRM_TYPE                      RTC_CR_TAMPALRM_TYPE_Msk                          
#define RTC_CR_OUT2EN_Pos                         (31U)
#define RTC_CR_OUT2EN_Msk                         (0x1UL<<RTC_CR_OUT2EN_Pos)                        /*!< 0x80000000 */
#define RTC_CR_OUT2EN                             RTC_CR_OUT2EN_Msk                                 

/********************************* Bit definition for RTC_WPR register **********************************************/
#define RTC_WPR_KEY_Pos                           (0U)
#define RTC_WPR_KEY_Msk                           (0xFFUL<<RTC_WPR_KEY_Pos)                         /*!< 0x000000FF */
#define RTC_WPR_KEY                               RTC_WPR_KEY_Msk

/********************************* Bit definition for RTC_CALR register *********************************************/
#define RTC_CALR_CALM_Pos                         (0U)
#define RTC_CALR_CALM_Msk                         (0x1FFUL<<RTC_CALR_CALM_Pos)                      /*!< 0x000001FF */
#define RTC_CALR_CALM                             RTC_CALR_CALM_Msk
#define RTC_CALR_CALW16_Pos                       (13U)
#define RTC_CALR_CALW16_Msk                       (0x1UL<<RTC_CALR_CALW16_Pos)                      /*!< 0x00002000 */
#define RTC_CALR_CALW16                           RTC_CALR_CALW16_Msk                               
#define RTC_CALR_CALW8_Pos                        (14U)
#define RTC_CALR_CALW8_Msk                        (0x1UL<<RTC_CALR_CALW8_Pos)                       /*!< 0x00004000 */
#define RTC_CALR_CALW8                            RTC_CALR_CALW8_Msk                                
#define RTC_CALR_CALP_Pos                         (15U)
#define RTC_CALR_CALP_Msk                         (0x1UL<<RTC_CALR_CALP_Pos)                        /*!< 0x00008000 */
#define RTC_CALR_CALP                             RTC_CALR_CALP_Msk                                 

/********************************* Bit definition for RTC_SHIFTR register *******************************************/
#define RTC_SHIFTR_SUBFS_Pos                      (0U)
#define RTC_SHIFTR_SUBFS_Msk                      (0x7FFFUL<<RTC_SHIFTR_SUBFS_Pos)                  /*!< 0x00007FFF */
#define RTC_SHIFTR_SUBFS                          RTC_SHIFTR_SUBFS_Msk
#define RTC_SHIFTR_ADD1S_Pos                      (31U)
#define RTC_SHIFTR_ADD1S_Msk                      (0x1UL<<RTC_SHIFTR_ADD1S_Pos)                     /*!< 0x80000000 */
#define RTC_SHIFTR_ADD1S                          RTC_SHIFTR_ADD1S_Msk                              

/********************************* Bit definition for RTC_TSTR register *********************************************/
#define RTC_TSTR_SU_Pos                           (0U)
#define RTC_TSTR_SU_Msk                           (0xFUL<<RTC_TSTR_SU_Pos)                          /*!< 0x0000000F */
#define RTC_TSTR_SU                               RTC_TSTR_SU_Msk
#define RTC_TSTR_SU_0                             (0x1UL<<RTC_TSTR_SU_Pos)                          /*!< 0x00000001 */
#define RTC_TSTR_SU_1                             (0x2UL<<RTC_TSTR_SU_Pos)                          /*!< 0x00000002 */
#define RTC_TSTR_SU_2                             (0x4UL<<RTC_TSTR_SU_Pos)                          /*!< 0x00000004 */
#define RTC_TSTR_SU_3                             (0x8UL<<RTC_TSTR_SU_Pos)                          /*!< 0x00000008 */
#define RTC_TSTR_ST_Pos                           (4U)
#define RTC_TSTR_ST_Msk                           (0x7UL<<RTC_TSTR_ST_Pos)                          /*!< 0x00000070 */
#define RTC_TSTR_ST                               RTC_TSTR_ST_Msk
#define RTC_TSTR_ST_0                             (0x1UL<<RTC_TSTR_ST_Pos)                          /*!< 0x00000010 */
#define RTC_TSTR_ST_1                             (0x2UL<<RTC_TSTR_ST_Pos)                          /*!< 0x00000020 */
#define RTC_TSTR_ST_2                             (0x4UL<<RTC_TSTR_ST_Pos)                          /*!< 0x00000040 */
#define RTC_TSTR_MNU_Pos                          (8U)
#define RTC_TSTR_MNU_Msk                          (0xFUL<<RTC_TSTR_MNU_Pos)                         /*!< 0x00000F00 */
#define RTC_TSTR_MNU                              RTC_TSTR_MNU_Msk
#define RTC_TSTR_MNU_0                            (0x1UL<<RTC_TSTR_MNU_Pos)                         /*!< 0x00000100 */
#define RTC_TSTR_MNU_1                            (0x2UL<<RTC_TSTR_MNU_Pos)                         /*!< 0x00000200 */
#define RTC_TSTR_MNU_2                            (0x4UL<<RTC_TSTR_MNU_Pos)                         /*!< 0x00000400 */
#define RTC_TSTR_MNU_3                            (0x8UL<<RTC_TSTR_MNU_Pos)                         /*!< 0x00000800 */
#define RTC_TSTR_MNT_Pos                          (12U)
#define RTC_TSTR_MNT_Msk                          (0x7UL<<RTC_TSTR_MNT_Pos)                         /*!< 0x00007000 */
#define RTC_TSTR_MNT                              RTC_TSTR_MNT_Msk
#define RTC_TSTR_MNT_0                            (0x1UL<<RTC_TSTR_MNT_Pos)                         /*!< 0x00001000 */
#define RTC_TSTR_MNT_1                            (0x2UL<<RTC_TSTR_MNT_Pos)                         /*!< 0x00002000 */
#define RTC_TSTR_MNT_2                            (0x4UL<<RTC_TSTR_MNT_Pos)                         /*!< 0x00004000 */
#define RTC_TSTR_HU_Pos                           (16U)
#define RTC_TSTR_HU_Msk                           (0xFUL<<RTC_TSTR_HU_Pos)                          /*!< 0x000F0000 */
#define RTC_TSTR_HU                               RTC_TSTR_HU_Msk
#define RTC_TSTR_HU_0                             (0x1UL<<RTC_TSTR_HU_Pos)                          /*!< 0x00010000 */
#define RTC_TSTR_HU_1                             (0x2UL<<RTC_TSTR_HU_Pos)                          /*!< 0x00020000 */
#define RTC_TSTR_HU_2                             (0x4UL<<RTC_TSTR_HU_Pos)                          /*!< 0x00040000 */
#define RTC_TSTR_HU_3                             (0x8UL<<RTC_TSTR_HU_Pos)                          /*!< 0x00080000 */
#define RTC_TSTR_HT_Pos                           (20U)
#define RTC_TSTR_HT_Msk                           (0x3UL<<RTC_TSTR_HT_Pos)                          /*!< 0x00300000 */
#define RTC_TSTR_HT                               RTC_TSTR_HT_Msk
#define RTC_TSTR_HT_0                             (0x1UL<<RTC_TSTR_HT_Pos)                          /*!< 0x00100000 */
#define RTC_TSTR_HT_1                             (0x2UL<<RTC_TSTR_HT_Pos)                          /*!< 0x00200000 */
#define RTC_TSTR_PM_Pos                           (22U)
#define RTC_TSTR_PM_Msk                           (0x1UL<<RTC_TSTR_PM_Pos)                          /*!< 0x00400000 */
#define RTC_TSTR_PM                               RTC_TSTR_PM_Msk                                   

/********************************* Bit definition for RTC_TSDR register *********************************************/
#define RTC_TSDR_DU_Pos                           (0U)
#define RTC_TSDR_DU_Msk                           (0xFUL<<RTC_TSDR_DU_Pos)                          /*!< 0x0000000F */
#define RTC_TSDR_DU                               RTC_TSDR_DU_Msk
#define RTC_TSDR_DU_0                             (0x1UL<<RTC_TSDR_DU_Pos)                          /*!< 0x00000001 */
#define RTC_TSDR_DU_1                             (0x2UL<<RTC_TSDR_DU_Pos)                          /*!< 0x00000002 */
#define RTC_TSDR_DU_2                             (0x4UL<<RTC_TSDR_DU_Pos)                          /*!< 0x00000004 */
#define RTC_TSDR_DU_3                             (0x8UL<<RTC_TSDR_DU_Pos)                          /*!< 0x00000008 */
#define RTC_TSDR_DT_Pos                           (4U)
#define RTC_TSDR_DT_Msk                           (0x3UL<<RTC_TSDR_DT_Pos)                          /*!< 0x00000030 */
#define RTC_TSDR_DT                               RTC_TSDR_DT_Msk
#define RTC_TSDR_DT_0                             (0x1UL<<RTC_TSDR_DT_Pos)                          /*!< 0x00000010 */
#define RTC_TSDR_DT_1                             (0x2UL<<RTC_TSDR_DT_Pos)                          /*!< 0x00000020 */
#define RTC_TSDR_MU_Pos                           (8U)
#define RTC_TSDR_MU_Msk                           (0xFUL<<RTC_TSDR_MU_Pos)                          /*!< 0x00000F00 */
#define RTC_TSDR_MU                               RTC_TSDR_MU_Msk
#define RTC_TSDR_MU_0                             (0x1UL<<RTC_TSDR_MU_Pos)                          /*!< 0x00000100 */
#define RTC_TSDR_MU_1                             (0x2UL<<RTC_TSDR_MU_Pos)                          /*!< 0x00000200 */
#define RTC_TSDR_MU_2                             (0x4UL<<RTC_TSDR_MU_Pos)                          /*!< 0x00000400 */
#define RTC_TSDR_MU_3                             (0x8UL<<RTC_TSDR_MU_Pos)                          /*!< 0x00000800 */
#define RTC_TSDR_MT_Pos                           (12U)
#define RTC_TSDR_MT_Msk                           (0x1UL<<RTC_TSDR_MT_Pos)                          /*!< 0x00001000 */
#define RTC_TSDR_MT                               RTC_TSDR_MT_Msk                                   
#define RTC_TSDR_WDU_Pos                          (13U)
#define RTC_TSDR_WDU_Msk                          (0x7UL<<RTC_TSDR_WDU_Pos)                         /*!< 0x0000E000 */
#define RTC_TSDR_WDU                              RTC_TSDR_WDU_Msk
#define RTC_TSDR_WDU_0                            (0x1UL<<RTC_TSDR_WDU_Pos)                         /*!< 0x00002000 */
#define RTC_TSDR_WDU_1                            (0x2UL<<RTC_TSDR_WDU_Pos)                         /*!< 0x00004000 */
#define RTC_TSDR_WDU_2                            (0x4UL<<RTC_TSDR_WDU_Pos)                         /*!< 0x00008000 */

/********************************* Bit definition for RTC_TSSSR register ********************************************/
#define RTC_TSSSR_SS_Pos                          (0U)
#define RTC_TSSSR_SS_Msk                          (0xFFFFUL<<RTC_TSSSR_SS_Pos)                      /*!< 0x0000FFFF */
#define RTC_TSSSR_SS                              RTC_TSSSR_SS_Msk

/********************************* Bit definition for RTC_ALRMAR register *******************************************/
#define RTC_ALRMAR_SU_Pos                         (0U)
#define RTC_ALRMAR_SU_Msk                         (0xFUL<<RTC_ALRMAR_SU_Pos)                        /*!< 0x0000000F */
#define RTC_ALRMAR_SU                             RTC_ALRMAR_SU_Msk
#define RTC_ALRMAR_SU_0                           (0x1UL<<RTC_ALRMAR_SU_Pos)                        /*!< 0x00000001 */
#define RTC_ALRMAR_SU_1                           (0x2UL<<RTC_ALRMAR_SU_Pos)                        /*!< 0x00000002 */
#define RTC_ALRMAR_SU_2                           (0x4UL<<RTC_ALRMAR_SU_Pos)                        /*!< 0x00000004 */
#define RTC_ALRMAR_SU_3                           (0x8UL<<RTC_ALRMAR_SU_Pos)                        /*!< 0x00000008 */
#define RTC_ALRMAR_ST_Pos                         (4U)
#define RTC_ALRMAR_ST_Msk                         (0x7UL<<RTC_ALRMAR_ST_Pos)                        /*!< 0x00000070 */
#define RTC_ALRMAR_ST                             RTC_ALRMAR_ST_Msk
#define RTC_ALRMAR_ST_0                           (0x1UL<<RTC_ALRMAR_ST_Pos)                        /*!< 0x00000010 */
#define RTC_ALRMAR_ST_1                           (0x2UL<<RTC_ALRMAR_ST_Pos)                        /*!< 0x00000020 */
#define RTC_ALRMAR_ST_2                           (0x4UL<<RTC_ALRMAR_ST_Pos)                        /*!< 0x00000040 */
#define RTC_ALRMAR_MSK1_Pos                       (7U)
#define RTC_ALRMAR_MSK1_Msk                       (0x1UL<<RTC_ALRMAR_MSK1_Pos)                      /*!< 0x00000080 */
#define RTC_ALRMAR_MSK1                           RTC_ALRMAR_MSK1_Msk                               
#define RTC_ALRMAR_MNU_Pos                        (8U)
#define RTC_ALRMAR_MNU_Msk                        (0xFUL<<RTC_ALRMAR_MNU_Pos)                       /*!< 0x00000F00 */
#define RTC_ALRMAR_MNU                            RTC_ALRMAR_MNU_Msk
#define RTC_ALRMAR_MNU_0                          (0x1UL<<RTC_ALRMAR_MNU_Pos)                       /*!< 0x00000100 */
#define RTC_ALRMAR_MNU_1                          (0x2UL<<RTC_ALRMAR_MNU_Pos)                       /*!< 0x00000200 */
#define RTC_ALRMAR_MNU_2                          (0x4UL<<RTC_ALRMAR_MNU_Pos)                       /*!< 0x00000400 */
#define RTC_ALRMAR_MNU_3                          (0x8UL<<RTC_ALRMAR_MNU_Pos)                       /*!< 0x00000800 */
#define RTC_ALRMAR_MNT_Pos                        (12U)
#define RTC_ALRMAR_MNT_Msk                        (0x7UL<<RTC_ALRMAR_MNT_Pos)                       /*!< 0x00007000 */
#define RTC_ALRMAR_MNT                            RTC_ALRMAR_MNT_Msk
#define RTC_ALRMAR_MNT_0                          (0x1UL<<RTC_ALRMAR_MNT_Pos)                       /*!< 0x00001000 */
#define RTC_ALRMAR_MNT_1                          (0x2UL<<RTC_ALRMAR_MNT_Pos)                       /*!< 0x00002000 */
#define RTC_ALRMAR_MNT_2                          (0x4UL<<RTC_ALRMAR_MNT_Pos)                       /*!< 0x00004000 */
#define RTC_ALRMAR_MSK2_Pos                       (15U)
#define RTC_ALRMAR_MSK2_Msk                       (0x1UL<<RTC_ALRMAR_MSK2_Pos)                      /*!< 0x00008000 */
#define RTC_ALRMAR_MSK2                           RTC_ALRMAR_MSK2_Msk                               
#define RTC_ALRMAR_HU_Pos                         (16U)
#define RTC_ALRMAR_HU_Msk                         (0xFUL<<RTC_ALRMAR_HU_Pos)                        /*!< 0x000F0000 */
#define RTC_ALRMAR_HU                             RTC_ALRMAR_HU_Msk
#define RTC_ALRMAR_HU_0                           (0x1UL<<RTC_ALRMAR_HU_Pos)                        /*!< 0x00010000 */
#define RTC_ALRMAR_HU_1                           (0x2UL<<RTC_ALRMAR_HU_Pos)                        /*!< 0x00020000 */
#define RTC_ALRMAR_HU_2                           (0x4UL<<RTC_ALRMAR_HU_Pos)                        /*!< 0x00040000 */
#define RTC_ALRMAR_HU_3                           (0x8UL<<RTC_ALRMAR_HU_Pos)                        /*!< 0x00080000 */
#define RTC_ALRMAR_HT_Pos                         (20U)
#define RTC_ALRMAR_HT_Msk                         (0x3UL<<RTC_ALRMAR_HT_Pos)                        /*!< 0x00300000 */
#define RTC_ALRMAR_HT                             RTC_ALRMAR_HT_Msk
#define RTC_ALRMAR_HT_0                           (0x1UL<<RTC_ALRMAR_HT_Pos)                        /*!< 0x00100000 */
#define RTC_ALRMAR_HT_1                           (0x2UL<<RTC_ALRMAR_HT_Pos)                        /*!< 0x00200000 */
#define RTC_ALRMAR_PM_Pos                         (22U)
#define RTC_ALRMAR_PM_Msk                         (0x1UL<<RTC_ALRMAR_PM_Pos)                        /*!< 0x00400000 */
#define RTC_ALRMAR_PM                             RTC_ALRMAR_PM_Msk                                 
#define RTC_ALRMAR_MSK3_Pos                       (23U)
#define RTC_ALRMAR_MSK3_Msk                       (0x1UL<<RTC_ALRMAR_MSK3_Pos)                      /*!< 0x00800000 */
#define RTC_ALRMAR_MSK3                           RTC_ALRMAR_MSK3_Msk                               
#define RTC_ALRMAR_DU_Pos                         (24U)
#define RTC_ALRMAR_DU_Msk                         (0xFUL<<RTC_ALRMAR_DU_Pos)                        /*!< 0x0F000000 */
#define RTC_ALRMAR_DU                             RTC_ALRMAR_DU_Msk
#define RTC_ALRMAR_DU_0                           (0x1UL<<RTC_ALRMAR_DU_Pos)                        /*!< 0x01000000 */
#define RTC_ALRMAR_DU_1                           (0x2UL<<RTC_ALRMAR_DU_Pos)                        /*!< 0x02000000 */
#define RTC_ALRMAR_DU_2                           (0x4UL<<RTC_ALRMAR_DU_Pos)                        /*!< 0x04000000 */
#define RTC_ALRMAR_DU_3                           (0x8UL<<RTC_ALRMAR_DU_Pos)                        /*!< 0x08000000 */
#define RTC_ALRMAR_DT_Pos                         (28U)
#define RTC_ALRMAR_DT_Msk                         (0x3UL<<RTC_ALRMAR_DT_Pos)                        /*!< 0x30000000 */
#define RTC_ALRMAR_DT                             RTC_ALRMAR_DT_Msk
#define RTC_ALRMAR_DT_0                           (0x1UL<<RTC_ALRMAR_DT_Pos)                        /*!< 0x10000000 */
#define RTC_ALRMAR_DT_1                           (0x2UL<<RTC_ALRMAR_DT_Pos)                        /*!< 0x20000000 */
#define RTC_ALRMAR_WDSEL_Pos                      (30U)
#define RTC_ALRMAR_WDSEL_Msk                      (0x1UL<<RTC_ALRMAR_WDSEL_Pos)                     /*!< 0x40000000 */
#define RTC_ALRMAR_WDSEL                          RTC_ALRMAR_WDSEL_Msk                              
#define RTC_ALRMAR_MSK4_Pos                       (31U)
#define RTC_ALRMAR_MSK4_Msk                       (0x1UL<<RTC_ALRMAR_MSK4_Pos)                      /*!< 0x80000000 */
#define RTC_ALRMAR_MSK4                           RTC_ALRMAR_MSK4_Msk                               

/********************************* Bit definition for RTC_ALRMASSR register *****************************************/
#define RTC_ALRMASSR_SS_Pos                       (0U)
#define RTC_ALRMASSR_SS_Msk                       (0x7FFFUL<<RTC_ALRMASSR_SS_Pos)                   /*!< 0x00007FFF */
#define RTC_ALRMASSR_SS                           RTC_ALRMASSR_SS_Msk
#define RTC_ALRMASSR_MASKSS_Pos                   (24U)
#define RTC_ALRMASSR_MASKSS_Msk                   (0xFUL<<RTC_ALRMASSR_MASKSS_Pos)                  /*!< 0x0F000000 */
#define RTC_ALRMASSR_MASKSS                       RTC_ALRMASSR_MASKSS_Msk
#define RTC_ALRMASSR_MASKSS_0                     (0x1UL<<RTC_ALRMASSR_MASKSS_Pos)                  /*!< 0x01000000 */
#define RTC_ALRMASSR_MASKSS_1                     (0x2UL<<RTC_ALRMASSR_MASKSS_Pos)                  /*!< 0x02000000 */
#define RTC_ALRMASSR_MASKSS_2                     (0x4UL<<RTC_ALRMASSR_MASKSS_Pos)                  /*!< 0x04000000 */
#define RTC_ALRMASSR_MASKSS_3                     (0x8UL<<RTC_ALRMASSR_MASKSS_Pos)                  /*!< 0x08000000 */

/********************************* Bit definition for RTC_ALRMBR register *******************************************/
#define RTC_ALRMBR_SU_Pos                         (0U)
#define RTC_ALRMBR_SU_Msk                         (0xFUL<<RTC_ALRMBR_SU_Pos)                        /*!< 0x0000000F */
#define RTC_ALRMBR_SU                             RTC_ALRMBR_SU_Msk
#define RTC_ALRMBR_SU_0                           (0x1UL<<RTC_ALRMBR_SU_Pos)                        /*!< 0x00000001 */
#define RTC_ALRMBR_SU_1                           (0x2UL<<RTC_ALRMBR_SU_Pos)                        /*!< 0x00000002 */
#define RTC_ALRMBR_SU_2                           (0x4UL<<RTC_ALRMBR_SU_Pos)                        /*!< 0x00000004 */
#define RTC_ALRMBR_SU_3                           (0x8UL<<RTC_ALRMBR_SU_Pos)                        /*!< 0x00000008 */
#define RTC_ALRMBR_ST_Pos                         (4U)
#define RTC_ALRMBR_ST_Msk                         (0x7UL<<RTC_ALRMBR_ST_Pos)                        /*!< 0x00000070 */
#define RTC_ALRMBR_ST                             RTC_ALRMBR_ST_Msk
#define RTC_ALRMBR_ST_0                           (0x1UL<<RTC_ALRMBR_ST_Pos)                        /*!< 0x00000010 */
#define RTC_ALRMBR_ST_1                           (0x2UL<<RTC_ALRMBR_ST_Pos)                        /*!< 0x00000020 */
#define RTC_ALRMBR_ST_2                           (0x4UL<<RTC_ALRMBR_ST_Pos)                        /*!< 0x00000040 */
#define RTC_ALRMBR_MSK1_Pos                       (7U)
#define RTC_ALRMBR_MSK1_Msk                       (0x1UL<<RTC_ALRMBR_MSK1_Pos)                      /*!< 0x00000080 */
#define RTC_ALRMBR_MSK1                           RTC_ALRMBR_MSK1_Msk                               
#define RTC_ALRMBR_MNU_Pos                        (8U)
#define RTC_ALRMBR_MNU_Msk                        (0xFUL<<RTC_ALRMBR_MNU_Pos)                       /*!< 0x00000F00 */
#define RTC_ALRMBR_MNU                            RTC_ALRMBR_MNU_Msk
#define RTC_ALRMBR_MNU_0                          (0x1UL<<RTC_ALRMBR_MNU_Pos)                       /*!< 0x00000100 */
#define RTC_ALRMBR_MNU_1                          (0x2UL<<RTC_ALRMBR_MNU_Pos)                       /*!< 0x00000200 */
#define RTC_ALRMBR_MNU_2                          (0x4UL<<RTC_ALRMBR_MNU_Pos)                       /*!< 0x00000400 */
#define RTC_ALRMBR_MNU_3                          (0x8UL<<RTC_ALRMBR_MNU_Pos)                       /*!< 0x00000800 */
#define RTC_ALRMBR_MNT_Pos                        (12U)
#define RTC_ALRMBR_MNT_Msk                        (0x7UL<<RTC_ALRMBR_MNT_Pos)                       /*!< 0x00007000 */
#define RTC_ALRMBR_MNT                            RTC_ALRMBR_MNT_Msk
#define RTC_ALRMBR_MNT_0                          (0x1UL<<RTC_ALRMBR_MNT_Pos)                       /*!< 0x00001000 */
#define RTC_ALRMBR_MNT_1                          (0x2UL<<RTC_ALRMBR_MNT_Pos)                       /*!< 0x00002000 */
#define RTC_ALRMBR_MNT_2                          (0x4UL<<RTC_ALRMBR_MNT_Pos)                       /*!< 0x00004000 */
#define RTC_ALRMBR_MSK2_Pos                       (15U)
#define RTC_ALRMBR_MSK2_Msk                       (0x1UL<<RTC_ALRMBR_MSK2_Pos)                      /*!< 0x00008000 */
#define RTC_ALRMBR_MSK2                           RTC_ALRMBR_MSK2_Msk                               
#define RTC_ALRMBR_HU_Pos                         (16U)
#define RTC_ALRMBR_HU_Msk                         (0xFUL<<RTC_ALRMBR_HU_Pos)                        /*!< 0x000F0000 */
#define RTC_ALRMBR_HU                             RTC_ALRMBR_HU_Msk
#define RTC_ALRMBR_HU_0                           (0x1UL<<RTC_ALRMBR_HU_Pos)                        /*!< 0x00010000 */
#define RTC_ALRMBR_HU_1                           (0x2UL<<RTC_ALRMBR_HU_Pos)                        /*!< 0x00020000 */
#define RTC_ALRMBR_HU_2                           (0x4UL<<RTC_ALRMBR_HU_Pos)                        /*!< 0x00040000 */
#define RTC_ALRMBR_HU_3                           (0x8UL<<RTC_ALRMBR_HU_Pos)                        /*!< 0x00080000 */
#define RTC_ALRMBR_HT_Pos                         (20U)
#define RTC_ALRMBR_HT_Msk                         (0x3UL<<RTC_ALRMBR_HT_Pos)                        /*!< 0x00300000 */
#define RTC_ALRMBR_HT                             RTC_ALRMBR_HT_Msk
#define RTC_ALRMBR_HT_0                           (0x1UL<<RTC_ALRMBR_HT_Pos)                        /*!< 0x00100000 */
#define RTC_ALRMBR_HT_1                           (0x2UL<<RTC_ALRMBR_HT_Pos)                        /*!< 0x00200000 */
#define RTC_ALRMBR_PM_Pos                         (22U)
#define RTC_ALRMBR_PM_Msk                         (0x1UL<<RTC_ALRMBR_PM_Pos)                        /*!< 0x00400000 */
#define RTC_ALRMBR_PM                             RTC_ALRMBR_PM_Msk                                 
#define RTC_ALRMBR_MSK3_Pos                       (23U)
#define RTC_ALRMBR_MSK3_Msk                       (0x1UL<<RTC_ALRMBR_MSK3_Pos)                      /*!< 0x00800000 */
#define RTC_ALRMBR_MSK3                           RTC_ALRMBR_MSK3_Msk                               
#define RTC_ALRMBR_DU_Pos                         (24U)
#define RTC_ALRMBR_DU_Msk                         (0xFUL<<RTC_ALRMBR_DU_Pos)                        /*!< 0x0F000000 */
#define RTC_ALRMBR_DU                             RTC_ALRMBR_DU_Msk
#define RTC_ALRMBR_DU_0                           (0x1UL<<RTC_ALRMBR_DU_Pos)                        /*!< 0x01000000 */
#define RTC_ALRMBR_DU_1                           (0x2UL<<RTC_ALRMBR_DU_Pos)                        /*!< 0x02000000 */
#define RTC_ALRMBR_DU_2                           (0x4UL<<RTC_ALRMBR_DU_Pos)                        /*!< 0x04000000 */
#define RTC_ALRMBR_DU_3                           (0x8UL<<RTC_ALRMBR_DU_Pos)                        /*!< 0x08000000 */
#define RTC_ALRMBR_DT_Pos                         (28U)
#define RTC_ALRMBR_DT_Msk                         (0x3UL<<RTC_ALRMBR_DT_Pos)                        /*!< 0x30000000 */
#define RTC_ALRMBR_DT                             RTC_ALRMBR_DT_Msk
#define RTC_ALRMBR_DT_0                           (0x1UL<<RTC_ALRMBR_DT_Pos)                        /*!< 0x10000000 */
#define RTC_ALRMBR_DT_1                           (0x2UL<<RTC_ALRMBR_DT_Pos)                        /*!< 0x20000000 */
#define RTC_ALRMBR_WDSEL_Pos                      (30U)
#define RTC_ALRMBR_WDSEL_Msk                      (0x1UL<<RTC_ALRMBR_WDSEL_Pos)                     /*!< 0x40000000 */
#define RTC_ALRMBR_WDSEL                          RTC_ALRMBR_WDSEL_Msk                              
#define RTC_ALRMBR_MSK4_Pos                       (31U)
#define RTC_ALRMBR_MSK4_Msk                       (0x1UL<<RTC_ALRMBR_MSK4_Pos)                      /*!< 0x80000000 */
#define RTC_ALRMBR_MSK4                           RTC_ALRMBR_MSK4_Msk                               

/********************************* Bit definition for RTC_ALRMBSSR register *****************************************/
#define RTC_ALRMBSSR_SS_Pos                       (0U)
#define RTC_ALRMBSSR_SS_Msk                       (0x7FFFUL<<RTC_ALRMBSSR_SS_Pos)                   /*!< 0x00007FFF */
#define RTC_ALRMBSSR_SS                           RTC_ALRMBSSR_SS_Msk
#define RTC_ALRMBSSR_MASKSS_Pos                   (24U)
#define RTC_ALRMBSSR_MASKSS_Msk                   (0xFUL<<RTC_ALRMBSSR_MASKSS_Pos)                  /*!< 0x0F000000 */
#define RTC_ALRMBSSR_MASKSS                       RTC_ALRMBSSR_MASKSS_Msk
#define RTC_ALRMBSSR_MASKSS_0                     (0x1UL<<RTC_ALRMBSSR_MASKSS_Pos)                  /*!< 0x01000000 */
#define RTC_ALRMBSSR_MASKSS_1                     (0x2UL<<RTC_ALRMBSSR_MASKSS_Pos)                  /*!< 0x02000000 */
#define RTC_ALRMBSSR_MASKSS_2                     (0x4UL<<RTC_ALRMBSSR_MASKSS_Pos)                  /*!< 0x04000000 */
#define RTC_ALRMBSSR_MASKSS_3                     (0x8UL<<RTC_ALRMBSSR_MASKSS_Pos)                  /*!< 0x08000000 */

/********************************* Bit definition for RTC_SR register ***********************************************/
#define RTC_SR_ALRAF_Pos                          (0U)
#define RTC_SR_ALRAF_Msk                          (0x1UL<<RTC_SR_ALRAF_Pos)                         /*!< 0x00000001 */
#define RTC_SR_ALRAF                              RTC_SR_ALRAF_Msk                                  
#define RTC_SR_ALRBF_Pos                          (1U)
#define RTC_SR_ALRBF_Msk                          (0x1UL<<RTC_SR_ALRBF_Pos)                         /*!< 0x00000002 */
#define RTC_SR_ALRBF                              RTC_SR_ALRBF_Msk                                  
#define RTC_SR_WUTF_Pos                           (2U)
#define RTC_SR_WUTF_Msk                           (0x1UL<<RTC_SR_WUTF_Pos)                          /*!< 0x00000004 */
#define RTC_SR_WUTF                               RTC_SR_WUTF_Msk                                   
#define RTC_SR_TSF_Pos                            (3U)
#define RTC_SR_TSF_Msk                            (0x1UL<<RTC_SR_TSF_Pos)                           /*!< 0x00000008 */
#define RTC_SR_TSF                                RTC_SR_TSF_Msk                                    
#define RTC_SR_TSOVF_Pos                          (4U)
#define RTC_SR_TSOVF_Msk                          (0x1UL<<RTC_SR_TSOVF_Pos)                         /*!< 0x00000010 */
#define RTC_SR_TSOVF                              RTC_SR_TSOVF_Msk                                  
#define RTC_SR_ITSF_Pos                           (5U)
#define RTC_SR_ITSF_Msk                           (0x1UL<<RTC_SR_ITSF_Pos)                          /*!< 0x00000020 */
#define RTC_SR_ITSF                               RTC_SR_ITSF_Msk                                   

/********************************* Bit definition for RTC_MISR register *********************************************/
#define RTC_MISR_ALRAMF_Pos                       (0U)
#define RTC_MISR_ALRAMF_Msk                       (0x1UL<<RTC_MISR_ALRAMF_Pos)                      /*!< 0x00000001 */
#define RTC_MISR_ALRAMF                           RTC_MISR_ALRAMF_Msk                               
#define RTC_MISR_ALRBMF_Pos                       (1U)
#define RTC_MISR_ALRBMF_Msk                       (0x1UL<<RTC_MISR_ALRBMF_Pos)                      /*!< 0x00000002 */
#define RTC_MISR_ALRBMF                           RTC_MISR_ALRBMF_Msk                               
#define RTC_MISR_WUTMF_Pos                        (2U)
#define RTC_MISR_WUTMF_Msk                        (0x1UL<<RTC_MISR_WUTMF_Pos)                       /*!< 0x00000004 */
#define RTC_MISR_WUTMF                            RTC_MISR_WUTMF_Msk                                
#define RTC_MISR_TSMF_Pos                         (3U)
#define RTC_MISR_TSMF_Msk                         (0x1UL<<RTC_MISR_TSMF_Pos)                        /*!< 0x00000008 */
#define RTC_MISR_TSMF                             RTC_MISR_TSMF_Msk                                 
#define RTC_MISR_TSOVMF_Pos                       (4U)
#define RTC_MISR_TSOVMF_Msk                       (0x1UL<<RTC_MISR_TSOVMF_Pos)                      /*!< 0x00000010 */
#define RTC_MISR_TSOVMF                           RTC_MISR_TSOVMF_Msk                               
#define RTC_MISR_ITSMF_Pos                        (5U)
#define RTC_MISR_ITSMF_Msk                        (0x1UL<<RTC_MISR_ITSMF_Pos)                       /*!< 0x00000020 */
#define RTC_MISR_ITSMF                            RTC_MISR_ITSMF_Msk                                

/********************************* Bit definition for RTC_SCR register **********************************************/
#define RTC_SCR_CALRAF_Pos                        (0U)
#define RTC_SCR_CALRAF_Msk                        (0x1UL<<RTC_SCR_CALRAF_Pos)                       /*!< 0x00000001 */
#define RTC_SCR_CALRAF                            RTC_SCR_CALRAF_Msk                                
#define RTC_SCR_CALRBF_Pos                        (1U)
#define RTC_SCR_CALRBF_Msk                        (0x1UL<<RTC_SCR_CALRBF_Pos)                       /*!< 0x00000002 */
#define RTC_SCR_CALRBF                            RTC_SCR_CALRBF_Msk                                
#define RTC_SCR_CWUTF_Pos                         (2U)
#define RTC_SCR_CWUTF_Msk                         (0x1UL<<RTC_SCR_CWUTF_Pos)                        /*!< 0x00000004 */
#define RTC_SCR_CWUTF                             RTC_SCR_CWUTF_Msk                                 
#define RTC_SCR_CTSF_Pos                          (3U)
#define RTC_SCR_CTSF_Msk                          (0x1UL<<RTC_SCR_CTSF_Pos)                         /*!< 0x00000008 */
#define RTC_SCR_CTSF                              RTC_SCR_CTSF_Msk                                  
#define RTC_SCR_CTSOVF_Pos                        (4U)
#define RTC_SCR_CTSOVF_Msk                        (0x1UL<<RTC_SCR_CTSOVF_Pos)                       /*!< 0x00000010 */
#define RTC_SCR_CTSOVF                            RTC_SCR_CTSOVF_Msk                                
#define RTC_SCR_CITSF_Pos                         (5U)
#define RTC_SCR_CITSF_Msk                         (0x1UL<<RTC_SCR_CITSF_Pos)                        /*!< 0x00000020 */
#define RTC_SCR_CITSF                             RTC_SCR_CITSF_Msk                                 

/********************************************************************************************************************/
/********************************* SPI ******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for SPI_CR1 register **********************************************/
#define SPI_CR1_CPHA_Pos                          (0U)
#define SPI_CR1_CPHA_Msk                          (0x1UL<<SPI_CR1_CPHA_Pos)                         /*!< 0x00000001 */
#define SPI_CR1_CPHA                              SPI_CR1_CPHA_Msk                                  /*!< Clock Phase */
#define SPI_CR1_CPOL_Pos                          (1U)
#define SPI_CR1_CPOL_Msk                          (0x1UL<<SPI_CR1_CPOL_Pos)                         /*!< 0x00000002 */
#define SPI_CR1_CPOL                              SPI_CR1_CPOL_Msk                                  /*!< Clock Polarity  */
#define SPI_CR1_MSTR_Pos                          (2U)
#define SPI_CR1_MSTR_Msk                          (0x1UL<<SPI_CR1_MSTR_Pos)                         /*!< 0x00000004 */
#define SPI_CR1_MSTR                              SPI_CR1_MSTR_Msk                                  /*!< Master Selection */
#define SPI_CR1_BR_Pos                            (3U)
#define SPI_CR1_BR_Msk                            (0x7UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000038 */
#define SPI_CR1_BR                                SPI_CR1_BR_Msk                                    /*!< Baud Rate Control */
#define SPI_CR1_BR_0                              (0x1UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000008 */
#define SPI_CR1_BR_1                              (0x2UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000010 */
#define SPI_CR1_BR_2                              (0x4UL<<SPI_CR1_BR_Pos)                           /*!< 0x00000020 */
#define SPI_CR1_SPE_Pos                           (6U)
#define SPI_CR1_SPE_Msk                           (0x1UL<<SPI_CR1_SPE_Pos)                          /*!< 0x00000040 */
#define SPI_CR1_SPE                               SPI_CR1_SPE_Msk                                   /*!< SPI Enable */
#define SPI_CR1_LSBFIRST_Pos                      (7U)
#define SPI_CR1_LSBFIRST_Msk                      (0x1UL<<SPI_CR1_LSBFIRST_Pos)                     /*!< 0x00000080 */
#define SPI_CR1_LSBFIRST                          SPI_CR1_LSBFIRST_Msk                              /*!<  Frame Format */
#define SPI_CR1_SSI_Pos                           (8U)
#define SPI_CR1_SSI_Msk                           (0x1UL<<SPI_CR1_SSI_Pos)                          /*!< 0x00000100 */
#define SPI_CR1_SSI                               SPI_CR1_SSI_Msk                                   /*!< Internal slave select */
#define SPI_CR1_SSM_Pos                           (9U)
#define SPI_CR1_SSM_Msk                           (0x1UL<<SPI_CR1_SSM_Pos)                          /*!< 0x00000200 */
#define SPI_CR1_SSM                               SPI_CR1_SSM_Msk                                   /*!< Software slave management  */
#define SPI_CR1_RXONLY_Pos                        (10U)
#define SPI_CR1_RXONLY_Msk                        (0x1UL<<SPI_CR1_RXONLY_Pos)                       /*!< 0x00000400 */
#define SPI_CR1_RXONLY                            SPI_CR1_RXONLY_Msk                                /*!< Receive only */
#define SPI_CR1_DFF_Pos                           (11U)
#define SPI_CR1_DFF_Msk                           (0x1UL<<SPI_CR1_DFF_Pos)                          /*!< 0x00000800 */
#define SPI_CR1_DFF                               SPI_CR1_DFF_Msk                                   /*!< Data frame format */
#define SPI_CR1_CRCNEXT_Pos                       (12U)
#define SPI_CR1_CRCNEXT_Msk                       (0x1UL<<SPI_CR1_CRCNEXT_Pos)                      /*!< 0x00001000 */
#define SPI_CR1_CRCNEXT                           SPI_CR1_CRCNEXT_Msk                               /*!< Transmit CRC next */
#define SPI_CR1_CRCEN_Pos                         (13U)
#define SPI_CR1_CRCEN_Msk                         (0x1UL<<SPI_CR1_CRCEN_Pos)                        /*!< 0x00002000 */
#define SPI_CR1_CRCEN                             SPI_CR1_CRCEN_Msk                                 /*!< Hardware CRC calculation enable */
#define SPI_CR1_BIDIOE_Pos                        (14U)
#define SPI_CR1_BIDIOE_Msk                        (0x1UL<<SPI_CR1_BIDIOE_Pos)                       /*!< 0x00004000 */
#define SPI_CR1_BIDIOE                            SPI_CR1_BIDIOE_Msk                                /*!< Output enable in bidirectional mode */
#define SPI_CR1_BIDIMODE_Pos                      (15U)
#define SPI_CR1_BIDIMODE_Msk                      (0x1UL<<SPI_CR1_BIDIMODE_Pos)                     /*!< 0x00008000 */
#define SPI_CR1_BIDIMODE                          SPI_CR1_BIDIMODE_Msk                              /*!< Bidirectional data mode enable */

/********************************* Bit definition for SPI_CR2 register **********************************************/
#define SPI_CR2_RXDMAEN_Pos                       (0U)
#define SPI_CR2_RXDMAEN_Msk                       (0x1UL<<SPI_CR2_RXDMAEN_Pos)                      /*!< 0x00000001 */
#define SPI_CR2_RXDMAEN                           SPI_CR2_RXDMAEN_Msk                               /*!< Rx buffer DMA enable */
#define SPI_CR2_TXDMAEN_Pos                       (1U)
#define SPI_CR2_TXDMAEN_Msk                       (0x1UL<<SPI_CR2_TXDMAEN_Pos)                      /*!< 0x00000002 */
#define SPI_CR2_TXDMAEN                           SPI_CR2_TXDMAEN_Msk                               /*!< Tx buffer DMA enable */
#define SPI_CR2_SSOE_Pos                          (2U)
#define SPI_CR2_SSOE_Msk                          (0x1UL<<SPI_CR2_SSOE_Pos)                         /*!< 0x00000004 */
#define SPI_CR2_SSOE                              SPI_CR2_SSOE_Msk                                  /*!< SS Output Enable  */
#define SPI_CR2_CLRTXFIFO_Pos                     (4U)
#define SPI_CR2_CLRTXFIFO_Msk                     (0x1UL<<SPI_CR2_CLRTXFIFO_Pos)                    /*!< 0x00000010 */
#define SPI_CR2_CLRTXFIFO                         SPI_CR2_CLRTXFIFO_Msk                             /*!< Clear FIFO */
#define SPI_CR2_ERRIE_Pos                         (5U)
#define SPI_CR2_ERRIE_Msk                         (0x1UL<<SPI_CR2_ERRIE_Pos)                        /*!< 0x00000020 */
#define SPI_CR2_ERRIE                             SPI_CR2_ERRIE_Msk                                 /*!< Error Interrupt Enable */
#define SPI_CR2_RXNEIE_Pos                        (6U)
#define SPI_CR2_RXNEIE_Msk                        (0x1UL<<SPI_CR2_RXNEIE_Pos)                       /*!< 0x00000040 */
#define SPI_CR2_RXNEIE                            SPI_CR2_RXNEIE_Msk                                /*!< RX buffer Not Empty Interrupt Enable */
#define SPI_CR2_TXEIE_Pos                         (7U)
#define SPI_CR2_TXEIE_Msk                         (0x1UL<<SPI_CR2_TXEIE_Pos)                        /*!< 0x00000080 */
#define SPI_CR2_TXEIE                             SPI_CR2_TXEIE_Msk                                 /*!< Tx buffer Empty Interrupt Enable */
#define SPI_CR2_FRXTH_Pos                         (12U)
#define SPI_CR2_FRXTH_Msk                         (0x1UL<<SPI_CR2_FRXTH_Pos)                        /*!< 0x00001000 */
#define SPI_CR2_FRXTH                             SPI_CR2_FRXTH_Msk                                 
#define SPI_CR2_FRF_Pos                           (15U)
#define SPI_CR2_FRF_Msk                           (0x1UL<<SPI_CR2_FRF_Pos)                          /*!< 0x00008000 */
#define SPI_CR2_FRF                               SPI_CR2_FRF_Msk                                   

/********************************* Bit definition for SPI_SR register ***********************************************/
#define SPI_SR_RXNE_Pos                           (0U)
#define SPI_SR_RXNE_Msk                           (0x1UL<<SPI_SR_RXNE_Pos)                          /*!< 0x00000001 */
#define SPI_SR_RXNE                               SPI_SR_RXNE_Msk                                   /*!< Receive buffer Not Empty  */
#define SPI_SR_TXE_Pos                            (1U)
#define SPI_SR_TXE_Msk                            (0x1UL<<SPI_SR_TXE_Pos)                           /*!< 0x00000002 */
#define SPI_SR_TXE                                SPI_SR_TXE_Msk                                    /*!< Transmit buffer Empty  */
#define SPI_SR_CHSIDE_Pos                         (2U)
#define SPI_SR_CHSIDE_Msk                         (0x1UL<<SPI_SR_CHSIDE_Pos)                        /*!< 0x00000004 */
#define SPI_SR_CHSIDE                             SPI_SR_CHSIDE_Msk                                 /*!< Channel side */
#define SPI_SR_UDR_Pos                            (3U)
#define SPI_SR_UDR_Msk                            (0x1UL<<SPI_SR_UDR_Pos)                           /*!< 0x00000008 */
#define SPI_SR_UDR                                SPI_SR_UDR_Msk                                    /*!< Underrun flag */
#define SPI_SR_CRCERR_Pos                         (4U)
#define SPI_SR_CRCERR_Msk                         (0x1UL<<SPI_SR_CRCERR_Pos)                        /*!< 0x00000010 */
#define SPI_SR_CRCERR                             SPI_SR_CRCERR_Msk                                 /*!< CRC error flag */
#define SPI_SR_MODF_Pos                           (5U)
#define SPI_SR_MODF_Msk                           (0x1UL<<SPI_SR_MODF_Pos)                          /*!< 0x00000020 */
#define SPI_SR_MODF                               SPI_SR_MODF_Msk                                   /*!< Mode fault */
#define SPI_SR_OVR_Pos                            (6U)
#define SPI_SR_OVR_Msk                            (0x1UL<<SPI_SR_OVR_Pos)                           /*!< 0x00000040 */
#define SPI_SR_OVR                                SPI_SR_OVR_Msk                                    /*!<  Overrun flag */
#define SPI_SR_BSY_Pos                            (7U)
#define SPI_SR_BSY_Msk                            (0x1UL<<SPI_SR_BSY_Pos)                           /*!< 0x00000080 */
#define SPI_SR_BSY                                SPI_SR_BSY_Msk                                    /*!< Busy flag  */
#define SPI_SR_FRE_Pos                            (8U)
#define SPI_SR_FRE_Msk                            (0x1UL<<SPI_SR_FRE_Pos)                           /*!< 0x00000100 */
#define SPI_SR_FRE                                SPI_SR_FRE_Msk                                    
#define SPI_SR_FRLVL_Pos                          (9U)
#define SPI_SR_FRLVL_Msk                          (0x3UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000600 */
#define SPI_SR_FRLVL                              SPI_SR_FRLVL_Msk                                  /*!< FIFO Reception Level */
#define SPI_SR_FRLVL_0                            (0x1UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000200 */
#define SPI_SR_FRLVL_1                            (0x2UL<<SPI_SR_FRLVL_Pos)                         /*!< 0x00000400 */
#define SPI_SR_FTLVL_Pos                          (11U)
#define SPI_SR_FTLVL_Msk                          (0x3UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00001800 */
#define SPI_SR_FTLVL                              SPI_SR_FTLVL_Msk                                  /*!< FIFO Transmission Level */
#define SPI_SR_FTLVL_0                            (0x1UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00000800 */
#define SPI_SR_FTLVL_1                            (0x2UL<<SPI_SR_FTLVL_Pos)                         /*!< 0x00001000 */

/********************************* Bit definition for SPI_DR register ***********************************************/
#define SPI_DR_DR_Pos                             (0U)
#define SPI_DR_DR_Msk                             (0xFFFFUL<<SPI_DR_DR_Pos)                         /*!< 0x0000FFFF */
#define SPI_DR_DR                                 SPI_DR_DR_Msk                                     /*!< Data Register */

/********************************* Bit definition for SPI_CRCPR register ********************************************/
#define SPI_CRCPR_CRCPOLY_Pos                     (0U)
#define SPI_CRCPR_CRCPOLY_Msk                     (0xFFFFUL<<SPI_CRCPR_CRCPOLY_Pos)                 /*!< 0x0000FFFF */
#define SPI_CRCPR_CRCPOLY                         SPI_CRCPR_CRCPOLY_Msk                             /*!< CRC polynomial register */

/********************************* Bit definition for SPI_RXCRCR register *******************************************/
#define SPI_RXCRCR_RXCRC_Pos                      (0U)
#define SPI_RXCRCR_RXCRC_Msk                      (0xFFFFUL<<SPI_RXCRCR_RXCRC_Pos)                  /*!< 0x0000FFFF */
#define SPI_RXCRCR_RXCRC                          SPI_RXCRCR_RXCRC_Msk                              /*!< rcv crc reg */

/********************************* Bit definition for SPI_TXCRCR register *******************************************/
#define SPI_TXCRCR_TXCRC_Pos                      (0U)
#define SPI_TXCRCR_TXCRC_Msk                      (0xFFFFUL<<SPI_TXCRCR_TXCRC_Pos)                  /*!< 0x0000FFFF */
#define SPI_TXCRCR_TXCRC                          SPI_TXCRCR_TXCRC_Msk                              /*!< tx crc reg  */

/********************************* Bit definition for SPI_I2S_CFGR register *****************************************/
#define SPI_I2S_CFGR_CHLEN_Pos                    (0U)
#define SPI_I2S_CFGR_CHLEN_Msk                    (0x1UL<<SPI_I2S_CFGR_CHLEN_Pos)                   /*!< 0x00000001 */
#define SPI_I2S_CFGR_CHLEN                        SPI_I2S_CFGR_CHLEN_Msk                            /*!< Channel length (number of bits per audio channel */
#define SPI_I2S_CFGR_DATLEN_Pos                   (1U)
#define SPI_I2S_CFGR_DATLEN_Msk                   (0x3UL<<SPI_I2S_CFGR_DATLEN_Pos)                  /*!< 0x00000006 */
#define SPI_I2S_CFGR_DATLEN                       SPI_I2S_CFGR_DATLEN_Msk                           /*!< Data length to be transferred */
#define SPI_I2S_CFGR_DATLEN_0                     (0x1UL<<SPI_I2S_CFGR_DATLEN_Pos)                  /*!< 0x00000002 */
#define SPI_I2S_CFGR_DATLEN_1                     (0x2UL<<SPI_I2S_CFGR_DATLEN_Pos)                  /*!< 0x00000004 */
#define SPI_I2S_CFGR_CKPOL_Pos                    (3U)
#define SPI_I2S_CFGR_CKPOL_Msk                    (0x1UL<<SPI_I2S_CFGR_CKPOL_Pos)                   /*!< 0x00000008 */
#define SPI_I2S_CFGR_CKPOL                        SPI_I2S_CFGR_CKPOL_Msk                            /*!< Steady state clock polarity */
#define SPI_I2S_CFGR_I2SSTD_Pos                   (4U)
#define SPI_I2S_CFGR_I2SSTD_Msk                   (0x3UL<<SPI_I2S_CFGR_I2SSTD_Pos)                  /*!< 0x00000030 */
#define SPI_I2S_CFGR_I2SSTD                       SPI_I2S_CFGR_I2SSTD_Msk                           /*!< I2S standard selection */
#define SPI_I2S_CFGR_I2SSTD_0                     (0x1UL<<SPI_I2S_CFGR_I2SSTD_Pos)                  /*!< 0x00000010 */
#define SPI_I2S_CFGR_I2SSTD_1                     (0x2UL<<SPI_I2S_CFGR_I2SSTD_Pos)                  /*!< 0x00000020 */
#define SPI_I2S_CFGR_PCMSYNC_Pos                  (7U)
#define SPI_I2S_CFGR_PCMSYNC_Msk                  (0x1UL<<SPI_I2S_CFGR_PCMSYNC_Pos)                 /*!< 0x00000080 */
#define SPI_I2S_CFGR_PCMSYNC                      SPI_I2S_CFGR_PCMSYNC_Msk                          /*!< PCM frame synchronization */
#define SPI_I2S_CFGR_I2SCFG_Pos                   (8U)
#define SPI_I2S_CFGR_I2SCFG_Msk                   (0x3UL<<SPI_I2S_CFGR_I2SCFG_Pos)                  /*!< 0x00000300 */
#define SPI_I2S_CFGR_I2SCFG                       SPI_I2S_CFGR_I2SCFG_Msk                           /*!< I2S configuration mode */
#define SPI_I2S_CFGR_I2SCFG_0                     (0x1UL<<SPI_I2S_CFGR_I2SCFG_Pos)                  /*!< 0x00000100 */
#define SPI_I2S_CFGR_I2SCFG_1                     (0x2UL<<SPI_I2S_CFGR_I2SCFG_Pos)                  /*!< 0x00000200 */
#define SPI_I2S_CFGR_I2SE_Pos                     (10U)
#define SPI_I2S_CFGR_I2SE_Msk                     (0x1UL<<SPI_I2S_CFGR_I2SE_Pos)                    /*!< 0x00000400 */
#define SPI_I2S_CFGR_I2SE                         SPI_I2S_CFGR_I2SE_Msk                             /*!<  I2S enable */
#define SPI_I2S_CFGR_I2SMOD_Pos                   (11U)
#define SPI_I2S_CFGR_I2SMOD_Msk                   (0x1UL<<SPI_I2S_CFGR_I2SMOD_Pos)                  /*!< 0x00000800 */
#define SPI_I2S_CFGR_I2SMOD                       SPI_I2S_CFGR_I2SMOD_Msk                           /*!< I2S mode selection */

/********************************* Bit definition for SPI_I2SPR register ********************************************/
#define SPI_I2SPR_I2SDIV_Pos                      (0U)
#define SPI_I2SPR_I2SDIV_Msk                      (0xFFUL<<SPI_I2SPR_I2SDIV_Pos)                    /*!< 0x000000FF */
#define SPI_I2SPR_I2SDIV                          SPI_I2SPR_I2SDIV_Msk                              /*!< I2S linear prescaler */
#define SPI_I2SPR_ODD_Pos                         (8U)
#define SPI_I2SPR_ODD_Msk                         (0x1UL<<SPI_I2SPR_ODD_Pos)                        /*!< 0x00000100 */
#define SPI_I2SPR_ODD                             SPI_I2SPR_ODD_Msk                                 /*!< Odd factor for the prescaler */
#define SPI_I2SPR_MCKOE_Pos                       (9U)
#define SPI_I2SPR_MCKOE_Msk                       (0x1UL<<SPI_I2SPR_MCKOE_Pos)                      /*!< 0x00000200 */
#define SPI_I2SPR_MCKOE                           SPI_I2SPR_MCKOE_Msk                               /*!<  Master clock output enable */

#define SPI_I2S_SUPPORT       /*!< I2S support */

/********************************************************************************************************************/
/********************************* SYSCFG ***************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for SYSCFG_CFGR1 register *****************************************/
#define SYSCFG_CFGR1_MEM_MODE_Pos                 (0U)
#define SYSCFG_CFGR1_MEM_MODE_Msk                 (0x3UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000003 */
#define SYSCFG_CFGR1_MEM_MODE                     SYSCFG_CFGR1_MEM_MODE_Msk
#define SYSCFG_CFGR1_MEM_MODE_0                   (0x1UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000001 */
#define SYSCFG_CFGR1_MEM_MODE_1                   (0x2UL<<SYSCFG_CFGR1_MEM_MODE_Pos)                /*!< 0x00000002 */
#define SYSCFG_CFGR1_TK_LP_TRIG_Pos               (8U)
#define SYSCFG_CFGR1_TK_LP_TRIG_Msk               (0x3UL<<SYSCFG_CFGR1_TK_LP_TRIG_Pos)              /*!< 0x00000300 */
#define SYSCFG_CFGR1_TK_LP_TRIG                   SYSCFG_CFGR1_TK_LP_TRIG_Msk
#define SYSCFG_CFGR1_TK_LP_TRIG_0                 (0x1UL<<SYSCFG_CFGR1_TK_LP_TRIG_Pos)              /*!< 0x00000100 */
#define SYSCFG_CFGR1_TK_LP_TRIG_1                 (0x2UL<<SYSCFG_CFGR1_TK_LP_TRIG_Pos)              /*!< 0x00000200 */
#define SYSCFG_CFGR1_RTC_TRIG_TKSEL_Pos           (10U)
#define SYSCFG_CFGR1_RTC_TRIG_TKSEL_Msk           (0x3UL<<SYSCFG_CFGR1_RTC_TRIG_TKSEL_Pos)          /*!< 0x00000C00 */
#define SYSCFG_CFGR1_RTC_TRIG_TKSEL               SYSCFG_CFGR1_RTC_TRIG_TKSEL_Msk
#define SYSCFG_CFGR1_RTC_TRIG_TKSEL_0             (0x1UL<<SYSCFG_CFGR1_RTC_TRIG_TKSEL_Pos)          /*!< 0x00000400 */
#define SYSCFG_CFGR1_RTC_TRIG_TKSEL_1             (0x2UL<<SYSCFG_CFGR1_RTC_TRIG_TKSEL_Pos)          /*!< 0x00000800 */
#define SYSCFG_CFGR1_LPTIM1_TRIG_TKEN_Pos         (12U)
#define SYSCFG_CFGR1_LPTIM1_TRIG_TKEN_Msk         (0x1UL<<SYSCFG_CFGR1_LPTIM1_TRIG_TKEN_Pos)        /*!< 0x00001000 */
#define SYSCFG_CFGR1_LPTIM1_TRIG_TKEN             SYSCFG_CFGR1_LPTIM1_TRIG_TKEN_Msk                 
#define SYSCFG_CFGR1_LPTIM2_TRIG_TKEN_Pos         (13U)
#define SYSCFG_CFGR1_LPTIM2_TRIG_TKEN_Msk         (0x1UL<<SYSCFG_CFGR1_LPTIM2_TRIG_TKEN_Pos)        /*!< 0x00002000 */
#define SYSCFG_CFGR1_LPTIM2_TRIG_TKEN             SYSCFG_CFGR1_LPTIM2_TRIG_TKEN_Msk                 
#define SYSCFG_CFGR1_GPIO_AHB_SEL_Pos             (24U)
#define SYSCFG_CFGR1_GPIO_AHB_SEL_Msk             (0x1UL<<SYSCFG_CFGR1_GPIO_AHB_SEL_Pos)            /*!< 0x01000000 */
#define SYSCFG_CFGR1_GPIO_AHB_SEL                 SYSCFG_CFGR1_GPIO_AHB_SEL_Msk                     

/********************************* Bit definition for SYSCFG_CFGR2 register *****************************************/
#define SYSCFG_CFGR2_CLL_Pos                      (0U)
#define SYSCFG_CFGR2_CLL_Msk                      (0x1UL<<SYSCFG_CFGR2_CLL_Pos)                     /*!< 0x00000001 */
#define SYSCFG_CFGR2_CLL                          SYSCFG_CFGR2_CLL_Msk                              
#define SYSCFG_CFGR2_SPL_Pos                      (1U)
#define SYSCFG_CFGR2_SPL_Msk                      (0x1UL<<SYSCFG_CFGR2_SPL_Pos)                     /*!< 0x00000002 */
#define SYSCFG_CFGR2_SPL                          SYSCFG_CFGR2_SPL_Msk                              
#define SYSCFG_CFGR2_PVDL_Pos                     (2U)
#define SYSCFG_CFGR2_PVDL_Msk                     (0x1UL<<SYSCFG_CFGR2_PVDL_Pos)                    /*!< 0x00000004 */
#define SYSCFG_CFGR2_PVDL                         SYSCFG_CFGR2_PVDL_Msk                             
#define SYSCFG_CFGR2_ECCL_Pos                     (3U)
#define SYSCFG_CFGR2_ECCL_Msk                     (0x1UL<<SYSCFG_CFGR2_ECCL_Pos)                    /*!< 0x00000008 */
#define SYSCFG_CFGR2_ECCL                         SYSCFG_CFGR2_ECCL_Msk                             

/********************************* Bit definition for SYSCFG_CFGR3 register *****************************************/
#define SYSCFG_CFGR3_DMA1_MAP_Pos                 (0U)
#define SYSCFG_CFGR3_DMA1_MAP_Msk                 (0x3FUL<<SYSCFG_CFGR3_DMA1_MAP_Pos)               /*!< 0x0000003F */
#define SYSCFG_CFGR3_DMA1_MAP                     SYSCFG_CFGR3_DMA1_MAP_Msk
#define SYSCFG_CFGR3_DMA1_MAP_0                   (0x1UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)                /*!< 0x00000001 */
#define SYSCFG_CFGR3_DMA1_MAP_1                   (0x2UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)                /*!< 0x00000002 */
#define SYSCFG_CFGR3_DMA1_MAP_2                   (0x4UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)                /*!< 0x00000004 */
#define SYSCFG_CFGR3_DMA1_MAP_3                   (0x8UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)                /*!< 0x00000008 */
#define SYSCFG_CFGR3_DMA1_MAP_4                   (0x10UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)               /*!< 0x00000010 */
#define SYSCFG_CFGR3_DMA1_MAP_5                   (0x20UL<<SYSCFG_CFGR3_DMA1_MAP_Pos)               /*!< 0x00000020 */
#define SYSCFG_CFGR3_DMA2_MAP_Pos                 (8U)
#define SYSCFG_CFGR3_DMA2_MAP_Msk                 (0x3FUL<<SYSCFG_CFGR3_DMA2_MAP_Pos)               /*!< 0x00003F00 */
#define SYSCFG_CFGR3_DMA2_MAP                     SYSCFG_CFGR3_DMA2_MAP_Msk
#define SYSCFG_CFGR3_DMA2_MAP_0                   (0x1UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)                /*!< 0x00000100 */
#define SYSCFG_CFGR3_DMA2_MAP_1                   (0x2UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)                /*!< 0x00000200 */
#define SYSCFG_CFGR3_DMA2_MAP_2                   (0x4UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)                /*!< 0x00000400 */
#define SYSCFG_CFGR3_DMA2_MAP_3                   (0x8UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)                /*!< 0x00000800 */
#define SYSCFG_CFGR3_DMA2_MAP_4                   (0x10UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)               /*!< 0x00001000 */
#define SYSCFG_CFGR3_DMA2_MAP_5                   (0x20UL<<SYSCFG_CFGR3_DMA2_MAP_Pos)               /*!< 0x00002000 */
#define SYSCFG_CFGR3_DMA3_MAP_Pos                 (16U)
#define SYSCFG_CFGR3_DMA3_MAP_Msk                 (0x3FUL<<SYSCFG_CFGR3_DMA3_MAP_Pos)               /*!< 0x003F0000 */
#define SYSCFG_CFGR3_DMA3_MAP                     SYSCFG_CFGR3_DMA3_MAP_Msk
#define SYSCFG_CFGR3_DMA3_MAP_0                   (0x1UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)                /*!< 0x00010000 */
#define SYSCFG_CFGR3_DMA3_MAP_1                   (0x2UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)                /*!< 0x00020000 */
#define SYSCFG_CFGR3_DMA3_MAP_2                   (0x4UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)                /*!< 0x00040000 */
#define SYSCFG_CFGR3_DMA3_MAP_3                   (0x8UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)                /*!< 0x00080000 */
#define SYSCFG_CFGR3_DMA3_MAP_4                   (0x10UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)               /*!< 0x00100000 */
#define SYSCFG_CFGR3_DMA3_MAP_5                   (0x20UL<<SYSCFG_CFGR3_DMA3_MAP_Pos)               /*!< 0x00200000 */
#define SYSCFG_CFGR3_DMA4_MAP_Pos                 (24U)
#define SYSCFG_CFGR3_DMA4_MAP_Msk                 (0x3FUL<<SYSCFG_CFGR3_DMA4_MAP_Pos)               /*!< 0x3F000000 */
#define SYSCFG_CFGR3_DMA4_MAP                     SYSCFG_CFGR3_DMA4_MAP_Msk
#define SYSCFG_CFGR3_DMA4_MAP_0                   (0x1UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)                /*!< 0x01000000 */
#define SYSCFG_CFGR3_DMA4_MAP_1                   (0x2UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)                /*!< 0x02000000 */
#define SYSCFG_CFGR3_DMA4_MAP_2                   (0x4UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)                /*!< 0x04000000 */
#define SYSCFG_CFGR3_DMA4_MAP_3                   (0x8UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)                /*!< 0x08000000 */
#define SYSCFG_CFGR3_DMA4_MAP_4                   (0x10UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)               /*!< 0x10000000 */
#define SYSCFG_CFGR3_DMA4_MAP_5                   (0x20UL<<SYSCFG_CFGR3_DMA4_MAP_Pos)               /*!< 0x20000000 */

/********************************* Bit definition for PA_ENS_REG register *******************************************/
#define PA_ENS_REG_PA_ENS_Pos                     (0U)
#define PA_ENS_REG_PA_ENS_Msk                     (0xFFFFUL<<PA_ENS_REG_PA_ENS_Pos)                 /*!< 0x0000FFFF */
#define PA_ENS_REG_PA_ENS                         PA_ENS_REG_PA_ENS_Msk

/********************************* Bit definition for PB_ENS_REG register *******************************************/
#define PB_ENS_REG_PB_ENS_Pos                     (0U)
#define PB_ENS_REG_PB_ENS_Msk                     (0xFFFFUL<<PB_ENS_REG_PB_ENS_Pos)                 /*!< 0x0000FFFF */
#define PB_ENS_REG_PB_ENS                         PB_ENS_REG_PB_ENS_Msk

/********************************* Bit definition for PC_ENS_REG register *******************************************/
#define PC_ENS_REG_PC_ENS_Pos                     (0U)
#define PC_ENS_REG_PC_ENS_Msk                     (0xFFFFUL<<PC_ENS_REG_PC_ENS_Pos)                 /*!< 0x0000FFFF */
#define PC_ENS_REG_PC_ENS                         PC_ENS_REG_PC_ENS_Msk

/********************************* Bit definition for PD_ENS_REG register *******************************************/
#define PD_ENS_REG_PD_ENS_Pos                     (0U)
#define PD_ENS_REG_PD_ENS_Msk                     (0x3FFFUL<<PD_ENS_REG_PD_ENS_Pos)                 /*!< 0x00003FFF */
#define PD_ENS_REG_PD_ENS                         PD_ENS_REG_PD_ENS_Msk

/********************************* Bit definition for SYSCFG_IOCFG register *****************************************/
#define SYSCFG_IOCFG_PA_EIIC_Pos                  (0U)
#define SYSCFG_IOCFG_PA_EIIC_Msk                  (0xFUL<<SYSCFG_IOCFG_PA_EIIC_Pos)                 /*!< 0x0000000F */
#define SYSCFG_IOCFG_PA_EIIC                      SYSCFG_IOCFG_PA_EIIC_Msk
#define SYSCFG_IOCFG_PA_EIIC_0                    (0x1UL<<SYSCFG_IOCFG_PA_EIIC_Pos)                 /*!< 0x00000001 */
#define SYSCFG_IOCFG_PA_EIIC_1                    (0x2UL<<SYSCFG_IOCFG_PA_EIIC_Pos)                 /*!< 0x00000002 */
#define SYSCFG_IOCFG_PA_EIIC_2                    (0x4UL<<SYSCFG_IOCFG_PA_EIIC_Pos)                 /*!< 0x00000004 */
#define SYSCFG_IOCFG_PA_EIIC_3                    (0x8UL<<SYSCFG_IOCFG_PA_EIIC_Pos)                 /*!< 0x00000008 */
#define SYSCFG_IOCFG_PA_I2C_FMP_Pos               (4U)
#define SYSCFG_IOCFG_PA_I2C_FMP_Msk               (0x1UL<<SYSCFG_IOCFG_PA_I2C_FMP_Pos)              /*!< 0x00000010 */
#define SYSCFG_IOCFG_PA_I2C_FMP                   SYSCFG_IOCFG_PA_I2C_FMP_Msk                       
#define SYSCFG_IOCFG_PB_I2C_FMP_Pos               (5U)
#define SYSCFG_IOCFG_PB_I2C_FMP_Msk               (0x1UL<<SYSCFG_IOCFG_PB_I2C_FMP_Pos)              /*!< 0x00000020 */
#define SYSCFG_IOCFG_PB_I2C_FMP                   SYSCFG_IOCFG_PB_I2C_FMP_Msk                       
#define SYSCFG_IOCFG_PC_I2C_FMP_Pos               (6U)
#define SYSCFG_IOCFG_PC_I2C_FMP_Msk               (0x1UL<<SYSCFG_IOCFG_PC_I2C_FMP_Pos)              /*!< 0x00000040 */
#define SYSCFG_IOCFG_PC_I2C_FMP                   SYSCFG_IOCFG_PC_I2C_FMP_Msk                       
#define SYSCFG_IOCFG_PD_I2C_FMP_Pos               (7U)
#define SYSCFG_IOCFG_PD_I2C_FMP_Msk               (0x1UL<<SYSCFG_IOCFG_PD_I2C_FMP_Pos)              /*!< 0x00000080 */
#define SYSCFG_IOCFG_PD_I2C_FMP                   SYSCFG_IOCFG_PD_I2C_FMP_Msk                       
#define SYSCFG_IOCFG_PB_EIIC_Pos                  (8U)
#define SYSCFG_IOCFG_PB_EIIC_Msk                  (0x3FUL<<SYSCFG_IOCFG_PB_EIIC_Pos)                /*!< 0x00003F00 */
#define SYSCFG_IOCFG_PB_EIIC                      SYSCFG_IOCFG_PB_EIIC_Msk
#define SYSCFG_IOCFG_PB_EIIC_0                    (0x1UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                 /*!< 0x00000100 */
#define SYSCFG_IOCFG_PB_EIIC_1                    (0x2UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                 /*!< 0x00000200 */
#define SYSCFG_IOCFG_PB_EIIC_2                    (0x4UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                 /*!< 0x00000400 */
#define SYSCFG_IOCFG_PB_EIIC_3                    (0x8UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                 /*!< 0x00000800 */
#define SYSCFG_IOCFG_PB_EIIC_4                    (0x10UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                /*!< 0x00001000 */
#define SYSCFG_IOCFG_PB_EIIC_5                    (0x20UL<<SYSCFG_IOCFG_PB_EIIC_Pos)                /*!< 0x00002000 */
#define SYSCFG_IOCFG_PC_EIIC_Pos                  (16U)
#define SYSCFG_IOCFG_PC_EIIC_Msk                  (0x1FFUL<<SYSCFG_IOCFG_PC_EIIC_Pos)               /*!< 0x01FF0000 */
#define SYSCFG_IOCFG_PC_EIIC                      SYSCFG_IOCFG_PC_EIIC_Msk
#define SYSCFG_IOCFG_PD_EIIC_Pos                  (28U)
#define SYSCFG_IOCFG_PD_EIIC_Msk                  (0x3UL<<SYSCFG_IOCFG_PD_EIIC_Pos)                 /*!< 0x30000000 */
#define SYSCFG_IOCFG_PD_EIIC                      SYSCFG_IOCFG_PD_EIIC_Msk
#define SYSCFG_IOCFG_PD_EIIC_0                    (0x1UL<<SYSCFG_IOCFG_PD_EIIC_Pos)                 /*!< 0x10000000 */
#define SYSCFG_IOCFG_PD_EIIC_1                    (0x2UL<<SYSCFG_IOCFG_PD_EIIC_Pos)                 /*!< 0x20000000 */
#define SYSCFG_IOCFG_PC_PU_IIC_Pos                (30U)
#define SYSCFG_IOCFG_PC_PU_IIC_Msk                (0x3UL<<SYSCFG_IOCFG_PC_PU_IIC_Pos)               /*!< 0xC0000000 */
#define SYSCFG_IOCFG_PC_PU_IIC                    SYSCFG_IOCFG_PC_PU_IIC_Msk
#define SYSCFG_IOCFG_PC_PU_IIC_0                  (0x1UL<<SYSCFG_IOCFG_PC_PU_IIC_Pos)               /*!< 0x40000000 */
#define SYSCFG_IOCFG_PC_PU_IIC_1                  (0x2UL<<SYSCFG_IOCFG_PC_PU_IIC_Pos)               /*!< 0x80000000 */

/********************************* Bit definition for SYSCFG_LEDCFG register ****************************************/
#define SYSCFG_LEDCFG_SEGIS_EN_Pos                (0U)
#define SYSCFG_LEDCFG_SEGIS_EN_Msk                (0xFUL<<SYSCFG_LEDCFG_SEGIS_EN_Pos)               /*!< 0x0000000F */
#define SYSCFG_LEDCFG_SEGIS_EN                    SYSCFG_LEDCFG_SEGIS_EN_Msk
#define SYSCFG_LEDCFG_SEGIS_EN_0                  (0x1UL<<SYSCFG_LEDCFG_SEGIS_EN_Pos)               /*!< 0x00000001 */
#define SYSCFG_LEDCFG_SEGIS_EN_1                  (0x2UL<<SYSCFG_LEDCFG_SEGIS_EN_Pos)               /*!< 0x00000002 */
#define SYSCFG_LEDCFG_SEGIS_EN_2                  (0x4UL<<SYSCFG_LEDCFG_SEGIS_EN_Pos)               /*!< 0x00000004 */
#define SYSCFG_LEDCFG_SEGIS_EN_3                  (0x8UL<<SYSCFG_LEDCFG_SEGIS_EN_Pos)               /*!< 0x00000008 */
#define SYSCFG_LEDCFG_PB_EHS_Pos                  (8U)
#define SYSCFG_LEDCFG_PB_EHS_Msk                  (0xFFUL<<SYSCFG_LEDCFG_PB_EHS_Pos)                /*!< 0x0000FF00 */
#define SYSCFG_LEDCFG_PB_EHS                      SYSCFG_LEDCFG_PB_EHS_Msk
#define SYSCFG_LEDCFG_PC_ENSEG_Pos                (30U)
#define SYSCFG_LEDCFG_PC_ENSEG_Msk                (0x3UL<<SYSCFG_LEDCFG_PC_ENSEG_Pos)               /*!< 0xC0000000 */
#define SYSCFG_LEDCFG_PC_ENSEG                    SYSCFG_LEDCFG_PC_ENSEG_Msk
#define SYSCFG_LEDCFG_PC_ENSEG_0                  (0x1UL<<SYSCFG_LEDCFG_PC_ENSEG_Pos)               /*!< 0x40000000 */
#define SYSCFG_LEDCFG_PC_ENSEG_1                  (0x2UL<<SYSCFG_LEDCFG_PC_ENSEG_Pos)               /*!< 0x80000000 */

/********************************* Bit definition for PA_ANA2EN_REG register ****************************************/
#define PA_ANA2EN_REG_PA_ANA2EN_Pos               (0U)
#define PA_ANA2EN_REG_PA_ANA2EN_Msk               (0xFFFFUL<<PA_ANA2EN_REG_PA_ANA2EN_Pos)           /*!< 0x0000FFFF */
#define PA_ANA2EN_REG_PA_ANA2EN                   PA_ANA2EN_REG_PA_ANA2EN_Msk

/********************************* Bit definition for PB_ANA2EN_REG register ****************************************/
#define PB_ANA2EN_REG_PB_ANA2EN_Pos               (0U)
#define PB_ANA2EN_REG_PB_ANA2EN_Msk               (0xFFFFUL<<PB_ANA2EN_REG_PB_ANA2EN_Pos)           /*!< 0x0000FFFF */
#define PB_ANA2EN_REG_PB_ANA2EN                   PB_ANA2EN_REG_PB_ANA2EN_Msk

/********************************* Bit definition for PC_ANA2EN_REG register ****************************************/
#define PC_ANA2EN_REG_PC_ANA2EN_Pos               (0U)
#define PC_ANA2EN_REG_PC_ANA2EN_Msk               (0xFFFFUL<<PC_ANA2EN_REG_PC_ANA2EN_Pos)           /*!< 0x0000FFFF */
#define PC_ANA2EN_REG_PC_ANA2EN                   PC_ANA2EN_REG_PC_ANA2EN_Msk

/********************************* Bit definition for PD_ANA2EN_REG register ****************************************/
#define PD_ANA2EN_REG_PD_ANA2EN_Pos               (0U)
#define PD_ANA2EN_REG_PD_ANA2EN_Msk               (0x3FFFUL<<PD_ANA2EN_REG_PD_ANA2EN_Pos)           /*!< 0x00003FFF */
#define PD_ANA2EN_REG_PD_ANA2EN                   PD_ANA2EN_REG_PD_ANA2EN_Msk

/********************************* Bit definition for PA_IORP register **********************************************/
#define PA_IORP_PA_IORP0_Pos                      (0U)
#define PA_IORP_PA_IORP0_Msk                      (0x3UL<<PA_IORP_PA_IORP0_Pos)                     /*!< 0x00000003 */
#define PA_IORP_PA_IORP0                          PA_IORP_PA_IORP0_Msk
#define PA_IORP_PA_IORP0_0                        (0x1UL<<PA_IORP_PA_IORP0_Pos)                     /*!< 0x00000001 */
#define PA_IORP_PA_IORP0_1                        (0x2UL<<PA_IORP_PA_IORP0_Pos)                     /*!< 0x00000002 */
#define PA_IORP_PA_IORP1_Pos                      (2U)
#define PA_IORP_PA_IORP1_Msk                      (0x3UL<<PA_IORP_PA_IORP1_Pos)                     /*!< 0x0000000C */
#define PA_IORP_PA_IORP1                          PA_IORP_PA_IORP1_Msk
#define PA_IORP_PA_IORP1_0                        (0x1UL<<PA_IORP_PA_IORP1_Pos)                     /*!< 0x00000004 */
#define PA_IORP_PA_IORP1_1                        (0x2UL<<PA_IORP_PA_IORP1_Pos)                     /*!< 0x00000008 */
#define PA_IORP_PA_IORP2_Pos                      (4U)
#define PA_IORP_PA_IORP2_Msk                      (0x3UL<<PA_IORP_PA_IORP2_Pos)                     /*!< 0x00000030 */
#define PA_IORP_PA_IORP2                          PA_IORP_PA_IORP2_Msk
#define PA_IORP_PA_IORP2_0                        (0x1UL<<PA_IORP_PA_IORP2_Pos)                     /*!< 0x00000010 */
#define PA_IORP_PA_IORP2_1                        (0x2UL<<PA_IORP_PA_IORP2_Pos)                     /*!< 0x00000020 */
#define PA_IORP_PA_IORP3_Pos                      (6U)
#define PA_IORP_PA_IORP3_Msk                      (0x3UL<<PA_IORP_PA_IORP3_Pos)                     /*!< 0x000000C0 */
#define PA_IORP_PA_IORP3                          PA_IORP_PA_IORP3_Msk
#define PA_IORP_PA_IORP3_0                        (0x1UL<<PA_IORP_PA_IORP3_Pos)                     /*!< 0x00000040 */
#define PA_IORP_PA_IORP3_1                        (0x2UL<<PA_IORP_PA_IORP3_Pos)                     /*!< 0x00000080 */
#define PA_IORP_PA_IORP4_Pos                      (8U)
#define PA_IORP_PA_IORP4_Msk                      (0x3UL<<PA_IORP_PA_IORP4_Pos)                     /*!< 0x00000300 */
#define PA_IORP_PA_IORP4                          PA_IORP_PA_IORP4_Msk
#define PA_IORP_PA_IORP4_0                        (0x1UL<<PA_IORP_PA_IORP4_Pos)                     /*!< 0x00000100 */
#define PA_IORP_PA_IORP4_1                        (0x2UL<<PA_IORP_PA_IORP4_Pos)                     /*!< 0x00000200 */
#define PA_IORP_PA_IORP5_Pos                      (10U)
#define PA_IORP_PA_IORP5_Msk                      (0x3UL<<PA_IORP_PA_IORP5_Pos)                     /*!< 0x00000C00 */
#define PA_IORP_PA_IORP5                          PA_IORP_PA_IORP5_Msk
#define PA_IORP_PA_IORP5_0                        (0x1UL<<PA_IORP_PA_IORP5_Pos)                     /*!< 0x00000400 */
#define PA_IORP_PA_IORP5_1                        (0x2UL<<PA_IORP_PA_IORP5_Pos)                     /*!< 0x00000800 */
#define PA_IORP_PA_IORP6_Pos                      (12U)
#define PA_IORP_PA_IORP6_Msk                      (0x3UL<<PA_IORP_PA_IORP6_Pos)                     /*!< 0x00003000 */
#define PA_IORP_PA_IORP6                          PA_IORP_PA_IORP6_Msk
#define PA_IORP_PA_IORP6_0                        (0x1UL<<PA_IORP_PA_IORP6_Pos)                     /*!< 0x00001000 */
#define PA_IORP_PA_IORP6_1                        (0x2UL<<PA_IORP_PA_IORP6_Pos)                     /*!< 0x00002000 */
#define PA_IORP_PA_IORP7_Pos                      (14U)
#define PA_IORP_PA_IORP7_Msk                      (0x3UL<<PA_IORP_PA_IORP7_Pos)                     /*!< 0x0000C000 */
#define PA_IORP_PA_IORP7                          PA_IORP_PA_IORP7_Msk
#define PA_IORP_PA_IORP7_0                        (0x1UL<<PA_IORP_PA_IORP7_Pos)                     /*!< 0x00004000 */
#define PA_IORP_PA_IORP7_1                        (0x2UL<<PA_IORP_PA_IORP7_Pos)                     /*!< 0x00008000 */
#define PA_IORP_PA_IORP8_Pos                      (16U)
#define PA_IORP_PA_IORP8_Msk                      (0x3UL<<PA_IORP_PA_IORP8_Pos)                     /*!< 0x00030000 */
#define PA_IORP_PA_IORP8                          PA_IORP_PA_IORP8_Msk
#define PA_IORP_PA_IORP8_0                        (0x1UL<<PA_IORP_PA_IORP8_Pos)                     /*!< 0x00010000 */
#define PA_IORP_PA_IORP8_1                        (0x2UL<<PA_IORP_PA_IORP8_Pos)                     /*!< 0x00020000 */
#define PA_IORP_PA_IORP9_Pos                      (18U)
#define PA_IORP_PA_IORP9_Msk                      (0x3UL<<PA_IORP_PA_IORP9_Pos)                     /*!< 0x000C0000 */
#define PA_IORP_PA_IORP9                          PA_IORP_PA_IORP9_Msk
#define PA_IORP_PA_IORP9_0                        (0x1UL<<PA_IORP_PA_IORP9_Pos)                     /*!< 0x00040000 */
#define PA_IORP_PA_IORP9_1                        (0x2UL<<PA_IORP_PA_IORP9_Pos)                     /*!< 0x00080000 */
#define PA_IORP_PA_IORP10_Pos                     (20U)
#define PA_IORP_PA_IORP10_Msk                     (0x3UL<<PA_IORP_PA_IORP10_Pos)                    /*!< 0x00300000 */
#define PA_IORP_PA_IORP10                         PA_IORP_PA_IORP10_Msk
#define PA_IORP_PA_IORP10_0                       (0x1UL<<PA_IORP_PA_IORP10_Pos)                    /*!< 0x00100000 */
#define PA_IORP_PA_IORP10_1                       (0x2UL<<PA_IORP_PA_IORP10_Pos)                    /*!< 0x00200000 */
#define PA_IORP_PA_IORP11_Pos                     (22U)
#define PA_IORP_PA_IORP11_Msk                     (0x3UL<<PA_IORP_PA_IORP11_Pos)                    /*!< 0x00C00000 */
#define PA_IORP_PA_IORP11                         PA_IORP_PA_IORP11_Msk
#define PA_IORP_PA_IORP11_0                       (0x1UL<<PA_IORP_PA_IORP11_Pos)                    /*!< 0x00400000 */
#define PA_IORP_PA_IORP11_1                       (0x2UL<<PA_IORP_PA_IORP11_Pos)                    /*!< 0x00800000 */
#define PA_IORP_PA_IORP12_Pos                     (24U)
#define PA_IORP_PA_IORP12_Msk                     (0x3UL<<PA_IORP_PA_IORP12_Pos)                    /*!< 0x03000000 */
#define PA_IORP_PA_IORP12                         PA_IORP_PA_IORP12_Msk
#define PA_IORP_PA_IORP12_0                       (0x1UL<<PA_IORP_PA_IORP12_Pos)                    /*!< 0x01000000 */
#define PA_IORP_PA_IORP12_1                       (0x2UL<<PA_IORP_PA_IORP12_Pos)                    /*!< 0x02000000 */
#define PA_IORP_PA_IORP13_Pos                     (26U)
#define PA_IORP_PA_IORP13_Msk                     (0x3UL<<PA_IORP_PA_IORP13_Pos)                    /*!< 0x0C000000 */
#define PA_IORP_PA_IORP13                         PA_IORP_PA_IORP13_Msk
#define PA_IORP_PA_IORP13_0                       (0x1UL<<PA_IORP_PA_IORP13_Pos)                    /*!< 0x04000000 */
#define PA_IORP_PA_IORP13_1                       (0x2UL<<PA_IORP_PA_IORP13_Pos)                    /*!< 0x08000000 */
#define PA_IORP_PA_IORP14_Pos                     (28U)
#define PA_IORP_PA_IORP14_Msk                     (0x3UL<<PA_IORP_PA_IORP14_Pos)                    /*!< 0x30000000 */
#define PA_IORP_PA_IORP14                         PA_IORP_PA_IORP14_Msk
#define PA_IORP_PA_IORP14_0                       (0x1UL<<PA_IORP_PA_IORP14_Pos)                    /*!< 0x10000000 */
#define PA_IORP_PA_IORP14_1                       (0x2UL<<PA_IORP_PA_IORP14_Pos)                    /*!< 0x20000000 */
#define PA_IORP_PA_IORP15_Pos                     (30U)
#define PA_IORP_PA_IORP15_Msk                     (0x3UL<<PA_IORP_PA_IORP15_Pos)                    /*!< 0xC0000000 */
#define PA_IORP_PA_IORP15                         PA_IORP_PA_IORP15_Msk
#define PA_IORP_PA_IORP15_0                       (0x1UL<<PA_IORP_PA_IORP15_Pos)                    /*!< 0x40000000 */
#define PA_IORP_PA_IORP15_1                       (0x2UL<<PA_IORP_PA_IORP15_Pos)                    /*!< 0x80000000 */

/********************************* Bit definition for PB_IORP register **********************************************/
#define PB_IORP_PB_IORP0_Pos                      (0U)
#define PB_IORP_PB_IORP0_Msk                      (0x3UL<<PB_IORP_PB_IORP0_Pos)                     /*!< 0x00000003 */
#define PB_IORP_PB_IORP0                          PB_IORP_PB_IORP0_Msk
#define PB_IORP_PB_IORP0_0                        (0x1UL<<PB_IORP_PB_IORP0_Pos)                     /*!< 0x00000001 */
#define PB_IORP_PB_IORP0_1                        (0x2UL<<PB_IORP_PB_IORP0_Pos)                     /*!< 0x00000002 */
#define PB_IORP_PB_IORP1_Pos                      (2U)
#define PB_IORP_PB_IORP1_Msk                      (0x3UL<<PB_IORP_PB_IORP1_Pos)                     /*!< 0x0000000C */
#define PB_IORP_PB_IORP1                          PB_IORP_PB_IORP1_Msk
#define PB_IORP_PB_IORP1_0                        (0x1UL<<PB_IORP_PB_IORP1_Pos)                     /*!< 0x00000004 */
#define PB_IORP_PB_IORP1_1                        (0x2UL<<PB_IORP_PB_IORP1_Pos)                     /*!< 0x00000008 */
#define PB_IORP_PB_IORP2_Pos                      (4U)
#define PB_IORP_PB_IORP2_Msk                      (0x3UL<<PB_IORP_PB_IORP2_Pos)                     /*!< 0x00000030 */
#define PB_IORP_PB_IORP2                          PB_IORP_PB_IORP2_Msk
#define PB_IORP_PB_IORP2_0                        (0x1UL<<PB_IORP_PB_IORP2_Pos)                     /*!< 0x00000010 */
#define PB_IORP_PB_IORP2_1                        (0x2UL<<PB_IORP_PB_IORP2_Pos)                     /*!< 0x00000020 */
#define PB_IORP_PB_IORP3_Pos                      (6U)
#define PB_IORP_PB_IORP3_Msk                      (0x3UL<<PB_IORP_PB_IORP3_Pos)                     /*!< 0x000000C0 */
#define PB_IORP_PB_IORP3                          PB_IORP_PB_IORP3_Msk
#define PB_IORP_PB_IORP3_0                        (0x1UL<<PB_IORP_PB_IORP3_Pos)                     /*!< 0x00000040 */
#define PB_IORP_PB_IORP3_1                        (0x2UL<<PB_IORP_PB_IORP3_Pos)                     /*!< 0x00000080 */
#define PB_IORP_PB_IORP4_Pos                      (8U)
#define PB_IORP_PB_IORP4_Msk                      (0x3UL<<PB_IORP_PB_IORP4_Pos)                     /*!< 0x00000300 */
#define PB_IORP_PB_IORP4                          PB_IORP_PB_IORP4_Msk
#define PB_IORP_PB_IORP4_0                        (0x1UL<<PB_IORP_PB_IORP4_Pos)                     /*!< 0x00000100 */
#define PB_IORP_PB_IORP4_1                        (0x2UL<<PB_IORP_PB_IORP4_Pos)                     /*!< 0x00000200 */
#define PB_IORP_PB_IORP5_Pos                      (10U)
#define PB_IORP_PB_IORP5_Msk                      (0x3UL<<PB_IORP_PB_IORP5_Pos)                     /*!< 0x00000C00 */
#define PB_IORP_PB_IORP5                          PB_IORP_PB_IORP5_Msk
#define PB_IORP_PB_IORP5_0                        (0x1UL<<PB_IORP_PB_IORP5_Pos)                     /*!< 0x00000400 */
#define PB_IORP_PB_IORP5_1                        (0x2UL<<PB_IORP_PB_IORP5_Pos)                     /*!< 0x00000800 */
#define PB_IORP_PB_IORP6_Pos                      (12U)
#define PB_IORP_PB_IORP6_Msk                      (0x3UL<<PB_IORP_PB_IORP6_Pos)                     /*!< 0x00003000 */
#define PB_IORP_PB_IORP6                          PB_IORP_PB_IORP6_Msk
#define PB_IORP_PB_IORP6_0                        (0x1UL<<PB_IORP_PB_IORP6_Pos)                     /*!< 0x00001000 */
#define PB_IORP_PB_IORP6_1                        (0x2UL<<PB_IORP_PB_IORP6_Pos)                     /*!< 0x00002000 */
#define PB_IORP_PB_IORP7_Pos                      (14U)
#define PB_IORP_PB_IORP7_Msk                      (0x3UL<<PB_IORP_PB_IORP7_Pos)                     /*!< 0x0000C000 */
#define PB_IORP_PB_IORP7                          PB_IORP_PB_IORP7_Msk
#define PB_IORP_PB_IORP7_0                        (0x1UL<<PB_IORP_PB_IORP7_Pos)                     /*!< 0x00004000 */
#define PB_IORP_PB_IORP7_1                        (0x2UL<<PB_IORP_PB_IORP7_Pos)                     /*!< 0x00008000 */
#define PB_IORP_PB_IORP8_Pos                      (16U)
#define PB_IORP_PB_IORP8_Msk                      (0x3UL<<PB_IORP_PB_IORP8_Pos)                     /*!< 0x00030000 */
#define PB_IORP_PB_IORP8                          PB_IORP_PB_IORP8_Msk
#define PB_IORP_PB_IORP8_0                        (0x1UL<<PB_IORP_PB_IORP8_Pos)                     /*!< 0x00010000 */
#define PB_IORP_PB_IORP8_1                        (0x2UL<<PB_IORP_PB_IORP8_Pos)                     /*!< 0x00020000 */
#define PB_IORP_PB_IORP9_Pos                      (18U)
#define PB_IORP_PB_IORP9_Msk                      (0x3UL<<PB_IORP_PB_IORP9_Pos)                     /*!< 0x000C0000 */
#define PB_IORP_PB_IORP9                          PB_IORP_PB_IORP9_Msk
#define PB_IORP_PB_IORP9_0                        (0x1UL<<PB_IORP_PB_IORP9_Pos)                     /*!< 0x00040000 */
#define PB_IORP_PB_IORP9_1                        (0x2UL<<PB_IORP_PB_IORP9_Pos)                     /*!< 0x00080000 */
#define PB_IORP_PB_IORP10_Pos                     (20U)
#define PB_IORP_PB_IORP10_Msk                     (0x3UL<<PB_IORP_PB_IORP10_Pos)                    /*!< 0x00300000 */
#define PB_IORP_PB_IORP10                         PB_IORP_PB_IORP10_Msk
#define PB_IORP_PB_IORP10_0                       (0x1UL<<PB_IORP_PB_IORP10_Pos)                    /*!< 0x00100000 */
#define PB_IORP_PB_IORP10_1                       (0x2UL<<PB_IORP_PB_IORP10_Pos)                    /*!< 0x00200000 */
#define PB_IORP_PB_IORP11_Pos                     (22U)
#define PB_IORP_PB_IORP11_Msk                     (0x3UL<<PB_IORP_PB_IORP11_Pos)                    /*!< 0x00C00000 */
#define PB_IORP_PB_IORP11                         PB_IORP_PB_IORP11_Msk
#define PB_IORP_PB_IORP11_0                       (0x1UL<<PB_IORP_PB_IORP11_Pos)                    /*!< 0x00400000 */
#define PB_IORP_PB_IORP11_1                       (0x2UL<<PB_IORP_PB_IORP11_Pos)                    /*!< 0x00800000 */
#define PB_IORP_PB_IORP12_Pos                     (24U)
#define PB_IORP_PB_IORP12_Msk                     (0x3UL<<PB_IORP_PB_IORP12_Pos)                    /*!< 0x03000000 */
#define PB_IORP_PB_IORP12                         PB_IORP_PB_IORP12_Msk
#define PB_IORP_PB_IORP12_0                       (0x1UL<<PB_IORP_PB_IORP12_Pos)                    /*!< 0x01000000 */
#define PB_IORP_PB_IORP12_1                       (0x2UL<<PB_IORP_PB_IORP12_Pos)                    /*!< 0x02000000 */
#define PB_IORP_PB_IORP13_Pos                     (26U)
#define PB_IORP_PB_IORP13_Msk                     (0x3UL<<PB_IORP_PB_IORP13_Pos)                    /*!< 0x0C000000 */
#define PB_IORP_PB_IORP13                         PB_IORP_PB_IORP13_Msk
#define PB_IORP_PB_IORP13_0                       (0x1UL<<PB_IORP_PB_IORP13_Pos)                    /*!< 0x04000000 */
#define PB_IORP_PB_IORP13_1                       (0x2UL<<PB_IORP_PB_IORP13_Pos)                    /*!< 0x08000000 */
#define PB_IORP_PB_IORP14_Pos                     (28U)
#define PB_IORP_PB_IORP14_Msk                     (0x3UL<<PB_IORP_PB_IORP14_Pos)                    /*!< 0x30000000 */
#define PB_IORP_PB_IORP14                         PB_IORP_PB_IORP14_Msk
#define PB_IORP_PB_IORP14_0                       (0x1UL<<PB_IORP_PB_IORP14_Pos)                    /*!< 0x10000000 */
#define PB_IORP_PB_IORP14_1                       (0x2UL<<PB_IORP_PB_IORP14_Pos)                    /*!< 0x20000000 */
#define PB_IORP_PB_IORP15_Pos                     (30U)
#define PB_IORP_PB_IORP15_Msk                     (0x3UL<<PB_IORP_PB_IORP15_Pos)                    /*!< 0xC0000000 */
#define PB_IORP_PB_IORP15                         PB_IORP_PB_IORP15_Msk
#define PB_IORP_PB_IORP15_0                       (0x1UL<<PB_IORP_PB_IORP15_Pos)                    /*!< 0x40000000 */
#define PB_IORP_PB_IORP15_1                       (0x2UL<<PB_IORP_PB_IORP15_Pos)                    /*!< 0x80000000 */

/********************************* Bit definition for PC_IORP register **********************************************/
#define PC_IORP_PC_IORP0_Pos                      (0U)
#define PC_IORP_PC_IORP0_Msk                      (0x3UL<<PC_IORP_PC_IORP0_Pos)                     /*!< 0x00000003 */
#define PC_IORP_PC_IORP0                          PC_IORP_PC_IORP0_Msk
#define PC_IORP_PC_IORP0_0                        (0x1UL<<PC_IORP_PC_IORP0_Pos)                     /*!< 0x00000001 */
#define PC_IORP_PC_IORP0_1                        (0x2UL<<PC_IORP_PC_IORP0_Pos)                     /*!< 0x00000002 */
#define PC_IORP_PC_IORP1_Pos                      (2U)
#define PC_IORP_PC_IORP1_Msk                      (0x3UL<<PC_IORP_PC_IORP1_Pos)                     /*!< 0x0000000C */
#define PC_IORP_PC_IORP1                          PC_IORP_PC_IORP1_Msk
#define PC_IORP_PC_IORP1_0                        (0x1UL<<PC_IORP_PC_IORP1_Pos)                     /*!< 0x00000004 */
#define PC_IORP_PC_IORP1_1                        (0x2UL<<PC_IORP_PC_IORP1_Pos)                     /*!< 0x00000008 */
#define PC_IORP_PC_IORP2_Pos                      (4U)
#define PC_IORP_PC_IORP2_Msk                      (0x3UL<<PC_IORP_PC_IORP2_Pos)                     /*!< 0x00000030 */
#define PC_IORP_PC_IORP2                          PC_IORP_PC_IORP2_Msk
#define PC_IORP_PC_IORP2_0                        (0x1UL<<PC_IORP_PC_IORP2_Pos)                     /*!< 0x00000010 */
#define PC_IORP_PC_IORP2_1                        (0x2UL<<PC_IORP_PC_IORP2_Pos)                     /*!< 0x00000020 */
#define PC_IORP_PC_IORP3_Pos                      (6U)
#define PC_IORP_PC_IORP3_Msk                      (0x3UL<<PC_IORP_PC_IORP3_Pos)                     /*!< 0x000000C0 */
#define PC_IORP_PC_IORP3                          PC_IORP_PC_IORP3_Msk
#define PC_IORP_PC_IORP3_0                        (0x1UL<<PC_IORP_PC_IORP3_Pos)                     /*!< 0x00000040 */
#define PC_IORP_PC_IORP3_1                        (0x2UL<<PC_IORP_PC_IORP3_Pos)                     /*!< 0x00000080 */
#define PC_IORP_PC_IORP4_Pos                      (8U)
#define PC_IORP_PC_IORP4_Msk                      (0x3UL<<PC_IORP_PC_IORP4_Pos)                     /*!< 0x00000300 */
#define PC_IORP_PC_IORP4                          PC_IORP_PC_IORP4_Msk
#define PC_IORP_PC_IORP4_0                        (0x1UL<<PC_IORP_PC_IORP4_Pos)                     /*!< 0x00000100 */
#define PC_IORP_PC_IORP4_1                        (0x2UL<<PC_IORP_PC_IORP4_Pos)                     /*!< 0x00000200 */
#define PC_IORP_PC_IORP5_Pos                      (10U)
#define PC_IORP_PC_IORP5_Msk                      (0x3UL<<PC_IORP_PC_IORP5_Pos)                     /*!< 0x00000C00 */
#define PC_IORP_PC_IORP5                          PC_IORP_PC_IORP5_Msk
#define PC_IORP_PC_IORP5_0                        (0x1UL<<PC_IORP_PC_IORP5_Pos)                     /*!< 0x00000400 */
#define PC_IORP_PC_IORP5_1                        (0x2UL<<PC_IORP_PC_IORP5_Pos)                     /*!< 0x00000800 */
#define PC_IORP_PC_IORP6_Pos                      (12U)
#define PC_IORP_PC_IORP6_Msk                      (0x3UL<<PC_IORP_PC_IORP6_Pos)                     /*!< 0x00003000 */
#define PC_IORP_PC_IORP6                          PC_IORP_PC_IORP6_Msk
#define PC_IORP_PC_IORP6_0                        (0x1UL<<PC_IORP_PC_IORP6_Pos)                     /*!< 0x00001000 */
#define PC_IORP_PC_IORP6_1                        (0x2UL<<PC_IORP_PC_IORP6_Pos)                     /*!< 0x00002000 */
#define PC_IORP_PC_IORP7_Pos                      (14U)
#define PC_IORP_PC_IORP7_Msk                      (0x3UL<<PC_IORP_PC_IORP7_Pos)                     /*!< 0x0000C000 */
#define PC_IORP_PC_IORP7                          PC_IORP_PC_IORP7_Msk
#define PC_IORP_PC_IORP7_0                        (0x1UL<<PC_IORP_PC_IORP7_Pos)                     /*!< 0x00004000 */
#define PC_IORP_PC_IORP7_1                        (0x2UL<<PC_IORP_PC_IORP7_Pos)                     /*!< 0x00008000 */
#define PC_IORP_PC_IORP8_Pos                      (16U)
#define PC_IORP_PC_IORP8_Msk                      (0x3UL<<PC_IORP_PC_IORP8_Pos)                     /*!< 0x00030000 */
#define PC_IORP_PC_IORP8                          PC_IORP_PC_IORP8_Msk
#define PC_IORP_PC_IORP8_0                        (0x1UL<<PC_IORP_PC_IORP8_Pos)                     /*!< 0x00010000 */
#define PC_IORP_PC_IORP8_1                        (0x2UL<<PC_IORP_PC_IORP8_Pos)                     /*!< 0x00020000 */
#define PC_IORP_PC_IORP9_Pos                      (18U)
#define PC_IORP_PC_IORP9_Msk                      (0x3UL<<PC_IORP_PC_IORP9_Pos)                     /*!< 0x000C0000 */
#define PC_IORP_PC_IORP9                          PC_IORP_PC_IORP9_Msk
#define PC_IORP_PC_IORP9_0                        (0x1UL<<PC_IORP_PC_IORP9_Pos)                     /*!< 0x00040000 */
#define PC_IORP_PC_IORP9_1                        (0x2UL<<PC_IORP_PC_IORP9_Pos)                     /*!< 0x00080000 */
#define PC_IORP_PC_IORP10_Pos                     (20U)
#define PC_IORP_PC_IORP10_Msk                     (0x3UL<<PC_IORP_PC_IORP10_Pos)                    /*!< 0x00300000 */
#define PC_IORP_PC_IORP10                         PC_IORP_PC_IORP10_Msk
#define PC_IORP_PC_IORP10_0                       (0x1UL<<PC_IORP_PC_IORP10_Pos)                    /*!< 0x00100000 */
#define PC_IORP_PC_IORP10_1                       (0x2UL<<PC_IORP_PC_IORP10_Pos)                    /*!< 0x00200000 */
#define PC_IORP_PC_IORP11_Pos                     (22U)
#define PC_IORP_PC_IORP11_Msk                     (0x3UL<<PC_IORP_PC_IORP11_Pos)                    /*!< 0x00C00000 */
#define PC_IORP_PC_IORP11                         PC_IORP_PC_IORP11_Msk
#define PC_IORP_PC_IORP11_0                       (0x1UL<<PC_IORP_PC_IORP11_Pos)                    /*!< 0x00400000 */
#define PC_IORP_PC_IORP11_1                       (0x2UL<<PC_IORP_PC_IORP11_Pos)                    /*!< 0x00800000 */
#define PC_IORP_PC_IORP12_Pos                     (24U)
#define PC_IORP_PC_IORP12_Msk                     (0x3UL<<PC_IORP_PC_IORP12_Pos)                    /*!< 0x03000000 */
#define PC_IORP_PC_IORP12                         PC_IORP_PC_IORP12_Msk
#define PC_IORP_PC_IORP12_0                       (0x1UL<<PC_IORP_PC_IORP12_Pos)                    /*!< 0x01000000 */
#define PC_IORP_PC_IORP12_1                       (0x2UL<<PC_IORP_PC_IORP12_Pos)                    /*!< 0x02000000 */
#define PC_IORP_PC_IORP13_Pos                     (26U)
#define PC_IORP_PC_IORP13_Msk                     (0x3UL<<PC_IORP_PC_IORP13_Pos)                    /*!< 0x0C000000 */
#define PC_IORP_PC_IORP13                         PC_IORP_PC_IORP13_Msk
#define PC_IORP_PC_IORP13_0                       (0x1UL<<PC_IORP_PC_IORP13_Pos)                    /*!< 0x04000000 */
#define PC_IORP_PC_IORP13_1                       (0x2UL<<PC_IORP_PC_IORP13_Pos)                    /*!< 0x08000000 */
#define PC_IORP_PC_IORP14_Pos                     (28U)
#define PC_IORP_PC_IORP14_Msk                     (0x3UL<<PC_IORP_PC_IORP14_Pos)                    /*!< 0x30000000 */
#define PC_IORP_PC_IORP14                         PC_IORP_PC_IORP14_Msk
#define PC_IORP_PC_IORP14_0                       (0x1UL<<PC_IORP_PC_IORP14_Pos)                    /*!< 0x10000000 */
#define PC_IORP_PC_IORP14_1                       (0x2UL<<PC_IORP_PC_IORP14_Pos)                    /*!< 0x20000000 */
#define PC_IORP_PC_IORP15_Pos                     (30U)
#define PC_IORP_PC_IORP15_Msk                     (0x3UL<<PC_IORP_PC_IORP15_Pos)                    /*!< 0xC0000000 */
#define PC_IORP_PC_IORP15                         PC_IORP_PC_IORP15_Msk
#define PC_IORP_PC_IORP15_0                       (0x1UL<<PC_IORP_PC_IORP15_Pos)                    /*!< 0x40000000 */
#define PC_IORP_PC_IORP15_1                       (0x2UL<<PC_IORP_PC_IORP15_Pos)                    /*!< 0x80000000 */

/********************************* Bit definition for PD_IORP register **********************************************/
#define PD_IORP_PD_IORP0_Pos                      (0U)
#define PD_IORP_PD_IORP0_Msk                      (0x3UL<<PD_IORP_PD_IORP0_Pos)                     /*!< 0x00000003 */
#define PD_IORP_PD_IORP0                          PD_IORP_PD_IORP0_Msk
#define PD_IORP_PD_IORP0_0                        (0x1UL<<PD_IORP_PD_IORP0_Pos)                     /*!< 0x00000001 */
#define PD_IORP_PD_IORP0_1                        (0x2UL<<PD_IORP_PD_IORP0_Pos)                     /*!< 0x00000002 */
#define PD_IORP_PD_IORP1_Pos                      (2U)
#define PD_IORP_PD_IORP1_Msk                      (0x3UL<<PD_IORP_PD_IORP1_Pos)                     /*!< 0x0000000C */
#define PD_IORP_PD_IORP1                          PD_IORP_PD_IORP1_Msk
#define PD_IORP_PD_IORP1_0                        (0x1UL<<PD_IORP_PD_IORP1_Pos)                     /*!< 0x00000004 */
#define PD_IORP_PD_IORP1_1                        (0x2UL<<PD_IORP_PD_IORP1_Pos)                     /*!< 0x00000008 */
#define PD_IORP_PD_IORP2_Pos                      (4U)
#define PD_IORP_PD_IORP2_Msk                      (0x3UL<<PD_IORP_PD_IORP2_Pos)                     /*!< 0x00000030 */
#define PD_IORP_PD_IORP2                          PD_IORP_PD_IORP2_Msk
#define PD_IORP_PD_IORP2_0                        (0x1UL<<PD_IORP_PD_IORP2_Pos)                     /*!< 0x00000010 */
#define PD_IORP_PD_IORP2_1                        (0x2UL<<PD_IORP_PD_IORP2_Pos)                     /*!< 0x00000020 */
#define PD_IORP_PD_IORP3_Pos                      (6U)
#define PD_IORP_PD_IORP3_Msk                      (0x3UL<<PD_IORP_PD_IORP3_Pos)                     /*!< 0x000000C0 */
#define PD_IORP_PD_IORP3                          PD_IORP_PD_IORP3_Msk
#define PD_IORP_PD_IORP3_0                        (0x1UL<<PD_IORP_PD_IORP3_Pos)                     /*!< 0x00000040 */
#define PD_IORP_PD_IORP3_1                        (0x2UL<<PD_IORP_PD_IORP3_Pos)                     /*!< 0x00000080 */
#define PD_IORP_PD_IORP4_Pos                      (8U)
#define PD_IORP_PD_IORP4_Msk                      (0x3UL<<PD_IORP_PD_IORP4_Pos)                     /*!< 0x00000300 */
#define PD_IORP_PD_IORP4                          PD_IORP_PD_IORP4_Msk
#define PD_IORP_PD_IORP4_0                        (0x1UL<<PD_IORP_PD_IORP4_Pos)                     /*!< 0x00000100 */
#define PD_IORP_PD_IORP4_1                        (0x2UL<<PD_IORP_PD_IORP4_Pos)                     /*!< 0x00000200 */
#define PD_IORP_PD_IORP5_Pos                      (10U)
#define PD_IORP_PD_IORP5_Msk                      (0x3UL<<PD_IORP_PD_IORP5_Pos)                     /*!< 0x00000C00 */
#define PD_IORP_PD_IORP5                          PD_IORP_PD_IORP5_Msk
#define PD_IORP_PD_IORP5_0                        (0x1UL<<PD_IORP_PD_IORP5_Pos)                     /*!< 0x00000400 */
#define PD_IORP_PD_IORP5_1                        (0x2UL<<PD_IORP_PD_IORP5_Pos)                     /*!< 0x00000800 */
#define PD_IORP_PD_IORP6_Pos                      (12U)
#define PD_IORP_PD_IORP6_Msk                      (0x3UL<<PD_IORP_PD_IORP6_Pos)                     /*!< 0x00003000 */
#define PD_IORP_PD_IORP6                          PD_IORP_PD_IORP6_Msk
#define PD_IORP_PD_IORP6_0                        (0x1UL<<PD_IORP_PD_IORP6_Pos)                     /*!< 0x00001000 */
#define PD_IORP_PD_IORP6_1                        (0x2UL<<PD_IORP_PD_IORP6_Pos)                     /*!< 0x00002000 */
#define PD_IORP_PD_IORP7_Pos                      (14U)
#define PD_IORP_PD_IORP7_Msk                      (0x3UL<<PD_IORP_PD_IORP7_Pos)                     /*!< 0x0000C000 */
#define PD_IORP_PD_IORP7                          PD_IORP_PD_IORP7_Msk
#define PD_IORP_PD_IORP7_0                        (0x1UL<<PD_IORP_PD_IORP7_Pos)                     /*!< 0x00004000 */
#define PD_IORP_PD_IORP7_1                        (0x2UL<<PD_IORP_PD_IORP7_Pos)                     /*!< 0x00008000 */
#define PD_IORP_PD_IORP8_Pos                      (16U)
#define PD_IORP_PD_IORP8_Msk                      (0x3UL<<PD_IORP_PD_IORP8_Pos)                     /*!< 0x00030000 */
#define PD_IORP_PD_IORP8                          PD_IORP_PD_IORP8_Msk
#define PD_IORP_PD_IORP8_0                        (0x1UL<<PD_IORP_PD_IORP8_Pos)                     /*!< 0x00010000 */
#define PD_IORP_PD_IORP8_1                        (0x2UL<<PD_IORP_PD_IORP8_Pos)                     /*!< 0x00020000 */
#define PD_IORP_PD_IORP9_Pos                      (18U)
#define PD_IORP_PD_IORP9_Msk                      (0x3UL<<PD_IORP_PD_IORP9_Pos)                     /*!< 0x000C0000 */
#define PD_IORP_PD_IORP9                          PD_IORP_PD_IORP9_Msk
#define PD_IORP_PD_IORP9_0                        (0x1UL<<PD_IORP_PD_IORP9_Pos)                     /*!< 0x00040000 */
#define PD_IORP_PD_IORP9_1                        (0x2UL<<PD_IORP_PD_IORP9_Pos)                     /*!< 0x00080000 */
#define PD_IORP_PD_IORP10_Pos                     (20U)
#define PD_IORP_PD_IORP10_Msk                     (0x3UL<<PD_IORP_PD_IORP10_Pos)                    /*!< 0x00300000 */
#define PD_IORP_PD_IORP10                         PD_IORP_PD_IORP10_Msk
#define PD_IORP_PD_IORP10_0                       (0x1UL<<PD_IORP_PD_IORP10_Pos)                    /*!< 0x00100000 */
#define PD_IORP_PD_IORP10_1                       (0x2UL<<PD_IORP_PD_IORP10_Pos)                    /*!< 0x00200000 */
#define PD_IORP_PD_IORP11_Pos                     (22U)
#define PD_IORP_PD_IORP11_Msk                     (0x3UL<<PD_IORP_PD_IORP11_Pos)                    /*!< 0x00C00000 */
#define PD_IORP_PD_IORP11                         PD_IORP_PD_IORP11_Msk
#define PD_IORP_PD_IORP11_0                       (0x1UL<<PD_IORP_PD_IORP11_Pos)                    /*!< 0x00400000 */
#define PD_IORP_PD_IORP11_1                       (0x2UL<<PD_IORP_PD_IORP11_Pos)                    /*!< 0x00800000 */
#define PD_IORP_PD_IORP12_Pos                     (24U)
#define PD_IORP_PD_IORP12_Msk                     (0x3UL<<PD_IORP_PD_IORP12_Pos)                    /*!< 0x03000000 */
#define PD_IORP_PD_IORP12                         PD_IORP_PD_IORP12_Msk
#define PD_IORP_PD_IORP12_0                       (0x1UL<<PD_IORP_PD_IORP12_Pos)                    /*!< 0x01000000 */
#define PD_IORP_PD_IORP12_1                       (0x2UL<<PD_IORP_PD_IORP12_Pos)                    /*!< 0x02000000 */
#define PD_IORP_PD_IORP13_Pos                     (26U)
#define PD_IORP_PD_IORP13_Msk                     (0x3UL<<PD_IORP_PD_IORP13_Pos)                    /*!< 0x0C000000 */
#define PD_IORP_PD_IORP13                         PD_IORP_PD_IORP13_Msk
#define PD_IORP_PD_IORP13_0                       (0x1UL<<PD_IORP_PD_IORP13_Pos)                    /*!< 0x04000000 */
#define PD_IORP_PD_IORP13_1                       (0x2UL<<PD_IORP_PD_IORP13_Pos)                    /*!< 0x08000000 */

/********************************* Bit definition for SYSCFG_SCSR register ******************************************/
#define SYSCFG_SCSR_PERR_RSTEN_Pos                (0U)
#define SYSCFG_SCSR_PERR_RSTEN_Msk                (0x1UL<<SYSCFG_SCSR_PERR_RSTEN_Pos)               /*!< 0x00000001 */
#define SYSCFG_SCSR_PERR_RSTEN                    SYSCFG_SCSR_PERR_RSTEN_Msk                        
#define SYSCFG_SCSR_SPF_Pos                       (8U)
#define SYSCFG_SCSR_SPF_Msk                       (0x1UL<<SYSCFG_SCSR_SPF_Pos)                      /*!< 0x00000100 */
#define SYSCFG_SCSR_SPF                           SYSCFG_SCSR_SPF_Msk                               

/********************************* Bit definition for SYSCFG_SWPR register ******************************************/
#define SYSCFG_SWPR_SYSCFG_SWPR_Pos               (0U)
#define SYSCFG_SWPR_SYSCFG_SWPR_Msk               (0xFFFFFFFFUL<<SYSCFG_SWPR_SYSCFG_SWPR_Pos)       /*!< 0xFFFFFFFF */
#define SYSCFG_SWPR_SYSCFG_SWPR                   SYSCFG_SWPR_SYSCFG_SWPR_Msk

/********************************************************************************************************************/
/********************************* TIM1 *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for TIM_CR1 register *********************************************/
#define TIM_CR1_CEN_Pos                          (0U)
#define TIM_CR1_CEN_Msk                          (0x1UL<<TIM_CR1_CEN_Pos)                         /*!< 0x00000001 */
#define TIM_CR1_CEN                              TIM_CR1_CEN_Msk                                  
#define TIM_CR1_UDIS_Pos                         (1U)
#define TIM_CR1_UDIS_Msk                         (0x1UL<<TIM_CR1_UDIS_Pos)                        /*!< 0x00000002 */
#define TIM_CR1_UDIS                             TIM_CR1_UDIS_Msk                                 
#define TIM_CR1_URS_Pos                          (2U)
#define TIM_CR1_URS_Msk                          (0x1UL<<TIM_CR1_URS_Pos)                         /*!< 0x00000004 */
#define TIM_CR1_URS                              TIM_CR1_URS_Msk                                  
#define TIM_CR1_OPM_Pos                          (3U)
#define TIM_CR1_OPM_Msk                          (0x1UL<<TIM_CR1_OPM_Pos)                         /*!< 0x00000008 */
#define TIM_CR1_OPM                              TIM_CR1_OPM_Msk                                  
#define TIM_CR1_DIR_Pos                          (4U)
#define TIM_CR1_DIR_Msk                          (0x1UL<<TIM_CR1_DIR_Pos)                         /*!< 0x00000010 */
#define TIM_CR1_DIR                              TIM_CR1_DIR_Msk                                  
#define TIM_CR1_CMS_Pos                          (5U)
#define TIM_CR1_CMS_Msk                          (0x3UL<<TIM_CR1_CMS_Pos)                         /*!< 0x00000060 */
#define TIM_CR1_CMS                              TIM_CR1_CMS_Msk
#define TIM_CR1_CMS_0                            (0x1UL<<TIM_CR1_CMS_Pos)                         /*!< 0x00000020 */
#define TIM_CR1_CMS_1                            (0x2UL<<TIM_CR1_CMS_Pos)                         /*!< 0x00000040 */
#define TIM_CR1_ARPE_Pos                         (7U)
#define TIM_CR1_ARPE_Msk                         (0x1UL<<TIM_CR1_ARPE_Pos)                        /*!< 0x00000080 */
#define TIM_CR1_ARPE                             TIM_CR1_ARPE_Msk                                 
#define TIM_CR1_CKD_Pos                          (8U)
#define TIM_CR1_CKD_Msk                          (0x3UL<<TIM_CR1_CKD_Pos)                         /*!< 0x00000300 */
#define TIM_CR1_CKD                              TIM_CR1_CKD_Msk
#define TIM_CR1_CKD_0                            (0x1UL<<TIM_CR1_CKD_Pos)                         /*!< 0x00000100 */
#define TIM_CR1_CKD_1                            (0x2UL<<TIM_CR1_CKD_Pos)                         /*!< 0x00000200 */

/********************************* Bit definition for TIM_CR2 register *********************************************/
#define TIM_CR2_CCPC_Pos                         (0U)
#define TIM_CR2_CCPC_Msk                         (0x1UL<<TIM_CR2_CCPC_Pos)                        /*!< 0x00000001 */
#define TIM_CR2_CCPC                             TIM_CR2_CCPC_Msk                                 
#define TIM_CR2_CCUS_Pos                         (2U)
#define TIM_CR2_CCUS_Msk                         (0x1UL<<TIM_CR2_CCUS_Pos)                        /*!< 0x00000004 */
#define TIM_CR2_CCUS                             TIM_CR2_CCUS_Msk                                 
#define TIM_CR2_CCDS_Pos                         (3U)
#define TIM_CR2_CCDS_Msk                         (0x1UL<<TIM_CR2_CCDS_Pos)                        /*!< 0x00000008 */
#define TIM_CR2_CCDS                             TIM_CR2_CCDS_Msk                                 
#define TIM_CR2_MMS_Pos                          (4U)
#define TIM_CR2_MMS_Msk                          (0x200007UL<<TIM_CR2_MMS_Pos)                    /*!< 0x02000070 */
#define TIM_CR2_MMS                              TIM_CR2_MMS_Msk
#define TIM_CR2_MMS_0                            (0x1UL<<TIM_CR2_MMS_Pos)                         /*!< 0x00000010 */
#define TIM_CR2_MMS_1                            (0x2UL<<TIM_CR2_MMS_Pos)                         /*!< 0x00000020 */
#define TIM_CR2_MMS_2                            (0x4UL<<TIM_CR2_MMS_Pos)                         /*!< 0x00000040 */
#define TIM_CR2_MMS_3                            (0x200000UL<<TIM_CR2_MMS_Pos)                    /*!< 0x02000000 */
#define TIM_CR2_TI1S_Pos                         (7U)
#define TIM_CR2_TI1S_Msk                         (0x1UL<<TIM_CR2_TI1S_Pos)                        /*!< 0x00000080 */
#define TIM_CR2_TI1S                             TIM_CR2_TI1S_Msk                                 
#define TIM_CR2_OIS1_Pos                         (8U)
#define TIM_CR2_OIS1_Msk                         (0x1UL<<TIM_CR2_OIS1_Pos)                        /*!< 0x00000100 */
#define TIM_CR2_OIS1                             TIM_CR2_OIS1_Msk                                 
#define TIM_CR2_OIS1N_Pos                        (9U)
#define TIM_CR2_OIS1N_Msk                        (0x1UL<<TIM_CR2_OIS1N_Pos)                       /*!< 0x00000200 */
#define TIM_CR2_OIS1N                            TIM_CR2_OIS1N_Msk                                
#define TIM_CR2_OIS2_Pos                         (10U)
#define TIM_CR2_OIS2_Msk                         (0x1UL<<TIM_CR2_OIS2_Pos)                        /*!< 0x00000400 */
#define TIM_CR2_OIS2                             TIM_CR2_OIS2_Msk                                 
#define TIM_CR2_OIS2N_Pos                        (11U)
#define TIM_CR2_OIS2N_Msk                        (0x1UL<<TIM_CR2_OIS2N_Pos)                       /*!< 0x00000800 */
#define TIM_CR2_OIS2N                            TIM_CR2_OIS2N_Msk                                
#define TIM_CR2_OIS3_Pos                         (12U)
#define TIM_CR2_OIS3_Msk                         (0x1UL<<TIM_CR2_OIS3_Pos)                        /*!< 0x00001000 */
#define TIM_CR2_OIS3                             TIM_CR2_OIS3_Msk                                 
#define TIM_CR2_OIS3N_Pos                        (13U)
#define TIM_CR2_OIS3N_Msk                        (0x1UL<<TIM_CR2_OIS3N_Pos)                       /*!< 0x00002000 */
#define TIM_CR2_OIS3N                            TIM_CR2_OIS3N_Msk                                
#define TIM_CR2_OIS4_Pos                         (14U)
#define TIM_CR2_OIS4_Msk                         (0x1UL<<TIM_CR2_OIS4_Pos)                        /*!< 0x00004000 */
#define TIM_CR2_OIS4                             TIM_CR2_OIS4_Msk                                 

/********************************* Bit definition for TIM_SMCR register ********************************************/
#define TIM_SMCR_SMS_Pos                         (0U)
#define TIM_SMCR_SMS_Msk                         (0x10007UL<<TIM_SMCR_SMS_Pos)                    /*!< 0x00010007 */
#define TIM_SMCR_SMS                             TIM_SMCR_SMS_Msk
#define TIM_SMCR_SMS_0                           (0x1UL<<TIM_SMCR_SMS_Pos)                        /*!< 0x00000001 */
#define TIM_SMCR_SMS_1                           (0x2UL<<TIM_SMCR_SMS_Pos)                        /*!< 0x00000002 */
#define TIM_SMCR_SMS_2                           (0x4UL<<TIM_SMCR_SMS_Pos)                        /*!< 0x00000004 */
#define TIM_SMCR_SMS_3                           (0x10000UL<<TIM_SMCR_SMS_Pos)                    /*!< 0x00010000 */
#define TIM_SMCR_OCCS_Pos                        (3U)
#define TIM_SMCR_OCCS_Msk                        (0x1UL<<TIM_SMCR_OCCS_Pos)                       /*!< 0x00000008 */
#define TIM_SMCR_OCCS                            TIM_SMCR_OCCS_Msk                                
#define TIM_SMCR_TS_Pos                          (4U)
#define TIM_SMCR_TS_Msk                          (0x30007UL<<TIM_SMCR_TS_Pos)                     /*!< 0x00300070 */
#define TIM_SMCR_TS                              TIM_SMCR_TS_Msk
#define TIM_SMCR_TS_0                            (0x1UL<<TIM_SMCR_TS_Pos)                         /*!< 0x00000010 */
#define TIM_SMCR_TS_1                            (0x2UL<<TIM_SMCR_TS_Pos)                         /*!< 0x00000020 */
#define TIM_SMCR_TS_2                            (0x4UL<<TIM_SMCR_TS_Pos)                         /*!< 0x00000040 */
#define TIM_SMCR_TS_3                            (0x10000UL<<TIM_SMCR_TS_Pos)                     /*!< 0x00100000 */
#define TIM_SMCR_TS_4                            (0x20000UL<<TIM_SMCR_TS_Pos)                     /*!< 0x00200000 */
#define TIM_SMCR_MSM_Pos                         (7U)
#define TIM_SMCR_MSM_Msk                         (0x1UL<<TIM_SMCR_MSM_Pos)                        /*!< 0x00000080 */
#define TIM_SMCR_MSM                             TIM_SMCR_MSM_Msk                                 
#define TIM_SMCR_ETF_Pos                         (8U)
#define TIM_SMCR_ETF_Msk                         (0xFUL<<TIM_SMCR_ETF_Pos)                        /*!< 0x00000F00 */
#define TIM_SMCR_ETF                             TIM_SMCR_ETF_Msk
#define TIM_SMCR_ETF_0                           (0x1UL<<TIM_SMCR_ETF_Pos)                        /*!< 0x00000100 */
#define TIM_SMCR_ETF_1                           (0x2UL<<TIM_SMCR_ETF_Pos)                        /*!< 0x00000200 */
#define TIM_SMCR_ETF_2                           (0x4UL<<TIM_SMCR_ETF_Pos)                        /*!< 0x00000400 */
#define TIM_SMCR_ETF_3                           (0x8UL<<TIM_SMCR_ETF_Pos)                        /*!< 0x00000800 */
#define TIM_SMCR_ETPS_Pos                        (12U)
#define TIM_SMCR_ETPS_Msk                        (0x3UL<<TIM_SMCR_ETPS_Pos)                       /*!< 0x00003000 */
#define TIM_SMCR_ETPS                            TIM_SMCR_ETPS_Msk
#define TIM_SMCR_ETPS_0                          (0x1UL<<TIM_SMCR_ETPS_Pos)                       /*!< 0x00001000 */
#define TIM_SMCR_ETPS_1                          (0x2UL<<TIM_SMCR_ETPS_Pos)                       /*!< 0x00002000 */
#define TIM_SMCR_ECE_Pos                         (14U)
#define TIM_SMCR_ECE_Msk                         (0x1UL<<TIM_SMCR_ECE_Pos)                        /*!< 0x00004000 */
#define TIM_SMCR_ECE                             TIM_SMCR_ECE_Msk                                 
#define TIM_SMCR_ETP_Pos                         (15U)
#define TIM_SMCR_ETP_Msk                         (0x1UL<<TIM_SMCR_ETP_Pos)                        /*!< 0x00008000 */
#define TIM_SMCR_ETP                             TIM_SMCR_ETP_Msk                                 

/********************************* Bit definition for TIM_DIER register ********************************************/
#define TIM_DIER_UIE_Pos                         (0U)
#define TIM_DIER_UIE_Msk                         (0x1UL<<TIM_DIER_UIE_Pos)                        /*!< 0x00000001 */
#define TIM_DIER_UIE                             TIM_DIER_UIE_Msk                                 
#define TIM_DIER_CC1IE_Pos                       (1U)
#define TIM_DIER_CC1IE_Msk                       (0x1UL<<TIM_DIER_CC1IE_Pos)                      /*!< 0x00000002 */
#define TIM_DIER_CC1IE                           TIM_DIER_CC1IE_Msk                               
#define TIM_DIER_CC2IE_Pos                       (2U)
#define TIM_DIER_CC2IE_Msk                       (0x1UL<<TIM_DIER_CC2IE_Pos)                      /*!< 0x00000004 */
#define TIM_DIER_CC2IE                           TIM_DIER_CC2IE_Msk                               
#define TIM_DIER_CC3IE_Pos                       (3U)
#define TIM_DIER_CC3IE_Msk                       (0x1UL<<TIM_DIER_CC3IE_Pos)                      /*!< 0x00000008 */
#define TIM_DIER_CC3IE                           TIM_DIER_CC3IE_Msk                               
#define TIM_DIER_CC4IE_Pos                       (4U)
#define TIM_DIER_CC4IE_Msk                       (0x1UL<<TIM_DIER_CC4IE_Pos)                      /*!< 0x00000010 */
#define TIM_DIER_CC4IE                           TIM_DIER_CC4IE_Msk                               
#define TIM_DIER_COMIE_Pos                       (5U)
#define TIM_DIER_COMIE_Msk                       (0x1UL<<TIM_DIER_COMIE_Pos)                      /*!< 0x00000020 */
#define TIM_DIER_COMIE                           TIM_DIER_COMIE_Msk                               
#define TIM_DIER_TIE_Pos                         (6U)
#define TIM_DIER_TIE_Msk                         (0x1UL<<TIM_DIER_TIE_Pos)                        /*!< 0x00000040 */
#define TIM_DIER_TIE                             TIM_DIER_TIE_Msk                                 
#define TIM_DIER_BIE_Pos                         (7U)
#define TIM_DIER_BIE_Msk                         (0x1UL<<TIM_DIER_BIE_Pos)                        /*!< 0x00000080 */
#define TIM_DIER_BIE                             TIM_DIER_BIE_Msk                                 
#define TIM_DIER_UDE_Pos                         (8U)
#define TIM_DIER_UDE_Msk                         (0x1UL<<TIM_DIER_UDE_Pos)                        /*!< 0x00000100 */
#define TIM_DIER_UDE                             TIM_DIER_UDE_Msk                                 
#define TIM_DIER_CC1DE_Pos                       (9U)
#define TIM_DIER_CC1DE_Msk                       (0x1UL<<TIM_DIER_CC1DE_Pos)                      /*!< 0x00000200 */
#define TIM_DIER_CC1DE                           TIM_DIER_CC1DE_Msk                               
#define TIM_DIER_CC2DE_Pos                       (10U)
#define TIM_DIER_CC2DE_Msk                       (0x1UL<<TIM_DIER_CC2DE_Pos)                      /*!< 0x00000400 */
#define TIM_DIER_CC2DE                           TIM_DIER_CC2DE_Msk                               
#define TIM_DIER_CC3DE_Pos                       (11U)
#define TIM_DIER_CC3DE_Msk                       (0x1UL<<TIM_DIER_CC3DE_Pos)                      /*!< 0x00000800 */
#define TIM_DIER_CC3DE                           TIM_DIER_CC3DE_Msk                               
#define TIM_DIER_CC4DE_Pos                       (12U)
#define TIM_DIER_CC4DE_Msk                       (0x1UL<<TIM_DIER_CC4DE_Pos)                      /*!< 0x00001000 */
#define TIM_DIER_CC4DE                           TIM_DIER_CC4DE_Msk                               
#define TIM_DIER_COMDE_Pos                       (13U)
#define TIM_DIER_COMDE_Msk                       (0x1UL<<TIM_DIER_COMDE_Pos)                      /*!< 0x00002000 */
#define TIM_DIER_COMDE                           TIM_DIER_COMDE_Msk                               
#define TIM_DIER_TDE_Pos                         (14U)
#define TIM_DIER_TDE_Msk                         (0x1UL<<TIM_DIER_TDE_Pos)                        /*!< 0x00004000 */
#define TIM_DIER_TDE                             TIM_DIER_TDE_Msk                                 

/********************************* Bit definition for TIM_SR register **********************************************/
#define TIM_SR_UIF_Pos                           (0U)
#define TIM_SR_UIF_Msk                           (0x1UL<<TIM_SR_UIF_Pos)                          /*!< 0x00000001 */
#define TIM_SR_UIF                               TIM_SR_UIF_Msk                                   
#define TIM_SR_CC1IF_Pos                         (1U)
#define TIM_SR_CC1IF_Msk                         (0x1UL<<TIM_SR_CC1IF_Pos)                        /*!< 0x00000002 */
#define TIM_SR_CC1IF                             TIM_SR_CC1IF_Msk                                 
#define TIM_SR_CC2IF_Pos                         (2U)
#define TIM_SR_CC2IF_Msk                         (0x1UL<<TIM_SR_CC2IF_Pos)                        /*!< 0x00000004 */
#define TIM_SR_CC2IF                             TIM_SR_CC2IF_Msk                                 
#define TIM_SR_CC3IF_Pos                         (3U)
#define TIM_SR_CC3IF_Msk                         (0x1UL<<TIM_SR_CC3IF_Pos)                        /*!< 0x00000008 */
#define TIM_SR_CC3IF                             TIM_SR_CC3IF_Msk                                 
#define TIM_SR_CC4IF_Pos                         (4U)
#define TIM_SR_CC4IF_Msk                         (0x1UL<<TIM_SR_CC4IF_Pos)                        /*!< 0x00000010 */
#define TIM_SR_CC4IF                             TIM_SR_CC4IF_Msk                                 
#define TIM_SR_COMIF_Pos                         (5U)
#define TIM_SR_COMIF_Msk                         (0x1UL<<TIM_SR_COMIF_Pos)                        /*!< 0x00000020 */
#define TIM_SR_COMIF                             TIM_SR_COMIF_Msk                                 
#define TIM_SR_TIF_Pos                           (6U)
#define TIM_SR_TIF_Msk                           (0x1UL<<TIM_SR_TIF_Pos)                          /*!< 0x00000040 */
#define TIM_SR_TIF                               TIM_SR_TIF_Msk                                   
#define TIM_SR_BIF_Pos                           (7U)
#define TIM_SR_BIF_Msk                           (0x1UL<<TIM_SR_BIF_Pos)                          /*!< 0x00000080 */
#define TIM_SR_BIF                               TIM_SR_BIF_Msk                                   
#define TIM_SR_B2IF_Pos                          (8U)
#define TIM_SR_B2IF_Msk                          (0x1UL<<TIM_SR_B2IF_Pos)                         /*!< 0x00000100 */
#define TIM_SR_B2IF                              TIM_SR_B2IF_Msk                                  
#define TIM_SR_CC1OF_Pos                         (9U)
#define TIM_SR_CC1OF_Msk                         (0x1UL<<TIM_SR_CC1OF_Pos)                        /*!< 0x00000200 */
#define TIM_SR_CC1OF                             TIM_SR_CC1OF_Msk                                 
#define TIM_SR_CC2OF_Pos                         (10U)
#define TIM_SR_CC2OF_Msk                         (0x1UL<<TIM_SR_CC2OF_Pos)                        /*!< 0x00000400 */
#define TIM_SR_CC2OF                             TIM_SR_CC2OF_Msk                                 
#define TIM_SR_CC3OF_Pos                         (11U)
#define TIM_SR_CC3OF_Msk                         (0x1UL<<TIM_SR_CC3OF_Pos)                        /*!< 0x00000800 */
#define TIM_SR_CC3OF                             TIM_SR_CC3OF_Msk                                 
#define TIM_SR_CC4OF_Pos                         (12U)
#define TIM_SR_CC4OF_Msk                         (0x1UL<<TIM_SR_CC4OF_Pos)                        /*!< 0x00001000 */
#define TIM_SR_CC4OF                             TIM_SR_CC4OF_Msk                                 
#define TIM_SR_SBIF_Pos                          (13U)
#define TIM_SR_SBIF_Msk                          (0x1UL<<TIM_SR_SBIF_Pos)                         /*!< 0x00002000 */
#define TIM_SR_SBIF                              TIM_SR_SBIF_Msk                                  
#define TIM_SR_IC1IR_Pos                         (18U)
#define TIM_SR_IC1IR_Msk                         (0x1UL<<TIM_SR_IC1IR_Pos)                        /*!< 0x00040000 */
#define TIM_SR_IC1IR                             TIM_SR_IC1IR_Msk                                 
#define TIM_SR_IC2IR_Pos                         (19U)
#define TIM_SR_IC2IR_Msk                         (0x1UL<<TIM_SR_IC2IR_Pos)                        /*!< 0x00080000 */
#define TIM_SR_IC2IR                             TIM_SR_IC2IR_Msk                                 
#define TIM_SR_IC3IR_Pos                         (24U)
#define TIM_SR_IC3IR_Msk                         (0x1UL<<TIM_SR_IC3IR_Pos)                        /*!< 0x01000000 */
#define TIM_SR_IC3IR                             TIM_SR_IC3IR_Msk                                 
#define TIM_SR_IC4IR_Pos                         (25U)
#define TIM_SR_IC4IR_Msk                         (0x1UL<<TIM_SR_IC4IR_Pos)                        /*!< 0x02000000 */
#define TIM_SR_IC4IR                             TIM_SR_IC4IR_Msk                                 
#define TIM_SR_IC1IF_Pos                         (26U)
#define TIM_SR_IC1IF_Msk                         (0x1UL<<TIM_SR_IC1IF_Pos)                        /*!< 0x04000000 */
#define TIM_SR_IC1IF                             TIM_SR_IC1IF_Msk                                 
#define TIM_SR_IC2IF_Pos                         (27U)
#define TIM_SR_IC2IF_Msk                         (0x1UL<<TIM_SR_IC2IF_Pos)                        /*!< 0x08000000 */
#define TIM_SR_IC2IF                             TIM_SR_IC2IF_Msk                                 
#define TIM_SR_IC3IF_Pos                         (28U)
#define TIM_SR_IC3IF_Msk                         (0x1UL<<TIM_SR_IC3IF_Pos)                        /*!< 0x10000000 */
#define TIM_SR_IC3IF                             TIM_SR_IC3IF_Msk                                 
#define TIM_SR_IC4IF_Pos                         (29U)
#define TIM_SR_IC4IF_Msk                         (0x1UL<<TIM_SR_IC4IF_Pos)                        /*!< 0x20000000 */
#define TIM_SR_IC4IF                             TIM_SR_IC4IF_Msk                                 

/********************************* Bit definition for TIM_EGR register *********************************************/
#define TIM_EGR_UG_Pos                           (0U)
#define TIM_EGR_UG_Msk                           (0x1UL<<TIM_EGR_UG_Pos)                          /*!< 0x00000001 */
#define TIM_EGR_UG                               TIM_EGR_UG_Msk                                   
#define TIM_EGR_CC1G_Pos                         (1U)
#define TIM_EGR_CC1G_Msk                         (0x1UL<<TIM_EGR_CC1G_Pos)                        /*!< 0x00000002 */
#define TIM_EGR_CC1G                             TIM_EGR_CC1G_Msk                                 
#define TIM_EGR_CC2G_Pos                         (2U)
#define TIM_EGR_CC2G_Msk                         (0x1UL<<TIM_EGR_CC2G_Pos)                        /*!< 0x00000004 */
#define TIM_EGR_CC2G                             TIM_EGR_CC2G_Msk                                 
#define TIM_EGR_CC3G_Pos                         (3U)
#define TIM_EGR_CC3G_Msk                         (0x1UL<<TIM_EGR_CC3G_Pos)                        /*!< 0x00000008 */
#define TIM_EGR_CC3G                             TIM_EGR_CC3G_Msk                                 
#define TIM_EGR_CC4G_Pos                         (4U)
#define TIM_EGR_CC4G_Msk                         (0x1UL<<TIM_EGR_CC4G_Pos)                        /*!< 0x00000010 */
#define TIM_EGR_CC4G                             TIM_EGR_CC4G_Msk                                 
#define TIM_EGR_COMG_Pos                         (5U)
#define TIM_EGR_COMG_Msk                         (0x1UL<<TIM_EGR_COMG_Pos)                        /*!< 0x00000020 */
#define TIM_EGR_COMG                             TIM_EGR_COMG_Msk                                 
#define TIM_EGR_TG_Pos                           (6U)
#define TIM_EGR_TG_Msk                           (0x1UL<<TIM_EGR_TG_Pos)                          /*!< 0x00000040 */
#define TIM_EGR_TG                               TIM_EGR_TG_Msk                                   
#define TIM_EGR_BG_Pos                           (7U)
#define TIM_EGR_BG_Msk                           (0x1UL<<TIM_EGR_BG_Pos)                          /*!< 0x00000080 */
#define TIM_EGR_BG                               TIM_EGR_BG_Msk                                   

/********************************* Bit definition for TIM_CCMR1 register *******************************************/
#define TIM_CCMR1_CC1S_Pos                       (0U)
#define TIM_CCMR1_CC1S_Msk                       (0x3UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000003 */
#define TIM_CCMR1_CC1S                           TIM_CCMR1_CC1S_Msk
#define TIM_CCMR1_CC1S_0                         (0x1UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000001 */
#define TIM_CCMR1_CC1S_1                         (0x2UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000002 */
#define TIM_CCMR1_OC1FE_Pos                      (2U)
#define TIM_CCMR1_OC1FE_Msk                      (0x1UL<<TIM_CCMR1_OC1FE_Pos)                     /*!< 0x00000004 */
#define TIM_CCMR1_OC1FE                          TIM_CCMR1_OC1FE_Msk                              
#define TIM_CCMR1_OC1PE_Pos                      (3U)
#define TIM_CCMR1_OC1PE_Msk                      (0x1UL<<TIM_CCMR1_OC1PE_Pos)                     /*!< 0x00000008 */
#define TIM_CCMR1_OC1PE                          TIM_CCMR1_OC1PE_Msk                              
#define TIM_CCMR1_OC1M_Pos                       (4U)
#define TIM_CCMR1_OC1M_Msk                       (0x1007UL<<TIM_CCMR1_OC1M_Pos)                      /*!< 0x00000070 */
#define TIM_CCMR1_OC1M                           TIM_CCMR1_OC1M_Msk
#define TIM_CCMR1_OC1M_0                         (0x1UL<<TIM_CCMR1_OC1M_Pos)                      /*!< 0x00000010 */
#define TIM_CCMR1_OC1M_1                         (0x2UL<<TIM_CCMR1_OC1M_Pos)                      /*!< 0x00000020 */
#define TIM_CCMR1_OC1M_2                         (0x4UL<<TIM_CCMR1_OC1M_Pos)                      /*!< 0x00000040 */
#define TIM_CCMR1_OC1M_3                         (0x1000UL<<TIM_CCMR1_OC1M_Pos)                   /*!< 0x00010000 */
#define TIM_CCMR1_OC1CE_Pos                      (7U)
#define TIM_CCMR1_OC1CE_Msk                      (0x1UL<<TIM_CCMR1_OC1CE_Pos)                     /*!< 0x00000080 */
#define TIM_CCMR1_OC1CE                          TIM_CCMR1_OC1CE_Msk                              
#define TIM_CCMR1_CC2S_Pos                       (8U)
#define TIM_CCMR1_CC2S_Msk                       (0x3UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000300 */
#define TIM_CCMR1_CC2S                           TIM_CCMR1_CC2S_Msk
#define TIM_CCMR1_CC2S_0                         (0x1UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000100 */
#define TIM_CCMR1_CC2S_1                         (0x2UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000200 */
#define TIM_CCMR1_OC2FE_Pos                      (10U)
#define TIM_CCMR1_OC2FE_Msk                      (0x1UL<<TIM_CCMR1_OC2FE_Pos)                     /*!< 0x00000400 */
#define TIM_CCMR1_OC2FE                          TIM_CCMR1_OC2FE_Msk                              
#define TIM_CCMR1_OC2PE_Pos                      (11U)
#define TIM_CCMR1_OC2PE_Msk                      (0x1UL<<TIM_CCMR1_OC2PE_Pos)                     /*!< 0x00000800 */
#define TIM_CCMR1_OC2PE                          TIM_CCMR1_OC2PE_Msk                              
#define TIM_CCMR1_OC2M_Pos                       (12U)
#define TIM_CCMR1_OC2M_Msk                       (0x1007UL<<TIM_CCMR1_OC2M_Pos)                   /*!< 0x01007000 */
#define TIM_CCMR1_OC2M                           TIM_CCMR1_OC2M_Msk
#define TIM_CCMR1_OC2M_0                         (0x1UL<<TIM_CCMR1_OC2M_Pos)                      /*!< 0x00001000 */
#define TIM_CCMR1_OC2M_1                         (0x2UL<<TIM_CCMR1_OC2M_Pos)                      /*!< 0x00002000 */
#define TIM_CCMR1_OC2M_2                         (0x4UL<<TIM_CCMR1_OC2M_Pos)                      /*!< 0x00004000 */
#define TIM_CCMR1_OC2M_3                         (0x1000UL<<TIM_CCMR1_OC2M_Pos)                   /*!< 0x01000000 */
#define TIM_CCMR1_OC2CE_Pos                      (15U)
#define TIM_CCMR1_OC2CE_Msk                      (0x1UL<<TIM_CCMR1_OC2CE_Pos)                     /*!< 0x00008000 */
#define TIM_CCMR1_OC2CE                          TIM_CCMR1_OC2CE_Msk                              

/*---------------------------------------------------------------------------*/

#define TIM_CCMR1_CC1S_Pos                       (0U)
#define TIM_CCMR1_CC1S_Msk                       (0x3UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000003 */
#define TIM_CCMR1_CC1S                           TIM_CCMR1_CC1S_Msk
#define TIM_CCMR1_CC1S_0                         (0x1UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000001 */
#define TIM_CCMR1_CC1S_1                         (0x2UL<<TIM_CCMR1_CC1S_Pos)                      /*!< 0x00000002 */
#define TIM_CCMR1_IC1PSC_Pos                     (2U)
#define TIM_CCMR1_IC1PSC_Msk                     (0x3UL<<TIM_CCMR1_IC1PSC_Pos)                    /*!< 0x0000000C */
#define TIM_CCMR1_IC1PSC                         TIM_CCMR1_IC1PSC_Msk
#define TIM_CCMR1_IC1PSC_0                       (0x1UL<<TIM_CCMR1_IC1PSC_Pos)                    /*!< 0x00000004 */
#define TIM_CCMR1_IC1PSC_1                       (0x2UL<<TIM_CCMR1_IC1PSC_Pos)                    /*!< 0x00000008 */
#define TIM_CCMR1_IC1F_Pos                       (4U)
#define TIM_CCMR1_IC1F_Msk                       (0xFUL<<TIM_CCMR1_IC1F_Pos)                      /*!< 0x000000F0 */
#define TIM_CCMR1_IC1F                           TIM_CCMR1_IC1F_Msk
#define TIM_CCMR1_IC1F_0                         (0x1UL<<TIM_CCMR1_IC1F_Pos)                      /*!< 0x00000010 */
#define TIM_CCMR1_IC1F_1                         (0x2UL<<TIM_CCMR1_IC1F_Pos)                      /*!< 0x00000020 */
#define TIM_CCMR1_IC1F_2                         (0x4UL<<TIM_CCMR1_IC1F_Pos)                      /*!< 0x00000040 */
#define TIM_CCMR1_IC1F_3                         (0x8UL<<TIM_CCMR1_IC1F_Pos)                      /*!< 0x00000080 */
#define TIM_CCMR1_CC2S_Pos                       (8U)
#define TIM_CCMR1_CC2S_Msk                       (0x3UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000300 */
#define TIM_CCMR1_CC2S                           TIM_CCMR1_CC2S_Msk
#define TIM_CCMR1_CC2S_0                         (0x1UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000100 */
#define TIM_CCMR1_CC2S_1                         (0x2UL<<TIM_CCMR1_CC2S_Pos)                      /*!< 0x00000200 */
#define TIM_CCMR1_IC2PSC_Pos                     (10U)
#define TIM_CCMR1_IC2PSC_Msk                     (0x3UL<<TIM_CCMR1_IC2PSC_Pos)                    /*!< 0x00000C00 */
#define TIM_CCMR1_IC2PSC                         TIM_CCMR1_IC2PSC_Msk
#define TIM_CCMR1_IC2PSC_0                       (0x1UL<<TIM_CCMR1_IC2PSC_Pos)                    /*!< 0x00000400 */
#define TIM_CCMR1_IC2PSC_1                       (0x2UL<<TIM_CCMR1_IC2PSC_Pos)                    /*!< 0x00000800 */
#define TIM_CCMR1_IC2F_Pos                       (12U)
#define TIM_CCMR1_IC2F_Msk                       (0xFUL<<TIM_CCMR1_IC2F_Pos)                      /*!< 0x0000F000 */
#define TIM_CCMR1_IC2F                           TIM_CCMR1_IC2F_Msk
#define TIM_CCMR1_IC2F_0                         (0x1UL<<TIM_CCMR1_IC2F_Pos)                      /*!< 0x00001000 */
#define TIM_CCMR1_IC2F_1                         (0x2UL<<TIM_CCMR1_IC2F_Pos)                      /*!< 0x00002000 */
#define TIM_CCMR1_IC2F_2                         (0x4UL<<TIM_CCMR1_IC2F_Pos)                      /*!< 0x00004000 */
#define TIM_CCMR1_IC2F_3                         (0x8UL<<TIM_CCMR1_IC2F_Pos)                      /*!< 0x00008000 */

/********************************* Bit definition for TIM_CCMR2 register ********************************************/
#define TIM_CCMR2_CC3S_Pos                        (0U)
#define TIM_CCMR2_CC3S_Msk                        (0x3UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000003 */
#define TIM_CCMR2_CC3S                            TIM_CCMR2_CC3S_Msk
#define TIM_CCMR2_CC3S_0                          (0x1UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000001 */
#define TIM_CCMR2_CC3S_1                          (0x2UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000002 */
#define TIM_CCMR2_OC3FE_Pos                       (2U)
#define TIM_CCMR2_OC3FE_Msk                       (0x1UL<<TIM_CCMR2_OC3FE_Pos)                      /*!< 0x00000004 */
#define TIM_CCMR2_OC3FE                           TIM_CCMR2_OC3FE_Msk                               
#define TIM_CCMR2_OC3PE_Pos                       (3U)
#define TIM_CCMR2_OC3PE_Msk                       (0x1UL<<TIM_CCMR2_OC3PE_Pos)                      /*!< 0x00000008 */
#define TIM_CCMR2_OC3PE                           TIM_CCMR2_OC3PE_Msk                               
#define TIM_CCMR2_OC3M_Pos                        (4U)
#define TIM_CCMR2_OC3M_Msk                        (0x1007UL<<TIM_CCMR2_OC3M_Pos)                    /*!< 0x00010070 */
#define TIM_CCMR2_OC3M                            TIM_CCMR2_OC3M_Msk
#define TIM_CCMR2_OC3M_0                          (0x1UL<<TIM_CCMR2_OC3M_Pos)                       /*!< 0x00000010 */
#define TIM_CCMR2_OC3M_1                          (0x2UL<<TIM_CCMR2_OC3M_Pos)                       /*!< 0x00000020 */
#define TIM_CCMR2_OC3M_2                          (0x4UL<<TIM_CCMR2_OC3M_Pos)                       /*!< 0x00000040 */
#define TIM_CCMR2_OC3M_3                          (0x1000UL<<TIM_CCMR2_OC3M_Pos)                    /*!< 0x00010000 */
#define TIM_CCMR2_OC3CE_Pos                       (7U)
#define TIM_CCMR2_OC3CE_Msk                       (0x1UL<<TIM_CCMR2_OC3CE_Pos)                      /*!< 0x00000080 */
#define TIM_CCMR2_OC3CE                           TIM_CCMR2_OC3CE_Msk                               
#define TIM_CCMR2_CC4S_Pos                        (8U)
#define TIM_CCMR2_CC4S_Msk                        (0x3UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000300 */
#define TIM_CCMR2_CC4S                            TIM_CCMR2_CC4S_Msk
#define TIM_CCMR2_CC4S_0                          (0x1UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000100 */
#define TIM_CCMR2_CC4S_1                          (0x2UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000200 */
#define TIM_CCMR2_OC4FE_Pos                       (10U)
#define TIM_CCMR2_OC4FE_Msk                       (0x1UL<<TIM_CCMR2_OC4FE_Pos)                      /*!< 0x00000400 */
#define TIM_CCMR2_OC4FE                           TIM_CCMR2_OC4FE_Msk                               
#define TIM_CCMR2_OC4PE_Pos                       (11U)
#define TIM_CCMR2_OC4PE_Msk                       (0x1UL<<TIM_CCMR2_OC4PE_Pos)                      /*!< 0x00000800 */
#define TIM_CCMR2_OC4PE                           TIM_CCMR2_OC4PE_Msk                               
#define TIM_CCMR2_OC4M_Pos                        (12U)
#define TIM_CCMR2_OC4M_Msk                        (0x1007UL<<TIM_CCMR2_OC4M_Pos)                    /*!< 0x01007000 */
#define TIM_CCMR2_OC4M                            TIM_CCMR2_OC4M_Msk
#define TIM_CCMR2_OC4M_0                          (0x1UL<<TIM_CCMR2_OC4M_Pos)                       /*!< 0x00001000 */
#define TIM_CCMR2_OC4M_1                          (0x2UL<<TIM_CCMR2_OC4M_Pos)                       /*!< 0x00002000 */
#define TIM_CCMR2_OC4M_2                          (0x4UL<<TIM_CCMR2_OC4M_Pos)                       /*!< 0x00004000 */
#define TIM_CCMR2_OC4M_3                          (0x1000UL<<TIM_CCMR2_OC4M_Pos)                    /*!< 0x01000000 */
#define TIM_CCMR2_OC4CE_Pos                       (15U)
#define TIM_CCMR2_OC4CE_Msk                       (0x1UL<<TIM_CCMR2_OC4CE_Pos)                      /*!< 0x00008000 */
#define TIM_CCMR2_OC4CE                           TIM_CCMR2_OC4CE_Msk                               
#define TIM_CCMR2_CC3S_Pos                        (0U)
#define TIM_CCMR2_CC3S_Msk                        (0x3UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000003 */
#define TIM_CCMR2_CC3S                            TIM_CCMR2_CC3S_Msk
#define TIM_CCMR2_CC3S_0                          (0x1UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000001 */
#define TIM_CCMR2_CC3S_1                          (0x2UL<<TIM_CCMR2_CC3S_Pos)                       /*!< 0x00000002 */
#define TIM_CCMR2_IC3PSC_Pos                      (2U)
#define TIM_CCMR2_IC3PSC_Msk                      (0x3UL<<TIM_CCMR2_IC3PSC_Pos)                     /*!< 0x0000000C */
#define TIM_CCMR2_IC3PSC                          TIM_CCMR2_IC3PSC_Msk
#define TIM_CCMR2_IC3PSC_0                        (0x1UL<<TIM_CCMR2_IC3PSC_Pos)                     /*!< 0x00000004 */
#define TIM_CCMR2_IC3PSC_1                        (0x2UL<<TIM_CCMR2_IC3PSC_Pos)                     /*!< 0x00000008 */
#define TIM_CCMR2_IC3F_Pos                        (4U)
#define TIM_CCMR2_IC3F_Msk                        (0xFUL<<TIM_CCMR2_IC3F_Pos)                       /*!< 0x000000F0 */
#define TIM_CCMR2_IC3F                            TIM_CCMR2_IC3F_Msk
#define TIM_CCMR2_IC3F_0                          (0x1UL<<TIM_CCMR2_IC3F_Pos)                       /*!< 0x00000010 */
#define TIM_CCMR2_IC3F_1                          (0x2UL<<TIM_CCMR2_IC3F_Pos)                       /*!< 0x00000020 */
#define TIM_CCMR2_IC3F_2                          (0x4UL<<TIM_CCMR2_IC3F_Pos)                       /*!< 0x00000040 */
#define TIM_CCMR2_IC3F_3                          (0x8UL<<TIM_CCMR2_IC3F_Pos)                       /*!< 0x00000080 */
#define TIM_CCMR2_CC4S_Pos                        (8U)
#define TIM_CCMR2_CC4S_Msk                        (0x3UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000300 */
#define TIM_CCMR2_CC4S                            TIM_CCMR2_CC4S_Msk
#define TIM_CCMR2_CC4S_0                          (0x1UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000100 */
#define TIM_CCMR2_CC4S_1                          (0x2UL<<TIM_CCMR2_CC4S_Pos)                       /*!< 0x00000200 */
#define TIM_CCMR2_IC4PSC_Pos                      (10U)
#define TIM_CCMR2_IC4PSC_Msk                      (0x3UL<<TIM_CCMR2_IC4PSC_Pos)                     /*!< 0x00000C00 */
#define TIM_CCMR2_IC4PSC                          TIM_CCMR2_IC4PSC_Msk
#define TIM_CCMR2_IC4PSC_0                        (0x1UL<<TIM_CCMR2_IC4PSC_Pos)                     /*!< 0x00000400 */
#define TIM_CCMR2_IC4PSC_1                        (0x2UL<<TIM_CCMR2_IC4PSC_Pos)                     /*!< 0x00000800 */
#define TIM_CCMR2_IC4F_Pos                        (12U)
#define TIM_CCMR2_IC4F_Msk                        (0xFUL<<TIM_CCMR2_IC4F_Pos)                       /*!< 0x0000F000 */
#define TIM_CCMR2_IC4F                            TIM_CCMR2_IC4F_Msk
#define TIM_CCMR2_IC4F_0                          (0x1UL<<TIM_CCMR2_IC4F_Pos)                       /*!< 0x00001000 */
#define TIM_CCMR2_IC4F_1                          (0x2UL<<TIM_CCMR2_IC4F_Pos)                       /*!< 0x00002000 */
#define TIM_CCMR2_IC4F_2                          (0x4UL<<TIM_CCMR2_IC4F_Pos)                       /*!< 0x00004000 */
#define TIM_CCMR2_IC4F_3                          (0x8UL<<TIM_CCMR2_IC4F_Pos)                       /*!< 0x00008000 */

/********************************* Bit definition for TIM_CCER register *********************************************/
#define TIM_CCER_CC1E_Pos                         (0U)
#define TIM_CCER_CC1E_Msk                         (0x1UL<<TIM_CCER_CC1E_Pos)                        /*!< 0x00000001 */
#define TIM_CCER_CC1E                             TIM_CCER_CC1E_Msk                                 
#define TIM_CCER_CC1P_Pos                         (1U)
#define TIM_CCER_CC1P_Msk                         (0x1UL<<TIM_CCER_CC1P_Pos)                        /*!< 0x00000002 */
#define TIM_CCER_CC1P                             TIM_CCER_CC1P_Msk                                 
#define TIM_CCER_CC1NE_Pos                        (2U)
#define TIM_CCER_CC1NE_Msk                        (0x1UL<<TIM_CCER_CC1NE_Pos)                       /*!< 0x00000004 */
#define TIM_CCER_CC1NE                            TIM_CCER_CC1NE_Msk                                
#define TIM_CCER_CC1NP_Pos                        (3U)
#define TIM_CCER_CC1NP_Msk                        (0x1UL<<TIM_CCER_CC1NP_Pos)                       /*!< 0x00000008 */
#define TIM_CCER_CC1NP                            TIM_CCER_CC1NP_Msk                                
#define TIM_CCER_CC2E_Pos                         (4U)
#define TIM_CCER_CC2E_Msk                         (0x1UL<<TIM_CCER_CC2E_Pos)                        /*!< 0x00000010 */
#define TIM_CCER_CC2E                             TIM_CCER_CC2E_Msk                                 
#define TIM_CCER_CC2P_Pos                         (5U)
#define TIM_CCER_CC2P_Msk                         (0x1UL<<TIM_CCER_CC2P_Pos)                        /*!< 0x00000020 */
#define TIM_CCER_CC2P                             TIM_CCER_CC2P_Msk                                 
#define TIM_CCER_CC2NE_Pos                        (6U)
#define TIM_CCER_CC2NE_Msk                        (0x1UL<<TIM_CCER_CC2NE_Pos)                       /*!< 0x00000040 */
#define TIM_CCER_CC2NE                            TIM_CCER_CC2NE_Msk                                
#define TIM_CCER_CC2NP_Pos                        (7U)
#define TIM_CCER_CC2NP_Msk                        (0x1UL<<TIM_CCER_CC2NP_Pos)                       /*!< 0x00000080 */
#define TIM_CCER_CC2NP                            TIM_CCER_CC2NP_Msk                                
#define TIM_CCER_CC3E_Pos                         (8U)
#define TIM_CCER_CC3E_Msk                         (0x1UL<<TIM_CCER_CC3E_Pos)                        /*!< 0x00000100 */
#define TIM_CCER_CC3E                             TIM_CCER_CC3E_Msk                                 
#define TIM_CCER_CC3P_Pos                         (9U)
#define TIM_CCER_CC3P_Msk                         (0x1UL<<TIM_CCER_CC3P_Pos)                        /*!< 0x00000200 */
#define TIM_CCER_CC3P                             TIM_CCER_CC3P_Msk                                 
#define TIM_CCER_CC3NE_Pos                        (10U)
#define TIM_CCER_CC3NE_Msk                        (0x1UL<<TIM_CCER_CC3NE_Pos)                       /*!< 0x00000400 */
#define TIM_CCER_CC3NE                            TIM_CCER_CC3NE_Msk                                
#define TIM_CCER_CC3NP_Pos                        (11U)
#define TIM_CCER_CC3NP_Msk                        (0x1UL<<TIM_CCER_CC3NP_Pos)                       /*!< 0x00000800 */
#define TIM_CCER_CC3NP                            TIM_CCER_CC3NP_Msk                                
#define TIM_CCER_CC4E_Pos                         (12U)
#define TIM_CCER_CC4E_Msk                         (0x1UL<<TIM_CCER_CC4E_Pos)                        /*!< 0x00001000 */
#define TIM_CCER_CC4E                             TIM_CCER_CC4E_Msk                                 
#define TIM_CCER_CC4P_Pos                         (13U)
#define TIM_CCER_CC4P_Msk                         (0x1UL<<TIM_CCER_CC4P_Pos)                        /*!< 0x00002000 */
#define TIM_CCER_CC4P                             TIM_CCER_CC4P_Msk                                 

/********************************* Bit definition for TIM_CNT register *********************************************/
#define TIM_CNT_CNT_Pos                          (0U)
#define TIM_CNT_CNT_Msk                          (0xFFFFUL<<TIM_CNT_CNT_Pos)                      /*!< 0x0000FFFF */
#define TIM_CNT_CNT                              TIM_CNT_CNT_Msk

/********************************* Bit definition for TIM_PSC register *********************************************/
#define TIM_PSC_PSC_Pos                          (0U)
#define TIM_PSC_PSC_Msk                          (0xFFFFUL<<TIM_PSC_PSC_Pos)                      /*!< 0x0000FFFF */
#define TIM_PSC_PSC                              TIM_PSC_PSC_Msk

/********************************* Bit definition for TIM_ARR register *********************************************/
#define TIM_ARR_ARR_Pos                          (0U)
#define TIM_ARR_ARR_Msk                          (0xFFFFUL<<TIM_ARR_ARR_Pos)                      /*!< 0x0000FFFF */
#define TIM_ARR_ARR                              TIM_ARR_ARR_Msk

/********************************* Bit definition for TIM_RCR register *********************************************/
#define TIM_RCR_REP_Pos                          (0U)
#define TIM_RCR_REP_Msk                          (0xFFUL<<TIM_RCR_REP_Pos)                        /*!< 0x000000FF */
#define TIM_RCR_REP                              TIM_RCR_REP_Msk

/********************************* Bit definition for TIM_CCR1 register ********************************************/
#define TIM_CCR1_CCR1_Pos                        (0U)
#define TIM_CCR1_CCR1_Msk                        (0xFFFFUL<<TIM_CCR1_CCR1_Pos)                    /*!< 0x0000FFFF */
#define TIM_CCR1_CCR1                            TIM_CCR1_CCR1_Msk

/********************************* Bit definition for TIM_CCR2 register ********************************************/
#define TIM_CCR2_CCR2_Pos                        (0U)
#define TIM_CCR2_CCR2_Msk                        (0xFFFFUL<<TIM_CCR2_CCR2_Pos)                    /*!< 0x0000FFFF */
#define TIM_CCR2_CCR2                            TIM_CCR2_CCR2_Msk

/********************************* Bit definition for TIM_CCR3 register ********************************************/
#define TIM_CCR3_CCR3_Pos                        (0U)
#define TIM_CCR3_CCR3_Msk                        (0xFFFFUL<<TIM_CCR3_CCR3_Pos)                    /*!< 0x0000FFFF */
#define TIM_CCR3_CCR3                            TIM_CCR3_CCR3_Msk

/********************************* Bit definition for TIM_CCR4 register ********************************************/
#define TIM_CCR4_CCR4_Pos                        (0U)
#define TIM_CCR4_CCR4_Msk                        (0xFFFFUL<<TIM_CCR4_CCR4_Pos)                    /*!< 0x0000FFFF */
#define TIM_CCR4_CCR4                            TIM_CCR4_CCR4_Msk

/********************************* Bit definition for TIM_BDTR register ********************************************/
#define TIM_BDTR_DTG_Pos                         (0U)
#define TIM_BDTR_DTG_Msk                         (0xFFUL<<TIM_BDTR_DTG_Pos)                       /*!< 0x000000FF */
#define TIM_BDTR_DTG                             TIM_BDTR_DTG_Msk
#define TIM_BDTR_LOCK_Pos                        (8U)
#define TIM_BDTR_LOCK_Msk                        (0x3UL<<TIM_BDTR_LOCK_Pos)                       /*!< 0x00000300 */
#define TIM_BDTR_LOCK                            TIM_BDTR_LOCK_Msk
#define TIM_BDTR_LOCK_0                          (0x1UL<<TIM_BDTR_LOCK_Pos)                       /*!< 0x00000100 */
#define TIM_BDTR_LOCK_1                          (0x2UL<<TIM_BDTR_LOCK_Pos)                       /*!< 0x00000200 */
#define TIM_BDTR_OSSI_Pos                        (10U)
#define TIM_BDTR_OSSI_Msk                        (0x1UL<<TIM_BDTR_OSSI_Pos)                       /*!< 0x00000400 */
#define TIM_BDTR_OSSI                            TIM_BDTR_OSSI_Msk                                
#define TIM_BDTR_OSSR_Pos                        (11U)
#define TIM_BDTR_OSSR_Msk                        (0x1UL<<TIM_BDTR_OSSR_Pos)                       /*!< 0x00000800 */
#define TIM_BDTR_OSSR                            TIM_BDTR_OSSR_Msk                                
#define TIM_BDTR_BKE_Pos                         (12U)
#define TIM_BDTR_BKE_Msk                         (0x1UL<<TIM_BDTR_BKE_Pos)                        /*!< 0x00001000 */
#define TIM_BDTR_BKE                             TIM_BDTR_BKE_Msk                                 
#define TIM_BDTR_BKP_Pos                         (13U)
#define TIM_BDTR_BKP_Msk                         (0x1UL<<TIM_BDTR_BKP_Pos)                        /*!< 0x00002000 */
#define TIM_BDTR_BKP                             TIM_BDTR_BKP_Msk                                 
#define TIM_BDTR_AOE_Pos                         (14U)
#define TIM_BDTR_AOE_Msk                         (0x1UL<<TIM_BDTR_AOE_Pos)                        /*!< 0x00004000 */
#define TIM_BDTR_AOE                             TIM_BDTR_AOE_Msk                                 
#define TIM_BDTR_MOE_Pos                         (15U)
#define TIM_BDTR_MOE_Msk                         (0x1UL<<TIM_BDTR_MOE_Pos)                        /*!< 0x00008000 */
#define TIM_BDTR_MOE                             TIM_BDTR_MOE_Msk                                 

/********************************* Bit definition for TIM_TISEL register *******************************************/
#define TIM_TISEL_TI1SEL_Pos                     (0U)
#define TIM_TISEL_TI1SEL_Msk                     (0xFUL<<TIM_TISEL_TI1SEL_Pos)                    /*!< 0x0000000F */
#define TIM_TISEL_TI1SEL                         TIM_TISEL_TI1SEL_Msk
#define TIM_TISEL_TI1SEL_0                       (0x1UL<<TIM_TISEL_TI1SEL_Pos)                    /*!< 0x00000001 */
#define TIM_TISEL_TI1SEL_1                       (0x2UL<<TIM_TISEL_TI1SEL_Pos)                    /*!< 0x00000002 */
#define TIM_TISEL_TI1SEL_2                       (0x4UL<<TIM_TISEL_TI1SEL_Pos)                    /*!< 0x00000004 */
#define TIM_TISEL_TI1SEL_3                       (0x8UL<<TIM_TISEL_TI1SEL_Pos)                    /*!< 0x00000008 */
#define TIM_TISEL_TI2SEL_Pos                     (8U)
#define TIM_TISEL_TI2SEL_Msk                     (0xFUL<<TIM_TISEL_TI2SEL_Pos)                    /*!< 0x00000F00 */
#define TIM_TISEL_TI2SEL                         TIM_TISEL_TI2SEL_Msk
#define TIM_TISEL_TI2SEL_0                       (0x1UL<<TIM_TISEL_TI2SEL_Pos)                    /*!< 0x00000100 */
#define TIM_TISEL_TI2SEL_1                       (0x2UL<<TIM_TISEL_TI2SEL_Pos)                    /*!< 0x00000200 */
#define TIM_TISEL_TI2SEL_2                       (0x4UL<<TIM_TISEL_TI2SEL_Pos)                    /*!< 0x00000400 */
#define TIM_TISEL_TI2SEL_3                       (0x8UL<<TIM_TISEL_TI2SEL_Pos)                    /*!< 0x00000800 */
#define TIM_TISEL_TI3SEL_Pos                     (16U)
#define TIM_TISEL_TI3SEL_Msk                     (0xFUL<<TIM_TISEL_TI3SEL_Pos)                    /*!< 0x000F0000 */
#define TIM_TISEL_TI3SEL                         TIM_TISEL_TI3SEL_Msk
#define TIM_TISEL_TI3SEL_0                       (0x1UL<<TIM_TISEL_TI3SEL_Pos)                    /*!< 0x00010000 */
#define TIM_TISEL_TI3SEL_1                       (0x2UL<<TIM_TISEL_TI3SEL_Pos)                    /*!< 0x00020000 */
#define TIM_TISEL_TI3SEL_2                       (0x4UL<<TIM_TISEL_TI3SEL_Pos)                    /*!< 0x00040000 */
#define TIM_TISEL_TI3SEL_3                       (0x8UL<<TIM_TISEL_TI3SEL_Pos)                    /*!< 0x00080000 */
#define TIM_TISEL_TI4SEL_Pos                     (24U)
#define TIM_TISEL_TI4SEL_Msk                     (0xFUL<<TIM_TISEL_TI4SEL_Pos)                    /*!< 0x0F000000 */
#define TIM_TISEL_TI4SEL                         TIM_TISEL_TI4SEL_Msk
#define TIM_TISEL_TI4SEL_0                       (0x1UL<<TIM_TISEL_TI4SEL_Pos)                    /*!< 0x01000000 */
#define TIM_TISEL_TI4SEL_1                       (0x2UL<<TIM_TISEL_TI4SEL_Pos)                    /*!< 0x02000000 */
#define TIM_TISEL_TI4SEL_2                       (0x4UL<<TIM_TISEL_TI4SEL_Pos)                    /*!< 0x04000000 */
#define TIM_TISEL_TI4SEL_3                       (0x8UL<<TIM_TISEL_TI4SEL_Pos)                    /*!< 0x08000000 */

/********************************* Bit definition for TIM_AF1 register *********************************************/
#define TIM_AF1_BKINE_Pos                        (0U)
#define TIM_AF1_BKINE_Msk                        (0x1UL<<TIM_AF1_BKINE_Pos)                       /*!< 0x00000001 */
#define TIM_AF1_BKINE                            TIM_AF1_BKINE_Msk                                
#define TIM_AF1_BKCMP1E_Pos                      (1U)
#define TIM_AF1_BKCMP1E_Msk                      (0x1UL<<TIM_AF1_BKCMP1E_Pos)                     /*!< 0x00000002 */
#define TIM_AF1_BKCMP1E                          TIM_AF1_BKCMP1E_Msk                              
#define TIM_AF1_BKCMP2E_Pos                      (2U)
#define TIM_AF1_BKCMP2E_Msk                      (0x1UL<<TIM_AF1_BKCMP2E_Pos)                     /*!< 0x00000004 */
#define TIM_AF1_BKCMP2E                          TIM_AF1_BKCMP2E_Msk                              
#define TIM_AF1_BKCMP3E_Pos                      (3U)
#define TIM_AF1_BKCMP3E_Msk                      (0x1UL<<TIM_AF1_BKCMP3E_Pos)                     /*!< 0x00000008 */
#define TIM_AF1_BKCMP3E                          TIM_AF1_BKCMP3E_Msk                              
#define TIM_AF1_BKINP_Pos                        (9U)
#define TIM_AF1_BKINP_Msk                        (0x1UL<<TIM_AF1_BKINP_Pos)                       /*!< 0x00000200 */
#define TIM_AF1_BKINP                            TIM_AF1_BKINP_Msk                                
#define TIM_AF1_BKCMP1P_Pos                      (10U)
#define TIM_AF1_BKCMP1P_Msk                      (0x1UL<<TIM_AF1_BKCMP1P_Pos)                     /*!< 0x00000400 */
#define TIM_AF1_BKCMP1P                          TIM_AF1_BKCMP1P_Msk                              
#define TIM_AF1_BKCMP2P_Pos                      (11U)
#define TIM_AF1_BKCMP2P_Msk                      (0x1UL<<TIM_AF1_BKCMP2P_Pos)                     /*!< 0x00000800 */
#define TIM_AF1_BKCMP2P                          TIM_AF1_BKCMP2P_Msk                              
#define TIM_AF1_BKCMP3P_Pos                      (12U)
#define TIM_AF1_BKCMP3P_Msk                      (0x1UL<<TIM_AF1_BKCMP3P_Pos)                     /*!< 0x00001000 */
#define TIM_AF1_BKCMP3P                          TIM_AF1_BKCMP3P_Msk                              
#define TIM_AF1_ETRSEL_Pos                       (14U)
#define TIM_AF1_ETRSEL_Msk                       (0xFUL<<TIM_AF1_ETRSEL_Pos)                      /*!< 0x0003C000 */
#define TIM_AF1_ETRSEL                           TIM_AF1_ETRSEL_Msk
#define TIM_AF1_ETRSEL_0                         (0x1UL<<TIM_AF1_ETRSEL_Pos)                      /*!< 0x00004000 */
#define TIM_AF1_ETRSEL_1                         (0x2UL<<TIM_AF1_ETRSEL_Pos)                      /*!< 0x00008000 */
#define TIM_AF1_ETRSEL_2                         (0x4UL<<TIM_AF1_ETRSEL_Pos)                      /*!< 0x00010000 */
#define TIM_AF1_ETRSEL_3                         (0x8UL<<TIM_AF1_ETRSEL_Pos)                      /*!< 0x00020000 */

/********************************* Bit definition for TIM_AF2 register *********************************************/
#define TIM_AF2_OCRSEL_Pos                       (16U)
#define TIM_AF2_OCRSEL_Msk                       (0x7UL<<TIM_AF2_OCRSEL_Pos)                      /*!< 0x00070000 */
#define TIM_AF2_OCRSEL                           TIM_AF2_OCRSEL_Msk
#define TIM_AF2_OCRSEL_0                         (0x1UL<<TIM_AF2_OCRSEL_Pos)                      /*!< 0x00010000 */
#define TIM_AF2_OCRSEL_1                         (0x2UL<<TIM_AF2_OCRSEL_Pos)                      /*!< 0x00020000 */
#define TIM_AF2_OCRSEL_2                         (0x4UL<<TIM_AF2_OCRSEL_Pos)                      /*!< 0x00040000 */

/********************************* Bit definition for TIM_DCR register *********************************************/
#define TIM_DCR_DBA_Pos                          (0U)
#define TIM_DCR_DBA_Msk                          (0x1FUL<<TIM_DCR_DBA_Pos)                        /*!< 0x0000001F */
#define TIM_DCR_DBA                              TIM_DCR_DBA_Msk
#define TIM_DCR_DBA_0                            (0x1UL<<TIM_DCR_DBA_Pos)                         /*!< 0x00000001 */
#define TIM_DCR_DBA_1                            (0x2UL<<TIM_DCR_DBA_Pos)                         /*!< 0x00000002 */
#define TIM_DCR_DBA_2                            (0x4UL<<TIM_DCR_DBA_Pos)                         /*!< 0x00000004 */
#define TIM_DCR_DBA_3                            (0x8UL<<TIM_DCR_DBA_Pos)                         /*!< 0x00000008 */
#define TIM_DCR_DBA_4                            (0x10UL<<TIM_DCR_DBA_Pos)                        /*!< 0x00000010 */
#define TIM_DCR_DBL_Pos                          (8U)
#define TIM_DCR_DBL_Msk                          (0x1FUL<<TIM_DCR_DBL_Pos)                        /*!< 0x00001F00 */
#define TIM_DCR_DBL                              TIM_DCR_DBL_Msk
#define TIM_DCR_DBL_0                            (0x1UL<<TIM_DCR_DBL_Pos)                         /*!< 0x00000100 */
#define TIM_DCR_DBL_1                            (0x2UL<<TIM_DCR_DBL_Pos)                         /*!< 0x00000200 */
#define TIM_DCR_DBL_2                            (0x4UL<<TIM_DCR_DBL_Pos)                         /*!< 0x00000400 */
#define TIM_DCR_DBL_3                            (0x8UL<<TIM_DCR_DBL_Pos)                         /*!< 0x00000800 */
#define TIM_DCR_DBL_4                            (0x10UL<<TIM_DCR_DBL_Pos)                        /*!< 0x00001000 */

/********************************* Bit definition for TIM_DMAR register ********************************************/
#define TIM_DMAR_DMAB_Pos                        (0U)
#define TIM_DMAR_DMAB_Msk                        (0xFFFFFFFFUL<<TIM_DMAR_DMAB_Pos)                /*!< 0xFFFFFFFF */
#define TIM_DMAR_DMAB                            TIM_DMAR_DMAB_Msk


// /*******************  Bit definition for TIM14_OR register  ********************/
// #define TIM14_OR_TI1_RMP_Pos      (0U)
// #define TIM14_OR_TI1_RMP_Msk      (0x3UL << TIM14_OR_TI1_RMP_Pos)              /*!< 0x00000003 */
// #define TIM14_OR_TI1_RMP          TIM14_OR_TI1_RMP_Msk                         /*!<TI1_RMP[1:0] bits (TIM14 Input 4 remap) */
// #define TIM14_OR_TI1_RMP_0        (0x1UL << TIM14_OR_TI1_RMP_Pos)              /*!< 0x00000001 */
// #define TIM14_OR_TI1_RMP_1        (0x2UL << TIM14_OR_TI1_RMP_Pos)              /*!< 0x00000002 */

/********************************************************************************************************************/
/********************************* TK *******************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for TK_CR1 register ***********************************************/
#define TK_CR1_EN_Pos                             (0U)
#define TK_CR1_EN_Msk                             (0x1UL<<TK_CR1_EN_Pos)                            /*!< 0x00000001 */
#define TK_CR1_EN                                 TK_CR1_EN_Msk                                     
#define TK_CR1_EXTCMODEN_Pos                      (3U)
#define TK_CR1_EXTCMODEN_Msk                      (0x1UL<<TK_CR1_EXTCMODEN_Pos)                     /*!< 0x00000008 */
#define TK_CR1_EXTCMODEN                          TK_CR1_EXTCMODEN_Msk                              
#define TK_CR1_ICOPA_Pos                          (4U)
#define TK_CR1_ICOPA_Msk                          (0x1UL<<TK_CR1_ICOPA_Pos)                         /*!< 0x00000010 */
#define TK_CR1_ICOPA                              TK_CR1_ICOPA_Msk                                  
#define TK_CR1_COPAMODE_Pos                       (5U)
#define TK_CR1_COPAMODE_Msk                       (0x1UL<<TK_CR1_COPAMODE_Pos)                      /*!< 0x00000020 */
#define TK_CR1_COPAMODE                           TK_CR1_COPAMODE_Msk                               
#define TK_CR1_COPADATA_Pos                       (6U)
#define TK_CR1_COPADATA_Msk                       (0x3UL<<TK_CR1_COPADATA_Pos)                      /*!< 0x000000C0 */
#define TK_CR1_COPADATA                           TK_CR1_COPADATA_Msk
#define TK_CR1_COPADATA_0                         (0x1UL<<TK_CR1_COPADATA_Pos)                      /*!< 0x00000040 */
#define TK_CR1_COPADATA_1                         (0x2UL<<TK_CR1_COPADATA_Pos)                      /*!< 0x00000080 */
#define TK_CR1_VCC1CR_Pos                         (11U)
#define TK_CR1_VCC1CR_Msk                         (0x7UL<<TK_CR1_VCC1CR_Pos)                        /*!< 0x00003800 */
#define TK_CR1_VCC1CR                             TK_CR1_VCC1CR_Msk
#define TK_CR1_VCC1CR_0                           (0x1UL<<TK_CR1_VCC1CR_Pos)                        /*!< 0x00000800 */
#define TK_CR1_VCC1CR_1                           (0x2UL<<TK_CR1_VCC1CR_Pos)                        /*!< 0x00001000 */
#define TK_CR1_VCC1CR_2                           (0x4UL<<TK_CR1_VCC1CR_Pos)                        /*!< 0x00002000 */
#define TK_CR1_MIDACSTEP_Pos                      (14U)
#define TK_CR1_MIDACSTEP_Msk                      (0x1UL<<TK_CR1_MIDACSTEP_Pos)                     /*!< 0x00004000 */
#define TK_CR1_MIDACSTEP                          TK_CR1_MIDACSTEP_Msk                              
#define TK_CR1_SIDACSTEP_Pos                      (15U)
#define TK_CR1_SIDACSTEP_Msk                      (0x1UL<<TK_CR1_SIDACSTEP_Pos)                     /*!< 0x00008000 */
#define TK_CR1_SIDACSTEP                          TK_CR1_SIDACSTEP_Msk                              
#define TK_CR1_CFT_Pos                            (16U)
#define TK_CR1_CFT_Msk                            (0x7UL<<TK_CR1_CFT_Pos)                           /*!< 0x00070000 */
#define TK_CR1_CFT                                TK_CR1_CFT_Msk
#define TK_CR1_CFT_0                              (0x1UL<<TK_CR1_CFT_Pos)                           /*!< 0x00010000 */
#define TK_CR1_CFT_1                              (0x2UL<<TK_CR1_CFT_Pos)                           /*!< 0x00020000 */
#define TK_CR1_CFT_2                              (0x4UL<<TK_CR1_CFT_Pos)                           /*!< 0x00040000 */
#define TK_CR1_INTCMODEN_Pos                      (20U)
#define TK_CR1_INTCMODEN_Msk                      (0x1UL<<TK_CR1_INTCMODEN_Pos)                     /*!< 0x00100000 */
#define TK_CR1_INTCMODEN                          TK_CR1_INTCMODEN_Msk                              
#define TK_CR1_CMPHYSEN_Pos                       (21U)
#define TK_CR1_CMPHYSEN_Msk                       (0x1UL<<TK_CR1_CMPHYSEN_Pos)                      /*!< 0x00200000 */
#define TK_CR1_CMPHYSEN                           TK_CR1_CMPHYSEN_Msk                               /*!< Comparetor Hysteresis Enable */
#define TK_CR1_BUFFEREN_Pos                       (22U)
#define TK_CR1_BUFFEREN_Msk                       (0x1UL<<TK_CR1_BUFFEREN_Pos)                      /*!< 0x00400000 */
#define TK_CR1_BUFFEREN                           TK_CR1_BUFFEREN_Msk                               /*!< Buffer Enable */
#define TK_CR1_DIV_Pos                            (24U)
#define TK_CR1_DIV_Msk                            (0x7UL<<TK_CR1_DIV_Pos)                           /*!< 0x07000000 */
#define TK_CR1_DIV                                TK_CR1_DIV_Msk                                    /*!< TK_CLK Division */
#define TK_CR1_DIV_0                              (0x1UL<<TK_CR1_DIV_Pos)                           /*!< 0x01000000 */
#define TK_CR1_DIV_1                              (0x2UL<<TK_CR1_DIV_Pos)                           /*!< 0x02000000 */
#define TK_CR1_DIV_2                              (0x4UL<<TK_CR1_DIV_Pos)                           /*!< 0x04000000 */
#define TK_CR1_INTEREN_Pos                        (31U)
#define TK_CR1_INTEREN_Msk                        (0x1UL<<TK_CR1_INTEREN_Pos)                       /*!< 0x80000000 */
#define TK_CR1_INTEREN                            TK_CR1_INTEREN_Msk                                /*!< Internal Enable */

/********************************* Bit definition for TK_CR2 register ***********************************************/
#define TK_CR2_START_Pos                          (0U)
#define TK_CR2_START_Msk                          (0x1UL<<TK_CR2_START_Pos)                         /*!< 0x00000001 */
#define TK_CR2_START                              TK_CR2_START_Msk                                  
#define TK_CR2_RSTB_Pos                           (1U)
#define TK_CR2_RSTB_Msk                           (0x1UL<<TK_CR2_RSTB_Pos)                          /*!< 0x00000002 */
#define TK_CR2_RSTB                               TK_CR2_RSTB_Msk                                   
#define TK_CR2_SWCTRL_Pos                         (2U)
#define TK_CR2_SWCTRL_Msk                         (0x1UL<<TK_CR2_SWCTRL_Pos)                        /*!< 0x00000004 */
#define TK_CR2_SWCTRL                             TK_CR2_SWCTRL_Msk                                 
#define TK_CR2_CMODCHRG_Pos                       (3U)
#define TK_CR2_CMODCHRG_Msk                       (0x1UL<<TK_CR2_CMODCHRG_Pos)                      /*!< 0x00000008 */
#define TK_CR2_CMODCHRG                           TK_CR2_CMODCHRG_Msk                               
#define TK_CR2_MIDACEN_Pos                        (5U)
#define TK_CR2_MIDACEN_Msk                        (0x1UL<<TK_CR2_MIDACEN_Pos)                       /*!< 0x00000020 */
#define TK_CR2_MIDACEN                            TK_CR2_MIDACEN_Msk                                
#define TK_CR2_SIDACEN_Pos                        (6U)
#define TK_CR2_SIDACEN_Msk                        (0x1UL<<TK_CR2_SIDACEN_Pos)                       /*!< 0x00000040 */
#define TK_CR2_SIDACEN                            TK_CR2_SIDACEN_Msk                                
#define TK_CR2_PWMM_Pos                           (8U)
#define TK_CR2_PWMM_Msk                           (0x1UL<<TK_CR2_PWMM_Pos)                          /*!< 0x00000100 */
#define TK_CR2_PWMM                               TK_CR2_PWMM_Msk                                   
#define TK_CR2_CTRLSOURCE_Pos                     (9U)
#define TK_CR2_CTRLSOURCE_Msk                     (0x1UL<<TK_CR2_CTRLSOURCE_Pos)                    /*!< 0x00000200 */
#define TK_CR2_CTRLSOURCE                         TK_CR2_CTRLSOURCE_Msk                             
#define TK_CR2_LPABNORMAL_Pos                     (10U)
#define TK_CR2_LPABNORMAL_Msk                     (0x1UL<<TK_CR2_LPABNORMAL_Pos)                    /*!< 0x00000400 */
#define TK_CR2_LPABNORMAL                         TK_CR2_LPABNORMAL_Msk                             
#define TK_CR2_SHORTSW_NUM_Pos                    (12U)
#define TK_CR2_SHORTSW_NUM_Msk                    (0x3UL<<TK_CR2_SHORTSW_NUM_Pos)                   /*!< 0x00003000 */
#define TK_CR2_SHORTSW_NUM                        TK_CR2_SHORTSW_NUM_Msk
#define TK_CR2_SHORTSW_NUM_0                      (0x1UL<<TK_CR2_SHORTSW_NUM_Pos)                   /*!< 0x00001000 */
#define TK_CR2_SHORTSW_NUM_1                      (0x2UL<<TK_CR2_SHORTSW_NUM_Pos)                   /*!< 0x00002000 */
#define TK_CR2_LPEN_Pos                           (14U)
#define TK_CR2_LPEN_Msk                           (0x1UL<<TK_CR2_LPEN_Pos)                          /*!< 0x00004000 */
#define TK_CR2_LPEN                               TK_CR2_LPEN_Msk                                   
#define TK_CR2_KVREF_Pos                          (16U)
#define TK_CR2_KVREF_Msk                          (0x3UL<<TK_CR2_KVREF_Pos)                         /*!< 0x00030000 */
#define TK_CR2_KVREF                              TK_CR2_KVREF_Msk
#define TK_CR2_KVREF_0                            (0x1UL<<TK_CR2_KVREF_Pos)                         /*!< 0x00010000 */
#define TK_CR2_KVREF_1                            (0x2UL<<TK_CR2_KVREF_Pos)                         /*!< 0x00020000 */
#define TK_CR2_LFSRW_Pos                          (20U)
#define TK_CR2_LFSRW_Msk                          (0xFUL<<TK_CR2_LFSRW_Pos)                         /*!< 0x00F00000 */
#define TK_CR2_LFSRW                              TK_CR2_LFSRW_Msk
#define TK_CR2_LFSRW_0                            (0x1UL<<TK_CR2_LFSRW_Pos)                         /*!< 0x00100000 */
#define TK_CR2_LFSRW_1                            (0x2UL<<TK_CR2_LFSRW_Pos)                         /*!< 0x00200000 */
#define TK_CR2_LFSRW_2                            (0x4UL<<TK_CR2_LFSRW_Pos)                         /*!< 0x00400000 */
#define TK_CR2_LFSRW_3                            (0x8UL<<TK_CR2_LFSRW_Pos)                         /*!< 0x00800000 */
#define TK_CR2_CMODCOPA_Pos                       (29U)
#define TK_CR2_CMODCOPA_Msk                       (0x1UL<<TK_CR2_CMODCOPA_Pos)                      /*!< 0x20000000 */
#define TK_CR2_CMODCOPA                           TK_CR2_CMODCOPA_Msk                               
#define TK_CR2_CMODCS_Pos                         (30U)
#define TK_CR2_CMODCS_Msk                         (0x1UL<<TK_CR2_CMODCS_Pos)                        /*!< 0x40000000 */
#define TK_CR2_CMODCS                             TK_CR2_CMODCS_Msk                                 
#define TK_CR2_CMODEN_Pos                         (31U)
#define TK_CR2_CMODEN_Msk                         (0x1UL<<TK_CR2_CMODEN_Pos)                        /*!< 0x80000000 */
#define TK_CR2_CMODEN                             TK_CR2_CMODEN_Msk                                 

/********************************* Bit definition for TK_IDACR register *********************************************/
#define TK_IDACR_MIDAC_Pos                        (0U)
#define TK_IDACR_MIDAC_Msk                        (0xFFUL<<TK_IDACR_MIDAC_Pos)                      /*!< 0x000000FF */
#define TK_IDACR_MIDAC                            TK_IDACR_MIDAC_Msk
#define TK_IDACR_SIDAC_Pos                        (16U)
#define TK_IDACR_SIDAC_Msk                        (0xFFUL<<TK_IDACR_SIDAC_Pos)                      /*!< 0x00FF0000 */
#define TK_IDACR_SIDAC                            TK_IDACR_SIDAC_Msk

/********************************* Bit definition for TK_KR register ************************************************/
#define TK_KR_KEN_Pos                             (0U)
#define TK_KR_KEN_Msk                             (0xFFFFFFFFUL<<TK_KR_KEN_Pos)                     /*!< 0xFFFFFFFF */
#define TK_KR_KEN                                 TK_KR_KEN_Msk

/********************************* Bit definition for TK_CSR register ***********************************************/
#define TK_CSR_CHS_Pos                            (0U)
#define TK_CSR_CHS_Msk                            (0xFFFFFFFFUL<<TK_CSR_CHS_Pos)                    /*!< 0xFFFFFFFF */
#define TK_CSR_CHS                                TK_CSR_CHS_Msk

/********************************* Bit definition for TK_UCR register ***********************************************/
#define TK_UCR_COPAMODEIO_Pos                     (0U)
#define TK_UCR_COPAMODEIO_Msk                     (0xFFFFFFFFUL<<TK_UCR_COPAMODEIO_Pos)             /*!< 0xFFFFFFFF */
#define TK_UCR_COPAMODEIO                         TK_UCR_COPAMODEIO_Msk

/********************************* Bit definition for TK_STR register ***********************************************/
#define TK_STR_SETTIME_Pos                        (0U)
#define TK_STR_SETTIME_Msk                        (0x3FFUL<<TK_STR_SETTIME_Pos)                     /*!< 0x000003FF */
#define TK_STR_SETTIME                            TK_STR_SETTIME_Msk

/********************************* Bit definition for TK_RESERVED register ******************************************/

/********************************* Bit definition for TK_WR register ************************************************/
#define TK_WR_WIN_Pos                             (0U)
#define TK_WR_WIN_Msk                             (0xFFFFUL<<TK_WR_WIN_Pos)                         /*!< 0x0000FFFF */
#define TK_WR_WIN                                 TK_WR_WIN_Msk

/********************************* Bit definition for TK_SDR register ***********************************************/
#define TK_SDR_SWDIV_Pos                          (0U)
#define TK_SDR_SWDIV_Msk                          (0x1FFUL<<TK_SDR_SWDIV_Pos)                       /*!< 0x000001FF */
#define TK_SDR_SWDIV                              TK_SDR_SWDIV_Msk

/********************************* Bit definition for TK_PRSR register **********************************************/
#define TK_PRSR_LFSRP_Pos                         (0U)
#define TK_PRSR_LFSRP_Msk                         (0xFFFFUL<<TK_PRSR_LFSRP_Pos)                     /*!< 0x0000FFFF */
#define TK_PRSR_LFSRP                             TK_PRSR_LFSRP_Msk
#define TK_PRSR_SEED_Pos                          (16U)
#define TK_PRSR_SEED_Msk                          (0xFFFFUL<<TK_PRSR_SEED_Pos)                      /*!< 0xFFFF0000 */
#define TK_PRSR_SEED                              TK_PRSR_SEED_Msk

/********************************* Bit definition for TK_DR register ************************************************/
#define TK_DR_DATA_Pos                            (0U)
#define TK_DR_DATA_Msk                            (0xFFFFUL<<TK_DR_DATA_Pos)                        /*!< 0x0000FFFF */
#define TK_DR_DATA                                TK_DR_DATA_Msk

/********************************* Bit definition for TK_SR register ************************************************/
#define TK_SR_SENSEDONE_Pos                       (0U)
#define TK_SR_SENSEDONE_Msk                       (0x1UL<<TK_SR_SENSEDONE_Pos)                      /*!< 0x00000001 */
#define TK_SR_SENSEDONE                           TK_SR_SENSEDONE_Msk                               
#define TK_SR_CMPOUT_Pos                          (1U)
#define TK_SR_CMPOUT_Msk                          (0x1UL<<TK_SR_CMPOUT_Pos)                         /*!< 0x00000002 */
#define TK_SR_CMPOUT                              TK_SR_CMPOUT_Msk                                  
#define TK_SR_SHORTSWERR_Pos                      (2U)
#define TK_SR_SHORTSWERR_Msk                      (0x1UL<<TK_SR_SHORTSWERR_Pos)                     /*!< 0x00000004 */
#define TK_SR_SHORTSWERR                          TK_SR_SHORTSWERR_Msk                              

/********************************* Bit definition for TK_IER register ***********************************************/
#define TK_IER_EVTINTEN_Pos                       (0U)
#define TK_IER_EVTINTEN_Msk                       (0x1UL<<TK_IER_EVTINTEN_Pos)                      /*!< 0x00000001 */
#define TK_IER_EVTINTEN                           TK_IER_EVTINTEN_Msk                               
#define TK_IER_ERRINTEN_Pos                       (1U)
#define TK_IER_ERRINTEN_Msk                       (0x1UL<<TK_IER_ERRINTEN_Pos)                      /*!< 0x00000002 */
#define TK_IER_ERRINTEN                           TK_IER_ERRINTEN_Msk                               

/********************************* Bit definition for TK_ICR register ***********************************************/
#define TK_ICR_EVTINTCLR_Pos                      (0U)
#define TK_ICR_EVTINTCLR_Msk                      (0x1UL<<TK_ICR_EVTINTCLR_Pos)                     /*!< 0x00000001 */
#define TK_ICR_EVTINTCLR                          TK_ICR_EVTINTCLR_Msk                              
#define TK_ICR_ERRINTCLR_Pos                      (1U)
#define TK_ICR_ERRINTCLR_Msk                      (0x1UL<<TK_ICR_ERRINTCLR_Pos)                     /*!< 0x00000002 */
#define TK_ICR_ERRINTCLR                          TK_ICR_ERRINTCLR_Msk                              

/********************************* Bit definition for TK_CDR register ***********************************************/
#define TK_CDR_CDATA_Pos                          (0U)
#define TK_CDR_CDATA_Msk                          (0xFFFFUL<<TK_CDR_CDATA_Pos)                      /*!< 0x0000FFFF */
#define TK_CDR_CDATA                              TK_CDR_CDATA_Msk

/********************************* Bit definition for TK_LPDR register **********************************************/
#define TK_LPDR_NORMALDELTA_Pos                   (0U)
#define TK_LPDR_NORMALDELTA_Msk                   (0x3FFUL<<TK_LPDR_NORMALDELTA_Pos)                /*!< 0x000003FF */
#define TK_LPDR_NORMALDELTA                       TK_LPDR_NORMALDELTA_Msk
#define TK_LPDR_ABNORMALDELTA_Pos                 (16U)
#define TK_LPDR_ABNORMALDELTA_Msk                 (0x3FFUL<<TK_LPDR_ABNORMALDELTA_Pos)              /*!< 0x03FF0000 */
#define TK_LPDR_ABNORMALDELTA                     TK_LPDR_ABNORMALDELTA_Msk

/********************************* Bit definition for TK_LPTR register **********************************************/
#define TK_LPTR_LPWAIT_Pos                        (0U)
#define TK_LPTR_LPWAIT_Msk                        (0x7FUL<<TK_LPTR_LPWAIT_Pos)                      /*!< 0x0000007F */
#define TK_LPTR_LPWAIT                            TK_LPTR_LPWAIT_Msk

/********************************* Bit definition for TK_OPR register ***********************************************/
#define TK_OPR_OP_Pos                             (0U)
#define TK_OPR_OP_Msk                             (0x3FUL<<TK_OPR_OP_Pos)                           /*!< 0x0000003F */
#define TK_OPR_OP                                 TK_OPR_OP_Msk
#define TK_OPR_OP_0                               (0x1UL<<TK_OPR_OP_Pos)                            /*!< 0x00000001 */
#define TK_OPR_OP_1                               (0x2UL<<TK_OPR_OP_Pos)                            /*!< 0x00000002 */
#define TK_OPR_OP_2                               (0x4UL<<TK_OPR_OP_Pos)                            /*!< 0x00000004 */
#define TK_OPR_OP_3                               (0x8UL<<TK_OPR_OP_Pos)                            /*!< 0x00000008 */
#define TK_OPR_OP_4                               (0x10UL<<TK_OPR_OP_Pos)                           /*!< 0x00000010 */
#define TK_OPR_OP_5                               (0x20UL<<TK_OPR_OP_Pos)                           /*!< 0x00000020 */

/********************************* Bit definition for TK_TR register ************************************************/
#define TK_TR_TMODE_Pos                           (0U)
#define TK_TR_TMODE_Msk                           (0x7UL<<TK_TR_TMODE_Pos)                          /*!< 0x00000007 */
#define TK_TR_TMODE                               TK_TR_TMODE_Msk
#define TK_TR_TMODE_0                             (0x1UL<<TK_TR_TMODE_Pos)                          /*!< 0x00000001 */
#define TK_TR_TMODE_1                             (0x2UL<<TK_TR_TMODE_Pos)                          /*!< 0x00000002 */
#define TK_TR_TMODE_2                             (0x4UL<<TK_TR_TMODE_Pos)                          /*!< 0x00000004 */
#define TK_TR_DBGSEL_Pos                          (16U)
#define TK_TR_DBGSEL_Msk                          (0xFUL<<TK_TR_DBGSEL_Pos)                         /*!< 0x000F0000 */
#define TK_TR_DBGSEL                              TK_TR_DBGSEL_Msk
#define TK_TR_DBGSEL_0                            (0x1UL<<TK_TR_DBGSEL_Pos)                         /*!< 0x00010000 */
#define TK_TR_DBGSEL_1                            (0x2UL<<TK_TR_DBGSEL_Pos)                         /*!< 0x00020000 */
#define TK_TR_DBGSEL_2                            (0x4UL<<TK_TR_DBGSEL_Pos)                         /*!< 0x00040000 */
#define TK_TR_DBGSEL_3                            (0x8UL<<TK_TR_DBGSEL_Pos)                         /*!< 0x00080000 */

/********************************* Bit definition for TK_DDR register ***********************************************/
#define TK_DDR_SHORTSW_DATA_Pos                   (0U)
#define TK_DDR_SHORTSW_DATA_Msk                   (0x3FFUL<<TK_DDR_SHORTSW_DATA_Pos)                /*!< 0x000003FF */
#define TK_DDR_SHORTSW_DATA                       TK_DDR_SHORTSW_DATA_Msk

/********************************************************************************************************************/
/********************************* 32BIT ****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for LPTIM_ISR register ********************************************/
#define LPTIM_ISR_CMPM_Pos                        (0U)
#define LPTIM_ISR_CMPM_Msk                        (0x1UL<<LPTIM_ISR_CMPM_Pos)                       /*!< 0x00000001 */
#define LPTIM_ISR_CMPM                            LPTIM_ISR_CMPM_Msk                                /*!< Compare match */
#define LPTIM_ISR_ARRM_Pos                        (1U)
#define LPTIM_ISR_ARRM_Msk                        (0x1UL<<LPTIM_ISR_ARRM_Pos)                       /*!< 0x00000002 */
#define LPTIM_ISR_ARRM                            LPTIM_ISR_ARRM_Msk                                /*!< Autoreload match */
#define LPTIM_ISR_EXTTRIG_Pos                     (2U)
#define LPTIM_ISR_EXTTRIG_Msk                     (0x1UL<<LPTIM_ISR_EXTTRIG_Pos)                    /*!< 0x00000004 */
#define LPTIM_ISR_EXTTRIG                         LPTIM_ISR_EXTTRIG_Msk                             /*!< External trigger edge event */
#define LPTIM_ISR_CMPOK_Pos                       (3U)
#define LPTIM_ISR_CMPOK_Msk                       (0x1UL<<LPTIM_ISR_CMPOK_Pos)                      /*!< 0x00000008 */
#define LPTIM_ISR_CMPOK                           LPTIM_ISR_CMPOK_Msk                               /*!< Compare register update OK */
#define LPTIM_ISR_ARROK_Pos                       (4U)
#define LPTIM_ISR_ARROK_Msk                       (0x1UL<<LPTIM_ISR_ARROK_Pos)                      /*!< 0x00000010 */
#define LPTIM_ISR_ARROK                           LPTIM_ISR_ARROK_Msk                               /*!< Autoreload register update OK */
#define LPTIM_ISR_UP_Pos                          (5U)
#define LPTIM_ISR_UP_Msk                          (0x1UL<<LPTIM_ISR_UP_Pos)                         /*!< 0x00000020 */
#define LPTIM_ISR_UP                              LPTIM_ISR_UP_Msk                                  /*!< Counter direction change down to up */
#define LPTIM_ISR_DOWN_Pos                        (6U)
#define LPTIM_ISR_DOWN_Msk                        (0x1UL<<LPTIM_ISR_DOWN_Pos)                       /*!< 0x00000040 */
#define LPTIM_ISR_DOWN                            LPTIM_ISR_DOWN_Msk                                /*!< Counter direction change up to down */

/********************************* Bit definition for LPTIM_ICR register ********************************************/
#define LPTIM_ICR_CMPMCF_Pos                      (0U)
#define LPTIM_ICR_CMPMCF_Msk                      (0x1UL<<LPTIM_ICR_CMPMCF_Pos)                     /*!< 0x00000001 */
#define LPTIM_ICR_CMPMCF                          LPTIM_ICR_CMPMCF_Msk                              /*!< Compare match clear flag */
#define LPTIM_ICR_ARRMCF_Pos                      (1U)
#define LPTIM_ICR_ARRMCF_Msk                      (0x1UL<<LPTIM_ICR_ARRMCF_Pos)                     /*!< 0x00000002 */
#define LPTIM_ICR_ARRMCF                          LPTIM_ICR_ARRMCF_Msk                              /*!< Autoreload match clear flag */
#define LPTIM_ICR_EXTTRIGCF_Pos                   (2U)
#define LPTIM_ICR_EXTTRIGCF_Msk                   (0x1UL<<LPTIM_ICR_EXTTRIGCF_Pos)                  /*!< 0x00000004 */
#define LPTIM_ICR_EXTTRIGCF                       LPTIM_ICR_EXTTRIGCF_Msk                           /*!< External trigger valid edge clear flag */
#define LPTIM_ICR_CMPOKCF_Pos                     (3U)
#define LPTIM_ICR_CMPOKCF_Msk                     (0x1UL<<LPTIM_ICR_CMPOKCF_Pos)                    /*!< 0x00000008 */
#define LPTIM_ICR_CMPOKCF                         LPTIM_ICR_CMPOKCF_Msk                             /*!< Compare register update OK clear flag */
#define LPTIM_ICR_ARROKCF_Pos                     (4U)
#define LPTIM_ICR_ARROKCF_Msk                     (0x1UL<<LPTIM_ICR_ARROKCF_Pos)                    /*!< 0x00000010 */
#define LPTIM_ICR_ARROKCF                         LPTIM_ICR_ARROKCF_Msk                             /*!< Autoreload register update OK clear flag */
#define LPTIM_ICR_UPCF_Pos                        (5U)
#define LPTIM_ICR_UPCF_Msk                        (0x1UL<<LPTIM_ICR_UPCF_Pos)                       /*!< 0x00000020 */
#define LPTIM_ICR_UPCF                            LPTIM_ICR_UPCF_Msk                                /*!< Direction change to UP clear flag */
#define LPTIM_ICR_DOWNCF_Pos                      (6U)
#define LPTIM_ICR_DOWNCF_Msk                      (0x1UL<<LPTIM_ICR_DOWNCF_Pos)                     /*!< 0x00000040 */
#define LPTIM_ICR_DOWNCF                          LPTIM_ICR_DOWNCF_Msk                              /*!< Direction change to down clear flag */

/********************************* Bit definition for LPTIM_IER register ********************************************/
#define LPTIM_IER_CMPMIE_Pos                      (0U)
#define LPTIM_IER_CMPMIE_Msk                      (0x1UL<<LPTIM_IER_CMPMIE_Pos)                     /*!< 0x00000001 */
#define LPTIM_IER_CMPMIE                          LPTIM_IER_CMPMIE_Msk                              /*!< Compare match Interrupt Enable */
#define LPTIM_IER_ARRMIE_Pos                      (1U)
#define LPTIM_IER_ARRMIE_Msk                      (0x1UL<<LPTIM_IER_ARRMIE_Pos)                     /*!< 0x00000002 */
#define LPTIM_IER_ARRMIE                          LPTIM_IER_ARRMIE_Msk                              /*!< Autoreload match Interrupt Enable */
#define LPTIM_IER_EXTTRIGIE_Pos                   (2U)
#define LPTIM_IER_EXTTRIGIE_Msk                   (0x1UL<<LPTIM_IER_EXTTRIGIE_Pos)                  /*!< 0x00000004 */
#define LPTIM_IER_EXTTRIGIE                       LPTIM_IER_EXTTRIGIE_Msk                           /*!< External trigger valid edge Interrupt Enable */
#define LPTIM_IER_CMPOKIE_Pos                     (3U)
#define LPTIM_IER_CMPOKIE_Msk                     (0x1UL<<LPTIM_IER_CMPOKIE_Pos)                    /*!< 0x00000008 */
#define LPTIM_IER_CMPOKIE                         LPTIM_IER_CMPOKIE_Msk                             /*!< Compare register update OK Interrupt Enable */
#define LPTIM_IER_ARROKIE_Pos                     (4U)
#define LPTIM_IER_ARROKIE_Msk                     (0x1UL<<LPTIM_IER_ARROKIE_Pos)                    /*!< 0x00000010 */
#define LPTIM_IER_ARROKIE                         LPTIM_IER_ARROKIE_Msk                             /*!< Autoreload register update OK Interrupt Enable */
#define LPTIM_IER_UPIE_Pos                        (5U)
#define LPTIM_IER_UPIE_Msk                        (0x1UL<<LPTIM_IER_UPIE_Pos)                       /*!< 0x00000020 */
#define LPTIM_IER_UPIE                            LPTIM_IER_UPIE_Msk                                /*!< Direction change to UP Interrupt Enable */
#define LPTIM_IER_DOWNIE_Pos                      (6U)
#define LPTIM_IER_DOWNIE_Msk                      (0x1UL<<LPTIM_IER_DOWNIE_Pos)                     /*!< 0x00000040 */
#define LPTIM_IER_DOWNIE                          LPTIM_IER_DOWNIE_Msk                              /*!< Direction change to down Interrupt Enable */

/********************************* Bit definition for LPTIM_CFGR register *******************************************/
#define LPTIM_CFGR_CKSEL_Pos                      (0U)
#define LPTIM_CFGR_CKSEL_Msk                      (0x1UL<<LPTIM_CFGR_CKSEL_Pos)                     /*!< 0x00000001 */
#define LPTIM_CFGR_CKSEL                          LPTIM_CFGR_CKSEL_Msk                              /*!< Clock selector */
#define LPTIM_CFGR_CKPOL_Pos                      (1U)
#define LPTIM_CFGR_CKPOL_Msk                      (0x3UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000006 */
#define LPTIM_CFGR_CKPOL                          LPTIM_CFGR_CKPOL_Msk                              /*!< Clock Polarity */
#define LPTIM_CFGR_CKPOL_0                        (0x1UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000002 */
#define LPTIM_CFGR_CKPOL_1                        (0x2UL<<LPTIM_CFGR_CKPOL_Pos)                     /*!< 0x00000004 */
#define LPTIM_CFGR_CKFLT_Pos                      (3U)
#define LPTIM_CFGR_CKFLT_Msk                      (0x3UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000018 */
#define LPTIM_CFGR_CKFLT                          LPTIM_CFGR_CKFLT_Msk                              /*!< Configurable digital filter for external clock */
#define LPTIM_CFGR_CKFLT_0                        (0x1UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000008 */
#define LPTIM_CFGR_CKFLT_1                        (0x2UL<<LPTIM_CFGR_CKFLT_Pos)                     /*!< 0x00000010 */
#define LPTIM_CFGR_TRGFLT_Pos                     (6U)
#define LPTIM_CFGR_TRGFLT_Msk                     (0x3UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x000000C0 */
#define LPTIM_CFGR_TRGFLT                         LPTIM_CFGR_TRGFLT_Msk                             /*!< Configurable digital filter for trigger */
#define LPTIM_CFGR_TRGFLT_0                       (0x1UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x00000040 */
#define LPTIM_CFGR_TRGFLT_1                       (0x2UL<<LPTIM_CFGR_TRGFLT_Pos)                    /*!< 0x00000080 */
#define LPTIM_CFGR_PRESC_Pos                      (9U)
#define LPTIM_CFGR_PRESC_Msk                      (0x7UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000E00 */
#define LPTIM_CFGR_PRESC                          LPTIM_CFGR_PRESC_Msk                              /*!< Clock prescaler */
#define LPTIM_CFGR_PRESC_0                        (0x1UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000200 */
#define LPTIM_CFGR_PRESC_1                        (0x2UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000400 */
#define LPTIM_CFGR_PRESC_2                        (0x4UL<<LPTIM_CFGR_PRESC_Pos)                     /*!< 0x00000800 */
#define LPTIM_CFGR_TRIGSEL_Pos                    (13U)
#define LPTIM_CFGR_TRIGSEL_Msk                    (0x7UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x0000E000 */
#define LPTIM_CFGR_TRIGSEL                        LPTIM_CFGR_TRIGSEL_Msk                            /*!< Trigger selector */
#define LPTIM_CFGR_TRIGSEL_0                      (0x1UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00002000 */
#define LPTIM_CFGR_TRIGSEL_1                      (0x2UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00004000 */
#define LPTIM_CFGR_TRIGSEL_2                      (0x4UL<<LPTIM_CFGR_TRIGSEL_Pos)                   /*!< 0x00008000 */
#define LPTIM_CFGR_TRIGEN_Pos                     (17U)
#define LPTIM_CFGR_TRIGEN_Msk                     (0x3UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00060000 */
#define LPTIM_CFGR_TRIGEN                         LPTIM_CFGR_TRIGEN_Msk                             /*!< Trigger enable and polarity */
#define LPTIM_CFGR_TRIGEN_0                       (0x1UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00020000 */
#define LPTIM_CFGR_TRIGEN_1                       (0x2UL<<LPTIM_CFGR_TRIGEN_Pos)                    /*!< 0x00040000 */
#define LPTIM_CFGR_TIMOUT_Pos                     (19U)
#define LPTIM_CFGR_TIMOUT_Msk                     (0x1UL<<LPTIM_CFGR_TIMOUT_Pos)                    /*!< 0x00080000 */
#define LPTIM_CFGR_TIMOUT                         LPTIM_CFGR_TIMOUT_Msk                             /*!<  Timeout enable */
#define LPTIM_CFGR_WAVE_Pos                       (20U)
#define LPTIM_CFGR_WAVE_Msk                       (0x1UL<<LPTIM_CFGR_WAVE_Pos)                      /*!< 0x00100000 */
#define LPTIM_CFGR_WAVE                           LPTIM_CFGR_WAVE_Msk                               /*!< Waveform shape */
#define LPTIM_CFGR_WAVPOL_Pos                     (21U)
#define LPTIM_CFGR_WAVPOL_Msk                     (0x1UL<<LPTIM_CFGR_WAVPOL_Pos)                    /*!< 0x00200000 */
#define LPTIM_CFGR_WAVPOL                         LPTIM_CFGR_WAVPOL_Msk                             /*!< Waveform shape polarity */
#define LPTIM_CFGR_PRELOAD_Pos                    (22U)
#define LPTIM_CFGR_PRELOAD_Msk                    (0x1UL<<LPTIM_CFGR_PRELOAD_Pos)                   /*!< 0x00400000 */
#define LPTIM_CFGR_PRELOAD                        LPTIM_CFGR_PRELOAD_Msk                            /*!< Registers update mode */
#define LPTIM_CFGR_COUNTMODE_Pos                  (23U)
#define LPTIM_CFGR_COUNTMODE_Msk                  (0x1UL<<LPTIM_CFGR_COUNTMODE_Pos)                 /*!< 0x00800000 */
#define LPTIM_CFGR_COUNTMODE                      LPTIM_CFGR_COUNTMODE_Msk                          /*!< counter mode enabled */
#define LPTIM_CFGR_ENC_Pos                        (24U)
#define LPTIM_CFGR_ENC_Msk                        (0x1UL<<LPTIM_CFGR_ENC_Pos)                       /*!< 0x01000000 */
#define LPTIM_CFGR_ENC                            LPTIM_CFGR_ENC_Msk                                /*!< Encoder mode enable */

/********************************* Bit definition for LPTIM_CR register *********************************************/
#define LPTIM_CR_ENABLE_Pos                       (0U)
#define LPTIM_CR_ENABLE_Msk                       (0x1UL<<LPTIM_CR_ENABLE_Pos)                      /*!< 0x00000001 */
#define LPTIM_CR_ENABLE                           LPTIM_CR_ENABLE_Msk                               /*!< LPTIM enable */
#define LPTIM_CR_SNGSTRT_Pos                      (1U)
#define LPTIM_CR_SNGSTRT_Msk                      (0x1UL<<LPTIM_CR_SNGSTRT_Pos)                     /*!< 0x00000002 */
#define LPTIM_CR_SNGSTRT                          LPTIM_CR_SNGSTRT_Msk                              /*!< LPTIM start in Single mode */
#define LPTIM_CR_CNTSTRT_Pos                      (2U)
#define LPTIM_CR_CNTSTRT_Msk                      (0x1UL<<LPTIM_CR_CNTSTRT_Pos)                     /*!< 0x00000004 */
#define LPTIM_CR_CNTSTRT                          LPTIM_CR_CNTSTRT_Msk                              /*!< Timer start in Continuous mode */
#define LPTIM_CR_COUNTRST_Pos                     (3U)
#define LPTIM_CR_COUNTRST_Msk                     (0x1UL<<LPTIM_CR_COUNTRST_Pos)                    /*!< 0x00000008 */
#define LPTIM_CR_COUNTRST                         LPTIM_CR_COUNTRST_Msk                             /*!< Counter reset */
#define LPTIM_CR_RSTARE_Pos                       (4U)
#define LPTIM_CR_RSTARE_Msk                       (0x1UL<<LPTIM_CR_RSTARE_Pos)                      /*!< 0x00000010 */
#define LPTIM_CR_RSTARE                           LPTIM_CR_RSTARE_Msk                               /*!< Reset after read enable */

/********************************* Bit definition for LPTIM_CMP register ********************************************/
#define LPTIM_CMP_CMP_Pos                         (0U)
#define LPTIM_CMP_CMP_Msk                         (0xFFFFFFFFUL<<LPTIM_CMP_CMP_Pos)                 /*!< 0xFFFFFFFF */
#define LPTIM_CMP_CMP                             LPTIM_CMP_CMP_Msk                                 /*!< Compare value */

/********************************* Bit definition for LPTIM_ARR register ********************************************/
#define LPTIM_ARR_ARR_Pos                         (0U)
#define LPTIM_ARR_ARR_Msk                         (0xFFFFFFFFUL<<LPTIM_ARR_ARR_Pos)                 /*!< 0xFFFFFFFF */
#define LPTIM_ARR_ARR                             LPTIM_ARR_ARR_Msk                                 /*!< Auto reload value */

/********************************* Bit definition for LPTIM_CNT register ********************************************/
#define LPTIM_CNT_CNT_Pos                         (0U)
#define LPTIM_CNT_CNT_Msk                         (0xFFFFFFFFUL<<LPTIM_CNT_CNT_Pos)                 /*!< 0xFFFFFFFF */
#define LPTIM_CNT_CNT                             LPTIM_CNT_CNT_Msk                                 /*!< Counter value */

/********************************* Bit definition for LPTIM_OR register *********************************************/
#define LPTIM_OR_IN1_Pos                          (0U)
#define LPTIM_OR_IN1_Msk                          (0xDUL<<LPTIM_OR_IN1_Pos)                         /*!< 0x0000000D */
#define LPTIM_OR_IN1                              LPTIM_OR_IN1_Msk                                  /*!< LPTimer input 1 remap extension */
#define LPTIM_OR_IN1_0                            (0x1UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000001 */
#define LPTIM_OR_IN1_1                            (0x4UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000004 */
#define LPTIM_OR_IN1_2                            (0x8UL<<LPTIM_OR_IN1_Pos)                         /*!< 0x00000008 */
#define LPTIM_OR_IN2_Pos                          (1U)
#define LPTIM_OR_IN2_Msk                          (0x19UL<<LPTIM_OR_IN2_Pos)                         /*!< 0x00000032 */
#define LPTIM_OR_IN2                              LPTIM_OR_IN2_Msk                                  /*!< LPTimer input 2 remap extension */
#define LPTIM_OR_IN2_0                            (0x1UL<<LPTIM_OR_IN2_Pos)                         /*!< 0x00000002 */
#define LPTIM_OR_IN2_1                            (0x8UL<<LPTIM_OR_IN2_Pos)                         /*!< 0x00000010 */
#define LPTIM_OR_IN2_2                            (0x10UL<<LPTIM_OR_IN2_Pos)                        /*!< 0x00000020 */

// /******************************************************************************/
// /*                                                                            */
// /*                      Analog Comparators (COMP)                             */
// /*                                                                            */
// /******************************************************************************/
// /**********************  Bit definition for COMP_CCSR register  ****************/
// #define COMP_CCSR_EN_Pos             (0U)
// #define COMP_CCSR_EN_Msk             (0x1UL << COMP_CCSR_EN_Pos)                    /*!< 0x00000001 */
// #define COMP_CCSR_EN                 COMP_CCSR_EN_Msk                               /*!< Comparator enable */
// #define COMP_CCSR_COMP1_EN           COMP_CCSR_EN
// #define COMP_CCSR_COMP2_EN           COMP_CCSR_EN

// #define COMP_CCSR_INMSEL_Pos         (5U)
// #define COMP_CCSR_INMSEL_Msk         (0x7UL << COMP_CCSR_INMSEL_Pos)                /*!< 0x000000F0 */
// #define COMP_CCSR_INMSEL             COMP_CCSR_INMSEL_Msk                           /*!< Comparator input minus selection */
// #define COMP_CCSR_INMSEL_0           (0x1UL << COMP_CCSR_INMSEL_Pos)                /*!< 0x00000020 */
// #define COMP_CCSR_INMSEL_1           (0x2UL << COMP_CCSR_INMSEL_Pos)                /*!< 0x00000040 */
// #define COMP_CCSR_INMSEL_2           (0x4UL << COMP_CCSR_INMSEL_Pos)                /*!< 0x00000080 */

// #define COMP_CCSR_INPSEL_Pos         (8U)
// #define COMP_CCSR_INPSEL_Msk         (0x7UL << COMP_CCSR_INPSEL_Pos)                /*!< 0x00000700 */
// #define COMP_CCSR_INPSEL             COMP_CCSR_INPSEL_Msk                           /*!< Comparator plus minus selection */
// #define COMP_CCSR_INPSEL_0           (0x1UL << COMP_CCSR_INPSEL_Pos)                /*!< 0x00000100 */
// #define COMP_CCSR_INPSEL_1           (0x2UL << COMP_CCSR_INPSEL_Pos)                /*!< 0x00000200 */
// #define COMP_CCSR_INPSEL_2           (0x4UL << COMP_CCSR_INPSEL_Pos)                /*!< 0x00000400 */

// #define COMP_CCSR_WINMODE_Pos        (11U)
// #define COMP_CCSR_WINMODE_Msk        (0x1UL << COMP_CCSR_WINMODE_Pos)               /*!< 0x00000800 */
// #define COMP_CCSR_WINMODE            COMP_CCSR_WINMODE_Msk                          /*!< Pair of comparators window mode. Bit intended to be used with COMP common instance (COMP_Common_TypeDef) */

// #define COMP_CCSR_POLARITY_Pos       (15U)
// #define COMP_CCSR_POLARITY_Msk       (0x1UL << COMP_CCSR_POLARITY_Pos)              /*!< 0x00008000 */
// #define COMP_CCSR_POLARITY           COMP_CCSR_POLARITY_Msk                         /*!< Comparator output polarity */

// #define COMP_CCSR_HYST_Pos           (16U)
// #define COMP_CCSR_HYST_Msk           (0x1UL << COMP_CCSR_HYST_Pos)                  /*!< 0x00010000 */
// #define COMP_CCSR_HYST               COMP_CCSR_HYST_Msk                             /*!< Comparator hysteresis enable */

// #define COMP_CCSR_PWRMODE_Pos        (18U)
// #define COMP_CCSR_PWRMODE_Msk        (0x3UL << COMP_CCSR_PWRMODE_Pos)               /*!< 0x000C0000 */
// #define COMP_CCSR_PWRMODE            COMP_CCSR_PWRMODE_Msk                          /*!< Comparator power mode */
// #define COMP_CCSR_PWRMODE_0          (0x1UL << COMP_CCSR_PWRMODE_Pos)               /*!< 0x00040000 */
// #define COMP_CCSR_PWRMODE_1          (0x2UL << COMP_CCSR_PWRMODE_Pos)               /*!< 0x00080000 */

// #define COMP_CCSR_VCDIV_Pos          (20U)
// #define COMP_CCSR_VCDIV_Msk          (0x3FUL << COMP_CCSR_VCDIV_Pos)                /*!< 0x003F0000 */
// #define COMP_CCSR_VCDIV              COMP_CCSR_VCDIV_Msk                            /*!< Comparator vcdiv */
// #define COMP_CCSR_VCDIV_0            (0x1UL << COMP_CCSR_VCDIV_Pos)                 /*!< 0x00010000 */
// #define COMP_CCSR_VCDIV_1            (0x2UL << COMP_CCSR_VCDIV_Pos)                 /*!< 0x00020000 */
// #define COMP_CCSR_VCDIV_2            (0x4UL << COMP_CCSR_VCDIV_Pos)                 /*!< 0x00040000 */
// #define COMP_CCSR_VCDIV_3            (0x8UL << COMP_CCSR_VCDIV_Pos)                 /*!< 0x00080000 */
// #define COMP_CCSR_VCDIV_4            (0x10UL << COMP_CCSR_VCDIV_Pos)                /*!< 0x00100000 */
// #define COMP_CCSR_VCDIV_5            (0x20UL << COMP_CCSR_VCDIV_Pos)                /*!< 0x00200000 */

// #define COMP_CCSR_VCDIV_EN_Pos       (26U)
// #define COMP_CCSR_VCDIV_EN_Msk       (0x1UL << COMP_CCSR_VCDIV_EN_Pos)              /*!< 0x04000000 */
// #define COMP_CCSR_VCDIV_EN           COMP_CCSR_VCDIV_EN_Msk                         /*!< Comparator vcdiv en */

// #define COMP_CCSR_VCSEL_Pos          (27U)
// #define COMP_CCSR_VCSEL_Msk          (0x1UL << COMP_CCSR_VCSEL_Pos)                 /*!< 0x08000000 */
// #define COMP_CCSR_VCSEL              COMP_CCSR_VCSEL_Msk                            /*!< Comparator vcsel */

// #define COMP_CCSR_COMP_OUT_Pos       (30U)
// #define COMP_CCSR_COMP_OUT_Msk       (0x1UL << COMP_CCSR_COMP_OUT_Pos)              /*!< 0x40000000 */
// #define COMP_CCSR_COMP_OUT           COMP_CCSR_COMP_OUT_Msk

// #define COMP_CCSR_LOCK_Pos           (31U)
// #define COMP_CCSR_LOCK_Msk           (0x1UL << COMP_CCSR_LOCK_Pos)                  /*!< 0x80000000 */
// #define COMP_CCSR_LOCK               COMP_CCSR_LOCK_Msk                             /*!< Comparator lock */

// /**********************  Bit definition for COMP_FR register  ****************/
// #define COMP_FR_FLTEN_Pos             (0U)
// #define COMP_FR_FLTEN_Msk             (0x1UL << COMP_FR_FLTEN_Pos)                /*!< 0x00000001 */
// #define COMP_FR_FLTEN                 COMP_FR_FLTEN_Msk                           /*!< Comparator filter enable */
// #define COMP_FR_FLTCNT_Pos            (16U)
// #define COMP_FR_FLTCNT_Msk            (0xFFFFUL << COMP_FR_FLTCNT_Pos)            /*!< 0xFFFF0000 */
// #define COMP_FR_FLTCNT                COMP_FR_FLTCNT_Msk                          /*!< Comparator filter counter */

// /******************************************************************************/
// /*                                                                            */
// /*                      Operational amplifier (OPA)                           */
// /*                                                                            */
// /******************************************************************************/
// /**********************  Bit definition for OPA_OENR register  ****************/
// #define OPA_OENR_OPA1OEN_Pos            (1U)
// #define OPA_OENR_OPA1OEN_Msk            (0x1UL << OPA_OENR_OPA1OEN_Pos)                 /*!< 0x00000002 */
// #define OPA_OENR_OPA1OEN                OPA_OENR_OPA1OEN_Msk                            /*!< OPA1 output enable */
// #define OPA_OENR_OPA2OEN_Pos            (6U)
// #define OPA_OENR_OPA2OEN_Msk            (0x1UL << OPA_OENR_OPA2OEN_Pos)                 /*!< 0x00000040 */
// #define OPA_OENR_OPA2OEN                OPA_OENR_OPA2OEN_Msk                            /*!< OPA2 output enable */

// /**********************  Bit definition for OPA_CR register  ****************/
// #define OPA_CR_OPA1EN_Pos             (5U)
// #define OPA_CR_OPA1EN_Msk             (0x1UL << OPA_CR_OPA1EN_Pos)                      /*!< 0x00000020 */
// #define OPA_CR_OPA1EN                 OPA_CR_OPA1EN_Msk                                 /*!< OPA1 enable */
// #define OPA_CR_OPA2EN_Pos             (6U)
// #define OPA_CR_OPA2EN_Msk             (0x1UL << OPA_CR_OPA2EN_Pos)                      /*!< 0x00000040 */
// #define OPA_CR_OPA2EN                 OPA_CR_OPA2EN_Msk                                 /*!< OPA2 enable */

// /******************************************************************************/
// /*                                                                            */
// /*                      Digital Co-process Operation (CORDIC)                 */
// /*                                                                            */
// /******************************************************************************/
// /*******************  Bit definition for CORDIC_CR register  *********************/
// #define CORDIC_CR_ITERATION_Pos                 (0U)
// #define CORDIC_CR_ITERATION_Msk                 (0x1FUL << CORDIC_CR_ITERATION_Pos)
// #define CORDIC_CR_ITERATION                     CORDIC_CR_ITERATION_Msk

// #define CORDIC_CR_CORDIC_INT_Pos                (5U)
// #define CORDIC_CR_CORDIC_INT_Msk                (0x1UL << CORDIC_CR_CORDIC_INT_Pos)
// #define CORDIC_CR_CORDIC_INT                    CORDIC_CR_CORDIC_INT_Msk

// #define CORDIC_CR_SQRT_INT_Pos                  (6U)
// #define CORDIC_CR_SQRT_INT_Msk                  (0x1UL << CORDIC_CR_SQRT_INT_Pos)
// #define CORDIC_CR_SQRT_INT                      CORDIC_CR_SQRT_INT_Msk

// #define CORDIC_CR_CORDIC_MODE_Pos               (7U)
// #define CORDIC_CR_CORDIC_MODE_Msk               (0x1UL << CORDIC_CR_CORDIC_MODE_Pos)
// #define CORDIC_CR_CORDIC_MODE                   CORDIC_CR_CORDIC_MODE_Msk

// #define CORDIC_CR_START_Pos                     (8U)
// #define CORDIC_CR_START_Msk                     (0x1UL << CORDIC_CR_START_Pos)
// #define CORDIC_CR_START                         CORDIC_CR_START_Msk

// #define CORDIC_CR_START_MODE_Pos                (9U)
// #define CORDIC_CR_START_MODE_Msk                (0x1UL << CORDIC_CR_START_MODE_Pos)
// #define CORDIC_CR_START_MODE                    CORDIC_CR_START_MODE_Msk

// #define CORDIC_CR_VECSIZE_Pos                   (20U)
// #define CORDIC_CR_VECSIZE_Msk                   (0x1UL << CORDIC_CR_VECSIZE_Pos)
// #define CORDIC_CR_VECSIZE                       CORDIC_CR_VECSIZE_Msk

// #define CORDIC_CR_RESSIZE_Pos                   (21U)
// #define CORDIC_CR_RESSIZE_Msk                   (0x1UL << CORDIC_CR_RESSIZE_Pos)
// #define CORDIC_CR_RESSIZE                       CORDIC_CR_RESSIZE_Msk

// #define CORDIC_CR_ARGSIZE_Pos                   (22U)
// #define CORDIC_CR_ARGSIZE_Msk                   (0x1UL << CORDIC_CR_ARGSIZE_Pos)
// #define CORDIC_CR_ARGSIZE                       CORDIC_CR_ARGSIZE_Msk

// #define CORDIC_CR_CORDIC_INT_MASK_Pos           (28U)
// #define CORDIC_CR_CORDIC_INT_MASK_Msk           (0x1UL << CORDIC_CR_CORDIC_INT_MASK_Pos)
// #define CORDIC_CR_CORDIC_INT_MASK               CORDIC_CR_CORDIC_INT_MASK_Msk

// #define CORDIC_CR_SQRT_INT_MASK_Pos             (29U)
// #define CORDIC_CR_SQRT_INT_MASK_Msk             (0x1UL << CORDIC_CR_SQRT_INT_MASK_Pos)
// #define CORDIC_CR_SQRT_INT_MASK                 CORDIC_CR_SQRT_INT_MASK_Msk

// #define CORDIC_CR_CORDIC_ERROR_INT_MASK_Pos     (30U)
// #define CORDIC_CR_CORDIC_ERROR_INT_MASK_Msk     (0x1UL << CORDIC_CR_CORDIC_ERROR_INT_MASK_Pos)
// #define CORDIC_CR_CORDIC_ERROR_INT_MASK         CORDIC_CR_CORDIC_ERROR_INT_MASK_Msk

// #define CORDIC_CR_ARCTAN_MOD_OV_MASK_Pos        (31U)
// #define CORDIC_CR_ARCTAN_MOD_OV_MASK_Msk        (0x1UL << CORDIC_CR_ARCTAN_MOD_OV_MASK_Pos)
// #define CORDIC_CR_ARCTAN_MOD_OV_MASK            CORDIC_CR_ARCTAN_MOD_OV_MASK_Msk



// /*******************  Bit definition for CORDIC_THETA register  *********************/
// #define CORDIC_THETA_THETA_Pos        (0U)
// #define CORDIC_THETA_THETA_Msk        (0xFFFFFFFFUL << CORDIC_THETA_THETA_Pos)  /*!< 0xFFFFFFFF */
// #define CORDIC_THETA_THETA            CORDIC_THETA_THETA_Msk                    /*!< Cordic SIN/COS input theta register bits */

// /*******************  Bit definition for CORDIC_SIN register  ***************  ******/
// #define CORDIC_SIN_SIN_Pos            (0U)
// #define CORDIC_SIN_SIN_Msk            (0xFFFFFFFFUL << CORDIC_SIN_SIN_Pos)      /*!< 0xFFFFFFFF */
// #define CORDIC_SIN_SIN                CORDIC_SIN_SIN_Msk                        /*!< Cordic SIN result register bits */

// /*******************  Bit definition for CORDIC_COS register  ***************  ******/
// #define CORDIC_COS_COS_Pos            (0U)
// #define CORDIC_COS_COS_Msk            (0xFFFFFFFFUL << CORDIC_COS_COS_Pos)      /*!< 0xFFFFFFFF */
// #define CORDIC_COS_COS                CORDIC_COS_COS_Msk                        /*!< Cordic COS result register bits */

// /*******************  Bit definition for CORDIC_X register  *****************  ****/
// #define CORDIC_X_X_Pos                (0U)
// #define CORDIC_X_X_Msk                (0xFFFFFFFFUL << CORDIC_X_X_Pos)          /*!< 0xFFFF0000 */
// #define CORDIC_X_X                    CORDIC_X_X_Msk                            /*!< Cordic arctan/modult X register bits */

// /*******************  Bit definition for CORDIC_Y register  *****************  ****/
// #define CORDIC_Y_Y_Pos                (0U)
// #define CORDIC_Y_Y_Msk                (0xFFFFFFFFUL << CORDIC_Y_Y_Pos)         /*!< 0xFFFFFFFF */
// #define CORDIC_Y_Y                    CORDIC_Y_Y_Msk                           /*!< Cordic arctan/modult Y register bits */

// /*******************  Bit definition for CORDIC_MOD register  ***************  ******/
// #define CORDIC_MOD_MOD_Pos            (0U)
// #define CORDIC_MOD_MOD_Msk            (0xFFFFFFFFUL << CORDIC_MOD_MOD_Pos)     /*!< 0xFFFFFFFF */
// #define CORDIC_MOD_MOD                CORDIC_MOD_MOD_Msk                       /*!< Cordic artan sqrt result register bits */

// /*******************  Bit definition for CORDIC_ARCTAN register  *********************/
// #define CORDIC_ARCTAN_ARCTAN_Pos      (0U)
// #define CORDIC_ARCTAN_ARCTAN_Msk      (0xFFFFFFFFUL << CORDIC_ARCTAN_ARCTAN_Pos) /*!< 0xFFFFFFFF */
// #define CORDIC_ARCTAN_ARCTAN          CORDIC_ARCTAN_ARCTAN_Msk                   /*!< Cordic arctan theta register bits */

// /*******************  Bit definition for DSP_RAD register  *********************/
// #define CORDIC_DSP_RAD_RAD_Pos        (0U)
// #define CORDIC_DSP_RAD_RAD_Msk        (0xFFFFFFFFUL << CORDIC_DSP_RAD_RAD_Pos)   /*!< 0xFFFFFFFF */
// #define CORDIC_DSP_RAD_RAD            CORDIC_DSP_RAD_RAD_Msk                     /*!< Cordic RAD register bits */

// /*******************  Bit definition for DSP_SQRT register  *********************/
// #define CORDIC_DSP_SQRT_SQRT_Pos      (0U)
// #define CORDIC_DSP_SQRT_SQRT_Msk      (0xFFFFUL << CORDIC_DSP_SQRT_SQRT_Pos)     /*!< 0x0000FFFF */
// #define CORDIC_DSP_SQRT_SQRT          CORDIC_DSP_SQRT_SQRT_Msk                   /*!< Cordic sqrt register bits */

// /*******************  Bit definition for CORDIC_SR register  *********************/
// #define CORDIC_SR_CCFF_Pos            (0U)
// #define CORDIC_SR_CCFF_Msk            (0x1UL << CORDIC_SR_CCFF_Pos)          /*!< 0x00000001 */
// #define CORDIC_SR_CCFF                CORDIC_SR_CCFF_Msk                     /*!< Cordic result interrupt generate flag */

// #define CORDIC_SR_SCFF_Pos            (1U)
// #define CORDIC_SR_SCFF_Msk            (0x1UL << CORDIC_SR_SCFF_Pos)
// #define CORDIC_SR_SCFF                CORDIC_SR_SCFF_Msk

// #define CORDIC_SR_CCEF_Pos            (2U)
// #define CORDIC_SR_CCEF_Msk            (0x1UL << CORDIC_SR_CCEF_Pos)
// #define CORDIC_SR_CCEF                CORDIC_SR_CCEF_Msk

// #define CORDIC_SR_ACEF_Pos            (3U)
// #define CORDIC_SR_ACEF_Msk            (0x1UL << CORDIC_SR_ACEF_Pos)
// #define CORDIC_SR_ACEF                CORDIC_SR_ACEF_Msk

// #define CORDIC_SR_BSY_Pos             (4U)
// #define CORDIC_SR_BSY_Msk             (0x1UL << CORDIC_SR_BSY_Pos)
// #define CORDIC_SR_BSY                 CORDIC_SR_BSY_Msk

// /******************************************************************************/
// /*                                                                            */
// /*                           Hardware Divider(HDIV)                           */
// /*                                                                            */
// /******************************************************************************/
// /*******************  Bit definition for HDIV_DEND register  ********************/
// #define HDIV_DEND_DEND_Pos              (0U)
// #define HDIV_DEND_DEND_Msk              (0xFFFFFFFFUL << HDIV_DEND_DEND_Pos)     /*!< 0xFFFFFFFF */
// #define HDIV_DEND_DEND                  HDIV_DEND_DEND_Msk

// /*******************  Bit definition for HDIV_SOR register  *********************/
// #define HDIV_SOR_SOR_Pos                (0U)
// #define HDIV_SOR_SOR_Msk                (0xFFFFFFFFUL << HDIV_SOR_SOR_Pos)       /*!< 0xFFFFFFFF */
// #define HDIV_SOR_SOR                    HDIV_SOR_SOR_Msk

// /*******************  Bit definition for HDIV_QUOT register  ********************/
// #define HDIV_QUOT_QUOT_Pos              (0U)
// #define HDIV_QUOT_QUOT_Msk              (0xFFFFFFFFUL << HDIV_QUOT_QUOT_Pos)     /*!< 0xFFFFFFFF */
// #define HDIV_QUOT_QUOT                  HDIV_QUOT_QUOT_Msk

// /*******************  Bit definition for HDIV_REMD register  ********************/
// #define HDIV_REMD_REMD_Pos              (0U)
// #define HDIV_REMD_REMD_Msk              (0xFFFFFFFFUL << HDIV_REMD_REMD_Pos)     /*!< 0xFFFFFFFF */
// #define HDIV_REMD_REMD                  HDIV_REMD_REMD_Msk

// /*******************  Bit definition for HDIV_SIGN register  ********************/
// #define HDIV_SIGN_SIGN_Pos              (0U)
// #define HDIV_SIGN_SIGN_Msk              (0x1UL << HDIV_SIGN_SIGN_Pos)            /*!< 0x00000001 */
// #define HDIV_SIGN_SIGN                  HDIV_SIGN_SIGN_Msk

// /*******************  Bit definition for HDIV_STAT register  ********************/
// #define HDIV_STAT_END_Pos               (0U)
// #define HDIV_STAT_END_Msk               (0x1UL << HDIV_STAT_END_Pos)            /*!< 0x00000001 */
// #define HDIV_STAT_END                   HDIV_STAT_END_Msk
// #define HDIV_STAT_ZERO_Pos              (1U)
// #define HDIV_STAT_ZERO_Msk              (0x1UL << HDIV_STAT_ZERO_Pos)           /*!< 0x00000002 */
// #define HDIV_STAT_ZERO                  HDIV_STAT_ZERO_Msk

/********************************************************************************************************************/
/********************************* UART *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for UART_DR register **********************************************/
#define UART_DR_DR_Pos                            (0U)
#define UART_DR_DR_Msk                            (0x1FFUL<<UART_DR_DR_Pos)                         /*!< 0x000001FF */
#define UART_DR_DR                                UART_DR_DR_Msk                                    /*!< Receive/send data register */

/********************************* Bit definition for UART_BRR register *********************************************/
#define UART_BRR_BRR_Pos                          (0U)
#define UART_BRR_BRR_Msk                          (0xFFFFUL<<UART_BRR_BRR_Pos)                      /*!< 0x0000FFFF */
#define UART_BRR_BRR                              UART_BRR_BRR_Msk                                  /*!< Baud rate register */

/********************************* Bit definition for UART_SR register **********************************************/
#define UART_SR_RXNE_Pos                          (0U)
#define UART_SR_RXNE_Msk                          (0x1UL<<UART_SR_RXNE_Pos)                         /*!< 0x00000001 */
#define UART_SR_RXNE                              UART_SR_RXNE_Msk                                  /*!< Receive register not empty */
#define UART_SR_ORE_Pos                           (1U)
#define UART_SR_ORE_Msk                           (0x1UL<<UART_SR_ORE_Pos)                          /*!< 0x00000002 */
#define UART_SR_ORE                               UART_SR_ORE_Msk                                   /*!< Overrun error bit */
#define UART_SR_PE_Pos                            (2U)
#define UART_SR_PE_Msk                            (0x1UL<<UART_SR_PE_Pos)                           /*!< 0x00000004 */
#define UART_SR_PE                                UART_SR_PE_Msk                                    /*!< Parity Error bit */
#define UART_SR_FE_Pos                            (3U)
#define UART_SR_FE_Msk                            (0x1UL<<UART_SR_FE_Pos)                           /*!< 0x00000008 */
#define UART_SR_FE                                UART_SR_FE_Msk                                    /*!< Framing Error bit */
#define UART_SR_BRI_Pos                           (4U)
#define UART_SR_BRI_Msk                           (0x1UL<<UART_SR_BRI_Pos)                          /*!< 0x00000010 */
#define UART_SR_BRI                               UART_SR_BRI_Msk                                   /*!< Break Interrupt bit */
#define UART_SR_TDRE_Pos                          (5U)
#define UART_SR_TDRE_Msk                          (0x1UL<<UART_SR_TDRE_Pos)                         /*!< 0x00000020 */
#define UART_SR_TDRE                              UART_SR_TDRE_Msk                                  /*!< Transmit Holding Register Empty bit */
#define UART_SR_TXE_Pos                           (6U)
#define UART_SR_TXE_Msk                           (0x1UL<<UART_SR_TXE_Pos)                          /*!< 0x00000040 */
#define UART_SR_TXE                               UART_SR_TXE_Msk                                   /*!< Transmitter Empty bit */
#define UART_SR_ADDR_RCVD_Pos                     (8U)
#define UART_SR_ADDR_RCVD_Msk                     (0x1UL<<UART_SR_ADDR_RCVD_Pos)                    /*!< 0x00000100 */
#define UART_SR_ADDR_RCVD                         UART_SR_ADDR_RCVD_Msk                             /*!< Address Received Bit */
#define UART_SR_BUSY_Pos                          (9U)
#define UART_SR_BUSY_Msk                          (0x1UL<<UART_SR_BUSY_Pos)                         /*!< 0x00000200 */
#define UART_SR_BUSY                              UART_SR_BUSY_Msk                                  /*!< UART Busy */
#define UART_SR_BUSY_ERR_Pos                      (10U)
#define UART_SR_BUSY_ERR_Msk                      (0x1UL<<UART_SR_BUSY_ERR_Pos)                     /*!< 0x00000400 */
#define UART_SR_BUSY_ERR                          UART_SR_BUSY_ERR_Msk                              /*!< Busy Detect Error */

/********************************* Bit definition for UART_CR1 register *********************************************/
#define UART_CR1_M_Pos                            (0U)
#define UART_CR1_M_Msk                            (0x3UL<<UART_CR1_M_Pos)                           /*!< 0x00000003 */
#define UART_CR1_M                                UART_CR1_M_Msk                                    /*!< Data Length Select */
#define UART_CR1_M_0                              (0x1UL<<UART_CR1_M_Pos)                           /*!< 0x00000001 */
#define UART_CR1_M_1                              (0x2UL<<UART_CR1_M_Pos)                           /*!< 0x00000002 */
#define UART_CR1_STOP_Pos                         (2U)
#define UART_CR1_STOP_Msk                         (0x1UL<<UART_CR1_STOP_Pos)                        /*!< 0x00000004 */
#define UART_CR1_STOP                             UART_CR1_STOP_Msk                                 /*!< Number of stop bits */
#define UART_CR1_PCE_Pos                          (3U)
#define UART_CR1_PCE_Msk                          (0x1UL<<UART_CR1_PCE_Pos)                         /*!< 0x00000008 */
#define UART_CR1_PCE                              UART_CR1_PCE_Msk                                  /*!< Parity Enable */
#define UART_CR1_PS_Pos                           (4U)
#define UART_CR1_PS_Msk                           (0x1UL<<UART_CR1_PS_Pos)                          /*!< 0x00000010 */
#define UART_CR1_PS                               UART_CR1_PS_Msk                                   /*!< Even Parity Select */
#define UART_CR1_SP_Pos                           (5U)
#define UART_CR1_SP_Msk                           (0x1UL<<UART_CR1_SP_Pos)                          /*!< 0x00000020 */
#define UART_CR1_SP                               UART_CR1_SP_Msk                                   /*!< Stick Parity */
#define UART_CR1_SBK_Pos                          (6U)
#define UART_CR1_SBK_Msk                          (0x1UL<<UART_CR1_SBK_Pos)                         /*!< 0x00000040 */
#define UART_CR1_SBK                              UART_CR1_SBK_Msk                                  /*!< Send Break */
#define UART_CR1_SWAP_Pos                         (8U)
#define UART_CR1_SWAP_Msk                         (0x1UL<<UART_CR1_SWAP_Pos)                        /*!< 0x00000100 */
#define UART_CR1_SWAP                             UART_CR1_SWAP_Msk                                 /*!< TX/RX pin swap */
#define UART_CR1_MSBFIRST_Pos                     (9U)
#define UART_CR1_MSBFIRST_Msk                     (0x1UL<<UART_CR1_MSBFIRST_Pos)                    /*!< 0x00000200 */
#define UART_CR1_MSBFIRST                         UART_CR1_MSBFIRST_Msk                             /*!< MSB first mode */

/********************************* Bit definition for UART_CR2 register *********************************************/
#define UART_CR2_RXNEIE_Pos                       (0U)
#define UART_CR2_RXNEIE_Msk                       (0x1UL<<UART_CR2_RXNEIE_Pos)                      /*!< 0x00000001 */
#define UART_CR2_RXNEIE                           UART_CR2_RXNEIE_Msk                               /*!< Enable Received Data Available Interrupt */
#define UART_CR2_TDREIE_Pos                       (1U)
#define UART_CR2_TDREIE_Msk                       (0x1UL<<UART_CR2_TDREIE_Pos)                      /*!< 0x00000002 */
#define UART_CR2_TDREIE                           UART_CR2_TDREIE_Msk                               /*!< Enable Transmit Holding Register Empty Interrupt */
#define UART_CR2_LSIE_Pos                         (2U)
#define UART_CR2_LSIE_Msk                         (0x1UL<<UART_CR2_LSIE_Pos)                        /*!< 0x00000004 */
#define UART_CR2_LSIE                             UART_CR2_LSIE_Msk                                 /*!< Enable Receiver Line Status Interrupt */
#define UART_CR2_BUSYERRIE_Pos                    (3U)
#define UART_CR2_BUSYERRIE_Msk                    (0x1UL<<UART_CR2_BUSYERRIE_Pos)                   /*!< 0x00000008 */
#define UART_CR2_BUSYERRIE                        UART_CR2_BUSYERRIE_Msk                            /*!< Enable Busyerr state interruption */
#define UART_CR2_TXEIE_Pos                        (4U)
#define UART_CR2_TXEIE_Msk                        (0x1UL<<UART_CR2_TXEIE_Pos)                       /*!< 0x00000010 */
#define UART_CR2_TXEIE                            UART_CR2_TXEIE_Msk                                /*!< Enable tx empty interruption */

/********************************* Bit definition for UART_CR3 register *********************************************/
#define UART_CR3_M_E_Pos                          (0U)
#define UART_CR3_M_E_Msk                          (0x1UL<<UART_CR3_M_E_Pos)                         /*!< 0x00000001 */
#define UART_CR3_M_E                              UART_CR3_M_E_Msk                                  /*!< Extension for DLS */
#define UART_CR3_ADDR_MATCH_Pos                   (1U)
#define UART_CR3_ADDR_MATCH_Msk                   (0x1UL<<UART_CR3_ADDR_MATCH_Pos)                  /*!< 0x00000002 */
#define UART_CR3_ADDR_MATCH                       UART_CR3_ADDR_MATCH_Msk                           /*!<  Address Match Mode */
#define UART_CR3_SEND_ADDR_Pos                    (2U)
#define UART_CR3_SEND_ADDR_Msk                    (0x1UL<<UART_CR3_SEND_ADDR_Pos)                   /*!< 0x00000004 */
#define UART_CR3_SEND_ADDR                        UART_CR3_SEND_ADDR_Msk                            /*!< Send address control bit */
#define UART_CR3_TX_MODE_Pos                      (3U)
#define UART_CR3_TX_MODE_Msk                      (0x1UL<<UART_CR3_TX_MODE_Pos)                     /*!< 0x00000008 */
#define UART_CR3_TX_MODE                          UART_CR3_TX_MODE_Msk                              /*!< Transmit mode control bit */
#define UART_CR3_DMAR_Pos                         (4U)
#define UART_CR3_DMAR_Msk                         (0x1UL<<UART_CR3_DMAR_Pos)                        /*!< 0x00000010 */
#define UART_CR3_DMAR                             UART_CR3_DMAR_Msk                                 /*!< DMA receive enable */
#define UART_CR3_DMAT_Pos                         (5U)
#define UART_CR3_DMAT_Msk                         (0x1UL<<UART_CR3_DMAT_Pos)                        /*!< 0x00000020 */
#define UART_CR3_DMAT                             UART_CR3_DMAT_Msk                                 /*!< DMA transmit enable */
#define UART_CR3_DMA_SOFT_ACK_Pos                 (6U)
#define UART_CR3_DMA_SOFT_ACK_Msk                 (0x1UL<<UART_CR3_DMA_SOFT_ACK_Pos)                /*!< 0x00000040 */
#define UART_CR3_DMA_SOFT_ACK                     UART_CR3_DMA_SOFT_ACK_Msk                         /*!< DMA software ack */

/********************************* Bit definition for UART_RAR register *********************************************/
#define UART_RAR_RAR_Pos                          (0U)
#define UART_RAR_RAR_Msk                          (0xFFUL<<UART_RAR_RAR_Pos)                        /*!< 0x000000FF */
#define UART_RAR_RAR                              UART_RAR_RAR_Msk                                  /*!< Receive address matching register */

/********************************* Bit definition for UART_TAR register *********************************************/
#define UART_TAR_TAR_Pos                          (0U)
#define UART_TAR_TAR_Msk                          (0xFFUL<<UART_TAR_TAR_Pos)                        /*!< 0x000000FF */
#define UART_TAR_TAR                              UART_TAR_TAR_Msk                                  /*!< Transmit address matching register */

/********************************* Bit definition for UART_BRRF register ********************************************/
#define UART_BRRF_BRRF_Pos                        (0U)
#define UART_BRRF_BRRF_Msk                        (0xFUL<<UART_BRRF_BRRF_Pos)                       /*!< 0x0000000F */
#define UART_BRRF_BRRF                            UART_BRRF_BRRF_Msk                                /*!< Baud rate fractional register */
#define UART_BRRF_BRRF_0                          (0x1UL<<UART_BRRF_BRRF_Pos)                       /*!< 0x00000001 */
#define UART_BRRF_BRRF_1                          (0x2UL<<UART_BRRF_BRRF_Pos)                       /*!< 0x00000002 */
#define UART_BRRF_BRRF_2                          (0x4UL<<UART_BRRF_BRRF_Pos)                       /*!< 0x00000004 */
#define UART_BRRF_BRRF_3                          (0x8UL<<UART_BRRF_BRRF_Pos)                       /*!< 0x00000008 */


/********************************************************************************************************************/
/********************************* USART ****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for USART_SR register *********************************************/
#define USART_SR_PE_Pos                           (0U)
#define USART_SR_PE_Msk                           (0x1UL<<USART_SR_PE_Pos)                          /*!< 0x00000001 */
#define USART_SR_PE                               USART_SR_PE_Msk                                   /*!< Parity error */
#define USART_SR_FE_Pos                           (1U)
#define USART_SR_FE_Msk                           (0x1UL<<USART_SR_FE_Pos)                          /*!< 0x00000002 */
#define USART_SR_FE                               USART_SR_FE_Msk                                   /*!< Framing error */
#define USART_SR_NE_Pos                           (2U)
#define USART_SR_NE_Msk                           (0x1UL<<USART_SR_NE_Pos)                          /*!< 0x00000004 */
#define USART_SR_NE                               USART_SR_NE_Msk                                   /*!< Noise error flag */
#define USART_SR_ORE_Pos                          (3U)
#define USART_SR_ORE_Msk                          (0x1UL<<USART_SR_ORE_Pos)                         /*!< 0x00000008 */
#define USART_SR_ORE                              USART_SR_ORE_Msk                                  /*!< overrun error */
#define USART_SR_IDLE_Pos                         (4U)
#define USART_SR_IDLE_Msk                         (0x1UL<<USART_SR_IDLE_Pos)                        /*!< 0x00000010 */
#define USART_SR_IDLE                             USART_SR_IDLE_Msk                                 /*!< idle */
#define USART_SR_RXNE_Pos                         (5U)
#define USART_SR_RXNE_Msk                         (0x1UL<<USART_SR_RXNE_Pos)                        /*!< 0x00000020 */
#define USART_SR_RXNE                             USART_SR_RXNE_Msk                                 /*!< Read data register not empty */
#define USART_SR_TC_Pos                           (6U)
#define USART_SR_TC_Msk                           (0x1UL<<USART_SR_TC_Pos)                          /*!< 0x00000040 */
#define USART_SR_TC                               USART_SR_TC_Msk                                   /*!< Transmission complete */
#define USART_SR_TXE_Pos                          (7U)
#define USART_SR_TXE_Msk                          (0x1UL<<USART_SR_TXE_Pos)                         /*!< 0x00000080 */
#define USART_SR_TXE                              USART_SR_TXE_Msk                                  /*!< Transmit data register empty */
#define USART_SR_LBD_Pos                          (8U)
#define USART_SR_LBD_Msk                          (0x1UL<<USART_SR_LBD_Pos)                         /*!< 0x00000100 */
#define USART_SR_LBD                              USART_SR_LBD_Msk                                  /*!< LIN break detection flag */
#define USART_SR_CTS_Pos                          (9U)
#define USART_SR_CTS_Msk                          (0x1UL<<USART_SR_CTS_Pos)                         /*!< 0x00000200 */
#define USART_SR_CTS                              USART_SR_CTS_Msk                                  /*!< cts flag */
#define USART_SR_ABRF_Pos                         (10U)
#define USART_SR_ABRF_Msk                         (0x1UL<<USART_SR_ABRF_Pos)                        /*!< 0x00000400 */
#define USART_SR_ABRF                             USART_SR_ABRF_Msk                                 /*!< auto baud rate flag */
#define USART_SR_ABRE_Pos                         (11U)
#define USART_SR_ABRE_Msk                         (0x1UL<<USART_SR_ABRE_Pos)                        /*!< 0x00000800 */
#define USART_SR_ABRE                             USART_SR_ABRE_Msk                                 /*!< auto baud rateerror */
#define USART_SR_ABRQ_Pos                         (12U)
#define USART_SR_ABRQ_Msk                         (0x1UL<<USART_SR_ABRQ_Pos)                        /*!< 0x00001000 */
#define USART_SR_ABRQ                             USART_SR_ABRQ_Msk                                 /*!< auto baud rate req */
#define USART_SR_EOBF_Pos                         (18U)
#define USART_SR_EOBF_Msk                         (0x1UL<<USART_SR_EOBF_Pos)                        /*!< 0x00040000 */
#define USART_SR_EOBF                             USART_SR_EOBF_Msk                                 /*!< end of block flag */
#define USART_SR_RTOF_Pos                         (19U)
#define USART_SR_RTOF_Msk                         (0x1UL<<USART_SR_RTOF_Pos)                        /*!< 0x00080000 */
#define USART_SR_RTOF                             USART_SR_RTOF_Msk                                 /*!< rto flag */
#define USART_SR_TXLSEF_Pos                       (31U)
#define USART_SR_TXLSEF_Msk                       (0x1UL<<USART_SR_TXLSEF_Pos)                      /*!< 0x80000000 */
#define USART_SR_TXLSEF                           USART_SR_TXLSEF_Msk                               

/********************************* Bit definition for USART_DR register *********************************************/
#define USART_DR_DR_Pos                           (0U)
#define USART_DR_DR_Msk                           (0x1FFUL<<USART_DR_DR_Pos)                        /*!< 0x000001FF */
#define USART_DR_DR                               USART_DR_DR_Msk                                   /*!< dr */

/********************************* Bit definition for USART_BRR register ********************************************/
#define USART_BRR_DIV_FRACTION_Pos                (0U)
#define USART_BRR_DIV_FRACTION_Msk                (0xFUL<<USART_BRR_DIV_FRACTION_Pos)               /*!< 0x0000000F */
#define USART_BRR_DIV_FRACTION                    USART_BRR_DIV_FRACTION_Msk                        /*!< DIV_Fraction */
#define USART_BRR_DIV_FRACTION_0                  (0x1UL<<USART_BRR_DIV_FRACTION_Pos)               /*!< 0x00000001 */
#define USART_BRR_DIV_FRACTION_1                  (0x2UL<<USART_BRR_DIV_FRACTION_Pos)               /*!< 0x00000002 */
#define USART_BRR_DIV_FRACTION_2                  (0x4UL<<USART_BRR_DIV_FRACTION_Pos)               /*!< 0x00000004 */
#define USART_BRR_DIV_FRACTION_3                  (0x8UL<<USART_BRR_DIV_FRACTION_Pos)               /*!< 0x00000008 */
#define USART_BRR_DIV_MANTISSA_Pos                (4U)
#define USART_BRR_DIV_MANTISSA_Msk                (0xFFFUL<<USART_BRR_DIV_MANTISSA_Pos)             /*!< 0x0000FFF0 */
#define USART_BRR_DIV_MANTISSA                    USART_BRR_DIV_MANTISSA_Msk                        /*!< DIV_Mantissa */

/********************************* Bit definition for USART_CR1 register ********************************************/
#define USART_CR1_SBK_Pos                         (0U)
#define USART_CR1_SBK_Msk                         (0x1UL<<USART_CR1_SBK_Pos)                        /*!< 0x00000001 */
#define USART_CR1_SBK                             USART_CR1_SBK_Msk                                 /*!< Send break */
#define USART_CR1_RWU_Pos                         (1U)
#define USART_CR1_RWU_Msk                         (0x1UL<<USART_CR1_RWU_Pos)                        /*!< 0x00000002 */
#define USART_CR1_RWU                             USART_CR1_RWU_Msk                                 /*!< Receiver wakeup */
#define USART_CR1_RE_Pos                          (2U)
#define USART_CR1_RE_Msk                          (0x1UL<<USART_CR1_RE_Pos)                         /*!< 0x00000004 */
#define USART_CR1_RE                              USART_CR1_RE_Msk                                  /*!< Receiver enable */
#define USART_CR1_TE_Pos                          (3U)
#define USART_CR1_TE_Msk                          (0x1UL<<USART_CR1_TE_Pos)                         /*!< 0x00000008 */
#define USART_CR1_TE                              USART_CR1_TE_Msk                                  /*!< Transmitter enable */
#define USART_CR1_IDLEIE_Pos                      (4U)
#define USART_CR1_IDLEIE_Msk                      (0x1UL<<USART_CR1_IDLEIE_Pos)                     /*!< 0x00000010 */
#define USART_CR1_IDLEIE                          USART_CR1_IDLEIE_Msk                              /*!< IDLE interrupt enable */
#define USART_CR1_RXNEIE_Pos                      (5U)
#define USART_CR1_RXNEIE_Msk                      (0x1UL<<USART_CR1_RXNEIE_Pos)                     /*!< 0x00000020 */
#define USART_CR1_RXNEIE                          USART_CR1_RXNEIE_Msk                              /*!< RXNE interrupt enable */
#define USART_CR1_TCIE_Pos                        (6U)
#define USART_CR1_TCIE_Msk                        (0x1UL<<USART_CR1_TCIE_Pos)                       /*!< 0x00000040 */
#define USART_CR1_TCIE                            USART_CR1_TCIE_Msk                                /*!< send complete int enable */
#define USART_CR1_TXEIE_Pos                       (7U)
#define USART_CR1_TXEIE_Msk                       (0x1UL<<USART_CR1_TXEIE_Pos)                      /*!< 0x00000080 */
#define USART_CR1_TXEIE                           USART_CR1_TXEIE_Msk                               /*!< txe  */
#define USART_CR1_PEIE_Pos                        (8U)
#define USART_CR1_PEIE_Msk                        (0x1UL<<USART_CR1_PEIE_Pos)                       /*!< 0x00000100 */
#define USART_CR1_PEIE                            USART_CR1_PEIE_Msk                                /*!< PE interrupt enable */
#define USART_CR1_PS_Pos                          (9U)
#define USART_CR1_PS_Msk                          (0x1UL<<USART_CR1_PS_Pos)                         /*!< 0x00000200 */
#define USART_CR1_PS                              USART_CR1_PS_Msk                                  /*!< Parity selection */
#define USART_CR1_PCE_Pos                         (10U)
#define USART_CR1_PCE_Msk                         (0x1UL<<USART_CR1_PCE_Pos)                        /*!< 0x00000400 */
#define USART_CR1_PCE                             USART_CR1_PCE_Msk                                 /*!< Parity control enable */
#define USART_CR1_WAKE_Pos                        (11U)
#define USART_CR1_WAKE_Msk                        (0x1UL<<USART_CR1_WAKE_Pos)                       /*!< 0x00000800 */
#define USART_CR1_WAKE                            USART_CR1_WAKE_Msk                                /*!< Wakeup method */
#define USART_CR1_M_Pos                           (12U)
#define USART_CR1_M_Msk                           (0x1UL<<USART_CR1_M_Pos)                          /*!< 0x00001000 */
#define USART_CR1_M                               USART_CR1_M_Msk                                   /*!< word length */
#define USART_CR1_UE_Pos                          (13U)
#define USART_CR1_UE_Msk                          (0x1UL<<USART_CR1_UE_Pos)                         /*!< 0x00002000 */
#define USART_CR1_UE                              USART_CR1_UE_Msk                                  /*!< uesart enable  */
#define USART_CR1_MSB_Pos                         (15U)
#define USART_CR1_MSB_Msk                         (0x1UL<<USART_CR1_MSB_Pos)                        /*!< 0x00008000 */
#define USART_CR1_MSB                             USART_CR1_MSB_Msk                                 /*!< msb */
#define USART_CR1_SWAP_Pos                        (18U)
#define USART_CR1_SWAP_Msk                        (0x1UL<<USART_CR1_SWAP_Pos)                       /*!< 0x00040000 */
#define USART_CR1_SWAP                            USART_CR1_SWAP_Msk                                /*!< swap */
#define USART_CR1_DATAINV_Pos                     (27U)
#define USART_CR1_DATAINV_Msk                     (0x1UL<<USART_CR1_DATAINV_Pos)                    /*!< 0x08000000 */
#define USART_CR1_DATAINV                         USART_CR1_DATAINV_Msk                             /*!< data inverse */

/********************************* Bit definition for USART_CR2 register ********************************************/
#define USART_CR2_LBDL_Pos                        (5U)
#define USART_CR2_LBDL_Msk                        (0x1UL<<USART_CR2_LBDL_Pos)                       /*!< 0x00000020 */
#define USART_CR2_LBDL                            USART_CR2_LBDL_Msk                                /*!< LIN break detection length */
#define USART_CR2_LBDIE_Pos                       (6U)
#define USART_CR2_LBDIE_Msk                       (0x1UL<<USART_CR2_LBDIE_Pos)                      /*!< 0x00000040 */
#define USART_CR2_LBDIE                           USART_CR2_LBDIE_Msk                               /*!< lin break enable */
#define USART_CR2_LBCL_Pos                        (8U)
#define USART_CR2_LBCL_Msk                        (0x1UL<<USART_CR2_LBCL_Pos)                       /*!< 0x00000100 */
#define USART_CR2_LBCL                            USART_CR2_LBCL_Msk                                /*!< lbcl  */
#define USART_CR2_CPHA_Pos                        (9U)
#define USART_CR2_CPHA_Msk                        (0x1UL<<USART_CR2_CPHA_Pos)                       /*!< 0x00000200 */
#define USART_CR2_CPHA                            USART_CR2_CPHA_Msk                                /*!< cpha */
#define USART_CR2_CPOL_Pos                        (10U)
#define USART_CR2_CPOL_Msk                        (0x1UL<<USART_CR2_CPOL_Pos)                       /*!< 0x00000400 */
#define USART_CR2_CPOL                            USART_CR2_CPOL_Msk                                /*!< cpol */
#define USART_CR2_CLKEN_Pos                       (11U)
#define USART_CR2_CLKEN_Msk                       (0x1UL<<USART_CR2_CLKEN_Pos)                      /*!< 0x00000800 */
#define USART_CR2_CLKEN                           USART_CR2_CLKEN_Msk                               /*!< clk en */
#define USART_CR2_STOP_Pos                        (12U)
#define USART_CR2_STOP_Msk                        (0x3UL<<USART_CR2_STOP_Pos)                       /*!< 0x00003000 */
#define USART_CR2_STOP                            USART_CR2_STOP_Msk                                /*!< stop bit */
#define USART_CR2_STOP_0                          (0x1UL<<USART_CR2_STOP_Pos)                       /*!< 0x00001000 */
#define USART_CR2_STOP_1                          (0x2UL<<USART_CR2_STOP_Pos)                       /*!< 0x00002000 */
#define USART_CR2_LINEN_Pos                       (14U)
#define USART_CR2_LINEN_Msk                       (0x1UL<<USART_CR2_LINEN_Pos)                      /*!< 0x00004000 */
#define USART_CR2_LINEN                           USART_CR2_LINEN_Msk                               /*!< lin enable */
#define USART_CR2_RTOEN_Pos                       (25U)
#define USART_CR2_RTOEN_Msk                       (0x1UL<<USART_CR2_RTOEN_Pos)                      /*!< 0x02000000 */
#define USART_CR2_RTOEN                           USART_CR2_RTOEN_Msk                               /*!< Receiver timeout enable */

/********************************* Bit definition for USART_CR3 register ********************************************/
#define USART_CR3_EIE_Pos                         (0U)
#define USART_CR3_EIE_Msk                         (0x1UL<<USART_CR3_EIE_Pos)                        /*!< 0x00000001 */
#define USART_CR3_EIE                             USART_CR3_EIE_Msk                                 /*!< error int enable */
#define USART_CR3_IREN_Pos                        (1U)
#define USART_CR3_IREN_Msk                        (0x1UL<<USART_CR3_IREN_Pos)                       /*!< 0x00000002 */
#define USART_CR3_IREN                            USART_CR3_IREN_Msk                                /*!< iren enable */
#define USART_CR3_IRLP_Pos                        (2U)
#define USART_CR3_IRLP_Msk                        (0x1UL<<USART_CR3_IRLP_Pos)                       /*!< 0x00000004 */
#define USART_CR3_IRLP                            USART_CR3_IRLP_Msk                                /*!< idra lowpower */
#define USART_CR3_HDSEL_Pos                       (3U)
#define USART_CR3_HDSEL_Msk                       (0x1UL<<USART_CR3_HDSEL_Pos)                      /*!< 0x00000008 */
#define USART_CR3_HDSEL                           USART_CR3_HDSEL_Msk                               /*!< hdsel  */
#define USART_CR3_NACK_Pos                        (4U)
#define USART_CR3_NACK_Msk                        (0x1UL<<USART_CR3_NACK_Pos)                       /*!< 0x00000010 */
#define USART_CR3_NACK                            USART_CR3_NACK_Msk                                /*!< scen nack enable */
#define USART_CR3_SCEN_Pos                        (5U)
#define USART_CR3_SCEN_Msk                        (0x1UL<<USART_CR3_SCEN_Pos)                       /*!< 0x00000020 */
#define USART_CR3_SCEN                            USART_CR3_SCEN_Msk                                /*!< scen enable */
#define USART_CR3_DMAR_Pos                        (6U)
#define USART_CR3_DMAR_Msk                        (0x1UL<<USART_CR3_DMAR_Pos)                       /*!< 0x00000040 */
#define USART_CR3_DMAR                            USART_CR3_DMAR_Msk                                /*!< dma rx enable */
#define USART_CR3_DMAT_Pos                        (7U)
#define USART_CR3_DMAT_Msk                        (0x1UL<<USART_CR3_DMAT_Pos)                       /*!< 0x00000080 */
#define USART_CR3_DMAT                            USART_CR3_DMAT_Msk                                /*!< dma tx enable */
#define USART_CR3_RTSE_Pos                        (8U)
#define USART_CR3_RTSE_Msk                        (0x1UL<<USART_CR3_RTSE_Pos)                       /*!< 0x00000100 */
#define USART_CR3_RTSE                            USART_CR3_RTSE_Msk                                /*!< rts enable */
#define USART_CR3_CTSE_Pos                        (9U)
#define USART_CR3_CTSE_Msk                        (0x1UL<<USART_CR3_CTSE_Pos)                       /*!< 0x00000200 */
#define USART_CR3_CTSE                            USART_CR3_CTSE_Msk                                /*!< cts enable */
#define USART_CR3_CTSIE_Pos                       (10U)
#define USART_CR3_CTSIE_Msk                       (0x1UL<<USART_CR3_CTSIE_Pos)                      /*!< 0x00000400 */
#define USART_CR3_CTSIE                           USART_CR3_CTSIE_Msk                               /*!< cts int enable */
#define USART_CR3_OVER8_Pos                       (11U)
#define USART_CR3_OVER8_Msk                       (0x1UL<<USART_CR3_OVER8_Pos)                      /*!< 0x00000800 */
#define USART_CR3_OVER8                           USART_CR3_OVER8_Msk                               /*!< oversampling  */
#define USART_CR3_ABREN_Pos                       (12U)
#define USART_CR3_ABREN_Msk                       (0x1UL<<USART_CR3_ABREN_Pos)                      /*!< 0x00001000 */
#define USART_CR3_ABREN                           USART_CR3_ABREN_Msk                               /*!< autobaud enable */
#define USART_CR3_ABRMOD_Pos                      (13U)
#define USART_CR3_ABRMOD_Msk                      (0x3UL<<USART_CR3_ABRMOD_Pos)                     /*!< 0x00006000 */
#define USART_CR3_ABRMOD                          USART_CR3_ABRMOD_Msk                              /*!< auto baud mod */
#define USART_CR3_ABRMOD_0                        (0x1UL<<USART_CR3_ABRMOD_Pos)                     /*!< 0x00002000 */
#define USART_CR3_ABRMOD_1                        (0x2UL<<USART_CR3_ABRMOD_Pos)                     /*!< 0x00004000 */
#define USART_CR3_RTOIE_Pos                       (29U)
#define USART_CR3_RTOIE_Msk                       (0x1UL<<USART_CR3_RTOIE_Pos)                      /*!< 0x20000000 */
#define USART_CR3_RTOIE                           USART_CR3_RTOIE_Msk                               /*!< rto int enable */
#define USART_CR3_EOBIE_Pos                       (30U)
#define USART_CR3_EOBIE_Msk                       (0x1UL<<USART_CR3_EOBIE_Pos)                      /*!< 0x40000000 */
#define USART_CR3_EOBIE                           USART_CR3_EOBIE_Msk                               /*!< eob int enable */
#define USART_CR3_TXLSEIE_Pos                     (31U)
#define USART_CR3_TXLSEIE_Msk                     (0x1UL<<USART_CR3_TXLSEIE_Pos)                    /*!< 0x80000000 */
#define USART_CR3_TXLSEIE                         USART_CR3_TXLSEIE_Msk                             /*!< tx register empty int enable */

/********************************* Bit definition for USART_GTPR register *******************************************/
#define USART_GTPR_PSC_Pos                        (0U)
#define USART_GTPR_PSC_Msk                        (0xFFUL<<USART_GTPR_PSC_Pos)                      /*!< 0x000000FF */
#define USART_GTPR_PSC                            USART_GTPR_PSC_Msk                                /*!< Prescaler value */
#define USART_GTPR_GT_Pos                         (8U)
#define USART_GTPR_GT_Msk                         (0xFFUL<<USART_GTPR_GT_Pos)                       /*!< 0x0000FF00 */
#define USART_GTPR_GT                             USART_GTPR_GT_Msk                                 /*!< Guard time value */

/********************************* Bit definition for USART_RTOR register *******************************************/
#define USART_RTOR_RTO_Pos                        (0U)
#define USART_RTOR_RTO_Msk                        (0xFFFFFFUL<<USART_RTOR_RTO_Pos)                  /*!< 0x00FFFFFF */
#define USART_RTOR_RTO                            USART_RTOR_RTO_Msk                                /*!< receive timeout */
#define USART_RTOR_BLEN_Pos                       (24U)
#define USART_RTOR_BLEN_Msk                       (0xFFUL<<USART_RTOR_BLEN_Pos)                     /*!< 0xFF000000 */
#define USART_RTOR_BLEN                           USART_RTOR_BLEN_Msk                               /*!< blocken length */

/********************************* Bit definition for USART_DEBUGSELR register **************************************/
#define USART_DEBUGSELR_DEBUG_SEL_Pos             (0U)
#define USART_DEBUGSELR_DEBUG_SEL_Msk             (0x3UL<<USART_DEBUGSELR_DEBUG_SEL_Pos)            /*!< 0x00000003 */
#define USART_DEBUGSELR_DEBUG_SEL                 USART_DEBUGSELR_DEBUG_SEL_Msk
#define USART_DEBUGSELR_DEBUG_SEL_0               (0x1UL<<USART_DEBUGSELR_DEBUG_SEL_Pos)            /*!< 0x00000001 */
#define USART_DEBUGSELR_DEBUG_SEL_1               (0x2UL<<USART_DEBUGSELR_DEBUG_SEL_Pos)            /*!< 0x00000002 */

/********************************************************************************************************************/
/********************************* WWDG *****************************************************************************/
/********************************************************************************************************************/

/********************************* Bit definition for WWDG_CR register **********************************************/
#define WWDG_CR_T_Pos                             (0U)
#define WWDG_CR_T_Msk                             (0x7FUL<<WWDG_CR_T_Pos)                           /*!< 0x0000007F */
#define WWDG_CR_T                                 WWDG_CR_T_Msk                                     /*!< wwdg counter bit6 */
#define WWDG_CR_WDGA_Pos                          (7U)
#define WWDG_CR_WDGA_Msk                          (0x1UL<<WWDG_CR_WDGA_Pos)                         /*!< 0x00000080 */
#define WWDG_CR_WDGA                              WWDG_CR_WDGA_Msk                                  /*!< WWDG Activation */

/********************************* Bit definition for WWDG_CFR register *********************************************/
#define WWDG_CFR_W_Pos                            (0U)
#define WWDG_CFR_W_Msk                            (0x7FUL<<WWDG_CFR_W_Pos)                          /*!< 0x0000007F */
#define WWDG_CFR_W                                WWDG_CFR_W_Msk                                    /*!< window value bit 6 */
#define WWDG_CFR_WDGTB_Pos                        (7U)
#define WWDG_CFR_WDGTB_Msk                        (0x3UL<<WWDG_CFR_WDGTB_Pos)                       /*!< 0x00000180 */
#define WWDG_CFR_WDGTB                            WWDG_CFR_WDGTB_Msk                                /*!< time base bit 1 */
#define WWDG_CFR_WDGTB_0                          (0x1UL<<WWDG_CFR_WDGTB_Pos)                       /*!< 0x00000080 */
#define WWDG_CFR_WDGTB_1                          (0x2UL<<WWDG_CFR_WDGTB_Pos)                       /*!< 0x00000100 */
#define WWDG_CFR_EWI_Pos                          (9U)
#define WWDG_CFR_EWI_Msk                          (0x1UL<<WWDG_CFR_EWI_Pos)                         /*!< 0x00000200 */
#define WWDG_CFR_EWI                              WWDG_CFR_EWI_Msk                                  /*!< Early wakeup interrupt */

/********************************* Bit definition for WWDG_SR register **********************************************/
#define WWDG_SR_EWIF_Pos                          (0U)
#define WWDG_SR_EWIF_Msk                          (0x1UL<<WWDG_SR_EWIF_Pos)                         /*!< 0x00000001 */
#define WWDG_SR_EWIF                              WWDG_SR_EWIF_Msk                                  /*!< Early wakeup interrupt flag */

/** @addtogroup Exported_macros
  * @{
  */

/******************************* ADC Instances ********************************/
#define IS_ADC_ALL_INSTANCE(INSTANCE) ((INSTANCE) == ADC1)

#define IS_ADC_COMMON_INSTANCE(INSTANCE) ((INSTANCE) == ADC)

/******************************* CRC Instances ********************************/
#define IS_CRC_ALL_INSTANCE(INSTANCE) ((INSTANCE) == CRC)


/******************************** DMA Instances *******************************/
#define IS_DMA_ALL_INSTANCE(INSTANCE) (((INSTANCE) == DMA1_Channel1) || \
                                       ((INSTANCE) == DMA1_Channel2) || \
                                       ((INSTANCE) == DMA1_Channel3))

/******************************* GPIO Instances *******************************/
#define IS_GPIO_ALL_INSTANCE(INSTANCE) (((INSTANCE) == GPIOA) || \
                                        ((INSTANCE) == GPIOB) || \
                                        ((INSTANCE) == GPIOF))

/********************** GPIO Alternate Function Instances *********************/
#define IS_GPIO_AF_INSTANCE(INSTANCE)   IS_GPIO_ALL_INSTANCE(INSTANCE)

/**************************** GPIO Lock Instances *****************************/
#define IS_GPIO_LOCK_INSTANCE(INSTANCE) IS_GPIO_ALL_INSTANCE(INSTANCE)

/****************************** I2C Instances ********************************/
#define IS_I2C_ALL_INSTANCE(INSTANCE)  (((INSTANCE) == I2C1) || \
                                        ((INSTANCE) == I2C2))

/****************************** RTC Instances *********************************/
#define IS_RTC_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == RTC)

/****************************** WAKEUP_FROMSTOP Instances *******************************/
#define IS_I2C_WAKEUP_FROMSTOP_INSTANCE(INSTANCE) (((INSTANCE) == I2C))

/******************************** SPI Instances *******************************/
#define IS_SPI_ALL_INSTANCE(INSTANCE) (((INSTANCE) == SPI1) || \
                                       ((INSTANCE) == SPI2))

#define IS_I2S_ALL_INSTANCE(INSTANCE) (((INSTANCE) == SPI1))                                       

/****************** LPTIM Instances : All supported instances *****************/
#define IS_LPTIM_INSTANCE(INSTANCE)     ((INSTANCE) == LPTIM)

/****************** LPTIM Instances : All supported instances *****************/
#define IS_LPTIM_ENCODER_INTERFACE_INSTANCE(INSTANCE) ((INSTANCE) == LPTIM)

/****************** TIM Instances : All supported instances *******************/
#define IS_TIM_INSTANCE(INSTANCE)       (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2)   || \
                                         ((INSTANCE) == TIM16)  || \
                                         ((INSTANCE) == TIM17))

/****************** TIM Instances : supporting the break function *************/
#define IS_TIM_BREAK_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1)    || \
                                            ((INSTANCE) == TIM16)   || \
                                            ((INSTANCE) == TIM17))

/************** TIM Instances : supporting Break source selection *************/
#define IS_TIM_BREAKSOURCE_INSTANCE(INSTANCE) (((INSTANCE) == TIM1)   || \
                                               ((INSTANCE) == TIM16)  || \
                                               ((INSTANCE) == TIM17))

/****************** TIM Instances : supporting 2 break inputs *****************/
#define IS_TIM_BKIN2_INSTANCE(INSTANCE)    ((INSTANCE) == TIM1)

/************* TIM Instances : at least 1 capture/compare channel *************/
#define IS_TIM_CC1_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2)   || \
                                         ((INSTANCE) == TIM16)  || \
                                         ((INSTANCE) == TIM17))

/************ TIM Instances : at least 2 capture/compare channels *************/
#define IS_TIM_CC2_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2))

/************ TIM Instances : at least 3 capture/compare channels *************/
#define IS_TIM_CC3_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2))

/************ TIM Instances : at least 4 capture/compare channels *************/
#define IS_TIM_CC4_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2))

/****************** TIM Instances : at least 5 capture/compare channels *******/
#define IS_TIM_CC5_INSTANCE(INSTANCE)   ((INSTANCE) == TIM1)

/****************** TIM Instances : at least 6 capture/compare channels *******/
#define IS_TIM_CC6_INSTANCE(INSTANCE)   ((INSTANCE) == TIM1)

/************ TIM Instances : DMA requests generation (TIM_DIER.COMDE) *******/
#define IS_TIM_CCDMA_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1)   || \
                                            ((INSTANCE) == TIM16)  || \
                                            ((INSTANCE) == TIM17))

/****************** TIM Instances : DMA requests generation (TIM_DIER.UDE) ***/
#define IS_TIM_DMA_INSTANCE(INSTANCE)      (((INSTANCE) == TIM1)   || \
                                            ((INSTANCE) == TIM2)   || \
                                            ((INSTANCE) == TIM16)  || \
                                            ((INSTANCE) == TIM17))

/************ TIM Instances : DMA requests generation (TIM_DIER.CCxDE) *******/
#define IS_TIM_DMA_CC_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)   || \
                                            ((INSTANCE) == TIM2)   || \
                                            ((INSTANCE) == TIM16)  || \
                                            ((INSTANCE) == TIM17))

/******************** TIM Instances : DMA burst feature ***********************/
#define IS_TIM_DMABURST_INSTANCE(INSTANCE) (((INSTANCE) == TIM1)   || \
                                            ((INSTANCE) == TIM2)   || \
                                            ((INSTANCE) == TIM16)  || \
                                            ((INSTANCE) == TIM17))

/******************* TIM Instances : output(s) available **********************/
#define IS_TIM_CCX_INSTANCE(INSTANCE, CHANNEL) \
    ((((INSTANCE) == TIM1) &&                  \
     (((CHANNEL) == TIM_CHANNEL_1) ||          \
      ((CHANNEL) == TIM_CHANNEL_2) ||          \
      ((CHANNEL) == TIM_CHANNEL_3) ||          \
      ((CHANNEL) == TIM_CHANNEL_4)))           \
     ||                                        \
     (((INSTANCE) == TIM2) &&                  \
     (((CHANNEL) == TIM_CHANNEL_1) ||          \
      ((CHANNEL) == TIM_CHANNEL_2) ||          \
      ((CHANNEL) == TIM_CHANNEL_3) ||          \
      ((CHANNEL) == TIM_CHANNEL_4)))           \
     ||                                        \
     (((INSTANCE) == TIM16) &&                 \
     (((CHANNEL) == TIM_CHANNEL_1)))           \
     ||                                        \
     (((INSTANCE) == TIM17) &&                 \
      (((CHANNEL) == TIM_CHANNEL_1))))

/****************** TIM Instances : supporting complementary output(s) ********/
#define IS_TIM_CCXN_INSTANCE(INSTANCE, CHANNEL) \
   ((((INSTANCE) == TIM1) &&                    \
     (((CHANNEL) == TIM_CHANNEL_1) ||           \
      ((CHANNEL) == TIM_CHANNEL_2) ||           \
      ((CHANNEL) == TIM_CHANNEL_3)))            \
    ||                                          \
    (((INSTANCE) == TIM16) &&                   \
     ((CHANNEL) == TIM_CHANNEL_1))              \
    ||                                          \
    (((INSTANCE) == TIM17) &&                   \
     ((CHANNEL) == TIM_CHANNEL_1)))

/****************** TIM Instances : supporting clock division *****************/
#define IS_TIM_CLOCK_DIVISION_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)    || \
                                                    ((INSTANCE) == TIM2)    || \
                                                    ((INSTANCE) == TIM16)   || \
                                                    ((INSTANCE) == TIM17))

/****** TIM Instances : supporting external clock mode 1 for ETRF input *******/
#define IS_TIM_CLOCKSOURCE_ETRMODE1_INSTANCE(INSTANCE) (((INSTANCE) == TIM1) || \
                                                        ((INSTANCE) == TIM2))

/****** TIM Instances : supporting external clock mode 2 for ETRF input *******/
#define IS_TIM_CLOCKSOURCE_ETRMODE2_INSTANCE(INSTANCE) (((INSTANCE) == TIM1) || \
                                                        ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting external clock mode 1 for TIX inputs*/
#define IS_TIM_CLOCKSOURCE_TIX_INSTANCE(INSTANCE)      (((INSTANCE) == TIM1) || \
                                                        ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting internal trigger inputs(ITRX) *******/
#define IS_TIM_CLOCKSOURCE_ITRX_INSTANCE(INSTANCE)     (((INSTANCE) == TIM1) || \
                                                        ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting combined 3-phase PWM mode ******/
#define IS_TIM_COMBINED3PHASEPWM_INSTANCE(INSTANCE)    ((INSTANCE) == TIM1)

/****************** TIM Instances : supporting commutation event generation ***/
#define IS_TIM_COMMUTATION_EVENT_INSTANCE(INSTANCE) (((INSTANCE) == TIM1)   || \
                                                     ((INSTANCE) == TIM16)  || \
                                                     ((INSTANCE) == TIM17))

/****************** TIM Instances : supporting counting mode selection ********/
#define IS_TIM_COUNTER_MODE_SELECT_INSTANCE(INSTANCE)  (((INSTANCE) == TIM1) || \
                                                        ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting encoder interface **************/
#define IS_TIM_ENCODER_INTERFACE_INSTANCE(INSTANCE)  (((INSTANCE) == TIM1)  || \
                                                      ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting Hall sensor interface **********/
#define IS_TIM_HALL_SENSOR_INTERFACE_INSTANCE(INSTANCE) (((INSTANCE) == TIM1)   || \
                                                         ((INSTANCE) == TIM2))

/**************** TIM Instances : external trigger input available ************/
#define IS_TIM_ETR_INSTANCE(INSTANCE)      (((INSTANCE) == TIM1)  || \
                                            ((INSTANCE) == TIM2))

/************* TIM Instances : supporting ETR source selection ***************/
#define IS_TIM_ETRSEL_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1)  || \
                                             ((INSTANCE) == TIM2))

/****** TIM Instances : Master mode available (TIM_CR2.MMS available )********/
#define IS_TIM_MASTER_INSTANCE(INSTANCE)   (((INSTANCE) == TIM1)  || \
                                            ((INSTANCE) == TIM2))

/*********** TIM Instances : Slave mode available (TIM_SMCR available )*******/
#define IS_TIM_SLAVE_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1)  || \
                                            ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting OCxREF clear *******************/
#define IS_TIM_OCXREF_CLEAR_INSTANCE(INSTANCE)        (((INSTANCE) == TIM1) || \
                                                       ((INSTANCE) == TIM2))

/****************** TIM Instances : remapping capability **********************/
#define IS_TIM_REMAP_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1)  || \
                                            ((INSTANCE) == TIM2))

/****************** TIM Instances : supporting repetition counter *************/
#define IS_TIM_REPETITION_COUNTER_INSTANCE(INSTANCE)  (((INSTANCE) == TIM1)  || \
                                                       ((INSTANCE) == TIM16) || \
                                                       ((INSTANCE) == TIM17))

/****************** TIM Instances : supporting synchronization ****************/
#define IS_TIM_SYNCHRO_INSTANCE(INSTANCE)  IS_TIM_MASTER_INSTANCE(INSTANCE)

/****************** TIM Instances : supporting ADC triggering through TRGO2 ***/
#define IS_TIM_TRGO2_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1))

/******************* TIM Instances : Timer input XOR function *****************/
#define IS_TIM_XOR_INSTANCE(INSTANCE)      (((INSTANCE) == TIM1)   || \
                                            ((INSTANCE) == TIM2))

/******************* TIM Instances : Timer input selection ********************/
#define IS_TIM_TISEL_INSTANCE(INSTANCE) (((INSTANCE) == TIM1)   || \
                                         ((INSTANCE) == TIM2)   || \
                                         ((INSTANCE) == TIM16)  || \
                                         ((INSTANCE) == TIM17))

/************ TIM Instances : Advanced timers  ********************************/
#define IS_TIM_ADVANCED_INSTANCE(INSTANCE)    (((INSTANCE) == TIM1))

/******************** UART Instances : Asynchronous mode **********************/
#define IS_UART_INSTANCE(INSTANCE) (((INSTANCE) == USART1) || \
                                    ((INSTANCE) == USART2) || \
                                    ((INSTANCE) == USART3))

/******************** USART Instances : Synchronous mode **********************/
#define IS_USART_INSTANCE(INSTANCE) (((INSTANCE) == USART1) || \
                                     ((INSTANCE) == USART2) || \
                                     ((INSTANCE) == USART3))

/****************** UART Instances : Hardware Flow control ********************/
#define IS_UART_HWFLOW_INSTANCE(INSTANCE) (((INSTANCE) == USART1) || \
                                           ((INSTANCE) == USART2) || \
                                           ((INSTANCE) == USART3))

/****************** UART Instances : Auto Baud Rate detection ****************/
#define IS_USART_AUTOBAUDRATE_DETECTION_INSTANCE(INSTANCE) (((INSTANCE) == USART1) || \
                                                            ((INSTANCE) == USART2) || \
                                                            ((INSTANCE) == USART3))

/******************** UART Instances : Half-Duplex mode **********************/
#define IS_UART_HALFDUPLEX_INSTANCE(INSTANCE)   (((INSTANCE) == USART1) || \
                                                 ((INSTANCE) == USART2) || \
                                                 ((INSTANCE) == USART3))

/******************** UART Instances : LIN mode **********************/
#define IS_UART_LIN_INSTANCE(INSTANCE)   ((INSTANCE) == USART1)

/******************** UART Instances : Wake-up from Stop mode **********************/
#define IS_UART_WAKEUP_FROMSTOP_INSTANCE(INSTANCE)   (((INSTANCE) == USART1) || \
                                                      ((INSTANCE) == USART2) || \
                                                      ((INSTANCE) == USART3))

/****************** UART Instances : Driver Enable *****************/
#define IS_UART_DRIVER_ENABLE_INSTANCE(INSTANCE)     (((INSTANCE) == USART1) || \
                                                      ((INSTANCE) == USART2) || \
                                                      ((INSTANCE) == USART3))

/****************** UART Instances : SPI Slave selection mode ***************/
#define IS_UART_SPI_SLAVE_INSTANCE(INSTANCE) (((INSTANCE) == USART1) || \
                                              ((INSTANCE) == USART2) || \
                                              ((INSTANCE) == USART3))

/****************** UART Instances : Driver Enable *****************/
#define IS_UART_FIFO_INSTANCE(INSTANCE)     (((INSTANCE) == USART1) || \
                                             ((INSTANCE) == USART2) || \
                                             ((INSTANCE) == USART3))
/****************************** IWDG Instances ********************************/
#define IS_IWDG_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == IWDG)

/****************************** WWDG Instances ********************************/
#define IS_WWDG_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == WWDG)

/****************************** LCD Instances ********************************/
#define IS_LCD_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == LCD)

/****************************** COMP Instances ********************************/
#define IS_COMP_ALL_INSTANCE(INSTANCE) (((INSTANCE) == COMP1) || \
                                        ((INSTANCE) == COMP2))

/****************************** HDIV Instances ********************************/
#define IS_HDIV_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == HDIV)

/****************************** CORDIC Instances ********************************/
#define IS_CORDIC_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == CORDIC)

/****************************** OPA Instances ********************************/
#define IS_OPA_ALL_INSTANCE(INSTANCE)  ((INSTANCE) == OPA)

/**
  * @}
  */

/**
 * @}
 */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __PY32T031X8_H */

/**
  * @}
  */

/**
* @}
*/

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/

